# clock.tcl --
#
# This file contains the code which handles the clock display of message time/date
#
#  Postilion software and its included text and images
#  Copyright (C) 1998 Nic Bernstein
#
#  The full text of the legal notices is contained in the file called
#  COPYING, included with this distribution.
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
# 
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

# Set these two values as you need
# set option(image_path) /home/nic/post/current/postilion/images
# set option(image_type) xpm

# ClkInitImages --
#
# Initialize all of the images
#
# Arguments:
proc ClkInitImages {} {
    global option

    # Create the month images
    foreach month {jan feb mar apr may jun jul aug sep oct nov dec} {
	image create photo $month -file $option(image_path)/$month.gif
    }
    # Create the day images
    foreach day {mon tue wed thu fri sat sun} {
	image create photo $day -file $option(image_path)/$day.gif
    }
    # Create the number images
    foreach num {0 1 2 3 4 5 6 7 8 9} {
	image create photo led$num -file $option(image_path)/led$num.gif
	image create photo date$num -file $option(image_path)/date$num.gif
    }
    # Create the misc. images
    image create photo ledcol -file $option(image_path)/ledcol.gif
    image create photo ledam -file $option(image_path)/ledam.gif
    image create photo ledpm -file $option(image_path)/ledpm.gif
    image create photo mask -file $option(image_path)/mask.gif
}

# ClkInitFrame --
#
# Initialize the frame, loading the mask and setting the coordinates registers
#
# Arguments:
# frame - the frame to put the clock into
# type - 24 or 12, which format to use
proc ClkInitFrame {frame type} {
    # Create the canvas, and save the tag
    set canvas [canvas $frame.clock -width 55 -height 57 -takefocus no]

    # We do this to be able to attach values to this canvas (in case there's
    # more than one)
    upvar #0 $canvas clkprt

    # Make the background
    set clkprt(clock) [$canvas create image 0 0 -image mask -anchor nw]

    # Make the space for the time display
    if ![string compare $type 12] {
	# This is for 12 hour mode, with am/pm indicator
	set clkprt(led1) [$canvas create image 1 2 -anchor nw]
	set clkprt(led2) [$canvas create image 10 2 -anchor nw]
	set clkprt(ledcol) [$canvas create image 19 2 -anchor nw]
	set clkprt(led3) [$canvas create image 22 2 -anchor nw]
	set clkprt(led4) [$canvas create image 31 2 -anchor nw]
	set clkprt(ampm) [$canvas create image 54 2 -anchor ne]
    } else {
	# This is for 24 hour mode
	set clkprt(led1) [$canvas create image 4 2 -anchor nw]
	set clkprt(led2) [$canvas create image 13 2 -anchor nw]
	set clkprt(ledcol) [$canvas create image 22 2 -anchor nw]
	set clkprt(led3) [$canvas create image 25 2 -anchor nw]
	set clkprt(led4) [$canvas create image 34 2 -anchor nw]
    }
    # Make the space for the day of the week
    set clkprt(day) [$canvas create image 16 21 -anchor nw]
    # Make the space for the date digits
    set clkprt(date1) [$canvas create image 18 30 -anchor nw]
    set clkprt(date2) [$canvas create image 27 30 -anchor nw]
    # Make the space for the month display
    set clkprt(mon) [$canvas create image 16 45 -anchor nw]
    # Save our type
    set clkprt(type) $type

    # Pack it
    pack $canvas

    # Return a handle to this widget
    return $canvas
}

# ClkSetTime --
#
# Display the time specified by time in the clock widget specified by canvas
#
# Arguments:
# shtime - the time/date to show
# canvas - the clock widget to display in
proc ClkSetTime {shtime canvas} {
    upvar #0 $canvas clkprt

    # We've been passed seconds since the epoch, in GMT
    if {$shtime >= 1} {
	# Convert it into our timezone (this is tortured)
	set otime [clock format $shtime -format {%a %b %d %H:%M:%S %Y}]
	set mtime [clock scan $otime -gmt true]
	set ftime [clock format $mtime -format {%b %a %p %I %H %M %d}]
    } else {
	set ftime [list jan mon am 00 00 00 00]
    }
    # Figure out which images to use for each field
    set mon [string tolower [lindex $ftime 0]]
    set day [string tolower [lindex $ftime 1]]
    if ![string compare $clkprt(type) 12] {
	set ampm [string tolower [lindex $ftime 2]]
        set led1 [string index [lindex $ftime 3] 0]
        set led2 [string index [lindex $ftime 3] 1]
    } else {
        set led1 [string index [lindex $ftime 4] 0]
        set led2 [string index [lindex $ftime 4] 1]
    }
    set led3 [string index [lindex $ftime 5] 0]
    set led4 [string index [lindex $ftime 5] 1]
    set date1 [string index [lindex $ftime 6] 0]
    set date2 [string index [lindex $ftime 6] 1]

    # Set all of the cells to use the correct images
    $canvas itemconfigure $clkprt(mon) -image $mon
    $canvas itemconfigure $clkprt(day) -image $day
    $canvas itemconfigure $clkprt(led1) -image led$led1
    $canvas itemconfigure $clkprt(led2) -image led$led2
    $canvas itemconfigure $clkprt(led3) -image led$led3
    $canvas itemconfigure $clkprt(led4) -image led$led4
    if ![string compare $clkprt(type) 12] {
	$canvas itemconfigure $clkprt(ampm) -image led$ampm
    }
    $canvas itemconfigure $clkprt(date1) -image date$date1
    $canvas itemconfigure $clkprt(date2) -image date$date2
    $canvas itemconfigure $clkprt(ledcol) -image ledcol
}

# These are what a program should call to get the job done:

ClkInitImages

# frame .test
# pack .test
# set mine [ClkInitFrame .test 12]
# ClkSetTime {5 Jan 1997 12:34} $mine 