/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  access.c:  access memory using virtual (seg:offset) ptr
 *                
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"

static void write_virtual_checks(vm_t *vm, unsigned sreg, Bit32u off, unsigned size);
static void read_virtual_checks(vm_t *vm, unsigned sreg, Bit32u off, unsigned size);




  void
read_virtual(vm_t *vm, unsigned sreg, Bit32u off, void *data, unsigned size)
{
  Bit32u laddr;

  cache_sreg(vm, sreg);
  laddr = vm->guest_cpu.desc_cache[sreg].base + off;

#if 1
  /* If linear access is not within one PDE slot away from monitor PDE,
   * then we can use a native seg:offset access.  Could use a check for
   * only the monitor PDE slot, but that would require two compares.
   */
  if ( (laddr & 0xffe00000) != (vm->mon_pde_mask | 0x00200000) ) {
    if (size==4) {
      if ( __readDWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==1) {
      if ( __readByteNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==2) {
      if ( __readWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    }
#endif

  read_virtual_checks(vm, sreg, off, size);
  access_linear(vm, laddr, size, G_CPL(vm)==3, OP_READ, data);
}

  void
write_virtual(vm_t *vm, unsigned sreg, Bit32u off, void *data, unsigned size)
{
  Bit32u laddr;

  cache_sreg(vm, sreg);
  laddr = vm->guest_cpu.desc_cache[sreg].base + off;

#if 1
  if ( (laddr & 0xffe00000) != (vm->mon_pde_mask | 0x00200000) ) {
    if (size==4) {
      if ( __writeDWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==1) {
      if ( __writeByteNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==2) {
      if ( __writeWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    }
#endif

  write_virtual_checks(vm, sreg, off, size);
  access_linear(vm, laddr, size, G_CPL(vm)==3, OP_WRITE, data);
}

  void
read_RMW_virtual(vm_t *vm, unsigned sreg, Bit32u off, void *data,
                 unsigned size)
{
  Bit32u laddr;

  cache_sreg(vm, sreg);
  laddr = vm->guest_cpu.desc_cache[sreg].base + off;

  /* Store information not passed by the write phase of RMW accesses. */
  vm->rmwArea.sreg   = sreg;
  vm->rmwArea.offset = off;
  vm->rmwArea.laddr  = laddr;
  vm->rmwArea.native = 1; /* Default to flagging that native access OK. */

#if 1
  /* If linear access is not within one PDE slot away from monitor PDE,
   * then we can use a native seg:offset access.  Could use a check for
   * only the monitor PDE slot, but that would require two compares.
   */
  if ( (laddr & 0xffe00000) != (vm->mon_pde_mask | 0x00200000) ) {
    if (size==4) {
      if ( __readRMWDWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==1) {
      if ( __readRMWByteNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    else if (size==2) {
      if ( __readRMWWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
        return; /* Handled by native seg:offset access. */
        }
      /* Fall through to slower emulation code. */
      }
    }
#endif

  vm->rmwArea.native = 0; /* Native access did not work. */

  write_virtual_checks(vm, sreg, off, size);
  access_linear(vm, laddr, size, G_CPL(vm)==3, OP_RW, data);
}

  void
write_RMW_virtual(vm_t *vm, void *data, unsigned size)
{
  if (vm->rmwArea.native) {
    unsigned sreg  = vm->rmwArea.sreg;
    Bit32u   off   = vm->rmwArea.offset;
    Bit32u   laddr = vm->rmwArea.laddr;

    if ( (laddr & 0xffe00000) != (vm->mon_pde_mask | 0x00200000) ) {
      if (size==4) {
        if ( __writeDWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
          return; /* Handled by native seg:offset access. */
          }
        /* Fall through to slower emulation code. */
        }
      else if (size==1) {
        if ( __writeByteNative(Selector(4+sreg,0,RPL3), off, data) ) {
          return; /* Handled by native seg:offset access. */
          }
        /* Fall through to slower emulation code. */
        }
      else if (size==2) {
        if ( __writeWordNative(Selector(4+sreg,0,RPL3), off, data) ) {
          return; /* Handled by native seg:offset access. */
          }
        /* Fall through to slower emulation code. */
        }
      }
    monpanic(vm, "write_RMW_virtual: native=1, but access failed?\n");
    }

  /* Read phase of access could not use native access.  Linear-to-physical
   * mappings are cached by the translation unit.  Use them, rather than
   * re-mapping address.
   */
  if (vm->guest_cpu.cr0.fields.pg) {
    if (vm->guest_cpu.address_xlation.pages == 1) {
      write_physical(vm, vm->guest_cpu.address_xlation.paddress1, size, data);
      }
    else {
      write_physical(vm, vm->guest_cpu.address_xlation.paddress1,
                     vm->guest_cpu.address_xlation.len1,
                     data);
      write_physical(vm, vm->guest_cpu.address_xlation.paddress2,
                     vm->guest_cpu.address_xlation.len2,
                     ((Bit8u *) data) + vm->guest_cpu.address_xlation.len1);
      }
    }
  else {
    write_physical(vm, vm->guest_cpu.address_xlation.paddress1, size, data);
    }
}


  void
write_virtual_checks(vm_t *vm, unsigned sreg, Bit32u off, unsigned size)
{
  Bit32u upper_limit;
  descriptor_cache_t *cache;

  cache = &vm->guest_cpu.desc_cache[sreg];
  if ( ProtectedMode(vm) ) {
    if (cache->desc.p == 0) { /* not present */
monpanic(vm, "write_virtual_checks(): segment not present\n");
      goto do_exception;
      }

    switch ( cache->desc.type & 0x0f ) {
      case 0: case 1:   /* read only */
      case 4: case 5:   /* read only, expand down */
      case 8: case 9:   /* execute only */
      case 10: case 11: /* execute/read */
      case 12: case 13: /* execute only, conforming */
      case 14: case 15: /* execute/read-only, conforming */
monpanic(vm, "write_virtual_checks(): no write access to seg\n");
        goto do_exception;

      case 2: case 3: /* read/write */
        if ( (off+size-1) > cache->limit_scaled ) {
monpanic(vm, "write_virtual_checks(): write beyond limit, r/w\n");
          goto do_exception;
          }
        break;

      case 6: case 7: /* read write, expand down */
        if (cache->desc.d_b)
          upper_limit = 0xffffffff;
        else
          upper_limit = 0x0000ffff;
        if ( (off <= cache->limit_scaled) ||
             (off > upper_limit) ||
             ((upper_limit - off) < (size - 1)) ) {
monpanic(vm, "write_virtual_checks(): write beyond limit, r/w ED\n");
          goto do_exception;
          }
        break;
      }

    return; /* OK */
    }

  else { /* real mode */
    if ( (off + size - 1)  >  cache->limit_scaled) {
      monprint(vm, "write_virtual_checks EXCEPTION: seg=%u %x:%x + %x\n",
        sreg, (unsigned) vm->guest_cpu.selector[sreg].raw,
        (unsigned) off, (unsigned) size);
      goto do_exception;
      }
    return; /* OK */
    }

do_exception:
  if (sreg == SRegSS)
    exception(vm, ExceptionSS, 0);
  else
    exception(vm, ExceptionGP, 0);
}

  void
read_virtual_checks(vm_t *vm, unsigned sreg, Bit32u off, unsigned size)
{
  Bit32u upper_limit;
  descriptor_cache_t *cache;

  cache = &vm->guest_cpu.desc_cache[sreg];
  if ( ProtectedMode(vm) ) {
    if (cache->desc.p == 0) { /* not present */
monpanic(vm, "read_virtual_checks(): segment not present\n");
      goto do_exception;
      }

    switch ( cache->desc.type & 0x0f ) {
      case 0: case 1: /* read only */
      case 2: case 3: /* read/write */
      case 10: case 11: /* execute/read */
      case 14: case 15: /* execute/read-only, conforming */
        if ( (off+size-1) > cache->limit_scaled ) {
monprint(vm, "read_virtual_checks: beyond limit, r/w\n");
          goto do_exception;
          }
        break;

      case 4: case 5: /* read only, expand down */
      case 6: case 7: /* read write, expand down */
        if (cache->desc.d_b)
          upper_limit = 0xffffffff;
        else
          upper_limit = 0x0000ffff;
        if ( (off <= cache->limit_scaled) ||
             (off > upper_limit) ||
             ((upper_limit - off) < (size - 1)) ) {
monpanic(vm, "read_virtual_checks: beyond limit, r/w ED\n");
          goto do_exception;
          }
        break;
      case 8: case 9: /* execute only */
      case 12: case 13: /* execute only, conforming */
        goto do_exception;
      }
    return; /* OK */
    }
  else {
    if ( (off + size - 1)  >  cache->limit_scaled) {
      monprint(vm, "read_virtual_checks EXCEPTION: seg=%u %x:%x + %x\n",
        sreg, (unsigned) vm->guest_cpu.selector[sreg].raw,
        (unsigned) off, (unsigned) size);
      goto do_exception;
      }
    return; /* OK */
    }

do_exception:
  if (sreg == SRegSS)
    exception(vm, ExceptionSS, 0);
  else
    exception(vm, ExceptionGP, 0);
}

