/*
 * Pidgin is the legal property of its developers, whose names are too numerous
 * to list here.  Please refer to the COPYRIGHT file distributed with this
 * source distribution.
 *
 * This program is free software; you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include "debug.h"

#include "gtkmenutray.h"

#include <gtk/gtkeventbox.h>
#include <gtk/gtkiconfactory.h>
#include <gtk/gtkversion.h>

/******************************************************************************
 * Enums
 *****************************************************************************/
enum {
	PROP_ZERO = 0,
	PROP_BOX
};

/******************************************************************************
 * Globals
 *****************************************************************************/
static GObjectClass *parent_class = NULL;

/******************************************************************************
 * Internal Stuff
 *****************************************************************************/

/******************************************************************************
 * Item Stuff
 *****************************************************************************/
static void
pidgin_menu_tray_select(GtkItem *item) {
	/* this may look like nothing, but it's really overriding the
	 * GtkMenuItem's select function so that it doesn't get highlighted like
	 * a normal menu item would.
	 */
}

static void
pidgin_menu_tray_deselect(GtkItem *item) {
	/* Probably not necessary, but I'd rather be safe than sorry.  We're
	 * overridding the select, so it makes sense to override deselect as well.
	 */
}

/******************************************************************************
 * Widget Stuff
 *****************************************************************************/

/******************************************************************************
 * Object Stuff
 *****************************************************************************/
static void
pidgin_menu_tray_get_property(GObject *obj, guint param_id, GValue *value,
								GParamSpec *pspec)
{
	PidginMenuTray *menu_tray = PIDGIN_MENU_TRAY(obj);

	switch(param_id) {
		case PROP_BOX:
			g_value_set_object(value, pidgin_menu_tray_get_box(menu_tray));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(obj, param_id, pspec);
			break;
	}
}

static void
pidgin_menu_tray_finalize(GObject *obj)
{
	PidginMenuTray *tray = PIDGIN_MENU_TRAY(obj);
#if 0
	/* This _might_ be leaking, but I have a sneaking suspicion that the widget is
	 * getting destroyed in GtkContainer's finalize function.  But if were are
	 * leaking here, be sure to figure out why this causes a crash.
	 *	-- Gary
	 */

	if(GTK_IS_WIDGET(tray->tray))
		gtk_widget_destroy(GTK_WIDGET(tray->tray));
#endif

	if (tray->tooltips) {
		gtk_object_sink(GTK_OBJECT(tray->tooltips));
	}

	G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
pidgin_menu_tray_class_init(PidginMenuTrayClass *klass) {
	GObjectClass *object_class = G_OBJECT_CLASS(klass);
	GtkItemClass *item_class = GTK_ITEM_CLASS(klass);
	GParamSpec *pspec;

	parent_class = g_type_class_peek_parent(klass);

	object_class->finalize = pidgin_menu_tray_finalize;
	object_class->get_property = pidgin_menu_tray_get_property;

	item_class->select = pidgin_menu_tray_select;
	item_class->deselect = pidgin_menu_tray_deselect;

	pspec = g_param_spec_object("box", "The box",
								"The box",
								GTK_TYPE_BOX,
								G_PARAM_READABLE);
	g_object_class_install_property(object_class, PROP_BOX, pspec);
}

static void
pidgin_menu_tray_init(PidginMenuTray *menu_tray) {
	GtkWidget *widget = GTK_WIDGET(menu_tray);
#if GTK_CHECK_VERSION(2,2,0)
	GtkSettings *settings;
#endif
	gint height = -1;

	gtk_menu_item_set_right_justified(GTK_MENU_ITEM(menu_tray), TRUE);

	if(!GTK_IS_WIDGET(menu_tray->tray))
		menu_tray->tray = gtk_hbox_new(FALSE, 0);

#if GTK_CHECK_VERSION(2,2,0)
	settings =
		gtk_settings_get_for_screen(gtk_widget_get_screen(widget));

	if(gtk_icon_size_lookup_for_settings(settings, GTK_ICON_SIZE_MENU,
										 NULL, &height))
#else
	if(gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, NULL, &height))
#endif
	{
		gtk_widget_set_size_request(widget, -1, height);
	}

	gtk_container_add(GTK_CONTAINER(menu_tray), menu_tray->tray);

	gtk_widget_show(menu_tray->tray);
}

/******************************************************************************
 * API
 *****************************************************************************/
GType
pidgin_menu_tray_get_gtype(void) {
	static GType type = 0;

	if(type == 0) {
		static const GTypeInfo info = {
			sizeof(PidginMenuTrayClass),
			NULL,
			NULL,
			(GClassInitFunc)pidgin_menu_tray_class_init,
			NULL,
			NULL,
			sizeof(PidginMenuTray),
			0,
			(GInstanceInitFunc)pidgin_menu_tray_init,
			NULL
		};

		type = g_type_register_static(GTK_TYPE_MENU_ITEM,
									  "PidginMenuTray",
									  &info, 0);
	}

	return type;
}

GtkWidget *
pidgin_menu_tray_new() {
	return g_object_new(PIDGIN_TYPE_MENU_TRAY, NULL);
}

GtkWidget *
pidgin_menu_tray_get_box(PidginMenuTray *menu_tray) {
	g_return_val_if_fail(PIDGIN_IS_MENU_TRAY(menu_tray), NULL);
	return menu_tray->tray;
}

static void
pidgin_menu_tray_add(PidginMenuTray *menu_tray, GtkWidget *widget,
					   const char *tooltip, gboolean prepend)
{
	g_return_if_fail(PIDGIN_IS_MENU_TRAY(menu_tray));
	g_return_if_fail(GTK_IS_WIDGET(widget));

	if (GTK_WIDGET_NO_WINDOW(widget))
	{
		GtkWidget *event;

		event = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(event), widget);
		gtk_widget_show(event);
		widget = event;
	}

	pidgin_menu_tray_set_tooltip(menu_tray, widget, tooltip);

	if (prepend)
		gtk_box_pack_start(GTK_BOX(menu_tray->tray), widget, FALSE, FALSE, 0);
	else
		gtk_box_pack_end(GTK_BOX(menu_tray->tray), widget, FALSE, FALSE, 0);
}

void
pidgin_menu_tray_append(PidginMenuTray *menu_tray, GtkWidget *widget, const char *tooltip)
{
	pidgin_menu_tray_add(menu_tray, widget, tooltip, FALSE);
}

void
pidgin_menu_tray_prepend(PidginMenuTray *menu_tray, GtkWidget *widget, const char *tooltip)
{
	pidgin_menu_tray_add(menu_tray, widget, tooltip, TRUE);
}

void
pidgin_menu_tray_set_tooltip(PidginMenuTray *menu_tray, GtkWidget *widget, const char *tooltip)
{
	if (!menu_tray->tooltips)
		menu_tray->tooltips = gtk_tooltips_new();

	/* Should we check whether widget is a child of menu_tray? */

	/*
	 * If the widget does not have it's own window, then it
	 * must have automatically been added to an event box
	 * when it was added to the menu tray.  If this is the
	 * case, we want to set the tooltip on the widget's parent,
	 * not on the widget itself.
	 */
	if (GTK_WIDGET_NO_WINDOW(widget))
		widget = widget->parent;

	gtk_tooltips_set_tip(menu_tray->tooltips, widget, tooltip, NULL);
}

