<?php
/**
 * Copyright 2010-2014 Horde LLC (http://www.horde.org/)
 *
 * See the enclosed file COPYING for license information (LGPL). If you
 * did not receive this file, see http://www.horde.org/licenses/lgpl21.
 *
 * @category  Horde
 * @copyright 2010-2014 Horde LLC
 * @license   http://www.horde.org/licenses/lgpl21 LGPL 2.1
 * @package   Cache
 */

/**
 * Cache storage in a PHP session.
 *
 * @author    Michael Slusarz <slusarz@horde.org>
 * @category  Horde
 * @copyright 2010-2014 Horde LLC
 * @license   http://www.horde.org/licenses/lgpl21 LGPL 2.1
 * @package   Cache
 */
class Horde_Cache_Storage_Session extends Horde_Cache_Storage_Base
{
    /**
     * Pointer to the session entry.
     *
     * @var array
     */
    protected $_sess;

    /**
     * Constructor.
     *
     * @param array $params  Optional parameters:
     * <pre>
     *   - session: (string) Store session data in this entry.
     *              DEFAULT: 'horde_cache_session'
     * </pre>
     */
    public function __construct(array $params = array())
    {
        $params = array_merge(array(
            'sess_name' => 'horde_cache_session'
        ), $params);

        parent::__construct($params);
    }

    /**
     * Do initialization tasks.
     */
    protected function _initOb()
    {
        if (!isset($_SESSION[$this->_params['sess_name']])) {
            $_SESSION[$this->_params['sess_name']] = array();
        }
        $this->_sess = &$_SESSION[$this->_params['sess_name']];
    }

    /**
     */
    public function get($key, $lifetime = 0)
    {
        return $this->exists($key, $lifetime)
            ? $this->_sess[$key]['d']
            : false;
    }

    /**
     */
    public function set($key, $data, $lifetime = 0)
    {
        $this->_sess[$key] = array(
            'd' => $data,
            'l' => $lifetime
        );
    }

    /**
     */
    public function exists($key, $lifetime = 0)
    {
        if (isset($this->_sess[$key])) {
            /* 0 means no expire. */
            if (($lifetime == 0) ||
                ((time() - $lifetime) <= $this->_sess[$key]['l'])) {
                return true;
            }

            unset($this->_sess[$key]);
        }

        return false;
    }

    /**
     */
    public function expire($key)
    {
        if (isset($this->_sess[$key])) {
            unset($this->_sess[$key]);
            return true;
        }

        return false;
    }

    /**
     */
    public function clear()
    {
        $this->_sess = array();
    }

}
