#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri2dCSR.c,v 1.5 2001/01/10 23:08:38 knepley Exp $";
#endif

/* CSR format for 2d triangular grids */
#include "src/mesh/impls/triangular/2d/2dimpl.h" /*I "mesh.h" I*/
#include "tri2dCSR.h"

#undef  __FUNCT__
#define __FUNCT__ "MeshCreateLocalCSR_Triangular_2D"
int MeshCreateLocalCSR_Triangular_2D(Mesh mesh, int *numV, int *numE, int **vertOffsets, int **vertNeighbors,
                                     double **vertCoords, int numBC, int *bcNodes, PetscTruth symmetric)
{
  Mesh_Triangular *tri         = (Mesh_Triangular *) mesh->data;
  int              numEdges    = mesh->numEdges;
  int             *edges       = tri->edges;
  int              numVertices = mesh->numVertices;
  int              numNodes    = mesh->numNodes;
  double          *nodes       = tri->nodes;
  int             *degrees     = tri->degrees;
  int             *vO;
  int             *vN;
  double          *vC;
  int              size, count, locCount, bcSeen, nBCSeen;
#ifdef PETSC_USE_BOPT_g
  int              globalRows;
#endif
  int              node, nNode, edge, row, bc;
  int              ierr;

  PetscFunctionBegin;
  /* Get the size of the neighbors array -- Remember that midnodes have degree 0 */
  for(node = 0, size = 0; node < numNodes; node++) {
    size += degrees[node]+1;
  }

  /* Allocate memory */
  ierr = PetscMalloc((numVertices-numBC+1) * sizeof(int),    &vO);                                        CHKERRQ(ierr);
  ierr = PetscMalloc(size                  * sizeof(int),    &vN);                                        CHKERRQ(ierr);
  ierr = PetscMalloc(numVertices*2         * sizeof(double), &vC);                                        CHKERRQ(ierr);

  /* Fill the neighbors array */
  vO[0] = 0;
  for(node = 0, row = 1, count = 0, bcSeen = 0; node < numNodes; node++) {
    /* Only process vertices */
    if (degrees[node] == 0)
      continue;
    /* Ignore constrained nodes */
    for(bc = 0; bc < numBC; bc++)
      if (node == bcNodes[bc])
        break;
    if (bc < numBC) {
      bcSeen++;
      continue;
    }

    /* Include a self edge */
    vN[count++] = node - bcSeen;
#ifdef PETSC_USE_BOPT_g
    if (node - bcSeen < 0) SETERRQ(PETSC_ERR_PLIB, "Invalid node number");
#endif

    for(edge = 0, locCount = 0; (edge < numEdges) && (locCount < degrees[node]); edge++)
    {
      if (edges[edge*2] == node)
        nNode = edges[edge*2+1];
      else if (edges[edge*2+1] == node)
        nNode = edges[edge*2];
      else
        continue;

      locCount++;
      /* Ignore edges which go off processor and check for symmetrized matrix */
      if ((nNode >= numNodes) || ((symmetric == PETSC_TRUE) && (nNode > node)))
        continue;
      /* Ignore constrained nodes and renumber */
      for(bc = 0, nBCSeen = 0; bc < numBC; bc++) {
        if (bcNodes[bc] < 0)
          continue;
        if (nNode == bcNodes[bc])
          break;
        else if (nNode > bcNodes[bc])
          nBCSeen++;
      }
      if (bc < numBC)
        continue;
      vN[count++] = nNode - nBCSeen;
#ifdef PETSC_USE_BOPT_g
      if (nNode - nBCSeen < 0) SETERRQ(PETSC_ERR_PLIB, "Invalid node number");
#endif
    }
    /* Get coordinates */
    vC[(row-1)*2]   = nodes[node*2];
    vC[(row-1)*2+1] = nodes[node*2+1];
    vO[row++]       = count;
  }
#ifdef PETSC_USE_BOPT_g
  ierr = MPI_Allreduce(&row, &globalRows, 1, MPI_INT, MPI_SUM, mesh->comm);                              CHKERRQ(ierr);
  if (globalRows != numVertices - numBC + mesh->part->numProcs) {
    SETERRQ(PETSC_ERR_PLIB, "Invalid number of vertices or bad boundary conditions");
  }
  ierr = PetscTrValid(__LINE__, __FUNCT__, __FILE__, __SDIR__);                                           CHKERRQ(ierr);
#endif
  if (count > size) SETERRQ(PETSC_ERR_PLIB, "Invalid connectivity");
  PetscLogInfo(mesh, "Local CSR Graph: %d vertices, %d edges\n", row-1, count-(row-1));
  *numV = row-1;
  *numE = count-(row-1);

#if 0
#ifdef PETSC_USE_BOPT_g
  {
    int neighbor, neighbor2, row2;

  /* Check local symmetry of the graph */
  for(row = 0; row < numNodes; row++)
    for(neighbor = vO[row]; neighbor < vO[row+1]; neighbor++)
    {
      row2 = vN[neighbor];
      if ((row2 < 0) || (row2 >= numNodes)) {
        SETERRQ(PETSC_ERR_PLIB, "Invalid local graph");
      } else {
        /* Check for companion edge */
        for(neighbor2 = vO[row2]; neighbor2 < vO[row2+1]; neighbor2++)
          if (vN[neighbor2] == row)
            break;
        if (neighbor2 == vO[row2+1]) SETERRQ(PETSC_ERR_PLIB, "Nonsymmetric local graph");
      }
    }
  }
#endif
#endif

  if (vertOffsets   != PETSC_NULL) {
    *vertOffsets   = vO;
  } else {
    ierr = PetscFree(vO);                                                                                CHKERRQ(ierr);
  }
  if (vertNeighbors != PETSC_NULL) {
    *vertNeighbors = vN;
  } else {
    ierr = PetscFree(vN);                                                                                CHKERRQ(ierr);
  }
  if (vertCoords    != PETSC_NULL) {
    *vertCoords    = vC;
  } else {
    ierr = PetscFree(vC);                                                                                CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCreateFullCSR_Triangular_2D"
int MeshCreateFullCSR_Triangular_2D(Mesh mesh, PetscTruth constrain, int *numV, int *numE, int **vertOffsets, int **vertNeighbors)
{
  Mesh_Triangular *tri         = (Mesh_Triangular *) mesh->data;
  int              numBd       = mesh->numBd;
  int              numElements = mesh->numFaces;
  int             *elements    = tri->faces;
  int              numCorners  = mesh->numCorners;
  int              numNodes    = mesh->numNodes;
  int             *markers     = tri->markers;
  int             *bdMarkers   = tri->bdMarkers;
  int             *bdBegin     = tri->bdBegin;
  int             *bdNodes     = tri->bdNodes;
  int             *nodeOffsets, *nodeNeighbors;
  int            **constNeighbors;
  int             *nodeDone, *neighbors, *constCount;
  int             *support;
  int              degree;
  PetscTruth       duplicate;
  int              maxNeighbors, size;
  int              bd, elem, corner, neighbor, node, anchorNode, sElem, sCorner, sNode, sBd, nNode, bdNode, count = 0;
  int              ierr;

  PetscFunctionBegin;
  /* Allocate adjacency structures */
  maxNeighbors  = mesh->maxDegree*numCorners;
  ierr = PetscMalloc(numNodes     * sizeof(int), &nodeDone);                                              CHKERRQ(ierr);
  ierr = PetscMalloc(maxNeighbors * sizeof(int), &nodeNeighbors);                                         CHKERRQ(ierr);
  ierr = PetscMalloc((numNodes+1) * sizeof(int), &nodeOffsets);                                           CHKERRQ(ierr);
  ierr = PetscMemzero(nodeOffsets, (numNodes+1) * sizeof(int));                                           CHKERRQ(ierr);
  ierr = PetscMemzero(nodeDone,     numNodes    * sizeof(int));                                           CHKERRQ(ierr);
  /* Allocate constraint structures */
  if (constrain == PETSC_TRUE) {
    ierr = PetscMalloc(numBd * sizeof(int *), &constNeighbors);                                           CHKERRQ(ierr);
    for(bd = 0; bd < numBd; bd++) {
      if (bdMarkers[bd] < 0) {
        ierr = PetscMalloc(maxNeighbors*(bdBegin[bd+1] - bdBegin[bd]) * sizeof(int), &constNeighbors[bd]); CHKERRQ(ierr);
      }
    }
  }
  /* Calculate the number of neighbors for each node */
  for(elem = 0; elem < numElements; elem++) {
    for(corner = 0; corner < numCorners; corner++) {
      node = elements[elem*numCorners+corner];
      if (nodeDone[node]) continue;
      nodeDone[node] = 1;

      /* Get array holding list of neighboring nodes */
      if ((constrain == PETSC_TRUE) && (markers[node] < 0)) {
        /* Constrained node: We maintain a list of neighbors seen */
        ierr = MeshGetBoundaryIndex(mesh, markers[node], &bd);                                            CHKERRQ(ierr);
        neighbors  = constNeighbors[bd];
        /* We let the first constrained node be a representative for all of them */
        anchorNode = bdNodes[bdBegin[bd]];
      } else {
        /* Normal node: Just use temporary storage since we only look at it once */
        neighbors  = nodeNeighbors;
        anchorNode = node;
      }

      /* Loop over nodes on each element in the support of the node */
      ierr = MeshGetNodeSupport(mesh, node, elem, &degree, &support);                                     CHKERRQ(ierr);
      for(sElem = 0, count = nodeOffsets[anchorNode+1]; sElem < degree; sElem++) {
        for(sCorner = 0; sCorner < numCorners; sCorner++) {
          sNode = elements[support[sElem]*numCorners+sCorner];

          /* Account for constrained nodes: link to anchor */
          if ((constrain == PETSC_TRUE) && (markers[sNode] < 0)) {
            ierr  = MeshGetBoundaryIndex(mesh, markers[sNode], &sBd);                                     CHKERRQ(ierr);
            sNode = bdNodes[bdBegin[sBd]];
          }

          /* Check for duplicate node and insert in order */
          for(neighbor = count-1, duplicate = PETSC_FALSE; neighbor >= 0; neighbor--) {
            nNode = neighbors[neighbor];
            if (sNode > nNode) {
              break;
            } else if (sNode == nNode) {
              duplicate = PETSC_TRUE;
              break;
            }
            neighbors[neighbor+1] = nNode;
          }
          if (duplicate == PETSC_FALSE) {
            neighbors[neighbor+1] = sNode;
            count++;
          } else {
            ierr = PetscMemmove(&neighbors[neighbor+1], &neighbors[neighbor+2], (count-1-neighbor) * sizeof(int));
            CHKERRQ(ierr);
          }
#ifdef PETSC_USE_BOPT_g
          if ((constrain == PETSC_TRUE) && (markers[node] < 0)) {
            if (count >= maxNeighbors*(bdBegin[bd+1] - bdBegin[bd]))
              SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Too many neighboring nodes %d", count);
          } else {
            if (count >= maxNeighbors)
              SETERRQ1(PETSC_ERR_ARG_OUTOFRANGE, "Too many neighboring nodes %d", count);
          }
#endif
        }
      }
      ierr = MeshRestoreNodeSupport(mesh, node, elem, &degree, &support);                                 CHKERRQ(ierr);

      /* Record number of neighbors */
      nodeOffsets[anchorNode+1] = count;
    }
  }
  ierr = PetscFree(nodeNeighbors);                                                                        CHKERRQ(ierr);
  /* Connect all the nodes on a given boundary */
  if (constrain == PETSC_TRUE) {
    for(bd = 0; bd < numBd; bd++) {
      if (bdMarkers[bd] < 0) {
        anchorNode = bdNodes[bdBegin[bd]];
        for(bdNode = bdBegin[bd]+1; bdNode < bdBegin[bd+1]; bdNode++)
          nodeOffsets[bdNodes[bdNode]+1] = nodeOffsets[anchorNode+1];
      }
    }
  }
  /* Do prefix sums */
  nodeOffsets[0] = 0;
  for(node = 1; node <= numNodes; node++) {
    nodeOffsets[node] += nodeOffsets[node-1];
  }
  /* Cleanup */
  if (constrain == PETSC_TRUE) {
    for(bd = 0; bd < numBd; bd++) {
      if (bdMarkers[bd] < 0) {
        ierr = PetscFree(constNeighbors[bd]);                                                             CHKERRQ(ierr);
      }
    }
    ierr = PetscFree(constNeighbors);                                                                     CHKERRQ(ierr);
  }

  /* Calculate adjacency list */
  ierr = PetscMalloc(nodeOffsets[numNodes] * sizeof(int), &nodeNeighbors);                                CHKERRQ(ierr);
  ierr = PetscMalloc(numBd                 * sizeof(int), &constCount);                                   CHKERRQ(ierr);
  ierr = PetscMemzero(nodeDone,   numNodes * sizeof(int));                                                CHKERRQ(ierr);
  ierr = PetscMemzero(constCount, numBd    * sizeof(int));                                                CHKERRQ(ierr);
  for(elem = 0; elem < numElements; elem++) {
    for(corner = 0; corner < numCorners; corner++) {
      node = elements[elem*numCorners+corner];
      if (nodeDone[node]) continue;
      nodeDone[node] = 1;

      /* Get array holding list of neighboring nodes */
      if ((constrain == PETSC_TRUE) && (markers[node] < 0)) {
        /* We let the first constrained node be a representative for all of them */
        ierr = MeshGetBoundaryIndex(mesh, markers[node], &bd);                                            CHKERRQ(ierr);
        anchorNode = bdNodes[bdBegin[bd]];
        count      = constCount[bd];
      } else {
        anchorNode = node;
        count      = 0;
      }
      neighbors    = &nodeNeighbors[nodeOffsets[anchorNode]];

      /* Loop over nodes on each element in the support of the node */
      ierr = MeshGetNodeSupport(mesh, node, elem, &degree, &support);                                     CHKERRQ(ierr);
      for(sElem = 0; sElem < degree; sElem++) {
        for(sCorner = 0; sCorner < numCorners; sCorner++) {
          if (count >= nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode]) break;
          sNode = elements[support[sElem]*numCorners+sCorner];

          /* Account for constrained nodes: link to anchor */
          if ((constrain == PETSC_TRUE) && (markers[sNode] < 0)) {
            ierr  = MeshGetBoundaryIndex(mesh, markers[sNode], &sBd);                                     CHKERRQ(ierr);
            sNode = bdNodes[bdBegin[sBd]];
          }

          /* Check for duplicate node and insert in order */
          for(neighbor = count-1, duplicate = PETSC_FALSE; neighbor >= 0; neighbor--) {
            nNode = neighbors[neighbor];
            if (sNode > nNode) {
              break;
            } else if (sNode == nNode) {
              duplicate = PETSC_TRUE;
              break;
            }
            neighbors[neighbor+1] = nNode;
          }
          if (duplicate == PETSC_FALSE) {
            neighbors[neighbor+1] = sNode;
            count++;
          } else {
            ierr = PetscMemmove(&neighbors[neighbor+1], &neighbors[neighbor+2], (count-1-neighbor) * sizeof(int));
            CHKERRQ(ierr);
          }
        }
      }
      ierr = MeshRestoreNodeSupport(mesh, node, elem, &degree, &support);                                 CHKERRQ(ierr);

      if ((constrain == PETSC_FALSE) || (markers[node] >= 0)) {
        if (count != nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode]) {
          SETERRQ2(PETSC_ERR_PLIB, "Invalid number of adjacent nodes %d should be %d",
                   count, nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode]);
        }
      } else {
        constCount[bd] = count;
      }
    }
  }
  /* Handle constrained boundaries */
  if (constrain == PETSC_TRUE) {
    for(bd = 0; bd < numBd; bd++) {
      if (bdMarkers[bd] < 0) {
        anchorNode = bdNodes[bdBegin[bd]];
        /* Check adjacency list */
        if (constCount[bd] != nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode]) {
          SETERRQ2(PETSC_ERR_PLIB, "Invalid number of adjacent nodes %d should be %d",
                   count, nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode]);
        }
        /* Connect all the nodes on a given boundary */
        for(bdNode = bdBegin[bd]+1; bdNode < bdBegin[bd+1]; bdNode++) {
          node = bdNodes[bdNode];
          size = nodeOffsets[anchorNode+1] - nodeOffsets[anchorNode];
          if (nodeOffsets[node+1] - nodeOffsets[node] != size) {
            SETERRQ2(PETSC_ERR_PLIB, "Invalid number of adjacent nodes %d should be %d",
                     nodeOffsets[node+1] - nodeOffsets[node], size);
          }
          ierr = PetscMemcpy(&nodeNeighbors[nodeOffsets[node]], &nodeNeighbors[nodeOffsets[anchorNode]], size*sizeof(int));
          CHKERRQ(ierr);
        }
      }
    }
  }
  /* Cleanup */
  ierr = PetscFree(nodeDone);                                                                             CHKERRQ(ierr);
  ierr = PetscFree(constCount);                                                                           CHKERRQ(ierr);

  *numV          = numNodes;
  *numE          = nodeOffsets[numNodes]/2;
  *vertOffsets   = nodeOffsets;
  *vertNeighbors = nodeNeighbors;
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshCreateDualCSR_Triangular_2D"
int MeshCreateDualCSR_Triangular_2D(Mesh mesh, int **elemOffsets, int **elemNeighbors, int **edgeWeights, int weight)
{
  Mesh_Triangular *tri          = (Mesh_Triangular *) mesh->data;
  Partition        p            = mesh->part;
  int              numNeighbors = 3;
  int              numCorners   = mesh->numCorners;
  int             *neighbors    = tri->neighbors;
  int             *elements     = tri->faces;
  int             *markers      = tri->markers;
  int             *eO;
  int             *eN;
  int             *eW;
  int              numLocElements;
  int              size, count;
  int              elem, neighbor, corner, row;
#ifdef PETSC_USE_BOPT_g
  int              neighbor2, row2;
#endif
  int              ierr;

  PetscFunctionBegin;
  /* Get the size of the neighbors array */
  for(elem = p->firstElement[p->rank], size = 0; elem < p->firstElement[p->rank+1]; elem++) {
    for(neighbor = 0; neighbor < numNeighbors; neighbor++) {
      if (neighbors[elem*numNeighbors + neighbor] >= 0) size++;
    }
  }

  /* Allocate memory */
  numLocElements = p->firstElement[p->rank+1] - p->firstElement[p->rank];
  ierr = PetscMalloc((numLocElements+1) * sizeof(int), &eO);                                              CHKERRQ(ierr);
  ierr = PetscMalloc(size               * sizeof(int), &eN);                                              CHKERRQ(ierr);

  if ((weight != 0) && (edgeWeights != PETSC_NULL)) {
    ierr = PetscMalloc(size * sizeof(int), &eW);                                                          CHKERRQ(ierr);
  } else {
    eW   = PETSC_NULL;
  }

  /* Fill the neighbors array */
  eO[0] = 0;
  for(elem = p->firstElement[p->rank], row = 1, count = 0; elem < p->firstElement[p->rank+1]; elem++, row++) {
    for(neighbor = 0; neighbor < numNeighbors; neighbor++) {
      if (neighbors[elem*numNeighbors + neighbor] >= 0) {
        eN[count] = neighbors[elem*numNeighbors + neighbor];
        if (eW != PETSC_NULL) {
          /* Check element for a node on an inner boundary */
          for(corner = 0; corner < 3; corner++) {
            if (markers[elements[elem*numCorners+corner]] < 0) break;
          }
          if (corner < 3) {
            /* Check neighbor for a node on an inner boundary */
            for(corner = 0; corner < 3; corner++) {
              if (markers[elements[neighbors[elem*numNeighbors + neighbor]*numCorners+corner]] < 0) break;
            }
            if (corner < 3) {
              eW[count] = weight;
            } else {
              eW[count] = 0;
            }
          } else {
            eW[count] = 0;
          }
        }
        count++;
      }
    }
    eO[row] = count;
  }
  if (count != size) SETERRQ(PETSC_ERR_PLIB, "Invalid adjacency matrix");
  PetscLogInfo(mesh, "Dual: %d elements, %d edges\n", numLocElements, size);

#ifdef PETSC_USE_BOPT_g
  /* Check local symmetry of the dual graph */
  for(row = 0; row < numLocElements; row++) {
    for(neighbor = eO[row]; neighbor < eO[row+1]; neighbor++) {
      row2 = eN[neighbor] - p->firstElement[p->rank];
      if ((row2 < 0) || (row2 >= numLocElements)) {
        /* PetscSynchronizedPrintf(p->comm, "[%3d]Cut: %6d --> %6d\n", p->rank, row + p->firstElement[p->rank], eN[neighbor]); */
      } else {
        /* Check for companion edge */
        for(neighbor2 = eO[row2]; neighbor2 < eO[row2+1]; neighbor2++) {
          if (eN[neighbor2] - p->firstElement[p->rank] == row) break;
        }
        if (neighbor2 == eO[row2+1]) SETERRQ(PETSC_ERR_PLIB, "Nonsymmetric dual graph");
      }
    }
  }
  /* PetscSynchronizedFlush(p->comm); */
#endif

  *elemOffsets   = eO;
  *elemNeighbors = eN;
  *edgeWeights   = eW;

  PetscFunctionReturn(0);
}

/*-------------------------------------- Replacement Backend Functions ----------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "PartitionCreate_CSR"
int PartitionCreate_CSR(Partition p)
{
  Partition_Triangular_2D *q        = (Partition_Triangular_2D *) p->data;
  Mesh                     mesh     = p->mesh;
  int                      numProcs = p->numProcs;
  int                      proc;
  int                      ierr;

  PetscFunctionBegin;
  p->numLocElements     = mesh->numFaces;
  p->numOverlapElements = p->numLocElements;
  q->numLocNodes        = mesh->numNodes;
  q->numOverlapNodes    = q->numLocNodes;
  q->numLocEdges        = mesh->numEdges;
  q->numLocBdNodes      = mesh->numBdNodes;
  q->numOverlapBdNodes  = q->numLocBdNodes;

  ierr = MPI_Allgather(&p->numLocElements, 1, MPI_INT, &p->firstElement[1], 1, MPI_INT, p->comm);        CHKERRQ(ierr);
  ierr = MPI_Allgather(&q->numLocNodes,    1, MPI_INT, &q->firstNode[1],    1, MPI_INT, p->comm);        CHKERRQ(ierr);
  ierr = MPI_Allgather(&q->numLocEdges,    1, MPI_INT, &q->firstEdge[1],    1, MPI_INT, p->comm);        CHKERRQ(ierr);
  ierr = MPI_Allgather(&q->numLocBdNodes,  1, MPI_INT, &q->firstBdNode[1],  1, MPI_INT, p->comm);        CHKERRQ(ierr);
  for(proc = 1; proc < numProcs; proc++) {
    p->firstElement[proc+1] += p->firstElement[proc];
    q->firstNode[proc+1]    += q->firstNode[proc];
    q->firstEdge[proc+1]    += q->firstEdge[proc];
    q->firstBdNode[proc+1]  += q->firstBdNode[proc];
  }

  p->numElements = p->firstElement[numProcs];
  q->numNodes    = q->firstNode[numProcs];
  q->numEdges    = q->firstEdge[numProcs];
  q->numBdNodes  = q->firstBdNode[numProcs];
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshPartition_Triangular_2D_CSR"
int MeshPartition_Triangular_2D_CSR(Mesh mesh)
{
  int ierr;

  PetscFunctionBegin;
  ierr = PartitionCreate(mesh, &mesh->part);                                                             CHKERRQ(ierr);
  ierr = PetscObjectComposeFunction((PetscObject) mesh->part, "PartitionTriangular2D_Create_C", "PartitionCreate_CSR",
                                    (void (*)(void)) PartitionCreate_CSR);
  CHKERRQ(ierr);
  ierr = PartitionSetType(mesh->part, PARTITION_TRIANGULAR_2D);                                          CHKERRQ(ierr);
  mesh->partitioned = 1;
  PetscFunctionReturn(ierr);
}

/*-------------------------------------------- Standard Interface ---------------------------------------------------*/
#undef  __FUNCT__
#define __FUNCT__ "MeshCreate_CSR"
/* MeshCreate_CSR
  This function creates a 2D unstructured mesh using an existing CSR representation.

  Collective on Mesh

  Input Parameter:
. numCorners  - The number of nodes in an element, here 3

  Input Parameters from bdCtx:
+ numBD       - The number of closed boundaries
. numVertices - The number of mesh nodes
. vertices    - The (x,y) coordinates of the mesh nodes
. markers     - The offset of the adjacency list for each node
. segMarkers  - The adjacency list of the mesh
. numSegments - The number of elements in the mesh
. segments    - The nodes in each element

  Output Parameter:
. mesh      - The new mesh

  Level: developer

.keywords mesh, CSR
.seealso MeshCreate_Triangle
*/
int MeshCreate_CSR(MeshBoundary2D *bdCtx, int numCorners, Mesh mesh)
{
  Mesh_Triangular *tri      = (Mesh_Triangular *) mesh->data;
  int              numNodes = bdCtx->numVertices;
  double          *nodes    = bdCtx->vertices;
  int             *offsets  = bdCtx->markers;
  int             *adj      = bdCtx->segMarkers;
  int              numEdges = offsets[numNodes];
  int              numFaces = bdCtx->numSegments;
  int             *faces    = bdCtx->segments;
  int              size, rank;
  int              node, adjNode, edge;
  int              ierr;

  PetscFunctionBegin;

  ierr = MPI_Comm_size(mesh->comm, &size);                                                               CHKERRQ(ierr);
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                               CHKERRQ(ierr);
  if (numCorners != 3) SETERRQ1(PETSC_ERR_SUP, "Each face must have 3 corners, not %d", numCorners);

  /* Setup function table */
  mesh->ops->partition = MeshPartition_Triangular_2D_CSR;

  /* Allocate arrays */
  tri->nodes         = PETSC_NULL;
  tri->markers       = PETSC_NULL;
  tri->edges         = PETSC_NULL;
  tri->edgemarkers   = PETSC_NULL;
  tri->faces         = PETSC_NULL;
  if (numNodes > 0) {
    ierr = PetscMalloc(numNodes*2 * sizeof(double), &tri->nodes);                                         CHKERRQ(ierr);
    ierr = PetscMalloc(numNodes   * sizeof(int),    &tri->markers);                                       CHKERRQ(ierr);
  }
  if (numEdges > 0) {
    ierr = PetscMalloc(numEdges*2 * sizeof(int),    &tri->edges);                                         CHKERRQ(ierr);
    ierr = PetscMalloc(numEdges   * sizeof(int),    &tri->edgemarkers);                                   CHKERRQ(ierr);
  }
  if (numFaces > 0) {
    ierr = PetscMalloc(numFaces*3 * sizeof(int),    &tri->faces);                                         CHKERRQ(ierr);
  }

  /* Remove redundant and self edges */
  for(node = 0, edge = 0; node < numNodes; node++) {
    for(adjNode = offsets[node]; adjNode < offsets[node+1]; adjNode++) {
      if (adj[adjNode] > node) {
        /* Could sort here */
        tri->edges[edge*2]   = node;
        tri->edges[edge*2+1] = adj[adjNode];
        edge++;
      }
    }
  }
  numEdges = edge;

  /* Store the information */
  mesh->numBd       = bdCtx->numBd;
  mesh->numNodes    = numNodes;
  mesh->numEdges    = numEdges;
  mesh->numVertices = (mesh->numNodes <= mesh->numEdges ? mesh->numNodes : mesh->numNodes - mesh->numEdges);
  mesh->numFaces    = numFaces;
  mesh->numCorners  = 3;
  mesh->numHoles    = 0;
  ierr = PetscMemcpy(tri->nodes, nodes, numNodes*2 * sizeof(double));                                    CHKERRQ(ierr);
  ierr = PetscMemcpy(tri->faces, faces, numFaces*3 * sizeof(int));                                       CHKERRQ(ierr);
  ierr = PetscMemzero(tri->markers,    numNodes   * sizeof(int));                                        CHKERRQ(ierr);
  ierr = PetscMemzero(tri->edgemarkers, numEdges * sizeof(int));                                         CHKERRQ(ierr);
  tri->neighbors   = PETSC_NULL;
  mesh->holes      = PETSC_NULL;
  PetscLogObjectMemory(mesh, (mesh->numNodes*2) * sizeof(double));
  PetscLogObjectMemory(mesh, (mesh->numNodes + mesh->numEdges*2 + mesh->numEdges + mesh->numFaces*mesh->numCorners) * sizeof(int));

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshRefine_CSR"
/*
  MeshRefine_CSR - This function refines a two dimensional unstructured mesh
  with a CSR representation using area constraints.
  
  Collective on Mesh
  
  Input Parameters:
+ oldMesh - The mesh begin refined
- area    - A function which gives an area constraint when evaluated inside an element
  
  Output Parameter:
. mesh    - The refined mesh
  
  Level: developer 
  
.keywords: mesh, CSR, refine 
.seealso: MeshRefine(), MeshCoarsen(), MeshReform()
*/
int MeshRefine_CSR(Mesh oldMesh, PointFunction area, Mesh mesh)
{ 
  SETERRQ(PETSC_ERR_SUP, "Not yet implemented");
}

