#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: tri1dView.c,v 1.4 2000/10/17 13:48:56 knepley Exp $";
#endif

/* Viewers for 1d triangular grids */
#include "src/mesh/impls/triangular/1d/1dimpl.h" /*I "mesh.h" I*/
#include "tri1dView.h"

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_1D_File"
static int MeshView_Triangular_1D_File(Mesh mesh, PetscViewer viewer) {
  Mesh_Triangular *tri = (Mesh_Triangular *) mesh->data;
  Partition        p   = mesh->part;
  FILE            *fd;
  int              i, j, d;
  int              ierr;

  PetscFunctionBegin;
  PetscViewerASCIIPrintf(viewer, "Mesh Object:\n");
  if (mesh->isPeriodic == PETSC_FALSE) {
    PetscViewerASCIIPrintf(viewer, "  Triangular 1D grid with\n");
  } else {
    PetscViewerASCIIPrintf(viewer, "  Periodic Triangular 1D grid with\n");
  }
  if (mesh->numBd == 1) {
    PetscViewerASCIIPrintf(viewer, "    %d closed boundary\n", mesh->numBd);
  } else {
    PetscViewerASCIIPrintf(viewer, "    %d closed boundaries\n", mesh->numBd);
  }
  for(i = 0; i < mesh->numBd; i++) {
    PetscViewerASCIIPrintf(viewer, "      Boundary %d: %d nodes\n", tri->bdMarkers[i], tri->bdBegin[i+1] - tri->bdBegin[i]);
  }
  PetscViewerASCIIPrintf(viewer, "      Total boundary nodes: %d\n", mesh->numBdNodes);
  ierr = PetscViewerASCIIGetPointer(viewer, &fd);                                                         CHKERRQ(ierr);
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: %d nodes\n", p->rank, mesh->numNodes);
  for(i = 0; i < mesh->numNodes; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d", i);
    for(d = 0; d < mesh->dim; d++) {
      PetscSynchronizedFPrintf(mesh->comm, fd, " %g", tri->nodes[i*mesh->dim+d]);
    }
    PetscSynchronizedFPrintf(mesh->comm, fd, " %d\n", tri->markers[i]);
  }
  PetscSynchronizedFlush(mesh->comm);
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local graph %d: %d edges with %d nodes per edge\n",
                           p->rank, mesh->numEdges, mesh->numCorners);
  for(i = 0; i < mesh->numEdges; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d", i);
    for(j = 0; j < mesh->numCorners; j++) PetscSynchronizedFPrintf(mesh->comm, fd, " %d", tri->edges[i*mesh->numCorners+j]);
    PetscSynchronizedFPrintf(mesh->comm, fd, "\n");
  }
  PetscSynchronizedFPrintf(mesh->comm, fd, "  Local dual graph %d: %d edges with %d neighbors per edge\n",
                           p->rank, mesh->numEdges, 2);
  for(i = 0; i < mesh->numEdges; i++) {
    PetscSynchronizedFPrintf(mesh->comm, fd, "    %d %d %d\n", i, tri->neighbors[i*2], tri->neighbors[i*2+1]);
  }
  PetscSynchronizedFlush(mesh->comm);
  
  if (mesh->partitioned) {
    ierr = PartitionView(mesh->part, viewer);                                                             CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_1D_Draw_Element"
static int MeshView_Triangular_1D_Draw_Element(Mesh mesh, PetscDraw draw, int edge, int color) {
  Mesh_Triangular *tri         = (Mesh_Triangular *) mesh->data;
  int              numCorners  = mesh->numCorners;
  int              numElements = mesh->numEdges;
  int             *elements    = tri->edges;
  double          *nodes       = tri->nodes;
  int             *markers     = tri->markers;
  int              node, corner;
  int              ierr;

  PetscFunctionBegin;
  if (numCorners == 2) {
    ierr = MeshDrawLine(mesh, draw, nodes[elements[edge*2]],   0.0, nodes[elements[edge*2+1]], 0.0, color);CHKERRQ(ierr);
  } else if (numCorners == 3) {
    ierr = MeshDrawLine(mesh, draw, nodes[elements[edge*3]],   0.0, nodes[elements[edge*3+2]], 0.0, color);CHKERRQ(ierr);
    ierr = MeshDrawLine(mesh, draw, nodes[elements[edge*3+2]], 0.0, nodes[elements[edge*3+1]], 0.0, color);CHKERRQ(ierr);
  } else {
    SETERRQ1(PETSC_ERR_SUP, "Invalid number of corners %d", numCorners);
  }
  for(corner = 0; corner < numCorners; corner++) {
    node = elements[edge*numCorners+corner];
    if (corner >= 2) {
      ierr = PetscDrawPointSetSize(draw, 0.0);CHKERRQ(ierr);
    } else {
      ierr = PetscDrawPointSetSize(draw, (1.0/numElements)/10.0);CHKERRQ(ierr);
    }
    if (markers[node] == 0) {
      ierr = PetscDrawPoint(draw, nodes[node], 0.0, PETSC_DRAW_BLACK);                                    CHKERRQ(ierr);
    } else {
      ierr = PetscDrawPoint(draw, nodes[node], 0.0, PETSC_DRAW_RED);                                      CHKERRQ(ierr);
    }
  }
  ierr = PetscDrawPointSetSize(draw, 0.0);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_1D_Draw_Mesh"
static int MeshView_Triangular_1D_Draw_Mesh(Mesh mesh, PetscDraw draw) {
  int numElements = mesh->numEdges;
  int hElement    = mesh->highlightElement;
  int color, rank;
  int elem;
  int ierr;

  PetscFunctionBegin;
  ierr = MPI_Comm_rank(mesh->comm, &rank);                                                                CHKERRQ(ierr);
  for(elem = 0; elem < numElements; elem++) {
    if (hElement == elem) {
      color = PETSC_DRAW_RED;
    } else {
      color = PETSC_DRAW_RED + rank + 2;
    }
    ierr = MeshView_Triangular_1D_Draw_Element(mesh, draw, elem, color);                                  CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_DrawStatusLine_Private"
/*
  Remember that this function sets the coordinates for the window.
*/
static int MeshView_DrawStatusLine_Private(Mesh mesh, PetscDraw draw, double startX, double startY, double endX,
                                           double endY, double offX, double offY) {
  Partition part = mesh->part;
  MPI_Comm  comm;
  char      statusLine[1024];
  PetscReal textWidth, textHeight;
  int       hElement, rank;
  int       ierr;

  PetscFunctionBegin;
  ierr = PetscObjectGetComm((PetscObject) mesh, &comm);                                                   CHKERRQ(ierr);
  ierr = MPI_Comm_rank(comm, &rank);                                                                      CHKERRQ(ierr);
  ierr = MeshGetHighlightElement(mesh, &hElement);                                                        CHKERRQ(ierr);
  ierr = PartitionLocalToGlobalElementIndex(part, hElement, &hElement);                                   CHKERRQ(ierr);
  sprintf(statusLine, "Element %d Proc: %d", hElement, rank);
  ierr = PetscDrawStringGetSize(draw, &textWidth, &textHeight);                                           CHKERRQ(ierr);
  ierr = PetscDrawSetCoordinates(draw, startX-offX, startY-offY-textHeight, endX+offX, endY+offY);        CHKERRQ(ierr);
  if (hElement >= 0) {
    ierr = PetscDrawString(draw, startX-offX, startY-offY-textHeight, PETSC_DRAW_BLACK, statusLine);      CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_1D_Draw"
static int MeshView_Triangular_1D_Draw(Mesh mesh, PetscViewer v) {
  double          startX = mesh->startX;
  double          endX   = mesh->endX;
  double          sizeX  = mesh->sizeX;
  double          sizeY  = sizeX/3.0;
  double          startY = 0.0;
  double          endY   = startY + sizeY;
  double          offX, offY;
  PetscDraw       draw;
  PetscTruth      isnull;
  PetscDrawButton button;
  int             pause, hElement;
  PetscReal       xc, yc, scale;
  int             xsize, ysize;
  int             ierr;

  PetscFunctionBegin;
  ierr = PetscViewerDrawGetDraw(v, 0, &draw);                                                             CHKERRQ(ierr);
  ierr = PetscDrawIsNull(draw, &isnull);                                                                  CHKERRQ(ierr);
  if (isnull) PetscFunctionReturn(0);
  if (sizeX > sizeY) {
    ysize = 300;
    if (sizeY > 0.0) {
      xsize = ysize * (int) (sizeX/sizeY);
    } else {
      PetscFunctionReturn(0);
    }
  } else {
    xsize = 300;
    if (sizeX > 0.0) {
      ysize = xsize * (int) (sizeY/sizeX);
    } else {
      PetscFunctionReturn(0);
    }
  }
  if ((xsize == 0) || (ysize == 0)) PetscFunctionReturn(0);
  offX = 0.02*sizeX;
  offY = 0.02*sizeY;
  ierr = PetscDrawResizeWindow(draw, xsize, ysize);                                                       CHKERRQ(ierr);
  ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);             CHKERRQ(ierr);
  ierr = MeshView_Triangular_1D_Draw_Mesh(mesh, draw);                                                    CHKERRQ(ierr);
  ierr = PetscDrawGetPause(draw, &pause);                                                                 CHKERRQ(ierr);
  if (pause >= 0) {
    PetscSleep(pause);
    PetscFunctionReturn(0);
  }

  /* Allow the mesh to zoom or shrink */
  ierr = PetscDrawCheckResizedWindow(draw);                                                               CHKERRQ(ierr);
  ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                              CHKERRQ(ierr);
  while (button != BUTTON_RIGHT) {
    ierr = PetscDrawSynchronizedClear(draw);                                                              CHKERRQ(ierr);
    switch (button)
    {
    case BUTTON_LEFT:
      scale = 0.5;
      break;
    case BUTTON_CENTER:
      scale = 2.0;
      break;
    case BUTTON_LEFT_SHIFT:
      scale = 1.0;
      ierr = MeshLocatePoint(mesh, xc, yc, 0.0, &hElement);                                               CHKERRQ(ierr);
      ierr = MeshSetHighlightElement(mesh, hElement);                                                     CHKERRQ(ierr);
      break;
    default:
      scale = 1.0;
      break;
    }
    if (scale != 1.0) {
      startX = scale*(startX + sizeX - xc) + xc - sizeX*scale;
      endX   = scale*(endX   - sizeX - xc) + xc + sizeX*scale;
      startY = scale*(startY + sizeY - yc) + yc - sizeY*scale;
      endY   = scale*(endY   - sizeY - yc) + yc + sizeY*scale;
      sizeX *= scale;
      sizeY *= scale;
      offX  *= scale;
      offY  *= scale;
    }

    ierr = MeshView_DrawStatusLine_Private(mesh, draw, startX, startY, endX, endY, offX, offY);           CHKERRQ(ierr);
    ierr = MeshView_Triangular_1D_Draw_Mesh(mesh, draw);                                                  CHKERRQ(ierr);
    ierr = PetscDrawCheckResizedWindow(draw);                                                             CHKERRQ(ierr);
    ierr = PetscDrawSynchronizedGetMouseButton(draw, &button, &xc, &yc, 0, 0);                            CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "MeshView_Triangular_1D"
int MeshView_Triangular_1D(Mesh mesh, PetscViewer viewer) {
  PetscTruth isascii, isdraw;
  int        ierr;

  PetscFunctionBegin;
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_ASCII, &isascii);                            CHKERRQ(ierr);
  ierr = PetscTypeCompare((PetscObject) viewer, PETSC_VIEWER_DRAW,  &isdraw);                             CHKERRQ(ierr);
  if (isascii == PETSC_TRUE) {
    ierr = MeshView_Triangular_1D_File(mesh, viewer);                                                     CHKERRQ(ierr);
  } else if (isdraw == PETSC_TRUE) {
    ierr = MeshView_Triangular_1D_Draw(mesh, viewer);                                                     CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}
