/*
 * filterdiff - extract (or exclude) a diff from a diff file
 * lsdiff - show which files are modified by a patch
 * grepdiff - show files modified by a patch containing a regexp
 * Copyright (C) 2001, 2002 Tim Waugh <twaugh@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_ERROR_H
# include <error.h>
#endif /* HAVE_ERROR_H */
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h> // for ssize_t
#endif /* HAVE_SYS_TYPES_H */
#include <fnmatch.h>
#include <getopt.h>
#include <locale.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "util.h"
#include "diff.h"

static struct patlist *pat_include = NULL;
static struct patlist *pat_exclude = NULL;

static int unzip = 0;
static enum {
	mode_filter,
	mode_list,
	mode_grep,
} mode;
static regex_t regex;
static int numbering = 0;
static int ignore_components = 0;
static int show_status = 0;

static int file_exists (const char *timestamp)
{
	struct tm t;
	time_t s;

	if (!strptime (timestamp, "%a %b %e %T %Y", &t))
		return 1;

	s = mktime (&t);

	/* If s is negative, or less that fifteen hours into 1970,
	 * and it's an exact multiple of 15 minutes, it's very likely
	 * to be the result of ctime(&zero). */
	if (s < (15 * 60 * 60) && (s % (15 * 60)) == 0)
		return 0;

	/* Otherwise, it's a real file timestamp. */
	return 1;
}

static const char *stripped (const char *name)
{
	const char *basename;
	int i = 0;
	if (!strncmp (name, "./", 2))
		name += 2;
	basename = strrchr (name, '/');
	if (!basename)
		basename = name;
	while (i < ignore_components &&
	       (name = strchr (name, '/')) != NULL) {
		while (*name == '/')
			name++;
		i++;
	}
	return name ? name : basename;
}

static void display_filename (unsigned long linenum, char status,
			      const char *filename)
{
	if (numbering)
		printf ("%6lu  ", linenum);
	if (show_status)
		printf ("%c ", status);
	puts (filename);
} 

static int do_unified (FILE *f, int match, char **line,
		       ssize_t *linelen, unsigned long *linenum,
		       unsigned long start_linenum, char status,
		       const char *bestname)
{
	/* Skip hunk. */
	unsigned long orig_lines, new_lines;

	if (strncmp (*line, "@@ ", 3))
		return 1;

	orig_lines = orig_num_lines (*line);
	new_lines = new_num_lines (*line);

	for (;;) {
		if (getline (line, linelen, f) == -1)
			return EOF;
		++*linenum;

		if (!orig_lines && !new_lines && **line != '\\') {
			if (strncmp (*line, "@@ ", 3))
				break;

			/* Next chunk. */
			orig_lines = orig_num_lines (*line);
			new_lines = new_num_lines (*line);

			if (match && mode == mode_filter)
				fputs (*line, stdout);

			continue;
		}

		if (orig_lines && **line != '+')
			orig_lines--;
		if (new_lines && **line != '-')
			new_lines--;

		if (match && mode == mode_filter)
			fputs (*line, stdout);

		if (match && mode == mode_grep &&
		    !regexec (&regex, *line + 1, 0, NULL, 0)) {
			display_filename (start_linenum,
					  status,
					  stripped (bestname));
			// Prevent further matching.
			match = 0;
		}
	}

	return 0;
}

static int do_context (FILE *f, int match, char **line,
		       ssize_t *linelen, unsigned long *linenum,
		       unsigned long start_linenum, char status,
		       const char *bestname)
{
	/* Skip before and after segments. */
	unsigned long line_start, line_end, line_count;
	char *n, *end;
	int i;
	int to_omitted;

	/* Context diff hunks are like this:
	 *
	 * ***************
	 * *** start[,end] ****
	 *   from lines... (omitted if there are only insertions)
	 * --- start[,end] ----
	 *   to lines... (omitted if there are only deletions)
	 */

	for (;;) {
		if (strcmp (*line, "***************\n"))
			return 1;

		to_omitted = 1; // until we know that to lines may not be

		for (i = 0; i < 2; i++) {
			int first = 1;

			if (getline (line, linelen, f) == -1)
				return EOF;

			++*linenum;
			if (match && mode == mode_filter)
				fputs (*line, stdout);

			if (strncmp (*line, i ? "--- " : "*** ", 4))
				return 1;

		do_line_counts:
			n = *line + 4;
			line_start = strtoul (n, &end, 10);
			if (n == end)
				return 1;

			if (*end == ',') {
				n = end + 1;
				line_end = strtoul (n, &end, 10);
				if (n == end)
					return 1;

				if (line_start > line_end)
					return 1;

				line_count = line_end - line_start + 1;
			} else {
				if (line_start)
					line_count = 1;
				else {
					to_omitted = 0;
					line_count = 0;
				}
			}

			if (i && to_omitted)
				break;

			while (line_count--) {
				if (getline (line, linelen, f) == -1)
					return EOF;

				++*linenum;
				if (match && mode == mode_filter)
					fputs (*line, stdout);

				if (match && mode == mode_grep &&
				    !regexec (&regex, *line + 2,
					      0, NULL, 0)) {
					display_filename (start_linenum,
							  status,
							  stripped (bestname));
					// Prevent further matching.
					match = 0;
				}

				if (!i && first) {
					first = 0;
					if (!strncmp (*line, "--- ", 4)) {
						/* From lines were
						 * omitted. */
						i++;
						to_omitted = 0;
						goto do_line_counts;
					}
				}

				if (!i && **line == '!')
					/* To lines may not be omitted. */
					to_omitted = 0;
			}
		}

		if (getline (line, linelen, f) == -1)
			return EOF;
	}

	return 0;
}

static int filterdiff (FILE *f)
{
	static unsigned long linenum = 0;
	char *names[2];
	char *origline = NULL;
	char *line = NULL;
	size_t linelen = 0;
	char *p;
	int match;

	for (;;) {
		int h;
		char status = '!';
		unsigned long start_linenum;
		int orig_file_exists;
		int is_context = 0;
		int (*do_diff) (FILE *, int, char **, ssize_t *,
				unsigned long *, unsigned long,
				char, const char *);

		orig_file_exists = 0; // shut gcc up

		// Search for start of patch ("--- " for unified diff,
		// "*** " for context).
		for (;;) {
			if (getline (&line, &linelen, f) == -1)
				goto eof;
			linenum++;

		check_orig:
			if (!strncmp (line, "--- ", 4)) {
				is_context = 0;
				break;
			}

			if (!strncmp (line, "*** ", 4)) {
				is_context = 1;
				break;
			}

			/* Show non-diff lines if excluding. */
			if (pat_exclude)
				fputs (line, stdout);
		}

		start_linenum = linenum;
		origline = xstrdup (line);
		h = strcspn (line + 4, "\t\n");
		names[0] = xstrndup (line + 4, h);
		h = 4 + h + strspn (line + 4 + h, "\t\n");
		if (mode != mode_filter && show_status)
			orig_file_exists = file_exists (line + h);

		if (getline (&line, &linelen, f) == -1) {
			free (names[0]);
			goto eof;
		}
		linenum++;

		if (strncmp (line, is_context ? "--- " : "+++ ", 4)) {
			free (names[0]);
			goto check_orig;
		}

		h = strcspn (line + 4, "\t\n");
		names[1] = xstrndup (line + 4, h);

		// Decide whether this matches this pattern.
		p = best_name (2, names);
		
		match = !patlist_match(pat_exclude, p);
		if (match && pat_include != NULL)
			match = patlist_match(pat_include, p);
		
		// print if it matches.
		if (match) {
			if (mode != mode_filter) {
				h = 4 + h + strspn (line + 4 + h, " \t\n");
				if (!orig_file_exists)
					status = '+';
				else if (!file_exists (line + h))
					status = '-';
			}

			if (mode == mode_list)
				display_filename (start_linenum, status,
						  stripped (p));
			else if (mode == mode_filter) {
				fputs (origline, stdout);
				fputs (line, stdout);
			}
		}

		free (origline);
		origline = NULL;

		if (getline (&line, &linelen, f) == -1) {
			free (names[0]);
			free (names[1]);
			goto eof;
		}

		linenum++;
		if (match && mode == mode_filter)
			fputs (line, stdout);

		if (is_context)
			do_diff = do_context;
		else
			do_diff = do_unified;

		switch (do_diff (f, match, &line,
				 &linelen, &linenum,
				 start_linenum, status, p)) {
		case EOF:
			free (names[0]);
			free (names[1]);
			goto eof;
		case 1:
			goto next_diff;
		}

	next_diff:
		free (names[0]);
		free (names[1]);
		goto check_orig;
	}

 eof:
	if (line)
		free (line);

	if (origline)
		free (origline);

	return 0;
}

static char * syntax_str =
"Options:\n"
"  -x PAT    exclude files matching PAT\n"
"  -i PAT    include only files matching PAT\n"
"  -z        decompress .gz and .bz2 files\n"
"  -n        show line numbers (lsdiff)\n"
"  -p N      pathname components to ignore (lsdiff)\n"
"  -s        show file additions and removals (lsdiff)\n"
"  --filter  run as 'filterdiff'\n"
"  --list    run as 'lsdiff'\n"
"  --grep    run as 'grepdiff'\n"
;

NORETURN
static void syntax (int err)
{
	const char *usage = "usage: %s [OPTION]... [files ...]\n";
	if (mode == mode_grep)
		usage = "usage: %s [OPTION]... REGEX [files ...]\n";
	fprintf (err ? stderr : stdout, usage, progname);
	fprintf (err ? stderr : stdout, syntax_str);
	exit (err);
}

static void set_list (void)
{
	/* This is lsdiff. */
	set_progname ("lsdiff");
	mode = mode_list;
}

static void set_filter (void)
{
	/* This is filterdiff. */
	set_progname ("filterdiff");
	mode = mode_filter;
}

static void set_grep (void)
{
	/* This is grepdiff. */
	set_progname ("grepdiff");
	mode = mode_grep;
}

static void determine_mode_from_name (const char *argv0)
{
	/* This is filterdiff, unless it is named 'lsdiff' or 'grepdiff'. */
	const char *p = strrchr (argv0, '/');
	if (!p++)
		p = argv0;
	if (strstr (p, "lsdiff"))
		set_list ();
	else if (strstr (p, "grepdiff"))
		set_grep ();
	else
		set_filter ();
}

int main (int argc, char *argv[])
{
	int i;
	FILE *f = stdin;

	setlocale (LC_TIME, "C");
	determine_mode_from_name (argv[0]);
	while (1) {
		static struct option long_options[] = {
	       		{"help", 0, 0, 'h'},
			{"version", 0, 0, 'v'},
			{"list", 0, 0, 'l'},
			{"filter", 0, 0, 'f'},
			{"grep", 0, 0, 'g'},
			{0, 0, 0, 0}
		};
		char *end;
		int c = getopt_long (argc, argv, "vhp:i:I:x:X:zns",
				long_options, NULL);
		if (c == -1)
			break;
		
		switch (c) {
		case 'g':
			set_grep ();
			break;
		case 'f':
			set_filter ();
			break;
		case 'l':
			set_list ();
			break;
		case 'v':
			printf("%s - patchutils version %s\n", progname,
			       VERSION);
			exit(0);
		case 'h':
			syntax (0);
			break;
		case 'p':
			ignore_components = strtoul (optarg, &end, 0);
			if (optarg == end)
				syntax (1);
			break;
		case 'x':
			patlist_add (&pat_exclude, optarg);
			break;
		case 'i':
			patlist_add (&pat_include, optarg);
			break;
		case 'z':
			unzip = 1;
			break;
		case 'n':
			numbering = 1;
			break;
		case 's':
			show_status = 1;
			break;
		default:
			syntax(1);
		}
				
	}

	if (mode == mode_grep) {
		int err;

		if (optind == argc)
			syntax (1);

		err = regcomp (&regex, argv[optind++], REG_NOSUB);
		if (err) {
			char errstr[300];
			regerror (err, &regex, errstr, sizeof (errstr));
			error (EXIT_FAILURE, 0, errstr);
			exit (1);
		}
	}
	
	if (optind == argc) {
		filterdiff(stdin);
	} else {
		for (i = optind; i < argc; i++) {
			if (unzip) {
				f = xopen_unzip (argv[i], "rb");
			} else {
				f = xopen(argv[i], "rb");
			}
			filterdiff(f);
			fclose(f);
		}
	}

	return 0;
}

