/*
**  SMTP.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#import <Pantomime/SMTP.h>

#import <Pantomime/Constants.h>
#import <Pantomime/GSMD5.h>
#import <Pantomime/Message.h>
#import <Pantomime/TCPConnection.h>

#define CR '\r'
#undef LF
#define LF '\n'

@implementation SMTP

//
//
//
- (id) initWithName: (NSString *) theName
               port: (int) thePort
{
  self = [super init];

  supportedAuthenticationMechanisms = [[NSMutableArray alloc] init];

  [self setName: theName];
  [self setPort: thePort];

  tcpConnection = [[TCPConnection alloc] initWithName: [self name]
					 port: thePort];


  if ( !tcpConnection )
    {
      AUTORELEASE(self);
      return nil;
    }
  
  NS_DURING
    {
      if ( ![self _responseFromServerIsEqualToCode: @"220"
		  verifyAuthenticationMechanism: NO] )
	{
	  AUTORELEASE(self);
	  NS_VALUERETURN(nil, id);
	}
      
      [[self tcpConnection] writeLine: @"EHLO localhost.localdomain"];
      
      if ( ![self _responseFromServerIsEqualToCode: @"250"
		  verifyAuthenticationMechanism: YES] )
	{
	  NSLog(@"SMTP: The server doesn't support the extended SMTP service.");
	  [[self tcpConnection] writeLine: @"HELO localhost.localdomain"];
	  
	  if ( ![self _responseFromServerIsEqualToCode: @"250"
		      verifyAuthenticationMechanism: NO] )
	    {
	      AUTORELEASE(self);
	      NS_VALUERETURN(nil, id);
	    }
	}
    }
  NS_HANDLER
    {
      NSLog(@"SMTP: An error occured while establishing the connection with the SMTP server. Aborting.");
      AUTORELEASE(self);
      NS_VALUERETURN(nil, id);
    }
  NS_ENDHANDLER

  return self;
}


//
//
//
- (id) initWithName: (NSString *) theName
{
  return [self initWithName: theName
	       port: 25];
}


//
//
//
- (void) dealloc
{
  RELEASE(supportedAuthenticationMechanisms);
  RELEASE(name);

  TEST_RELEASE(tcpConnection);
  
  [super dealloc];
}


//
//
//
- (BOOL) authenticateWithUsername: (NSString *) username
                         password: (NSString *) password
{
  return [self authenticateWithUsername: username
	       password: password
	       mechanism: nil];
}

//
// This method is used to authenticate ourself to the SMTP server.
//
- (BOOL) authenticateWithUsername: (NSString *) username
                         password: (NSString *) password
			mechanism: (NSString *) mechanism
{
 
  // If the mechanism is nil, we go from the 'best' one to the 'worst' one
  if ( ! mechanism )
    {
      NSLog(@"SMTP authentication mechanism is nil - FIXME!");
    }
  else if ( [mechanism caseInsensitiveCompare: @"PLAIN"] == NSOrderedSame )
    {
      return [self _plainAuthenticationUsingUsername: username
		   password: password];
    }
  else if ( [mechanism caseInsensitiveCompare: @"LOGIN"] == NSOrderedSame )
    {
      return [self _loginAuthenticationUsingUsername: username
		   password: password];
    }
  else if ( [mechanism caseInsensitiveCompare: @"CRAM-MD5"] == NSOrderedSame )
    {
      return [self _cramMD5AuthenticationUsingUsername: username
		   password: password];
    }
  
  NSLog(@"Unsupported SMTP authentication method.");
  
  return NO;
}		  

//
//
//
- (NSString *) name
{
  return name;
}


//
//
//
- (void) setName: (NSString *) theName
{
  RETAIN(theName);
  RELEASE(name);
  name = theName;
}


- (int) port
{
  return port;
}

- (void) setPort: (int) thePort
{
  port = thePort;
}

//
//
//
- (TCPConnection *) tcpConnection
{
  return tcpConnection;
}


//
//
//
- (BOOL) sendMessage: (Message *) theMessage
{
  return [self sendMessage: theMessage
	       withRawSource: [theMessage dataUsingSendingMode: SEND_USING_SMTP]];
}


//
// The message received has \n for each lines, we must put \r\n 
// at the end of each line instead.
//
- (BOOL) sendMessageFromRawSource: (NSData *) theData
{
  Message *aMessage;
  BOOL aBOOL;

  aMessage = [[Message alloc] initWithData: theData];

  // FIXME
  // We must be careful about \n -> \r\n.. We should do more testing
  // with various SMTP server.
  aBOOL = [self sendMessage: aMessage 
		withRawSource: theData];

  DESTROY(aMessage);
  
  return aBOOL;
}


//
//
//
- (void) close
{
  [[self tcpConnection] writeLine: @"QUIT"];

  if ( ![self _responseFromServerIsEqualToCode: @"221"
	      verifyAuthenticationMechanism: NO] )
    {
      NSLog(@"SMTP: An error occured while ending the conversation with the SMTP server.");
      [[self tcpConnection] close];
    }
}

//
//
//
- (NSArray *) supportedAuthenticationMechanisms
{
  return [NSArray arrayWithArray: supportedAuthenticationMechanisms];
}

@end


//
// Private methods
//
@implementation SMTP (Private)


//
// This method receives a string like: AUTH PLAIN LOGIN
//                                     AUTH=PLAIN LOGIN X-NETSCAPE-HAS-BUGS                              
// It decode the string to build a list of supported authentication mechanisms.
//
- (void) _decodeSupportedAuthenticationMechanismFromString: (NSString *) theString
{
  NSRange aRange;
  
  // If we already have elements in our supportedAuthenticationMechanisms, that means
  // we already decoded an AUTH parameter. Let's not decode again.
  if ( [supportedAuthenticationMechanisms count] > 0 )
    {
      return;
    }

  aRange = [theString rangeOfString: @"AUTH"
		      options: NSCaseInsensitiveSearch];
  
  if ( aRange.length )
    {
      NSString *aString;

      // We trim the AUTH (and the ' ' or '=' following it)
      aString = [theString substringFromIndex: (aRange.location+aRange.length) + 1]; 
      
      // We trim our \r\n
      aString = [aString substringToIndex: ([aString length] - 2)];
      
      [supportedAuthenticationMechanisms addObjectsFromArray: [aString componentsSeparatedByString: @" "]];
    }
}

//
//
//
- (BOOL) _responseFromServerIsEqualToCode: (NSString *) theCode
	    verifyAuthenticationMechanism: (BOOL) aBOOL
{
  char prev, c[2], buf[4];
  BOOL multiLine;
  
  NSString *aString;

  multiLine = YES;
  
  while ( multiLine == YES )
    {
      // We read the code that our SMTP server sent us back
      aString = [[self tcpConnection] readStringOfLength: 3];
      
      if ( !aString )
	{
	  NSLog(@"SMTP: Error on reading the code.");
          return NO;
	}
      
      // We get the cString of our string in our buffer buf
      [aString getCString: buf];
      
      // We verify if our response is multiline or not ?
      aString = [[self tcpConnection] readStringOfLength: 1];

      if ( !aString )
	{
	  NSLog(@"SMTP: Error on reading to see if we got a multiline response or not.");
	  return NO;
	}

      // We get the cString of our string in our buffer c
      [aString getCString: c];
      
      // We verify if we have a multiline response from our SMTP server.
      if ( c[0] == '-' )
        {
          multiLine = YES;
        }
      else 
        {
          multiLine = NO;
        }
      
      // We read the response to the end
      do
        {
          prev = c[0];
	  
	  aString = [[self tcpConnection] readLine];
     
	  // If we MUST verify for the supported authentication mechanism
	  if ( aBOOL )
	    {
	      [self _decodeSupportedAuthenticationMechanismFromString: aString];
	    }
	  
	  c[0] = [aString characterAtIndex: ([aString length] - 1)];
	 

//  	  aString = [[self tcpConnection] readStringOfLength: 1];
	  
//  	  if ( !aString )
//  	    {
//  	      NSLog(@"SMTP: Error while reading to the end of the multiline response.");
//  	      return NO;
//  	    }

//  	  // We get the cString of our string in our buffer c
//  	  [aString getCString: c];
        }
      while ( (prev != CR) && (c[0] != LF) );
    }

  if ( ! [theCode isEqualToString: [NSString stringWithCString: buf]] )
    {
      return NO;
    }

  return YES;
}


//
//
//
- (BOOL) sendMessage: (Message *) theMessage
       withRawSource: (NSData *) theRawSource
{
  BOOL isBouncedMessage;

  if ( ! theMessage )
    {
      return NO;
    }
  
  // We first verify if it's a bounced message
  if ( [theMessage resentFrom ])
    {
      isBouncedMessage = YES;
      [[self tcpConnection] writeLine: [NSString stringWithFormat: @"MAIL FROM:<%@>", 
						 [[theMessage resentFrom] address]] ];
    }
  else
    {
      isBouncedMessage = NO;
      [[self tcpConnection] writeLine: [NSString stringWithFormat: @"MAIL FROM:<%@>", 
						 [[theMessage from] address]] ];
    }

  // We verify the response from our server from the MAIL FROM command
  if ( ![self _responseFromServerIsEqualToCode: @"250"
	      verifyAuthenticationMechanism: NO] )
    {
      return NO;
    }

  if (! [self writeRecipients: [theMessage recipients] 
	      usingBouncingMode: isBouncedMessage] )
    {
      return NO;
    }

  if ( [self writeMessageFromRawSource: theRawSource] )
    {
      if ( ![self _responseFromServerIsEqualToCode: @"250"
		  verifyAuthenticationMechanism: NO] )
	{
	  return NO;
	}
    }
  else
    {
      return NO;
    }

  return YES;
}

//
//
//
- (BOOL) writeRecipients: (NSArray *) recipients
       usingBouncingMode: (BOOL) aBOOL;
{
  NSEnumerator *recipientsEnumerator;
  InternetAddress *theAddress;
  NSString *aString;
  
  recipientsEnumerator = [recipients objectEnumerator];
  
  // We verify if we have at least one recipient
  if (!recipients || [recipients count] == 0)
    {
      NSLog(@"SMTP: No recipients were found, aborting.");
      return NO;
    }

  while ( (theAddress = [recipientsEnumerator nextObject]) )
    {
      // If it's a bounced message...
      if ( aBOOL )
	{
	  // We only get the bounced recipients
	  if ( [theAddress type] > 3 )
	    {
	      aString = [NSString stringWithFormat:@"RCPT TO:<%@>", [theAddress address]];
	    }
	  else
	    {
	      aString = nil;
	    }
	}
      else
	{
	  // Otherwise, we get the real recipients
	  if ( [theAddress type] < 4 )
	    {
	      aString = [NSString stringWithFormat:@"RCPT TO:<%@>", [theAddress address]];
	    }
	  else
	    {
	      aString = nil;
	    }
	}
      
      // If we have a recipient to write, let's write the string to the socket!
      if ( aString )
	{
	  [[self tcpConnection] writeLine: aString];

	  // We verify if the server accepted this recipient.
	  if (! [self _responseFromServerIsEqualToCode: @"250"
		      verifyAuthenticationMechanism: NO] )
	    {
	      return NO;
	    }
	}
    }

  return YES;
}


//
//
//
- (BOOL) writeMessageFromRawSource: (NSData *) theRawSource
{  
  [[self tcpConnection] writeLine: @"DATA"];
  
  if (! [self _responseFromServerIsEqualToCode: @"354"
	      verifyAuthenticationMechanism: NO] )
    {
      NSLog(@"SMTP: an error occured while writing the DATA command, we abort.");
      return NO;
    }

  [[self tcpConnection] writeData: theRawSource];
  [[self tcpConnection] writeString: @"\r\n.\r\n"];
  
  return YES;
}


//
// PLAIN authentication mechanism (RFC2595)
//
- (BOOL) _plainAuthenticationUsingUsername: (NSString *) username
                                  password: (NSString *) password
{  
  [[self tcpConnection] writeLine: @"AUTH PLAIN"];
  
  if ( [self _responseFromServerIsEqualToCode: @"334"
	     verifyAuthenticationMechanism: NO] )
    {
      NSMutableData *aMutableData;
      NSString *aString;
      
      int len_username, len_password;
      
      len_username = [username length];
  
      if (! password )
	{
	  len_password = 0;
	}
      else
	{
	  len_password = [password length];
	}
      
      // We create our phrase
      aMutableData = [NSMutableData dataWithLength: (len_username + len_password + 2)];
      
      [aMutableData replaceBytesInRange: NSMakeRange(1,len_username)
		    withBytes: [[username dataUsingEncoding: NSASCIIStringEncoding] bytes]];
      
      
      [aMutableData replaceBytesInRange: NSMakeRange(2 + len_username, len_password)
		    withBytes: [[password dataUsingEncoding: NSASCIIStringEncoding] bytes]];
      
      aString = [[NSString alloc] initWithData: [MimeUtility encodeBase64: aMutableData
							     lineLength: 0]
				  encoding: NSASCIIStringEncoding];
      
      [[self tcpConnection] writeLine: aString];
      RELEASE(aString);
      
      if ( [self _responseFromServerIsEqualToCode: @"235"
		 verifyAuthenticationMechanism: NO] )
	{
	  NSLog(@"PLAIN Authentication successful");
	  return YES;
	}
    }

  return NO;
}


//
// LOGIN authentication mechanism (undocumented but easy to figure out)
//
- (BOOL) _loginAuthenticationUsingUsername: (NSString *) username
                                  password: (NSString *) password
{
  NSString *aString;
  
  [[self tcpConnection] writeLine: @"AUTH LOGIN"];

  aString = [[self tcpConnection] readLine];
 
  if ( [aString hasPrefix: @"334"] )
    {
      NSString *un, *pw;

      un = [[NSString alloc] initWithData: [MimeUtility encodeBase64: [username dataUsingEncoding: NSASCIIStringEncoding]
							lineLength: 0]
			     encoding: NSASCIIStringEncoding];
      
      [[self tcpConnection] writeLine: un];
      RELEASE(un);
      
      aString = [[self tcpConnection] readLine];

      if ( [aString hasPrefix: @"334"] )
	{
	  pw = [[NSString alloc] initWithData: [MimeUtility encodeBase64: [password dataUsingEncoding: NSASCIIStringEncoding]
							    lineLength: 0]
				 encoding: NSASCIIStringEncoding];
	  
	  [[self tcpConnection] writeLine: pw];
	  RELEASE(pw);
	  
	  if ( [self _responseFromServerIsEqualToCode: @"235"
		     verifyAuthenticationMechanism: NO] )
	    {
	      NSLog(@"LOGIN Authentication successful");
	      return YES;
	    }
	}
    }
  
  return NO;
}


//
// CRAM-MD5 authentication mechanism (2195)
//
- (BOOL) _cramMD5AuthenticationUsingUsername: (NSString *) username
                                    password: (NSString *) password
{
  NSString *aString;
  
  [[self tcpConnection] writeLine: @"AUTH CRAM-MD5"];
  
  aString = [[self tcpConnection] readLine];
  
  if ( [aString hasPrefix: @"334"] )
    {
      GSMD5 *md5;
      
      // We trim the "334 " and we keep the challenge phrase
      aString = [aString substringFromIndex: 4];
      
      // We trim our \r\n
      aString = [aString substringToIndex: ([aString length] - 2)];
      
      aString = [[NSString alloc] initWithData: [MimeUtility decodeBase64: [aString dataUsingEncoding: NSASCIIStringEncoding]]
    				  encoding: NSASCIIStringEncoding];;
      
      
      md5 = [[GSMD5 alloc] init];
      [md5 updateWithString: [NSString stringWithFormat: @"%@%@", password, aString]
    	   usingEncoding: NSASCIIStringEncoding];
      [md5 calculateDigest];
      RELEASE(aString);
     
      aString = DataToHexString( [md5 digest] ); 
      aString = [[NSString alloc] initWithData: [MimeUtility encodeBase64: [aString dataUsingEncoding: NSASCIIStringEncoding]
    							     lineLength: 0]
    				  encoding: NSASCIIStringEncoding];
      RELEASE(md5);

      [[self tcpConnection] writeLine: aString];
      RELEASE(aString);

      if ( [self _responseFromServerIsEqualToCode: @"235"
		 verifyAuthenticationMechanism: NO] )
	{
	  NSLog(@"CRAM-MD5 Authentication successful");
	  return YES;
	}
    }

  return NO;
}

@end
