unit Grt;

interface

uses
  gnugettext,
{$IFDEF GrtForms}
  XGrtForms,
{$ENDIF}
  Windows, SysUtils, Classes, TntClasses, Forms, TntSysUtils,
  myx_public_interface,
  myx_grt_public_interface, myx_grt_builtin_module_public_interface,
{$IFDEF GrtGc}
  myx_grt_gc_public_interface,
{$ENDIF}
  UniCodeConsole, SyncObjs, AuxFuncs;


type
  TGrt = class;
  TGrtEngine = class;

  GrtLoaderState = (
    GrtLsNotInitialized,
    GrtLsInitialized,
    GrtLsInitializeFailed,
    GrtLsModulesLoaded
  );

  TGrtProcessOutput = procedure(Text: WideString) of object;
  TGrtProcessMessages = function(Messages: TMYX_GRT_MSGS): integer of object;
  TGrtOnShutDown = procedure(Grt: TGrt) of object;


  TGrt = class
  protected
    GrtEngineInitializedEvent: TEvent;

    function GetJavaLoaderState: GrtLoaderState;
    function GetLuaLoaderState: GrtLoaderState;
    function GetBuiltinLoaderState: GrtLoaderState;
    function GetPhpLoaderState: GrtLoaderState;
    function GetUiLoaderState: GrtLoaderState;
    function GetGcLoaderState: GrtLoaderState;

    procedure OutputCommandLine(Text: WideString);

    function GrtPGrt: Pointer;

    function GetGlobal(const Path: WideString): Pointer;
    procedure SetGlobal(const Path: WideString; NewValue: Pointer);
    function GetGlobalAsString(const Path: WideString): WideString;
    procedure SetGlobalAsString(const Path: WideString; NewValue: WideString);
    function GetGlobalAsInt(const Path: WideString): Integer;
    procedure SetGlobalAsInt(const Path: WideString; NewValue: Integer);
    function GetGlobalAsReal(const Path: WideString): Double;
    procedure SetGlobalAsReal(const Path: WideString; NewValue: Double);

    function GetListItem(const List: Pointer; const I: Integer): Pointer;
    procedure SetListItem(const List: Pointer; const I: Integer; NewValue: Pointer);
    function GetListString(const List: Pointer; const I: Integer): WideString;
    procedure SetListString(const List: Pointer; const I: Integer; NewValue: WideString);

    function GetDictItem(const Dict: Pointer; const Key: WideString): Pointer;
    procedure SetDictItem(const Dict: Pointer; const Key: WideString; NewValue: Pointer);
    function GetDictString(const Dict: Pointer; const Key: WideString): WideString;
    procedure SetDictString(const Dict: Pointer; const Key: WideString; NewValue: WideString);
    function GetDictInt(const Dict: Pointer; const Key: WideString): Integer;
    procedure SetDictInt(const Dict: Pointer; const Key: WideString; NewValue: Integer);
    function GetDictReal(const Dict: Pointer; const Key: WideString): Double;
    procedure SetDictReal(const Dict: Pointer; const Key: WideString; NewValue: Double);
  private
    FGrtEngine: TGrtEngine;

    FOnShutDown: TGrtOnShutDown;

    FStructIconsList: TTntStringList;

    FGrtFormsFuncNameCache: TStringList;

{$IFDEF GrtForms}
    FGrtForms: TXGrtFormApplication;
{$ENDIF}

    FConsole: TUniCodeConsole;

    FVerbose: Boolean;
    FRemoteDebug: Boolean;
    FJvmLibrary: WideString;
  public
    // Initialization
    constructor Create(Console: TUniCodeConsole;
      ScriptFilename: WideString; Verbose: Boolean = False;
      RemoteDebug: Boolean = False; JvmLibrary: WideString = '');
    destructor Destroy; override;

    property PGrt: Pointer read GrtPGrt;
    property Console: TUniCodeConsole read FConsole;

    property OnShutDown: TGrtOnShutDown read FOnShutDown write FOnShutDown;

    // Options
    property Verbose: Boolean read FVerbose write FVerbose;
    property RemoteDebug: Boolean read FRemoteDebug write FRemoteDebug;
    property JvmLibrary: WideString read FJvmLibrary write FJvmLibrary;

    // Loaders
    property JavaLoaderState: GrtLoaderState read GetJavaLoaderState;
    property LuaLoaderState: GrtLoaderState read GetLuaLoaderState;
    property BuiltinLoaderState: GrtLoaderState read GetBuiltinLoaderState;
    property PhpLoaderState: GrtLoaderState read GetPhpLoaderState;
    property UiLoaderState: GrtLoaderState read GetUiLoaderState;
    property GcLoaderState: GrtLoaderState read GetGcLoaderState;

    // Access values
    property Global[const Path: WideString]: Pointer read GetGlobal write SetGlobal;
    property GlobalAsString[const Path: WideString]: WideString read GetGlobalAsString write SetGlobalAsString;
    property GlobalAsInt[const Path: WideString]: Integer read GetGlobalAsInt write SetGlobalAsInt;
    property GlobalAsReal[const Path: WideString]: Double read GetGlobalAsReal write SetGlobalAsReal;

    function ValueDuplicate(Value: Pointer): Pointer;

    // Access lists
    property ListItem[const List: Pointer; const I: Integer]: Pointer read GetListItem write SetListItem;
    property ListString[const List: Pointer; const I: Integer]: WideString read GetListString write SetListString;

    function ListCount(List: Pointer): Integer;
    procedure ListAdd(List: Pointer; Value: Pointer; IncreaseRefCount: Boolean = True);
    function ListDel(List: Pointer; Index: Integer): Boolean;

    // Access dicts
    property DictItem[const Dict: Pointer; const Key: WideString]: Pointer read GetDictItem write SetDictItem;
    property DictString[const Dict: Pointer; const Key: WideString]: WideString read GetDictString write SetDictString;
    property DictInt[const Dict: Pointer; const Key: WideString]: Integer read GetDictInt write SetDictInt;
    property DictReal[const Dict: Pointer; const Key: WideString]: Double read GetDictReal write SetDictReal;

    // Function
    function ExecuteModalFunction(ModulName: WideString;
      FunctionName: WideString;
      FunctionArguments: array of const;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      ProcessMessagesFunction: TGrtProcessMessages = nil;
      AllowNullAsResult: Boolean = False;
      SearchParent: Boolean = True;
      TimeOutMS: Integer = -1): Pointer;
    function ExecuteModalShellFunction(Cmd: WideString;
      ProcessOutputFunction: TGrtProcessOutput = nil;
      TimeOutMS: Integer = -1): MYX_GRT_SHELL_COMMAND;

    // Caches
    property StructIconsList: TTntStringList read FStructIconsList;
    property GrtFormsFuncNameCache: TStringList read FGrtFormsFuncNameCache;

    // Helper functions
    function BuildGrtParamList(Params: array of const): Pointer;
    function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;
  end;

  TGrtEngine = class(TThread)
    constructor Create(CreateSuspended: Boolean; Grt: TGrt);
    destructor Destroy; override;
  protected
    procedure Execute; override;

    procedure InitializeGrt;

    function GetPGrt: Pointer;

    function GetExecutionFinished: Boolean;
    function GetExecutionTime: TDateTime;

    procedure DoOutputText;
    procedure DoProcessMessages;

    function InitDelphiLoader(Error: PMYX_GRT_ERROR): Pointer;
    function AddDelphiModules(Loader: Pointer): Integer;
  private
    FGrt: TGrt;
    FPGrt: Pointer;

    FFunctionSynchronizer,
      FVclDataSynchronizer: TCriticalSection;
    FFunctionStartedEvent,
      FFunctionFinishedEvent: TEvent;
    FExecutionFinished: Boolean;
    FModulName,
      FFunctionName: WideString;
    FSearchParent: Boolean;
    FFunctionArgument: Pointer;
    FPError: PMYX_GRT_ERROR;
    FResult: Pointer;
    FShellCmd: WideString;
    FShellResult: MYX_GRT_SHELL_COMMAND;
    FStartTime,
      FEndTime: TDateTime;

    FProcessOutputFunction: TGrtProcessOutput;
    FTextForOutput: WideString;

    FProcessMessagesFunction: TGrtProcessMessages;
    FPMessages: PMYX_GRT_MSGS;
    FMessageReturnValue: Integer;

    FJavaLoaderState,
      FLuaLoaderState,
      FBuiltinLoaderState,
      FPhpLoaderState,
      FUiLoaderState,
      FGcLoaderState: GrtLoaderState;

    FDelphiGrtMessages: TMYX_GRT_MSGS;

    procedure OutputModuleStatus(LoaderName: WideString;
      LoadedModuleCount: Integer; Error: MYX_GRT_ERROR);
  public
    property PGrt: Pointer read GetPGrt;
    property ExecutionFinished: Boolean read GetExecutionFinished;
    property ExecutionTime: TDateTime read GetExecutionTime;

    property DelphiGrtMessages: TMYX_GRT_MSGS read FDelphiGrtMessages;

    function ExecuteModalFunction(ModulName: WideString;
      FunctionName: WideString;
      FunctionArgument: Pointer; Error: PMYX_GRT_ERROR;
      ProcessOutputFunction: TGrtProcessOutput;
      ProcessMessagesFunction: TGrtProcessMessages;
      SearchParent: Boolean;
      TimeOutMS: Integer = -1): Pointer;
    function ExecuteModalShellFunction(Cmd: WideString;
      ProcessOutputFunction: TGrtProcessOutput;
      TimeOutMS: Integer = -1): MYX_GRT_SHELL_COMMAND;

    procedure OutputText(S: WideString);
    function ProcessMessages(PMessages: PMYX_GRT_MSGS): integer;
  end;

  EGrtError = class(Exception)
  private
    FErrorNumber: Integer;
    FDescription: WideString;
  public
    constructor Create(ErrorNumber: Integer; Description: WideString);

    property ErrorNumber: Integer read FErrorNumber;
    property Description: WideString read FDescription;
  end;

function GetListMemberCount(Grt: Pointer; StructName: WideString;
  OnlyCheck: Boolean): Integer;
function GetListMember(Grt: Pointer; StructName: WideString;
  Index: Integer): Pointer;
function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;


implementation

// -----------------------------------------------------------------------------

procedure ProcessGrtOutput(text: PChar; userdata: Pointer) cdecl;

var
  GrtEngine: TGrtEngine;
  S: WideString;

begin
  GrtEngine := userdata;
  S := UTF8Decode(text);

  GrtEngine.OutputText(S);
end;

// -----------------------------------------------------------------------------

function ProcessGrtMessages(PMessages: PMYX_GRT_MSGS; userdata: Pointer): integer cdecl;

var
  GrtEngine: TGrtEngine;

begin
  GrtEngine := userdata;

  Result := GrtEngine.ProcessMessages(PMessages);
end;

// -----------------------------------------------------------------------------

constructor TGrt.Create(Console: TUniCodeConsole;
  ScriptFilename: WideString; Verbose: Boolean;
  RemoteDebug: Boolean; JvmLibrary: WideString);

var
  TimeStart: TDateTime;
  TimeOut: Boolean;
  TimeOutInterval: TDateTime;
  WaitResult: Cardinal;
  WaitHandle: THandle;

begin
  inherited Create;

  FStructIconsList := TTntStringList.Create;

  FOnShutDown := nil;

  FGrtFormsFuncNameCache := nil;
{$IFDEF GrtForms}
  FGrtForms := nil;
{$ENDIF}

  FVerbose := Verbose;
  FRemoteDebug := RemoteDebug;
  FJvmLibrary := JvmLibrary;

  FConsole := Console;

  GrtEngineInitializedEvent := TEvent.Create(nil, False, False, '');

  FGrtEngine := TGrtEngine.Create(False, self);
  FGrtEngine.FreeOnTerminate := True;

  // Wait for 8 sek
  TimeOut := False;
  TimeStart := Now;
  TimeOutInterval := (1 / 86400) * 8;
  WaitHandle := GrtEngineInitializedEvent.Handle;

  while (Not(TimeOut)) do
  begin
    // Wait for the initialisation to finish but every
    // 100 milliseonds check if the timeout value has been reached.
    // Process any incomming message while we wait.
    WaitResult := MsgWaitForMultipleObjects(
      1, WaitHandle, false, 100, QS_ALLEVENTS);
    if WaitResult = WAIT_OBJECT_0 then
      Break;

    Application.ProcessMessages;

    if (Now - TimeStart > TimeOutInterval) then
      TimeOut := True;
  end;

  if (TimeOut) then
    raise Exception.Create(_('Could not initialize the GRT Environment. ' +
      'A timeout occured during the initalization.'));

  GrtEngineInitializedEvent.Free;
  GrtEngineInitializedEvent := nil;

  if (ScriptFilename <> '') and (FGrtEngine.PGrt <> nil) then
    myx_grt_lua_shell_run_file(FGrtEngine.PGrt, ScriptFilename, 1);
end;

// -----------------------------------------------------------------------------

destructor TGrt.Destroy;

var
  I: Integer;

begin
  if (Assigned(FOnShutDown)) then
    FOnShutDown(self);

  // Free FGrtStructIconsList
  for I := 0 to FStructIconsList.Count - 1 do
    FStructIconsList.Objects[I].Free;
  FStructIconsList.Free;

{$IFDEF GrtForms}
  if (FGrtForms <> nil) then
    FGrtForms.Free;
{$ENDIF}

  if (FGrtFormsFuncNameCache <> nil) then
    FGrtFormsFuncNameCache.Free;

  FGrtEngine.Terminate;
end;

// ----------------------------------------------------------------------------

function TGrt.GetJavaLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FJavaLoaderState;
end;

// ----------------------------------------------------------------------------

function TGrt.GetLuaLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FLuaLoaderState;
end;

// ----------------------------------------------------------------------------

function TGrt.GetBuiltinLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FBuiltinLoaderState;
end;

// ----------------------------------------------------------------------------

function TGrt.GetPhpLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FUiLoaderState;
end;

// ----------------------------------------------------------------------------

function TGrt.GetUiLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FUiLoaderState;
end;

// ----------------------------------------------------------------------------

function TGrt.GetGcLoaderState: GrtLoaderState;

begin
  Result := FGrtEngine.FGcLoaderState;
end;

// ----------------------------------------------------------------------------

procedure TGrt.OutputCommandLine(Text: WideString);

begin
  if (FConsole <> nil) then
    FConsole.AddOutput(#13#10 + Text);
end;

// ----------------------------------------------------------------------------

function TGrt.GrtPGrt: Pointer;

begin
  Result := FGrtEngine.PGrt;
end;

// ----------------------------------------------------------------------------

function TGrt.ExecuteModalFunction(ModulName: WideString;
  FunctionName: WideString;
  FunctionArguments: array of const;
  ProcessOutputFunction: TGrtProcessOutput;
  ProcessMessagesFunction: TGrtProcessMessages;
  AllowNullAsResult: Boolean;
  SearchParent: Boolean;
  TimeOutMS: Integer): Pointer;

var
  Args: Pointer;
  Error: MYX_GRT_ERROR;
  FunctionErrorString: WideString;

begin
  Args := BuildGrtParamList(FunctionArguments);

  //Execute the function
  Result := FGrtEngine.ExecuteModalFunction(ModulName,
    FunctionName,
    Args, @Error,
    ProcessOutputFunction,
    ProcessMessagesFunction,
    SearchParent,
    TimeOutMS);

  //Check the result for error keys
  FunctionErrorString := myx_grt_function_check_error(Result,
    Ord(AllowNullAsResult));

  //Show error
  if (Error<>MYX_GRT_NO_ERROR) or
    (FunctionErrorString<>'') then
    raise EGrtError.Create(Ord(Error), FunctionErrorString);

  //If there was no error we have a value key
  //Return the contents of the key as result
  if (Result<>nil) then
    Result := myx_grt_dict_item_get_value(Result, 'value');
end;

// ----------------------------------------------------------------------------

function TGrt.ExecuteModalShellFunction(Cmd: WideString;
  ProcessOutputFunction: TGrtProcessOutput;
  TimeOutMS: Integer): MYX_GRT_SHELL_COMMAND;

begin
  Result := FGrtEngine.ExecuteModalShellFunction(Cmd,
    ProcessOutputFunction,
    TimeOutMS);
end;

// -----------------------------------------------------------------------------

function TGrt.BuildGrtParamList(Params: array of const): Pointer;

var
  ParamList: Pointer;
  Param: Pointer;
  I: Integer;

begin
  ParamList := myx_grt_list_new(MYX_ANY_VALUE, '');

  for I:=0 to High(Params) do
  begin
    with Params[I] do
      case VType of
        vtInteger:
          Param := myx_grt_value_from_int(VInteger);
        vtExtended:
          Param := myx_grt_value_from_real(VExtended^);
        vtWideString:
          Param := myx_grt_value_from_string(WideString(VWideString));
        vtString:
          Param := myx_grt_value_from_string(VString^);
        vtAnsiString:
          Param := myx_grt_value_from_string(string(VAnsiString));
        vtPChar:
          Param := myx_grt_value_from_string(VPChar);
        vtPointer:
          Param := VPointer;
      else
        raise EInOutError.Create(_('BuildGrtParamList called with unsupported parameter type.'));
      end;

    myx_grt_list_item_add(ParamList, Param);
  end;

  Result := ParamList;
end;

// -----------------------------------------------------------------------------

function TGrt.GetGlobal(const Path: WideString): Pointer;

begin
  Result := myx_grt_dict_item_get_by_path(myx_grt_get_root(PGrt), Path);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetGlobal(const Path: WideString; NewValue: Pointer);

begin
  if (myx_grt_dict_item_set_by_path(myx_grt_get_root(PGrt), Path, NewValue) <> 0) then
    raise Exception.Create(Format(
      _('The value %s cannot be set.'),
      [Path]));
end;

// -----------------------------------------------------------------------------

function TGrt.GetGlobalAsString(const Path: WideString): WideString;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_STRING_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a string value.'),
        [Path]));

    Result := myx_grt_value_as_string(PValue);
  end
  else
    Result := '';
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetGlobalAsString(const Path: WideString; NewValue: WideString);

begin
  Global[Path] := myx_grt_value_from_string(NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetGlobalAsInt(const Path: WideString): Integer;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_INT_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a string value.'),
        [Path]));

    Result := myx_grt_value_as_int(PValue);
  end
  else
    Result := 0;
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetGlobalAsInt(const Path: WideString; NewValue: Integer);

begin
  Global[Path] := myx_grt_value_from_int(NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetGlobalAsReal(const Path: WideString): Double;

var
  PValue: Pointer;

begin
  PValue := Global[Path];

  if (PValue <> nil) then
  begin
    if (myx_grt_value_get_type(PValue) <> MYX_INT_VALUE) then
      raise Exception.Create(Format(
        _('The value %s is not a string value.'),
        [Path]));

    Result := myx_grt_value_as_real(PValue);
  end
  else
    Result := 0;
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetGlobalAsReal(const Path: WideString; NewValue: Double);

begin
  Global[Path] := myx_grt_value_from_real(NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.ValueDuplicate(Value: Pointer): Pointer;

begin
  Result := myx_grt_value_dup(Value);
end;

// -----------------------------------------------------------------------------

function TGrt.FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;

begin
  Result := FormatGrtMessagesAsString(Msgs);
end;

// -----------------------------------------------------------------------------

function TGrt.ListCount(List: Pointer): Integer;

begin
  Result := myx_grt_list_item_count(List);
end;

// -----------------------------------------------------------------------------

function TGrt.GetListItem(const List: Pointer; const I: Integer): Pointer;

begin
  Result := myx_grt_list_item_get(List, I);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetListItem(const List: Pointer; const I: Integer; NewValue: Pointer);

begin
  myx_grt_list_set_item(List, I, NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetListString(const List: Pointer; const I: Integer): WideString;

begin
  Result := myx_grt_list_item_get_as_string(List, I);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetListString(const List: Pointer; const I: Integer; NewValue: WideString);

var
  Value: Pointer;

begin
  Value := myx_grt_value_from_string(NewValue);

  myx_grt_list_set_item(List, I, Value);

  myx_grt_value_release(Value);
end;

// -----------------------------------------------------------------------------

procedure TGrt.ListAdd(List: Pointer; Value: Pointer; IncreaseRefCount: Boolean);

begin
  myx_grt_list_item_add(List, Value);

  // as myx_grt_list_item_add increases refcount,
  // release it one time if it should not be increased
  if (Not(IncreaseRefCount)) then
    myx_grt_value_release(Value);
end;

// -----------------------------------------------------------------------------

function TGrt.ListDel(List: Pointer; Index: Integer): Boolean;

begin
  Result := (myx_grt_list_item_del(List, Index) = 0);
end;

// -----------------------------------------------------------------------------

function TGrt.GetDictItem(const Dict: Pointer; const Key: WideString): Pointer;

begin
  Result := myx_grt_dict_item_get_value(Dict, Key);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetDictItem(const Dict: Pointer; const Key: WideString; NewValue: Pointer);

begin
  myx_grt_dict_item_set_value(Dict, Key, NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetDictString(const Dict: Pointer; const Key: WideString): WideString;

begin
  Result := myx_grt_dict_item_get_as_string(Dict, Key);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetDictString(const Dict: Pointer; const Key: WideString; NewValue: WideString);

begin
  myx_grt_dict_item_set_value_from_string(Dict, Key, NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetDictInt(const Dict: Pointer; const Key: WideString): Integer;

begin
  Result := myx_grt_dict_item_get_as_int(Dict, Key);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetDictInt(const Dict: Pointer; const Key: WideString; NewValue: Integer);

begin
  myx_grt_dict_item_set_value_from_int(Dict, Key, NewValue);
end;

// -----------------------------------------------------------------------------

function TGrt.GetDictReal(const Dict: Pointer; const Key: WideString): Double;

begin
  Result := myx_grt_dict_item_get_as_real(Dict, Key);
end;

// -----------------------------------------------------------------------------

procedure TGrt.SetDictReal(const Dict: Pointer; const Key: WideString; NewValue: Double);

begin
  myx_grt_dict_item_set_value_from_real(Dict, Key, NewValue);
end;


// -----------------------------------------------------------------------------

constructor TGrtEngine.Create(CreateSuspended: Boolean; Grt: TGrt);

begin
  inherited Create(CreateSuspended);

  FGrt := Grt;
  FPGrt := nil;

  FJavaLoaderState := GrtLsNotInitialized;
  FLuaLoaderState := GrtLsNotInitialized;
  FBuiltinLoaderState := GrtLsNotInitialized;
  FPhpLoaderState := GrtLsNotInitialized;
  FUiLoaderState := GrtLsNotInitialized;
  FGcLoaderState := GrtLsNotInitialized;

  FFunctionSynchronizer := TCriticalSection.Create;
  FFunctionStartedEvent := TEvent.Create(nil, False, False, '');
  FFunctionFinishedEvent := TEvent.Create(nil, False, False, '');

  FVclDataSynchronizer := TCriticalSection.Create;

  FDelphiGrtMessages := TMYX_GRT_MSGS.Create;
end;

// ----------------------------------------------------------------------------

destructor TGrtEngine.Destroy;

begin
  if FPGrt<>nil then
    myx_grt_finalize(FPGrt);

  FFunctionFinishedEvent.Free;
  FFunctionStartedEvent.Free;
  FFunctionSynchronizer.Free;

  FVclDataSynchronizer.Free;

  FDelphiGrtMessages.Free;

  inherited;
end;

// ----------------------------------------------------------------------------

procedure TGrtEngine.Execute;

var
  FunctionStartedWaitResult: TWaitResult;
  ErrorDetails: TMYX_STRINGLIST;
  GrtOptions: Integer;

begin
  GrtOptions := 0;
  if (FGrt.Verbose) then
    GrtOptions := GrtOptions + Ord(MYX_GRT_VERBOSE);
  if (FGrt.RemoteDebug) then
    GrtOptions := GrtOptions + Ord(MYX_GRT_REMOTE_DEBUG);

  FPGrt := myx_grt_initialize(GrtOptions);

  InitializeGrt;

  while (Not(Terminated)) do
  begin
    FunctionStartedWaitResult :=
      FFunctionStartedEvent.WaitFor(1000);

    if (FunctionStartedWaitResult = wrSignaled) then
    begin
      FStartTime := Now;

      try
        // if the function name is set, it is a GrtFunction
        if (FFunctionName <> '') then
          FResult := myx_grt_function_get_and_call(FPGrt,
            FModulName, FFunctionName, Ord(FSearchParent),
            FFunctionArgument, FPError)
        else
          // otherwise its a shell function
          FShellResult := myx_grt_lua_shell_execute(FPGrt, FShellCmd);
      except
        on X: Exception do
        begin
          ErrorDetails := TMYX_STRINGLIST.Create;
          try
            ErrorDetails.strings.Text := Format(
              _('The call of the function %s:%s cause an exception.'),
              [FModulName, FFunctionName]);

            myx_grt_messages_stack_add(FPGrt, 1, X.Message,
              ErrorDetails.get_record_pointer, 1, -1);

            myx_grt_messages_stack_flush(FPGrt, 0);
          finally
            ErrorDetails.Free;
          end;
        end;
      end;

      FEndTime := Now;

      FFunctionFinishedEvent.SetEvent;
    end;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.InitializeGrt;

var
  Loader,
    NativeLoader: Pointer;
  Error: MYX_GRT_ERROR;
  I: integer;

begin
  // Register print output function
  myx_grt_set_output_callback(FPGrt, self, @ProcessGrtOutput);

  // Register message processing function
  myx_grt_set_message_callback(FPGrt, self, @ProcessGrtMessages);


  myx_grt_shell_print_welcome(FPGrt);

  // -------------------------------------------------
  // Load Structs

  I := myx_grt_scan_for_structs(FPGrt, './xml', @Error);
  if (Error<>MYX_GRT_NO_ERROR) then
    FGrt.OutputCommandLine('  ' + Format(
      _('Error while loading struct definitions (%d).'),
      [Ord(Error)]))
  else
    if (I = 1) then
      FGrt.OutputCommandLine(
        _('Registered 1 struct definition file.'))
    else
      FGrt.OutputCommandLine(Format(
        _('Registered %d struct definition files.'),
        [I]));


  // -------------------------------------------------
  // Initialized Loaders

  // Init Delphi loader
  FGrt.OutputCommandLine(_('Initializing native loader...'));
  FUiLoaderState := GrtLsInitializeFailed;
  NativeLoader := InitDelphiLoader(@Error);
  if (NativeLoader<>nil) then
  begin
    if (myx_grt_register_module_loader(FPGrt, NativeLoader)<>MYX_GRT_NO_ERROR) then
      FGrt.OutputCommandLine('  ' + _('Could not register Delphi modules.'))
    else
      FUiLoaderState := GrtLsInitialized;
  end
  else
    FGrt.OutputCommandLine('  ' + Format(
      _('Error initializing Delphi modules (%d)'),
      [Ord(error)]));


  // Init Java loader
  if (WideDirectoryExists(GetApplDir + 'java')) then
  begin
    FGrt.OutputCommandLine(_('Initializing Java loader...'));
    if (FGrt.Verbose) then
      FGrt.OutputCommandLine('');

    FJavaLoaderState := GrtLsInitializeFailed;

    Loader := myx_java_init_loader(FPGrt, 'java', @Error, FGrt.JvmLibrary);

    if (Loader<>nil) then
    begin
      FJavaLoaderState := GrtLsInitialized;

      if (myx_grt_register_module_loader(FPGrt, Loader)<>MYX_GRT_NO_ERROR) then
        FGrt.OutputCommandLine('  ' + _('Could not register Java module loader.'))
      else
        FJavaLoaderState := GrtLsInitialized;
    end
    else
      FGrt.OutputCommandLine('  ' + Format(
        _('Error initializing Java module loader (%d)'),
        [Ord(error)]));
  end;


  // Init PHP loader
  if (WideDirectoryExists(GetApplDir + 'php')) then
  begin
    FGrt.OutputCommandLine(_('Initializing PHP loader...'));
    FPhpLoaderState := GrtLsInitializeFailed;

    Loader := myx_php_init_loader(FPGrt, @Error);
    if (Loader<>nil) then
    begin
      if (myx_grt_register_module_loader(FPGrt, Loader)<>MYX_GRT_NO_ERROR) then
        FGrt.OutputCommandLine('  ' + _('Could not register PHP module loader.'))
      else
        FPhpLoaderState := GrtLsInitialized;
    end
    else
      FGrt.OutputCommandLine('  ' + Format(
        _('Error initializing PHP module loader (%d)'),
        [Ord(error)]));
  end;


  // Init lua loader
  FGrt.OutputCommandLine(_('Initializing Lua loader...'));
  FLuaLoaderState := GrtLsInitializeFailed;
  Loader := myx_lua_init_loader(FPGrt, @Error);
  if (Loader<>nil) then
  begin
    if (myx_grt_register_module_loader(FPGrt, Loader)<>MYX_GRT_NO_ERROR) then
      FGrt.OutputCommandLine('  ' + _('Could not register Lua module loader.'))
    else
      FLuaLoaderState := GrtLsInitialized;
  end
  else
    FGrt.OutputCommandLine('  ' + Format(
      _('Error initializing Lua module loader (%d)'),
      [Ord(error)]));

  // -------------------------------------------------
  // Load modules

  //Load builtin modules
  FBuiltinLoaderState := GrtLsInitializeFailed;
  if (myx_register_builtin_grt_module_base(FPGrt) <> nil) and
    (myx_register_builtin_grt_module_reverse_engineer_mysql(FPGrt) <> nil) and
    (myx_register_builtin_grt_module_transformation_mysql(FPGrt) <> nil) then
  begin
    FBuiltinLoaderState := GrtLsModulesLoaded;

    OutputModuleStatus('builtin', 3, MYX_GRT_NO_ERROR);
  end;

{$IFDEF GrtGc}
  //Load canvas module
  FGcLoaderState := GrtLsInitializeFailed;
  if (myx_register_grt_gc_module(FPGrt) = MYX_GRT_NO_ERROR) then
  begin
    FGcLoaderState := GrtLsModulesLoaded;

    OutputModuleStatus('Generic Canvas', 1, MYX_GRT_NO_ERROR);
  end;
{$ENDIF}

  // Native modules
  if (FUiLoaderState = GrtLsInitialized) then
  begin
    //Add modules
    I := AddDelphiModules(NativeLoader);
    OutputModuleStatus('native', I, MYX_GRT_NO_ERROR);

    FUiLoaderState := GrtLsModulesLoaded;
  end;

  // Java modules
  if (FJavaLoaderState = GrtLsInitialized) then
  begin
    if (FGrt.Verbose) then
      FGrt.OutputCommandLine('');

    //Scan for Java plugins
    I := myx_grt_scan_for_modules(FPGrt, './java/com/mysql/grt/modules', @Error);
    OutputModuleStatus('Java', I, Error);

    FJavaLoaderState := GrtLsModulesLoaded;
  end;

  // Php modules
  if (FPhpLoaderState = GrtLsInitialized) then
  begin
    //Scan for PHP plugins
    I := myx_grt_scan_for_modules(FPGrt, './php/modules', @Error);
    OutputModuleStatus('PHP', I, Error);
  end;

  // Lua modules
  if (FLuaLoaderState = GrtLsInitialized) and
    (WideDirectoryExists(GetApplDir + 'lua')) then
  begin
    //Scan for Lua plugins
    I := myx_grt_scan_for_modules(FPGrt, './lua', @Error);
    OutputModuleStatus('Lua', I, Error);
  end;

  FGrt.OutputCommandLine('');


  //Init lua shell
  myx_grt_init_lua_shell(FPGrt);

  FGrt.GrtEngineInitializedEvent.SetEvent;
end;

// ----------------------------------------------------------------------------

procedure TGrtEngine.OutputModuleStatus(LoaderName: WideString;
  LoadedModuleCount: Integer; Error: MYX_GRT_ERROR);
begin
  if (Error<>MYX_GRT_NO_ERROR) then
    FGrt.OutputCommandLine(Format(
      _('Error while loading %s modules (%d).'),
      [LoaderName, Ord(Error)]))
  else
    if (LoadedModuleCount = 1) then
      FGrt.OutputCommandLine(Format(
        _('Registered 1 %s module.'), [LoaderName]))
    else if (LoadedModuleCount > 0) then
      FGrt.OutputCommandLine(Format(
        _('Registered %d %s modules.'), [LoadedModuleCount, LoaderName]))
end;

// ----------------------------------------------------------------------------

function TGrtEngine.GetPGrt: Pointer;

begin
  Result := FPGrt;
end;

// ----------------------------------------------------------------------------

function TGrtEngine.ExecuteModalFunction(ModulName: WideString;
  FunctionName: WideString;
  FunctionArgument: Pointer; Error: PMYX_GRT_ERROR;
  ProcessOutputFunction: TGrtProcessOutput;
  ProcessMessagesFunction: TGrtProcessMessages;
  SearchParent: Boolean;
  TimeOutMS: Integer): Pointer;

var
  StartTime: TDateTime;
  TimeOut: Boolean;
  TimeOutTime: TDateTime;
  WaitHandle: THandle;
  WaitResult: DWORD;

begin
  FResult := nil;

  // Allow only one function at a time
  FFunctionSynchronizer.Acquire;
  try
    FExecutionFinished := False;
    FModulName := ModulName;
    FFunctionName := FunctionName;
    FSearchParent := SearchParent;
    FFunctionArgument := FunctionArgument;
    FPError := Error;

    // Set callback functions
    FProcessOutputFunction := ProcessOutputFunction;
    FProcessMessagesFunction := ProcessMessagesFunction;

    // Trigger the function execution
    FFunctionStartedEvent.SetEvent;

    StartTime := Now;
    TimeOut := False;
    TimeOutTime := 0;
    if (TimeOutMS > -1) then
      TimeOutTime := TimeOutMS / (24 * 60 * 60 * 1000);

    WaitHandle := FFunctionFinishedEvent.Handle;
    while (Not(TimeOut)) do
    begin
      // Wait for the function to finish but every 100 milliseonds check if the timeout value has been reached.
      // Process any incomming message while we wait.
      WaitResult := MsgWaitForMultipleObjects(1, WaitHandle, false, 100, QS_ALLEVENTS);
      if WaitResult = WAIT_OBJECT_0 then
        Break;

      Application.ProcessMessages;

      if (TimeOutMS > -1) then
      begin
        TimeOut := (Now - StartTime > TimeOutTime);

        if (TimeOut) then
          if (ShowModalDialog(_('Function Timeout Occured'),
            Format(_('The executed function %s:%s has not returned ' +
                'after %s minutes. Do you want to wait for another ' +
                '%s minutes?'),
              [FModulName,
                FFunctionName,
                FormatDateTime('n:sss', TimeOutTime),
                FormatDateTime('n:sss', TimeOutTime)]),
              myx_mtInformation, _('Yes')+#13#10+_('No')) = 2) then
          begin
            raise Exception.Create(
              Format(_('The function %s:%s ' +
                  'did not complete in %s minutes.'),
                  [FModulName,
                    FFunctionName,
                    FormatDateTime('n:sss', TimeOutTime)]));
          end
          else
            StartTime := Now;
      end;
    end;
  finally
    FProcessOutputFunction := nil;
    FProcessMessagesFunction := nil;

    FFunctionSynchronizer.Release;

    FFunctionName := '';
  end;

  Result := FResult;
end;

// -----------------------------------------------------------------------------

function TGrtEngine.ExecuteModalShellFunction(Cmd: WideString;
  ProcessOutputFunction: TGrtProcessOutput;
  TimeOutMS: Integer): MYX_GRT_SHELL_COMMAND;

var
  StartTime: TDateTime;
  TimeOut: Boolean;
  TimeOutTime: TDateTime;
  WaitHandle: THandle;
  WaitResult: DWORD;

begin
  FShellResult := MYX_GRT_SHELL_COMMAND_UNKNOWN;

  // Allow only one function at a time
  FFunctionSynchronizer.Acquire;
  try
    FShellCmd := Cmd;

    // Set output function
    FProcessOutputFunction := ProcessOutputFunction;

    // Trigger the function execution
    FFunctionStartedEvent.SetEvent;

    StartTime := Now;
    TimeOut := False;
    TimeOutTime := 0;
    if (TimeOutMS > -1) then
      TimeOutTime := TimeOutMS / (24 * 60 * 60 * 1000);

    WaitHandle := FFunctionFinishedEvent.Handle;
    while (Not(TimeOut)) do
    begin
      // Wait for the function to finish but every 100 milliseonds check if the timeout value has been reached.
      // Process any incomming message while we wait.
      WaitResult := MsgWaitForMultipleObjects(1, WaitHandle, false, 100, QS_ALLEVENTS);
      if WaitResult = WAIT_OBJECT_0 then
        Break;

      Application.ProcessMessages;

      if (TimeOutMS > -1) then
      begin
        TimeOut := (Now - StartTime > TimeOutTime);

        if (TimeOut) then
          if (ShowModalDialog(_('Function Timeout Occured'),
            Format(_('The executed function %s:%s has not returned ' +
                'after %s minutes. Do you want to wait for another ' +
                '%s minutes?'),
              [FModulName,
                FFunctionName,
                FormatDateTime('n:sss', TimeOutTime),
                FormatDateTime('n:sss', TimeOutTime)]),
              myx_mtInformation, _('Yes')+#13#10+_('No')) = 2) then
          begin
            raise Exception.Create(
              Format(_('The function %s:%s ' +
                  'did not complete in %s minutes.'),
                  [FModulName,
                    FFunctionName,
                    FormatDateTime('n:sss', TimeOutTime)]));
          end
          else
            StartTime := Now;
      end;
    end;
  finally
    FProcessOutputFunction := nil;

    FFunctionSynchronizer.Release;
  end;

  Result := FShellResult;
end;

// -----------------------------------------------------------------------------

function TGrtEngine.GetExecutionFinished: Boolean;

begin
  Result := FExecutionFinished;
end;

// -----------------------------------------------------------------------------

function TGrtEngine.GetExecutionTime: TDateTime;

begin
  Result := FEndTime - FStartTime;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.OutputText(S: WideString);

begin
  FVclDataSynchronizer.Acquire;
  try
    FTextForOutput := S;
    Synchronize(DoOutputText);
  finally
    FVclDataSynchronizer.Release;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.DoOutputText;

begin
  if (Assigned(FProcessOutputFunction)) then
    FProcessOutputFunction(FTextForOutput)
  else
  begin
    FGrt.Console.AddOutput(FTextForOutput);
    FGrt.Console.Invalidate;
  end;
end;

// ----------------------------------------------------------------------------

function TGrtEngine.ProcessMessages(PMessages: PMYX_GRT_MSGS): integer;

begin
  FVclDataSynchronizer.Acquire;
  try
    FPMessages := PMessages;
    Synchronize(DoProcessMessages);
    Result := FMessageReturnValue;
  finally
    FVclDataSynchronizer.Release;
  end;
end;

// -----------------------------------------------------------------------------

procedure TGrtEngine.DoProcessMessages;

var
  Msgs: TMYX_GRT_MSGS;

begin
  Msgs := TMYX_GRT_MSGS.Create(FPMessages);
  try
    if (Assigned(FProcessMessagesFunction)) then
      FMessageReturnValue := FProcessMessagesFunction(Msgs)
    else
    begin
      // print the message
      FTextForOutput := FormatGrtMessagesAsString(Msgs);
      DoOutputText;

      FMessageReturnValue := 0;
    end;
  finally
    Msgs.Free;
  end;
end;


// -----------------------------------------------------------------------------

function DelphiGrtCallFunction(func: PMYX_GRT_FUNCTION; argument: Pointer; retval: PPointer): MYX_GRT_ERROR cdecl;

var
  GrtEngine: TGrtEngine;
{$IFDEF GrtForms}
  XGrtForms: TXGrtFormApplication;
  FormsCacheIndex: Integer;
{$ENDIF}

begin
  Result := MYX_GRT_NO_ERROR;

  GrtEngine := func.module.priv;

  retval^ := nil;

  try
{$IFDEF GrtForms}
    if (func.module.name = 'Forms') then
    begin
      XGrtForms := func.priv;

      FormsCacheIndex :=
        Grt.GrtFormsFuncNameCache.IndexOf(func.name);

      case FormsCacheIndex of
        0: retval^ := XGrtForms.CreateProxyRoot(argument);
        1: retval^ := XGrtForms.ValueGet(argument);
        2: retval^ := XGrtForms.ValueSet(argument);
        3: retval^ := XGrtForms.ListSize(argument);
        4: retval^ := XGrtForms.ListItemGet(argument);
        5: retval^ := XGrtForms.ListItemAdd(argument);
        6: retval^ := XGrtForms.ListItemRemove(argument);
        7: retval^ := XGrtForms.DictItemGet(argument);
        8: retval^ := XGrtForms.DictItemSet(argument);
        9: retval^ := XGrtForms.DictItemCount(argument);
        10: retval^ := XGrtForms.DictItemKey(argument);
      end;
    end;
{$ENDIF}
  except
    on x: Exception do
    begin
      // Log error
      GrtEngine.DelphiGrtMessages.msgs.Add(
        TMYX_GRT_MSG.Create(1, x.Message, -1, nil));
    end;
  end;
end;

// -----------------------------------------------------------------------------

function TGrtEngine.InitDelphiLoader(Error: PMYX_GRT_ERROR): Pointer;

begin
  Result := myx_grt_module_loader_create(FPGrt,
    MYX_DELPHI_MODULE_TYPE, nil,
    nil, @DelphiGrtCallFunction,
    self);
end;

// ----------------------------------------------------------------------------

function TGrtEngine.AddDelphiModules(Loader: Pointer): integer;

{$IFDEF GrtForms}
var
  PModule: PMYX_GRT_MODULE;
  I: Integer;
{$ENDIF}

begin
  Result := 0;

{$IFDEF GrtForms}
  PModule := myx_grt_module_create(Loader, 'Forms', '', '', self);

  FGrtForms := TXGrtFormApplication.Create(FGrtEngine.PGrt, PModule);

  FGrtFormsFuncNameCache := TStringList.Create;
  FGrtFormsFuncNameCache.Text :=
    'create'#13#10 +
    '_valueGet'#13#10 +
    '_valueSet'#13#10 +
    '_listSize'#13#10 +
    '_listItemGet'#13#10 +
    '_listItemAdd'#13#10 +
    '_listItemRemove'#13#10 +
    '_dictItemGet'#13#10 +
    '_dictItemSet'#13#10 +
    '_dictItemCount'#13#10 +
    '_dictItemKey'#13#10;

  for I := 0 to FGrtFormsFuncNameCache.Count - 1 do
    myx_grt_module_add_function(PModule, FGrtFormsFuncNameCache[I], '', '', FGrtForms);

  myx_grt_add_module(FGrtEngine.PGrt, PModule);

  Result := Result + 1;
{$ENDIF}
end;

// -----------------------------------------------------------------------------

constructor EGrtError.Create(ErrorNumber: Integer; Description: WideString);

begin
  FErrorNumber := ErrorNumber;
  FDescription := Description;

  if (FErrorNumber <> 0) then
    Message := FDescription + ' Error Nr.' + IntToStr(FErrorNumber)
  else
    Message := FDescription;
end;

// -----------------------------------------------------------------------------

function GetListMemberCount(Grt: Pointer; StructName: WideString;
  OnlyCheck: Boolean): Integer;

var
  PStruct,
    PMember: Pointer;
  i,
    count: integer;

begin
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i:=0 to myx_grt_struct_get_member_count_total_excluding_struct(
    Grt, PStruct, 'db.DatabaseObject')-1 do
  begin
    PMember := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(PMember) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(PMember) = MYX_DICT_VALUE) then
    begin
      inc(count);

      if (OnlyCheck) then
        break;
    end;
  end;

  Result := count;
end;

// -----------------------------------------------------------------------------

function GetListMember(Grt: Pointer; StructName: WideString;
  Index: Integer): Pointer;

var
  PStruct: Pointer;
  i,
    Count: integer;

begin
  Result := nil;
  PStruct := myx_grt_struct_get(Grt, StructName);

  count := 0;
  for i:=0 to myx_grt_struct_get_member_count_total(Grt, PStruct)-1 do
  begin
    Result := myx_grt_struct_get_member_by_index_total(
      Grt, PStruct, i);

    if (myx_grt_struct_member_get_type(Result) = MYX_LIST_VALUE) and
      (myx_grt_struct_member_get_content_type(Result) = MYX_DICT_VALUE) then
    begin
      inc(Count);

      if (Index = Count-1) then
        break;
    end;
  end;
end;

// ----------------------------------------------------------------------------

function FormatGrtMessagesAsString(Msgs: TMYX_GRT_MSGS): WideString;

var
  S: WideString;
  I,
    J: Integer;

begin
  S := '';

  for I := 0 to Msgs.msgs.Count-1 do
  begin
    // do not log progress messages
    if (Msgs.msgs[I].msg_type = 2) then
      Continue;

    if (Msgs.msgs[I].msg_type = 1) then
      S := S + 'ERROR: ';

    S := S +
      WideStringReplace(
        WideStringReplace(
          Msgs.msgs[I].msg, #13#10, #10, [rfReplaceAll], False),
        #10, #13#10, [rfReplaceAll], False) +
      #13#10;

    if (Msgs.msgs[I].msg_detail <> nil) then
    begin
      for j:=0 to Msgs.msgs[I].msg_detail.strings.Count-1 do
      begin
        S := S +
          WideStringReplace(
            WideStringReplace(
              Msgs.msgs[I].msg_detail.strings[J], #13#10, #10, [rfReplaceAll], False),
            #10, #13#10, [rfReplaceAll], False) +
          #13#10;
      end;
    end;
  end;

  Result := S;
end;

// -----------------------------------------------------------------------------

end.
