/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_font_manager.cpp 
 * @brief A class that manages shared display lists used for text output
 * 
 */

#ifndef __GC_FONT_MANAGER_H__
#define __GC_FONT_MANAGER_H__

#include "myx_gc.h"

// FTGL
#include <FTGLOutlineFont.h>
#include <FTGLPolygonFont.h>
#include <FTGLBitmapFont.h>
#include <FTGLTextureFont.h>
#include <FTGLPixmapFont.h>
#include <FTGLExtrdFont.h>

#include "myx_gc_utilities.h"

//----------------------------------------------------------------------------------------------------------------------

typedef hash_map<string, FTFont*> Fonts;

typedef struct tagFontFileEntry
{
  int UseCount;
  string Entries[2][2];
} FontFileEntry;

typedef hash_map<string, FontFileEntry*> FontFiles;

// CFontManager is a helper class for text output in the generic canvas. It maps a description string for a font with attributes to a 
// display list. If there is no display for a given font then one is created.
// The font manager is basically a singleton class. We only need one instance of it.
class CFontManager
{
private:
  Fonts FFonts;
  FontFiles FFiles;
protected:
  string GetFontFile(string Family, string Style, int Weight);
public:
  CFontManager(void);
  virtual ~CFontManager(void);

  void BoundingBox(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize, TBoundingBox* Box);
  void Clear(void);
  string CreateLookupKey(const string& Family, const string& Style, int Weight);
  void TextOut(const wstring& Output, string FontFamily, string FontStyle, int Weight, int FontSize);
};

//----------------------------------------------------------------------------------------------------------------------

int ConvertFontWeight(const string Weight);

/** Returns the singleton font manager instance. */
CFontManager* FontManager(void);
/** Increase lock count for the manager. */
void LockFontManager(void);
/** Decrease lock count for the manager. */
void UnlockFontManager(void);

#endif // #ifdef __GC_FONT_MANAGER_H__

