// processfun.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/

#ifdef __GNUG__
#pragma implementation
#endif

#include "localdefs.h"
#include "data.h"
#include "crossfader.h"		// for undo
#include "processfun.h"
#include "request.h"
#include "requester.h"
#include "range.h"
#include "valuerequester.h"

int
SimpleFunction::doApply(Data *data) {
	return data->apply(*this);
}

//********

SingleFactorFunction::SingleFactorFunction(Data* d,
										   double defaultVal, 
										   bool needUndo)
    : SimpleFunction(d, needUndo), _factor(defaultVal) {}

//**********

double Scaler::_savedFactor = 1.0;

Scaler::Scaler(Data* d, double val, bool needUndo)
		: SingleFactorFunction(d, val, needUndo) {
	initialize();
}

Scaler::Scaler(Data* d, bool needUndo)
		: SingleFactorFunction(d, _savedFactor, needUndo) {}

Requester *
Scaler::createRequester() {
	return new ValueRequester<double>("Scale Values in Selected Region:",
	                                 "Amplitude Factor:", _factor);
}

void
Scaler::saveConfig() {
    _savedFactor = _factor;
}

Modifier *
Scaler::create(DataEditor* de) {
	return nil;
}

Status 
Scaler::doConfigure(Controller *) {
	setNeedUndo(_factor == 0.0 || target()->dataType() <= IntData);
	return Succeed;
}

Modifier *
Scaler::createUndo() {
	return (_factor != 0.0 && target()->dataType() > IntData) ? 
		new Scaler(target(), 1.0/_factor, true) : Super::createUndo();
}

void
Scaler::operator () (double *input, int count) {
    const double fac = _factor;
    for (int n = 0; n < count; n++)
        input[n] *= fac;
}

//**********

double Offset::_savedOffset = 0.0;

Offset::Offset(Data* d, double val) : SingleFactorFunction(d, val, false) {
	initialize();
}

Offset::Offset(Data* d) : SingleFactorFunction(d, _savedOffset, false) {}

Requester *
Offset::createRequester() {
	return new ValueRequester<double>("Offset Selected Region:",
	                                  "Offset By:", 
									  _factor);
}

void
Offset::saveConfig() {
    _savedOffset = _factor;
}

Modifier *
Offset::create(DataEditor* de) {
	return nil;
}

Status 
Offset::doConfigure(Controller *) {
	setNeedUndo(target()->dataType() < FloatData);
	return Succeed;
}

Modifier *
Offset::createUndo() {
	return (target()->dataType() >= FloatData) ? 
			new Offset(target(), -_factor) : Super::createUndo();
}

void
Offset::operator () (double *input, int count) {
    const double fac = _factor;
    for (int n = 0; n < count; n++)
        input[n] += fac;
}

//**********

Eraser::Eraser(Data* d) : Scaler(d, 0.0, true) {}

Modifier *
Eraser::create(DataEditor* de) {
	return nil;
}

Modifier *
Eraser::createUndo() {
	return (getUndoBuffer() != nil) ? new Mixer(target(), getUndoBuffer()) : nil;
}


