/*
 *   dspmem.c - DSP memory related routines 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include "dspmgr.h"

#include "dspmem.h"                    // Include function prototypes
#include "dspglist.h"                  // Include function prototypes
#include "dspmem.h"                    // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes
#include "dsppcmem.h"                  // Include function prototypes

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AddFreeMemList                                          */
/*                                                                          */
/* FUNCTION: Add a Free MEM block to the Free MEM list.                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMEM   pprmemNew   - Ptr to MEM block to add to Free list.       */
/*                              Block has been removed from Allocated List  */
/*                              prior to calling this routine.              */
/*        PRMEM * pprmemFreeTail - Ptr to MEM block Free list.              */
/*                                                                          */
/* OUTPUT: (NONE)                                                           */
/*                                                                          */
/* SIDE EFFECTS:                                                            */
/*     - Makes Use Count 0, assumes this is definitely free MEM block       */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

VOID AddFreeMemList(PRMEM prmem,PRDSP prdsp,USHORT usFlag)

{
   PRMEM      *pprmemFree;             /* Free List                         */
   PRMEM      *pprmemList;             /* Memory List                       */
   PRMEM      prmemN;                  /* Neighbor                          */
   PRMEM      prmemT;                  /* Traversal Pointer                 */
   BOOL       bFound;                  /* Found Indicator                   */


   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspmem::AddFreeMemList entry prmem %x prdsp %x\n",(int)prmem,(int)prdsp);

   prmem->MEM_usUseCount = 0;          /* Set Use Count to Zero             */
   switch (usFlag) {
      case  MEM_DATA :
         pprmemFree = &prdsp->DSP_prmemDFreeTail;
         pprmemList = &prdsp->DSP_prmemDTail;
         break;
      case  MEM_INST :
         pprmemFree = &prdsp->DSP_prmemIFreeTail;
         pprmemList = &prdsp->DSP_prmemITail;
         break;
      default  :
         return ;                      /* Error Condition, should never get */
                                       /* here                              */
   }

/*                                                                          */
/* Perform Left Coalesce                                                    */
/*                                                                          */
/* If not first on memory list and previous neighbor is free                */
/*   unhook neighbor from free list for repositioning.                      */
/*   unhook block being freed from memory list.                             */
/*   add length of block being freed to neighbor.                           */
/*   neighbor next = block being freed next for later use                   */
/*   free control block for block being freed                               */
/*   neighbor now block being freed.                                        */
/*                                                                          */

   if ((*pprmemList)->MEM_prmemNext != prmem &&
      prmem->MEM_prmemPrev->MEM_usUseCount == 0) {

/* Unhook Neighbor from Free List                                           */

      prmemN = prmem->MEM_prmemPrev;

      CDLLDelNode((PVOID *)&prmemN->MEM_prmemListNext, (PVOID *)
         &prmemN->MEM_prmemListPrev, (PVOID *)
         &prmemN->MEM_prmemListPrev->MEM_prmemListNext, (PVOID *)
         &prmemN->MEM_prmemListNext->MEM_prmemListPrev, (PVOID *)pprmemFree);

/* Unhook Block being freed from memory list                                */

      CDLLDelNode((PVOID *)&prmem->MEM_prmemNext, (PVOID *)
         &prmem->MEM_prmemPrev, (PVOID *)&prmem->MEM_prmemPrev->MEM_prmemNext,
         (PVOID *)&prmem->MEM_prmemNext->MEM_prmemPrev, (PVOID *)pprmemList);

/* Add Length to Neighbor                                                   */

      prmemN->MEM_ulMemSize += prmem->MEM_ulMemSize;

/* Neighbor Next = Block being Freed Next                                   */

      prmemN->MEM_prmemNext = prmem->MEM_prmemNext;

/* Free control block being freed                                           */

      FreeRMemory(prmem);

/* Block being freed is now combined block                                  */

      prmem = prmemN;
   }

/*                                                                          */
/* Perform Right Coalesce                                                   */
/*                                                                          */
/* If not last on memory list and right adjacent neighbor is free           */
/*   unhook neighbor from free list for repositioning.                      */
/*   unhook neighbor from memory list.                                      */
/*   add length of neighbor to block being freed.                           */
/*   free control neighbor                                                  */
/*                                                                          */

   if (*pprmemList != prmem && prmem->MEM_prmemNext->MEM_usUseCount == 0) {

/* Unhook Neighbor from Free List                                           */

      prmemN = prmem->MEM_prmemNext;

      CDLLDelNode((PVOID *)&prmemN->MEM_prmemListNext, (PVOID *)
         &prmemN->MEM_prmemListPrev, (PVOID *)
         &prmemN->MEM_prmemListPrev->MEM_prmemListNext, (PVOID *)
         &prmemN->MEM_prmemListNext->MEM_prmemListPrev, (PVOID *)pprmemFree);

/* Unhook Neighbor from memory list                                         */

      CDLLDelNode((PVOID *)&prmemN->MEM_prmemNext, (PVOID *)
         &prmemN->MEM_prmemPrev, (PVOID *)
         &prmemN->MEM_prmemPrev->MEM_prmemNext, (PVOID *)
         &prmemN->MEM_prmemNext->MEM_prmemPrev, (PVOID *)pprmemList);

/* Add Length of Neighbor to block being freed                              */

      prmem->MEM_ulMemSize += prmemN->MEM_ulMemSize;

/* Free Neighbor                                                            */

      FreeRMemory(prmemN);
   }

/*                                                                          */
/* Add block being freed to free list.  The free list is sorted by size of  */
/* block.                                                                   */
/*                                                                          */

   if (*pprmemFree == NULL)            /* Empty Free List                   */
      {
      *pprmemFree = prmem;             /* Make us only element on list      */
      prmem->MEM_prmemListNext = prmem;/* Doubly Link us to ourself         */
      prmem->MEM_prmemListPrev = prmem;
   }

/*                                                                          */
/* Traverse the Free List looking for a smaller element or End of List.     */
/* At the completion of the traverse, prmemT will point to the right        */
/* neighbor of the new element (if bFound is set) or the left neighbor      */
/* (if bFound is not set).  When bFound is not set, move right one to       */
/* normalize the loop ending condition.  Then, we can hook the new element  */
/* into the list easily.  If bFound is not set, the new element alsoe       */
/* becoms the list tail so *pprmemFree must be updated to reflect this.     */
/*                                                                          */

   else {
      prmemT = *pprmemFree;
      bFound = 0;

      do {
         prmemT = (PRMEM)prmemT->MEM_prmemListNext;/* Move to Next Element  */
         bFound = (BOOL)(prmem->MEM_ulMemSize <= prmemT->MEM_ulMemSize);/*  */
                                       /* Hit Indicator                     */
      }  while ((*pprmemFree != prmemT) && (!bFound));
      if (!bFound) {
         prmemT = prmemT->MEM_prmemListNext;/* Move to Next Element         */
         *pprmemFree = prmem;          /* Update tail pointer               */
      }

      prmem->MEM_prmemListNext = prmemT;/* Hook us in                       */
      prmem->MEM_prmemListPrev = prmemT->MEM_prmemListPrev;
      prmemT->MEM_prmemListPrev = prmem;
      prmem->MEM_prmemListPrev->MEM_prmemListNext = prmem;
   }

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspmem::AddFreeMemList exit ulRC=0\n");
   return ;
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: SplitMemBlock                                           */
/*                                                                          */
/* FUNCTION: A free MEM block has been found for a newly allocated segment, */
/*           but the MEM block is too big. This routine splits the existing */
/*           MEM block into two MEM blocks and attaches one representing    */
/*           the newly allocated segment to the DSP Used MEM chain and      */
/*           the DSP address MEM chain (flat chain).                        */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMEM   prmemOld    - Ptr to MEM block to split.                  */
/*        PRMEM * pprmemNew   - Ptr to new MEM block after split.           */
/*        ULONG   ulMemSize   - Size of new segment MEM block.              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMEM * pprmemNew   - Ptr to new segemnt MEM block.               */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC SplitMemBlock(PRMEM prmemOld,PRMEM *pprmemNew,ULONG ulMemSize)

{
   RC         ulRC;                    // error return code
   PRMEM      prmemNew;

   /* allocate a new MEM structure for split                                */
   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspmem::SplitMemBlock entry prmemOld %x\n",(int)prmemOld);

   if ((ulRC = AllocRMemory(&prmemNew)) == DSP_NOERROR) {

      /* Adjust sizes of memory blocks and DSP block addresses              */

      prmemNew->MEM_ulMemSize = ulMemSize;
      prmemOld->MEM_ulMemSize -= ulMemSize;

      prmemNew->MEM_ulDspAddr = prmemOld->MEM_ulDspAddr;
      prmemOld->MEM_ulDspAddr += ulMemSize;
      prmemOld->MEM_ulSegAddr = prmemOld->MEM_ulDspAddr;

      /* Add new MEM structure (prmemNew) into DSP Address MEM chain        */

      prmemNew->MEM_prmemPrev = prmemOld->MEM_prmemPrev;
      prmemNew->MEM_prmemNext = prmemOld;
      prmemNew->MEM_prmemPrev->MEM_prmemNext = prmemNew;
      prmemOld->MEM_prmemPrev = prmemNew;

      *pprmemNew = prmemNew;
   }

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspmem::SplitMemBlock exit ulRC %lx\n",ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: AllocateMem                                             */
/*                                                                          */
/* FUNCTION: Allocates DSP memory blocks. This routine will traverse the    */
/*           free MEM chain to find a MEM block that is => than the size    */
/*           we need. If one is not found, return error. If one is found,   */
/*           we un-link from the free MEM chain.                            */
/*                                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP   prdsp       - Ptr to DSP to allocate memory from.         */
/*        ULONG   ulMemSize   - Size of block to allocate in either         */
/*                              instructions WORDs or data WORDs.           */
/*        USHORT  usFlag      - Flags - data or instruction memory.         */
/*        ULONG   ulAlign     - Align start of segment (End of TCB) on this */
/*                              boundary. (Default Data Segment only)       */
/*                                for INSTRUCTION memory:                   */
/*                                   0 == ILLEGAL                           */
/*                                   1 == Instruction WORD alignment        */
/*                                for DATA memory:                          */
/*                                   0 == ILLEGAL                           */
/*                                   1 == Data WORD alignment               */
/*                                  32 == 32 Data WORD alignment            */
/*                                  64 == 64 Data WORD alignment            */
/*                                 128 == 128 Data WORD alignment           */
/*        ULONG   ulPrefix    - Prefix is size of TCB in WORDs.             */
/*                              (Default Data Segment only)                 */
/*        PRMEM * pprmemBlock - Ptr to MEM structure allocated.             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMEM * pprmemBlock - Ptr to MEM structure updated.               */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC AllocateMem(PRDSP prdsp,ULONG ulMemSize,USHORT usFlag,ULONG ulAlign,ULONG
                ulPrefix,PRMEM *pprmemBlock)

{
   RC         ulRC = DSP_NOERROR;      // error return code
   ULONG      ulSize,ulWorkingSize;    // reuested size + alignment size
   ULONG      ulSmallest;              // smallest split size
   LONG       lSplitSize;              // split block size
   ULONG      ulSegAddr;               // DSP segment address
   PRMEM      prmem;                   // current free MEM structure
   PRMEM      prmemOld;                // Old free MEM structure
   PRMEM      prmemHead;               // Head of free MEM list
   PRMEM      *pprmemTail;             // Address of DSP mem tail ptr
   BOOL       fDataSeg;                // Is segment a data segment ?
   BOOL       fFound;                  // Found a MEM block ?
   ULONG      ulFence = 0;


   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspmem::AllocateMem entry prdsp %x\n",(int)prdsp);
   MW_SYSLOG_5(TRACE_MANAGER_CORE,"  ulMemSize %lx usFlag %x ulAlign %lx ulPrefix %lx\n",
             ulMemSize,usFlag,ulAlign,ulPrefix);
   if (ulMemSize == 0) {
      *pprmemBlock = NULL;
      return (DSP_NOERROR);            // Return Nothing!
   }

   if (ulAlign == 0)
      return (DSP_INV_PARAMETER);      // Illegal request

   ulSmallest = (ULONG)SMALLEST_ALIGN_SPLIT;

   /* Determine if data or instruction memory                               */

   fDataSeg = (BOOL)((usFlag&(USHORT)MEM_MEM_MASK) == (USHORT)MEM_DATA);
   if (fDataSeg) {
      pprmemTail = &prdsp->DSP_prmemDFreeTail; // data memory chain
      ulSize = ulMemSize *2; // Convert size to bytes if data only
      ulPrefix *= 2;                 // Convert size to bytes if data only
      ulAlign = (ulAlign *2)-1;      // Convert size to bytes if data only
      ulSmallest *= 2;               // Convert size to bytes if data only
   }
   else {
      pprmemTail = &prdsp->DSP_prmemIFreeTail; // instruction memory chain
      ulSize = ulMemSize;
      --ulAlign;                       // Make 0 based.
   }

   /* Traverse the MEM chain, searching for free MEM structure              */

   fFound = FALSE;
   if (*pprmemTail != NULL)            /* check for mem completely          */
                                       /* used                              */
      {
      prmemHead = (*pprmemTail)->MEM_prmemListNext;
      prmem = prmemHead;

      do {                             /*                                   */
                                       /* ************************************
                                       *************************            */

         /* Compute possible segment address in this DSP memory block.      */
         /* The segment address is the DSP memory block address + the       */
         /* prefix value, other than 1, + any alignment difference.         */
         /* Some segments need to be aligned on a multiple WORD boundary    */
         /*******************************************************************/

         ulSegAddr = prmem->MEM_ulDspAddr+ulPrefix;
         ulSegAddr = (ulSegAddr+ulAlign)&~ulAlign; // Round to Alignment boundary

         /*-----------------------------------------------------------------*/
         /* Since we may make several passes of this loop before we find    */
         /* a free block large enough to satisfy ulMemSize, don't screw     */
         /* up ulSize, use a working variable ulWorkingSize                 */
         /*-----------------------------------------------------------------*/

         ulWorkingSize = ulSize+(ulSegAddr-prmem->MEM_ulDspAddr);

         /* Is this block big enough ?                                      */

         if (((lSplitSize = prmem->MEM_ulMemSize-ulWorkingSize) >= 0) &&
                                       (prmem->MEM_ulDspAddr >= ulFence)){

            /* Yes, Found a block that we can use, unlink from the free chain*/

            fFound = TRUE;
            CDLLDelNode((PVOID *)&prmem->MEM_prmemListNext, (PVOID *)
               &prmem->MEM_prmemListPrev, (PVOID *)
               &prmem->MEM_prmemListPrev->MEM_prmemListNext, (PVOID *)
               &prmem->MEM_prmemListNext->MEM_prmemListPrev, (PVOID *)
               pprmemTail);

            /* Mark our newly found block as used                           */

            prmem->MEM_usUseCount = 1;

            /* Is block too big for allocation ?                            */

            if (lSplitSize > (LONG)ulSmallest) {

               /* Yes, split it into two MEM blocks                         */

               prmemOld = prmem;
               if ((ulRC = SplitMemBlock(prmemOld, &prmem, ulWorkingSize)) !=
                  DSP_NOERROR)
                  return (ulRC);

               /* Add unsed portion to the free list                        */

               AddFreeMemList(prmemOld, prdsp, usFlag); // Need to coalesce
            }

            /****************************************************************/
            /* Now, determine if we can split the block and save some space */
            /* because of any alignment that we have done. If Align is one, */
            /* essentially, no alignment took place.                        */
            /****************************************************************/

            if (ulAlign > 1 && (ulWorkingSize = (ulSegAddr-ulPrefix)-
               prmem->MEM_ulDspAddr) > ulSmallest)
               {

/* Yes, we can split the block and save at least 9 WORDs (instruction or data)*/

               if ((ulRC = SplitMemBlock(prmem, &prmemOld, ulWorkingSize)) !=
                  DSP_NOERROR)
                  return (ulRC);

               /* Add unsed portion to the free list                        */

               AddFreeMemList(prmemOld, prdsp, usFlag); // Need to coalesce
            }

            prmem->MEM_ulSegAddr = ulSegAddr; // Fill in Segment Address of new segment
            prmem->MEM_ulSegSize = ulMemSize; // Fill in Actual Segment Size
         }
         else
            prmem = prmem->MEM_prmemListNext; // go to next in list
      }  while (prmem != prmemHead && !fFound); // Back at head yet ?
   }

   /* if memory found, add MEM structure to Used chain                      */

   if (fFound) {

      if (fDataSeg)
         pprmemTail = &prdsp->DSP_prmemDUsedTail; // data memory chain
      else
         pprmemTail = &prdsp->DSP_prmemIUsedTail; // instruction memory chain

      /*  Add new MEM structure to Used chain                               */

      if (*pprmemTail == NULL)         // Check for empty list
         prmem->MEM_prmemListNext = prmem->MEM_prmemListPrev = *pprmemTail =
            prmem;
      else
         CDLLInsTail((PVOID)prmem, (PVOID *)&prmem->MEM_prmemListNext, (PVOID
            *)&prmem->MEM_prmemListPrev, (PVOID *)&(*pprmemTail)->
            MEM_prmemListNext, (PVOID *)&(*pprmemTail)->
            MEM_prmemListNext->MEM_prmemListPrev, (PVOID *)pprmemTail);

      *pprmemBlock = prmem;
   }

   /* Not enough DSP memory to allocate                                     */

   else {
      if (fDataSeg)
         ulRC = DSP_INSUFF_DATA_MEM;
      else
         ulRC = DSP_INSUFF_INST_MEM;
   }

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspmem::AllocateMem exit pprmemBlock %x ulRC %lx\n",(int)*pprmemBlock,ulRC);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: FreeMem                                                 */
/*                                                                          */
/* FUNCTION: Returns a DSP Memory block pointed to by a PRMEM to the        */
/*           memory pool that is owned by a DSP.                            */
/*                                                                          */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP   prdsp       - Ptr to DSP to free memory to.               */
/*        PRMEM   prmem       - Ptr to Memory block being freed.            */
/*        USHORT  usFlag      - Indicates Data or Instruction Memory.       */
/*                               MEM_DATA = Data Memory                     */
/*                               MEM_INST = Instruction Memory              */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC FreeMem(PRDSP prdsp,PRMEM prmem,USHORT usFlag)
{
   PRMEM      *pprmemAllocTail;        /* Anchor of Allocated List          */

   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspmem::FreeMem entry prdsp %x prmem %x usFlag %x\n",(int)prdsp,(int)prmem,usFlag);
   if (prmem->MEM_usUseCount != 1)     /* Check if Last User of Memory      */
      return (DSP_NOERROR);

   if (prdsp->DSP_achValid != DSPVALIDATE)
      return (DSP_INV_HANDLE);

/*                                                                          */
/* Locate Appropriate Tail Pointers                                         */
/*                                                                          */

   switch (usFlag) {
      case  MEM_DATA :
         pprmemAllocTail = &prdsp->DSP_prmemDUsedTail;
         break;
      case  MEM_INST :
         pprmemAllocTail = &prdsp->DSP_prmemIUsedTail;
         break;
      default  :
         return (DSP_INV_PARAMETER);
   }

/*                                                                          */
/* Remove block from allocated list                                         */
/*                                                                          */

   CDLLDelNode((PVOID *)&prmem->MEM_prmemListNext, (PVOID *)
      &prmem->MEM_prmemListPrev, (PVOID *)
      &prmem->MEM_prmemListPrev->MEM_prmemListNext, (PVOID *)
      &prmem->MEM_prmemListNext->MEM_prmemListPrev, (PVOID *)pprmemAllocTail);

/*                                                                          */
/* AddFreeMemList does the real work                                        */
/*                                                                          */

   AddFreeMemList(prmem, prdsp, usFlag);

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspmem::FreeMem exit ulRC=0\n");
   return (DSP_NOERROR);
}
