//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: mtscale.cpp,v 1.1.1.1 2003/10/29 10:06:27 wschweer Exp $
//  (C) Copyright 1999 Werner Schweer (ws@seh.de)
//=========================================================

#include <cmath>
#include <qpainter.h>
#include <qtooltip.h>

#include "mtscale.h"
#include "midieditor.h"
#include "globals.h"
#include "song.h"
#include "../marker/marker.h"
#include "icons.h"

//---------------------------------------------------------
//   MTScale
//    Midi Time Scale
//---------------------------------------------------------

MTScale::MTScale(int* r, QWidget* parent, int xs, bool _mode)
   : View(parent, xs, 1)
      {
      waveMode = _mode;
      QToolTip::add(this, tr("bar scale"));
      barLocator = false;
      raster = r;
      if (waveMode) {
            pos[0] = int(tempomap.tick2time(song->cpos()) * sampleRate);
            pos[1] = int(tempomap.tick2time(song->lpos()) * sampleRate);
            pos[2] = int(tempomap.tick2time(song->rpos()) * sampleRate);
            }
      else {
            pos[0] = song->cpos();
            pos[1] = song->lpos();
            pos[2] = song->rpos();
            }
      pos[3] = -1;            // do not show
      button = QMouseEvent::NoButton;
      setMouseTracking(true);
      connect(song, SIGNAL(posChanged(int, int, bool)), SLOT(setPos(int, int, bool)));
      connect(song, SIGNAL(songChanged(int)), SLOT(songChanged(int)));
      connect(song, SIGNAL(markerChanged(int)), SLOT(redraw()));
      setFixedHeight(28);
      setBg(QColor(0xe0, 0xe0, 0xe0));
      }

//---------------------------------------------------------
//   songChanged
//---------------------------------------------------------

void MTScale::songChanged(int type)
      {
      if (type & (SC_SIG|SC_TEMPO)) {
           if ((type & SC_TEMPO) && waveMode) {
                  pos[0] = int(tempomap.tick2time(song->cpos())*sampleRate);
                  pos[1] = int(tempomap.tick2time(song->lpos())*sampleRate);
                  pos[2] = int(tempomap.tick2time(song->rpos())*sampleRate);
                  }
            redraw();
            }
      }

//---------------------------------------------------------
//   setPos
//---------------------------------------------------------

void MTScale::setPos(int idx, int val, bool)
      {
      if (waveMode)
            val = int(tempomap.tick2time(val) * sampleRate);
      if (val == pos[idx])
            return;
      int opos = mapx(pos[idx] == -1 ? val : pos[idx]);
      pos[idx] = val;
      if (!isVisible())
            return;
      val   = mapx(val);
      int x = -9;
      int w = 18;
      if (opos > val) {
            w += opos - val;
            x += val;
            }
      else {
            w += val - opos;
            x += opos;
            }
      redraw(QRect(x, 0, w, height()));
      }

//---------------------------------------------------------
//   viewMousePressEvent
//---------------------------------------------------------

void MTScale::viewMousePressEvent(QMouseEvent* event)
      {
      button = event->button();
      viewMouseMoveEvent(event);
      }

//---------------------------------------------------------
//   viewMouseReleaseEvent
//---------------------------------------------------------

void MTScale::viewMouseReleaseEvent(QMouseEvent*)
      {
      button = QMouseEvent::NoButton;
      }

//---------------------------------------------------------
//   viewMouseMoveEvent
//---------------------------------------------------------

void MTScale::viewMouseMoveEvent(QMouseEvent* event)
      {
      int x = event->x();
      if (waveMode)
            x = tempomap.time2tick(double(x)/double(sampleRate));
      x = sigmap.raster(x, *raster);
      if (x < 0)
            x = 0;
      emit timeChanged(x);
      int i;
      switch (button) {
            case QMouseEvent::LeftButton:
                  i = 0;
                  break;
            case QMouseEvent::MidButton:
                  i = 1;
                  break;
            case QMouseEvent::RightButton:
                  i = 2;
                  break;
            default:
                  return;
            }
      song->setPos(i, x);
      }

//---------------------------------------------------------
//   leaveEvent
//---------------------------------------------------------

void MTScale::leaveEvent(QEvent*)
      {
      emit timeChanged(-1);
      }

//---------------------------------------------------------
//   draw
//---------------------------------------------------------

void MTScale::pdraw(QPainter& p, const QRect& r)
      {
      int x = r.x();
      int w = r.width();

      x -= 20;
      w += 40;    // wg. Text

      //---------------------------------------------------
      //    draw Marker
      //---------------------------------------------------

      int y = 12;
      p.setPen(black);
      p.setFont(font4);
      p.drawLine(r.x(), y+1, r.x() + r.width(), y+1);
      QRect tr(r);
      tr.setHeight(12);
      MarkerList* marker = song->marker();
      for (iMarker m = marker->begin(); m != marker->end(); ++m) {
            int xp = mapx(m->second.posTick());
            if (xp > x+w)
                  break;
            int xe = r.x() + r.width();
            iMarker mm = m;
            ++mm;
            if (mm != marker->end()) {
                  xe = mapx(mm->first);
                  }
            QRect tr(xp, 0, xe-xp, 13);
            if (m->second.current()) {
                  p.fillRect(tr, white);
                  }
            if (r.intersects(tr)) {
                  int x2;
                  iMarker mm = m;
                  ++mm;
                  if (mm != marker->end())
                        x2 = mapx(mm->first);
                  else
                        x2 = xp+200;
                  QRect r  = QRect(xp+10, 0, x2-xp, 12);
                  p.drawPixmap(xp, 0, *flagIconS);
                  p.setPen(black);
                  p.drawText(r, AlignLeft|AlignVCenter, m->second.name());
                  p.setPen(green);
                  p.drawLine(xp, y, xp, height());
                  }
            }

      //---------------------------------------------------
      //    draw location marker
      //---------------------------------------------------

      int h = height()-12;

      if (barLocator) {
            p.setPen(red);
            int xp = mapx(pos[0]);
            if (xp >= x && xp < x+w)
                  p.drawLine(xp, y, xp, h);
            p.setPen(blue);
            xp = mapx(pos[1]);
            if (xp >= x && xp < x+w)
                  p.drawLine(xp, y, xp, h);
            xp = mapx(pos[2]);
            if (xp >= x && xp < x+w)
                  p.drawLine(xp, y, xp, h);
            }
      else {
            for (int i = 0; i < 3; ++i) {
                  int xp = mapx(pos[i]);
                  if (xp >= x && xp < x+w) {
                        QPixmap* pm = markIcon[i];
                        p.drawPixmap(xp - pm->width()/2, y-1, *pm);
                        }
                  }
            }
      p.setPen(black);
      if (pos[3] != -1) {
            int xp = mapx(pos[3]);
            if (xp >= x && xp < x+w)
                  p.drawLine(xp, 0, xp, height());
            }

      int ctick;
      int bar1, bar2, beat, tick;

      if (waveMode) {
            ctick = tempomap.time2tick(double(mapxDev(x))/double(sampleRate));
            sigmap.tickValues(ctick, &bar1, &beat, &tick);
            sigmap.tickValues(tempomap.time2tick(double(mapxDev(x+w))/double(sampleRate)),
               &bar2, &beat, &tick);
            }
      else {
            ctick = mapxDev(x);
            sigmap.tickValues(ctick, &bar1, &beat, &tick);
            sigmap.tickValues(mapxDev(x+w), &bar2, &beat, &tick);
            }

//printf("bar %d  %d-%d=%d\n", bar, ntick, stick, ntick-stick);

      int stick = sigmap.bar2tick(bar1, 0, 0);
      int ntick;
      for (int bar = bar1; bar <= bar2; bar++, stick = ntick) {
            ntick     = sigmap.bar2tick(bar+1, 0, 0);
            int tpix, a, b=0;
            if (waveMode) {
                  a = lrint(tempomap.tick2time(ntick) * double(sampleRate));
                  b = lrint(tempomap.tick2time(stick) * double(sampleRate));
                  tpix  = rmapx(a - b);
                  }
            else {
                  tpix  = rmapx(ntick - stick);
                  }
            if (tpix < 64) {
                  // dont show beats if measure is this small
                  int n = 1;
                  if (tpix < 32)
                        n = 2;
                  if (tpix <= 16)
                        n = 4;
                  if (tpix < 8)
                        n = 8;
                  if (tpix <= 4)
                        n = 16;
                  if (tpix <= 2)
                        n = 32;
                  if (bar % n)
                        continue;
                  p.setFont(font3);
                  int x = mapx(waveMode ? b : stick);
                  QString s;
                  s.setNum(bar + 1);
                  p.drawLine(x, y+1, x, y+1+h);
//                  QRect r = QRect(x+2, y, 0, h);
                  QRect r = QRect(x+2, y, 1000, h);
                  p.drawText(r, AlignLeft|AlignVCenter|DontClip, s);
                  }
            else {
                  int z, n;
                  sigmap.timesig(stick, z, n);
                  for (int beat = 0; beat < z; beat++) {
                        int xx = sigmap.bar2tick(bar, beat, 0);
                        if (waveMode)
                              xx = lrint(tempomap.tick2time(xx) * double(sampleRate));
                        int xp = mapx(xx);
                        QString s;
                        QRect r(xp+2, y, 1000, h);
                        int y1;
                        int num;
                        if (beat == 0) {
                              num = bar + 1;
                              y1  = y + 1;
                              p.setFont(font3);
                              }
                        else {
                              num = beat + 1;
                              y1  = y + 7;
                              p.setFont(font1);
                              r.setY(y+3);
                              }
                        s.setNum(num);
                        p.drawLine(xp, y1, xp, y+1+h);
                        p.drawText(r, AlignLeft|AlignVCenter|DontClip, s);
                        }
                  }
            }
      }

