//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: track.h,v 1.2 2001/11/20 15:19:32 muse Exp $
//
//  (C) Copyright 1999/2000 Werner Schweer (ws@seh.de)
//=========================================================

#ifndef __TRACK_H__
#define __TRACK_H__

#include <qstring.h>
#include <vector>
#include <map>
#include "part.h"
#include "key.h"
#include "audionode.h"
#include "ssource.h"
#include "mixdowndialog.h"
#include "audiothread.h"

class Pipeline;
class Xml;
class AudioChannel;
class Device;
class SndFile;

enum StaffMode { Single, Split };

//---------------------------------------------------------
//   Track
//---------------------------------------------------------

class Track {
      static int snGen;
      bool _recordFlag;        // enables recording for this track
      QString _comment;
      int _outPort;
      int _outChannel;
      int _inPortMask;        // bitmask of accepted record ports
      int _inChannelMask;     // bitmask of accepted record channels
      QString _name;
      int _sn;

   public:
      enum TrackType { MIDI=0, DRUM, WAVE };

   private:
      TrackType _type;
      bool _locked;
      bool _selected;
      PartList _parts;

      void init();

   public:
      Track(const Track&);
      Track(TrackType);
      Track(const QString&, TrackType);
      virtual ~Track();
      int sn() const                  { return _sn; }
      virtual void setSn(int n);
      int newSn()                     { return snGen++; }
      QString comment() const         { return _comment; }
      void setComment(const QString& s) { _comment = s; }

      bool selected() const           { return _selected; }
      void setSelected(bool f)        { _selected = f; }
      bool locked() const             { return _locked; }
      void setLocked(bool b)          { _locked = b; }

      bool recordFlag() const         { return _recordFlag; }
      void setRecordFlag(bool f)      { _recordFlag = f; }

      const QString& tname() const    { return _name; }
      void setTName(const QString& s) { _name = s; }
      TrackType type() const          { return _type; }
      void setType(TrackType t)       { _type = t; }
      PartList* parts()               { return &_parts; }
      const PartList* cparts() const  { return &_parts; }
      Part* findPart(int tick);
      iPart addPart(Part* p);

      virtual void read(Xml&);
      virtual void write(int, Xml&) const;
      virtual Track* newTrack() const = 0;
      virtual Track* clone() const = 0;
      virtual void copy(const Track*) = 0;

      void setOutChannel(int i);
      void setInChannelMask(int i);
      void setOutPort(int i);
      void setInPortMask(int i);
      int outPort() const             { return _outPort;     }
      int inPortMask() const          { return _inPortMask;  }
      int outChannel() const          { return _outChannel;  }
      int inChannelMask() const       { return _inChannelMask; }

      virtual Device* outDevice() const = 0;
      virtual Device* inDevice() const = 0;

      virtual Part* newPart(Part*p=0) = 0;
      void dump() const;

      virtual void splitPart(Part*, int, Part*&, Part*&);
      };

//---------------------------------------------------------
//   MidiTrack
//---------------------------------------------------------

class MidiTrack : public Track, public SoundSource {
      EventList* _events;     // tmp Events during midi import
                              // and recording
      bool _midiThruFlag;     // echo received events to outPort

   protected:

   public:
      MidiTrack();
      MidiTrack(const MidiTrack&);
      MidiTrack(const QString&);
      virtual ~MidiTrack();

      void init();

      NKey key;               // Notenschlssel
      Scale scale;            // Tonart
      NKey keyL;              // lower System (mode == Split)
      Scale scaleL;

      int noteQuant;          // Noten Darstellungs-Quantisierung
      int restQuant;          // Pausen Darstellungs-Quantisierung
      StaffMode mode;
      int splitpoint;

      // play parameter
      int transposition;
      int velocity;
      int delay;
      int len;
      int compression;

      EventList* events() const        { return _events; }

      virtual void read(Xml&);
      virtual void write(int, Xml&) const;
      void readStaff(Xml& xml);

      bool midiThruFlag() const     { return _midiThruFlag; }
      void setMidiThruFlag(bool f)  { _midiThruFlag = f; }

      virtual MidiTrack* newTrack() const { return new MidiTrack(); }
      virtual MidiTrack* clone() const;
      virtual void copy(const Track* t);
      virtual Part* newPart(Part*p=0);
      virtual Device* outDevice() const;
      virtual Device* inDevice() const;
      };

//---------------------------------------------------------
//   WaveTrack
//---------------------------------------------------------

class WaveTrack : public Track, public AudioNode {
      SndFile* _recFile;

      void readRecfile(Xml&);
      void init();

      float* buffer[PREFETCH_BUFFER];

   public:
      WaveTrack();
      WaveTrack(const WaveTrack&);
      virtual ~WaveTrack();
      WaveTrack(QString const& s);

      virtual void read(Xml&);
      virtual void write(int, Xml&) const;

      virtual WaveTrack* newTrack() const { return new WaveTrack(); }
      virtual WaveTrack* clone() const;
      virtual void copy(const Track* t);
      virtual Part* newPart(Part*p=0);

      virtual Device* outDevice() const;
      virtual Device* inDevice() const;

      virtual void getData(float*, int);
      virtual void fetchData(int bufferNo, int samplePos, bool realloc);

      SndFile* recFile() const { return _recFile; }
      void setRecFile(SndFile*);

      virtual void setSn(int n);
      };

//---------------------------------------------------------
//   TrackList
//---------------------------------------------------------

struct TrackList : public std::list<Track*> {
      typedef TrackList::iterator iTrack;
      Track* add(Track*);
      void deselect();
      unsigned index(Track*);
      Track* sfind(int sn);   // search serial number
      Track* findIdx(int sn);   // search index
      iTrack find(int n);
      };

typedef TrackList::iterator iTrack;
typedef TrackList::const_iterator ciTrack;

#endif

