//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: sfont.cpp,v 1.1 2002/01/30 12:08:39 muse Exp $
//
//  This file is derived from IIWU Synth and modified
//    for MusE.
//  Parts of IIWU are derived from Smurf Sound Font Editor.
//  Parts of Smurf Sound Font Editor are derived from
//    awesfx utilities
//  Smurf:  Copyright (C) 1999-2000 Josh Green
//  IIWU:   Copyright (C) 2001 Peter Hanappe
//  MusE:   Copyright (C) 2001 Werner Schweer
//  awesfx: Copyright (C) 1996-1999 Takashi Iwai
//=========================================================

#include "sfont.h"
#include <stdarg.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>

enum {
      UNKN_ID, RIFF_ID, LIST_ID, SFBK_ID,
      INFO_ID, SDTA_ID, PDTA_ID,	/* info/sample/preset */

      IFIL_ID, ISNG_ID, INAM_ID, IROM_ID, /* info ids (1st byte of info strings) */
      IVER_ID, ICRD_ID, IENG_ID, IPRD_ID,	/* more info ids */
      ICOP_ID, ICMT_ID, ISFT_ID,	/* and yet more info ids */

      SNAM_ID, SMPL_ID,		/* sample ids */
      PHDR_ID, PBAG_ID, PMOD_ID, PGEN_ID,	/* preset ids */
      IHDR_ID, IBAG_ID, IMOD_ID, IGEN_ID,	/* instrument ids */
      SHDR_ID			/* sample info */
      };

//---------------------------------------------------------
//   gen_init_array
//    Set an array of generators to their initial value
//---------------------------------------------------------

void gen_init_array(Gen* gen)
      {
      int i;
      for (i = 0; i < GEN_LAST; i++) {
            gen[i].flags = GEN_UNUSED;
            gen[i].mod = 0.0;
            gen[i].val = 0.0;
            }
      gen[GEN_FILTERFC].val           = 13500;
      gen[GEN_MODLFODELAY].val        = -12000;
      gen[GEN_VIBLFODELAY].val        = -12000;
      gen[GEN_MODENVDELAY].val        = -12000;
      gen[GEN_MODENVATTACK].val       = -12000;
      gen[GEN_MODENVHOLD].val         = -12000;
      gen[GEN_MODENVDECAY].val        = -12000;
      gen[GEN_MODENVRELEASE].val      = -12000;
      gen[GEN_VOLENVDELAY].val        = -12000;
      gen[GEN_VOLENVATTACK].val       = -12000;
      gen[GEN_VOLENVHOLD].val         = -12000;
      gen[GEN_VOLENVDECAY].val        = -12000;
      gen[GEN_VOLENVRELEASE].val      = -12000;
      gen[GEN_KEYNUM].val             = -1;
      gen[GEN_VELOCITY].val           = -1;
      gen[GEN_SCALETUNE].val          = 100;
      gen[GEN_OVERRIDEROOTKEY].val    = -1;
      }

//---------------------------------------------------------
//   inside_range
//---------------------------------------------------------

int Zone::inside_range(int key, int vel)
      {
      return ((keylo <= key) && (keyhi >= key) && (vello <= vel) && (velhi >= vel));
      }

//---------------------------------------------------------
//   Preset
//---------------------------------------------------------

Preset::Preset()
      {
      next        = 0;
      name        = 0;
      hbank       = 0;
      lbank       = 0;
      prog        = 0;
      zone        = 0;
      _sfont      = 0;
      }

//---------------------------------------------------------
//   Preset
//---------------------------------------------------------

Preset::~Preset()
      {
      Zone* z = zone;
      while (z) {
            Zone* zz = z->next;
            delete z;
            z = zz;
            }
      }

//---------------------------------------------------------
//   Zone
//---------------------------------------------------------

Zone::Zone()
      {
      next       = 0;
      instsampno = 0;
      samp       = 0;
      sfgen      = 0;
      mod        = 0;
      inst       = 0;
      keylo      = 0;
      vello      = 0;
      velhi      = keyhi = 128;
      gen_init_array(gen);
      }

//---------------------------------------------------------
//   ~Zone
//---------------------------------------------------------

Zone::~Zone()
      {
      SFMod* m = mod;
      while (m) {				// Free mod chunks for this zone
            SFMod* nm = m->next;
            delete m;
            m = nm;
            }
      }

//---------------------------------------------------------
//   chunkid
//---------------------------------------------------------

static int chunkid(unsigned id)
      {
      static char idlist[] = {
            "RIFFLISTsfbkINFOsdtapdtaifilisngINAMiromiverICRDIENGIPRD"
            "ICOPICMTISFTsnamsmplphdrpbagpmodpgeninstibagimodigenshdr"
            };
      unsigned* p = (unsigned *) & idlist;

      for (unsigned i = 0; i < sizeof (idlist) / sizeof (int); ++i, ++p)
            if (*p == id)
                  return (i + 1);
      return UNKN_ID;
      }

//---------------------------------------------------------
//   gen_valid
//    check validity of instrument generator
//    return true if valid
//---------------------------------------------------------

static bool gen_valid(unsigned gen)
      {
      static unsigned badgen[] = {
            Gen_Unused1, Gen_Unused2, Gen_Unused3, Gen_Unused4,
            Gen_Reserved1, Gen_Reserved2, Gen_Reserved3, 0
            };

      if (gen > Gen_MaxValid)
            return false;
      unsigned i;
      for (i = 0; i < sizeof(badgen)/sizeof(*badgen); ++i)
            if (badgen[i] == gen)
                  return false;
      return true;
      }

//---------------------------------------------------------
//   gen_validp
//    check validity of preset generator
//---------------------------------------------------------

static bool gen_validp(unsigned int gen)
      {				/* is preset generator valid? */
      static unsigned badpgen[] = {
            Gen_StartAddrOfs, Gen_EndAddrOfs, Gen_StartLoopAddrOfs,
            Gen_EndLoopAddrOfs, Gen_StartAddrCoarseOfs, Gen_EndAddrCoarseOfs,
            Gen_StartLoopAddrCoarseOfs, Gen_Keynum, Gen_Velocity,
            Gen_EndLoopAddrCoarseOfs, Gen_SampleModes, Gen_ExclusiveClass,
            Gen_OverrideRootKey, 0
            };

      if (!gen_valid (gen))
            return false;
      int i = 0;
      while (badpgen[i] && badpgen[i] != (unsigned) gen)
            i++;
      return (badpgen[i] == 0);
      }

//---------------------------------------------------------
//   gerr
//    Logging function
//---------------------------------------------------------

static void gerr(char * fmt, ...)
      {
      va_list args;

      va_start (args, fmt);
      vprintf(fmt, args);
      va_end (args);

      printf("\n");
      }

//---------------------------------------------------------
//   sfont_zone_delete
//    delete zone from zone list
//---------------------------------------------------------

static void sfont_zone_delete(Zone** zlist, Zone* zone)
      {
      if (zone == 0)
            return;
      Zone* prev = 0;
      for (Zone* tmp  = *zlist; tmp; tmp = tmp->next) {
            if (tmp == zone) {
                  if (prev)
                        prev->next = tmp->next;
                  if (*zlist == tmp)
                        *zlist = (*zlist)->next;
                  tmp->next = 0;
                  break;
                  }
            prev = tmp;
            }
      delete zone;
      }

static void SLADVREMZ(Zone*& list, Zone*& item)
      {
      Zone* _temp = item;
      if (item) {
            item = item->next;
            sfont_zone_delete(&list, _temp);
            }
      }

//---------------------------------------------------------
//   safe_fread
//---------------------------------------------------------

void SFont::safe_fread(void *buf, int count)
      {
      if (fread (buf, count, 1, fd) != 1) { // size_t = count, nmemb = 1
            if (feof (fd))
                  longjmp(env, 51);
            else
                  longjmp(env, 52);
            }
      }

//---------------------------------------------------------
//   readid
//---------------------------------------------------------

void SFont::readid(unsigned* var)
      {
      safe_fread(var, 4);
      }

//---------------------------------------------------------
//   readstr
//---------------------------------------------------------

void SFont::readstr(char* var)
      {
      safe_fread(var, 20);
      var[20] = '\0';
      }

//---------------------------------------------------------
//   read_listchunk
//---------------------------------------------------------

void SFont::read_listchunk(SFChunk* chunk)
      {
      safe_fread(chunk, 8);
      if (chunkid (chunk->id) != LIST_ID)	/* error if ! list chunk */
            longjmp(env, 7);
      readid(&chunk->id);	/* read id string */
      chunk->size -= 4;
      }

//---------------------------------------------------------
//   readw
//---------------------------------------------------------

int SFont::readw()
      {
      short s;
      safe_fread(&s, 2);
      return s;
      }

//---------------------------------------------------------
//   readWord
//---------------------------------------------------------

unsigned int SFont::readWord()
      {
      unsigned short s;
      safe_fread(&s, 2);
      return s;
      }

//---------------------------------------------------------
//   readd
//---------------------------------------------------------

int SFont::readd()
      {
      unsigned s;
      safe_fread(&s, 4);
      return s;
      }

//---------------------------------------------------------
//   readb
//---------------------------------------------------------

int SFont::readb()
      {
      char var;
      safe_fread(&var, 1);
      return var;
      }

//---------------------------------------------------------
//   readByte
//    read unsigned char
//---------------------------------------------------------

unsigned int SFont::readByte()
      {
      unsigned char var;
      safe_fread(&var, 1);
      return var;
      }

//---------------------------------------------------------
//   fskip
//---------------------------------------------------------

void SFont::fskip(int size)
      {
      if (fseek (fd, size, SEEK_CUR) == -1)
            longjmp(env, 53);
      }

//---------------------------------------------------------
//   SFont
//---------------------------------------------------------

SFont::SFont()
      {
      file       = 0;
      next       = 0;
      samplepos  = 0;
      samplesize = 0;
      sample     = 0;
      sampledata = 0;
      info       = 0;
      inst       = 0;
      preset     = 0;
      }

//---------------------------------------------------------
//   ~SFont
//---------------------------------------------------------

SFont::~SFont()
      {
      if (file)
            delete file;
      while (sample) {
            Sample* s = sample->next;
            delete sample;
            sample = s;
            }
      if (sampledata)
            delete sampledata;

      for (SFInfo* i = info; i; ) {
            SFInfo* ni = i->next;
            delete i;
            i = ni;
            }

      while (preset) {
            Preset* nz = preset->next;
            delete preset;
            preset = nz;
            }

      while (inst) {
            Inst* ni = inst->next;
            delete inst;
            inst = ni;
            }
      }

//---------------------------------------------------------
//   process_info
//---------------------------------------------------------

void SFont::process_info(int size)
      {
      SFChunk chunk;

      SFInfo* ni = info;
      while (size > 0) {
            safe_fread(&chunk, 8);
            size -= 8;

            unsigned char id = chunkid(chunk.id);

            if (id == IFIL_ID) {          // sound font version chunk?
                  if (chunk.size != 4)
                        longjmp(env, 1);

                  version.major = readw();
                  version.minor = readw();

                  if (version.major < 2)
                        longjmp(env, 2);
                  if (version.major > 2)
                        gerr ("Sound font version is %d.%d which is newer than"
                           " what this version of Smurf was designed for (v2.0x)");
                  }
            else if (id == IVER_ID) {     // ROM version chunk?
                  if (chunk.size != 4)
                        longjmp(env, 3);
                  romver.major = readw();
                  romver.minor = readw();
                  }
            else if (id != UNKN_ID) {
                  if ((id != ICMT_ID && chunk.size > 256) || (chunk.size > 65536)
                     || (chunk.size % 2))
                        longjmp(env, 4);

                  /* alloc for chunk id and da chunk */
                  SFInfo* item = new SFInfo;

                  item->data = new char[chunk.size+1];

                  /* attach to INFO list, sfont_close will cleanup if FAIL occurs */
                  item->next = 0;
                  if (ni)
                        ni->next = item;
                  else
                        info = item;
                  ni = item;

                  item->data[0] = id;
                  safe_fread (&item->data[1], chunk.size);

                  /* force terminate info item (don't forget uchar info ID) */
                  *(item->data + chunk.size) = '\0';
                  }
            else
                  longjmp(env, 5);
            size -= chunk.size;
            }

      if (size < 0)
            longjmp(env, 6);
      }

//---------------------------------------------------------
//   load_body
//---------------------------------------------------------

void SFont::load_body()
      {
      SFChunk chunk;

      safe_fread(&chunk, 8);
      if (chunkid (chunk.id) != RIFF_ID) {	/* error if not RIFF */
            longjmp(env, 8);
            }

      readid(&chunk.id);	            // load file ID */
      if (chunkid (chunk.id) != SFBK_ID)
            longjmp(env, 9);
      if (chunk.size != unsigned(fsize - 8))
            longjmp(env, 10);

      // Process INFO block
      read_listchunk(&chunk);
      if (chunkid (chunk.id) != INFO_ID)
            longjmp(env, 11);
      process_info(chunk.size);

      // Process sample chunk
      read_listchunk(&chunk);
      if (chunkid (chunk.id) != SDTA_ID)
            longjmp(env, 12);
      process_sdta(chunk.size);

      // process HYDRA chunk
      read_listchunk(&chunk);
      if (chunkid (chunk.id) != PDTA_ID)
            longjmp(env, 13);
      process_pdta(chunk.size);

      fixup_pgen();
      fixup_igen();
      fixup_sample();

      /* sort preset list by bank, preset # */
//TODO      preset = g_slist_sort(preset, (GCompareFunc) sfont_preset_compare_func);
      load_sampledata();
      }

//---------------------------------------------------------
//   process_sdta
//---------------------------------------------------------

void SFont::process_sdta(int size)
      {
      SFChunk chunk;

      if (size == 0)
            return;		/* no sample data? */

      /* read sub chunk */
      safe_fread(&chunk, 8);
      size -= 8;

      if (chunkid (chunk.id) != SMPL_ID)
            longjmp(env, 14);

      if ((size - chunk.size) != 0)
            longjmp(env, 15);

      /* sample data follows */
      samplepos = ftell(fd);

      /* used in fixup_sample() to check validity of sample headers */
      sdtachunk_size = chunk.size;
      samplesize     = chunk.size;
      fskip(chunk.size);
      }

//---------------------------------------------------------
//   pdtahelper
//---------------------------------------------------------

void SFont::pdtahelper(unsigned expid, unsigned reclen, SFChunk* chunk, int* size)
      {
      unsigned id;

      safe_fread(chunk, 8);
      *size -= 8;

      if ((id = chunkid (chunk->id)) != expid)
            longjmp(env, 16);
      if (chunk->size % reclen)	/* valid chunk size? */
            longjmp(env, 17);
      if ((*size -= chunk->size) < 0)
            longjmp(env, 18);
      }

//---------------------------------------------------------
//   load_phdr
//    preset header loader
//---------------------------------------------------------

void SFont::load_phdr(int size)
      {
      int i2;

      if (size % SFPHDRSIZE || size == 0)
            longjmp(env, 19);

      int i = size / SFPHDRSIZE - 1;
      if (i == 0) {				/* at least one preset + term record */
            gerr("File contains no presets");
            fskip(SFPHDRSIZE);
            return;
            }

      Preset* pr = preset;	      // ptr to previous preset
      int pzndx  = 0;

      for (; i > 0; i--) {
            Preset* p = new Preset();
            p->setSFont(this);
            char buffer[21];
            readstr(buffer);
            p->name = strdup(buffer);
            p->prog   = readWord();
            int lbank = readWord();
            if (lbank == 128) {  // drum set
                  p->hbank = 1;
                  }
            else {
                  p->lbank = lbank;
                  }
            int zndx    = readWord();
            p->zoneIndex = zndx;
            p->libr     = readd();
            p->genre    = readd();
            p->morph    = readd();

            if (pr) {			/* not first preset? */
                  pr->next = p;
                  if (zndx < pzndx)
                        longjmp(env, 20);
                  i2 = zndx - pzndx;
                  while (i2--) {
                        Zone* z = new Zone;
                        z->next   = pr->zone;
                        pr->zone  = z;
                        }
                  }
            else {
                  preset = p;
                  if (zndx > 0)     /* 1st preset, warn if ofs >0 */
                        longjmp(env, 21);
                  }
            pr    = p;			/* update preset ptr */
            pzndx = zndx;
            }

      fskip(24);
      int zndx  = readWord();        /* Read terminal generator index */
      fskip(12);

      if (zndx < pzndx)
            longjmp(env, 22);
      i2 = zndx - pzndx;
      while (i2--) {
            Zone* z = new Zone;
            z->next   = pr->zone;
            pr->zone  = z;
            }
      }

//---------------------------------------------------------
//   process_pdta
//---------------------------------------------------------

void SFont::process_pdta(int size)
      {
      SFChunk chunk;

      pdtahelper(PHDR_ID, SFPHDRSIZE, &chunk, &size);
      load_phdr(chunk.size);

      pdtahelper(PBAG_ID, SFBAGSIZE, &chunk, &size);
      load_pbag(chunk.size);

      pdtahelper(PMOD_ID, SFMODSIZE, &chunk, &size);
      load_pmod(chunk.size);

      pdtahelper(PGEN_ID, SFGENSIZE, &chunk, &size);
      load_pgen(chunk.size);

      pdtahelper(IHDR_ID, SFIHDRSIZE, &chunk, &size);
      load_ihdr(chunk.size);

      pdtahelper(IBAG_ID, SFBAGSIZE, &chunk, &size);
      load_ibag(chunk.size);

      pdtahelper(IMOD_ID, SFMODSIZE, &chunk, &size);
      load_imod(chunk.size);

      pdtahelper(IGEN_ID, SFGENSIZE, &chunk, &size);
      load_igen(chunk.size);

      pdtahelper(SHDR_ID, SFSHDRSIZE, &chunk, &size);
      load_shdr(chunk.size);
      }

//---------------------------------------------------------
//   load_pbag
//    preset bag loader
//---------------------------------------------------------

void SFont::load_pbag(int size)
      {
      Zone* pz = 0;
      unsigned genndx, modndx;
      unsigned pgenndx=0, pmodndx = 0;

      if ((size % SFBAGSIZE) || size == 0)  // size is multiple of SFBAGSIZE?
            longjmp(env, 23);

      for (Preset* p = preset; p; p = p->next) {
            for (Zone* z = p->zone; z; z = z->next) {
                  size -= SFBAGSIZE;
                  if (size < 0)
                        longjmp(env, 24);
                  genndx      = readWord();
                  modndx      = readWord();

                  if (pz) {			/* if not first zone */
                        if (genndx < pgenndx)
                              longjmp(env, 25);
                        if (modndx < pmodndx)
                              longjmp(env, 26);
// printf("preset <%s> Gen:%d Mod:%d\n", p->name, genndx-pgenndx, modndx-pmodndx);
	                  int i = genndx - pgenndx;
                        pz->sfgen += i;
                        i = modndx - pmodndx;
                        while (i--) {
                              SFMod* m = new SFMod;
                              m->next = pz->mod;
                              pz->mod = m;
                              }
                        }
                  pz      = z;		// update previous zone ptr
                  pgenndx = genndx;	/* update previous zone gen index */
                  pmodndx = modndx;	/* update previous zone mod index */
                  }
            }
      size -= SFBAGSIZE;
      if (size != 0)
            longjmp(env, 24);

      genndx = readWord();
      modndx = readWord();

      if (!pz) {
            if (genndx > 0)
                  longjmp(env, 27);
            if (modndx > 0)
                  longjmp(env, 28);
            return;
            }

      if (genndx < pgenndx)
            longjmp(env, 29);
      if (modndx < pmodndx)
            longjmp(env, 30);
      int i = genndx - pgenndx;

      pz->sfgen += i;
      i = modndx - pmodndx;
      while (i--) {
            SFMod* m = new SFMod;
            m->next = pz->mod;
            pz->mod = m;
            }
      }

//---------------------------------------------------------
//   load_pmod
//    preset modulator loader
//---------------------------------------------------------

void SFont::load_pmod(int size)
      {
      for (Preset* p = preset; p; p = p->next) {
            for (Zone* z = p->zone; z; z = z->next) {
                  for (SFMod* m = z->mod; m; m = m->next) {
                        if ((size -= SFMODSIZE) < 0)
                              longjmp(env, 31);
                        m->src    = readWord();
                        m->dest   = readWord();
                        m->amount = readw();
                        m->amtsrc = readWord();
                        m->trans  = readWord();
                        }
                  }
            }
      /*
       * If there isn't even a terminal record
       * Hmmm, the specs say there should be one, but..
       */
      if (size == 0)
            return;

      size -= SFMODSIZE;
      if (size != 0)
            longjmp(env, 32);
      fskip(SFMODSIZE);	/* terminal mod */
      }

//---------------------------------------------------------
//   load_ihdr
//    instrument header loader
//---------------------------------------------------------

void SFont::load_ihdr(int size)
      {
      if (size % SFIHDRSIZE || size == 0)	/* chunk size is valid? */
            longjmp(env, 34);

      size = size / SFIHDRSIZE - 1;
      if (size == 0) {				/* at least one preset + term record */
            gerr ("File contains no instruments");
            fskip(SFIHDRSIZE);
            return;
            }
      Inst *pr = inst;	   // ptr to current & previous instrument
      int pzndx  = 0;

      for (int i = 0; i < size; i++) { /* load all instrument headers */
            Inst* p = new Inst;
            p->next = 0;
            p->zone  = 0;		/* For proper cleanup if fail (sfont_close) */
            readstr(p->name);	
            int zndx = readWord();
            if (pr) {			/* not first instrument? */
                  pr->next = p;
                  if (zndx < pzndx)
                        longjmp(env, 35);
                  int i2 = zndx - pzndx;
                  while (i2--) {
                        Zone* z = new Zone;
                        z->next = pr->zone;
                        pr->zone = z;
                        }
                  }
            else {
                  inst = p;
                  if (zndx > 0)     /* 1st inst, warn if ofs >0 */
                        gerr ("%d instrument zones not referenced, discarding", zndx);
                  }
            pzndx = zndx;
            pr    = p;			/* update instrument ptr */
            }

      fskip(20);
      int zndx = readWord();

      if (zndx < pzndx)
            longjmp(env, 35);
      int i2 = zndx - pzndx;
      while (i2--) {
            Zone* z = new Zone;
            z->next = pr->zone;
            pr->zone = z;
            }
      }

//---------------------------------------------------------
//   load_ibag
//    instrument bag loader
//---------------------------------------------------------

void SFont::load_ibag(int size)
      {
      Zone *pz = 0;
      unsigned int genndx, modndx;
      unsigned int pgenndx=0, pmodndx=0;

      if (size % SFBAGSIZE || size == 0)	/* size is multiple of SFBAGSIZE? */
            longjmp(env, 36);

      for (Inst* p = inst; p; p = p->next) {
            for (Zone* z = p->zone; z; z = z->next) {
                  if ((size -= SFBAGSIZE) < 0)
                        longjmp(env, 37);
                  genndx = readWord();
                  modndx = readWord();

                  if (pz) {			/* if not first zone */
                        if (genndx < pgenndx)
                              longjmp(env, 38);
                        if (modndx < pmodndx)
                              longjmp(env, 39);
                        pz->sfgen += genndx - pgenndx;
                        for (unsigned i = pmodndx; i < modndx; ++i) {
                              SFMod* m = new SFMod;
                              m->next = pz->mod;
                              pz->mod = m;
                              }
                        }
                  pz      = z;	   /* update previous zone ptr */
                  pgenndx = genndx;
                  pmodndx = modndx;
                  }
            }

      size -= SFBAGSIZE;
      if (size != 0)
            longjmp(env, 40);

      genndx = readWord();
      modndx = readWord();

      if (!pz) {				/* in case that all are no zoners */
            if (genndx > 0)
                  gerr ("No instrument generators and terminal index not 0");
            if (modndx > 0)
                  gerr ("No instrument modulators and terminal index not 0");
            return;
            }

      if (genndx < pgenndx)
            longjmp(env, 41);
      if (modndx < pmodndx)
            longjmp(env, 42);

      pz->sfgen += genndx - pgenndx;
      for (unsigned i = pmodndx; i < modndx; ++i) {
            SFMod* m = new SFMod;
            m->next = pz->mod;
            pz->mod = m;
            }
      }

//---------------------------------------------------------
//   load_imod
//    instrument modulator loader
//---------------------------------------------------------

void SFont::load_imod(int size)
      {
      for (Inst* p = inst; p; p = p->next) {
            for (Zone* p2 = p->zone; p2; p2 = p2->next) {
                  for (SFMod* m = p2->mod; m; m = m->next) {
                        if ((size -= SFMODSIZE) < 0)
                              longjmp(env, 43);
                        m->src    = readWord();
                        m->dest   = readWord();
                        m->amount = readw();
                        m->amtsrc = readWord();
                        m->trans  = readWord();
                        }
                  }
            }
      /*
         If there isn't even a terminal record
         Hmmm, the specs say there should be one, but..
       */
      if (size == 0)
            return;

      size -= SFMODSIZE;
      if (size != 0)
            longjmp(env, 44);
      fskip(SFMODSIZE);	/* terminal mod */
      }

//---------------------------------------------------------
//   load_pgen
//    preset generator loader
//    generator (per preset) loading rules:
//    Zones with no generators or modulators shall be annihilated
//    Global zone must be 1st zone, discard additional ones (instrumentless zones)
//
//    generator (per zone) loading rules (in order of decreasing precedence):
//    KeyRange is 1st in list (if exists), else discard
//    if a VelRange exists only
//          preceded by a KeyRange, else discard
//    if a generator follows an instrument
//          discard it
//    if a duplicate generator exists
//          replace previous one
//---------------------------------------------------------

void SFont::load_pgen(int size)
      {
      for (Preset* p = preset; p; p = p->next) {
            Zone** hz = 0;
            if (p->zone)
                  hz = &p->zone;

            bool gzone = false;
            for (Zone* zone = p->zone; zone; zone = zone->next) {
                  int level  = 0;
                  for (int gen = 0; gen < zone->sfgen; ++gen) {
                        if ((size -= SFGENSIZE) < 0)
                              longjmp(env, 32);

                        if (level == 3) {
                              fskip(SFGENSIZE);
                              continue;
                              }

                        unsigned genid = readw();

                        if (genid == Gen_KeyRange) {
                              unsigned char a = readByte();
                              unsigned char b = readByte();
                              if (level == 0) {
                                    level = 1;
                                    zone->keylo = a;
                                    zone->keyhi = b;
                                    }
                              }
                        else if (genid == Gen_VelRange) {
                              unsigned char a = readByte();
                              unsigned char b = readByte();
                              if (level <= 1) {
                                    level = 2;
                                    zone->vello = a;
                                    zone->velhi = b;
                                    }
                              }
                        else if (genid == Gen_Instrument) {	// inst is last gen
                              level = 3;
                              zone->instsampno = readWord() + 1;
                              }
                        else {
                              level = 2;
                              int data = readw();
                              if (gen_validp(genid)) {
                                    zone->gen[genid].val   = double(data);
                                    zone->gen[genid].flags = GEN_SET;
                                    }
                              }
                        }

                  if (level != 3) {
                        if (!gzone) {	// Prior global zones?
                              gzone = true;

                              // if global zone is not 1st zone, relocate
                              if (*hz != zone) {
                                    Zone* save = zone;
                                    Zone* pp = zone;
                                    zone = zone->next;
                                    sfont_zone_delete(hz, pp);
                                    save->next = *hz;
                                    *hz = save;
                                    continue;
                                    }
                              }
                        else {    // previous global zone exists, discard
                              sfont_zone_delete(hz, zone);
                              }
                        }
                  }
            }
      // in case there isn't a terminal record
      if (size == 0)
            return;

      size -= SFGENSIZE;
      if (size != 0)
            longjmp(env, 33);
      fskip(SFGENSIZE);
      }

//---------------------------------------------------------
//   load_igen
//    load instrument generators
//    (see load_pgen for loading rules)
//---------------------------------------------------------

void SFont::load_igen(int size)
      {
      for (Inst* ip = inst; ip; ip = ip->next) {
            Zone** hz = 0;
            if (ip->zone)
                  hz = &ip->zone;

            bool gzone = false;
            for (Zone* zone = ip->zone; zone; zone = zone->next) {
                  int level = 0;
                  for (int gen = 0; gen < zone->sfgen; ++gen) {
                        if ((size -= SFGENSIZE) < 0)
                              longjmp(env, 45);

                        if (level == 3) {
                              fskip(SFGENSIZE);
                              continue;
                              }

                        unsigned genid = readWord();

                        if (genid == Gen_KeyRange) {
                              unsigned char a = readByte();
                              unsigned char b = readByte();
                              if (level == 0) {
                                    level = 1;
                                    zone->keylo = a;
                                    zone->keyhi = b;
                                    }
                              }
                        else if (genid == Gen_VelRange) {
                              unsigned char a = readByte();
                              unsigned char b = readByte();
                              if (level <= 1) {
                                    level = 2;
                                    zone->vello = a;
                                    zone->velhi = b;
                                    }
                              }
                        else if (genid == Gen_SampleId) {
                              level = 3;
                              zone->instsampno = readWord() + 1;
                              }
                        else {
                              level = 2;
                              int data = readw();
                              if (gen_valid(genid)) {		/* gen valid? */
                                    zone->gen[genid].val   = double(data);
                                    zone->gen[genid].flags = GEN_SET;
                                    }
                              }
                        }

                  if (level != 3) {
                        if (!gzone) {
                              gzone = true;

                              /* if global zone is not 1st zone, relocate */
                              if (*hz != zone) {
                                    Zone* save = zone;
                                    SLADVREMZ (*hz, zone);
                                    save->next = *hz;
                                    *hz = save;
                                    continue;
                                    }
                              }
                        else {
                              sfont_zone_delete (hz, zone);
                              }
                        }
                  }
            }

      /* for those non-terminal record cases, grr! */
      if (size == 0)
            return;

      size -= SFGENSIZE;
      if (size != 0)
            longjmp(env, 47);
      fskip(SFGENSIZE);	/* terminal gen */
      }

//---------------------------------------------------------
//   load_shdr
//    sample header loader
//---------------------------------------------------------

void SFont::load_shdr(int size)
      {
      if (size % SFSHDRSIZE || size == 0)	/* size is multiple of SHDR size? */
            longjmp(env, 48);

      size = size / SFSHDRSIZE - 1;
      if (size == 0) {				/* at least one sample + term record? */
            gerr ("File contains no samples");
            fskip(SFSHDRSIZE);
            return;
            }
      /* load all sample headers */
      Sample* ps = sample;
      while (ps && ps->next)
            ps = ps->next;

      for (int i = 0; i < size; i++) {
            Sample* p = new Sample;
            p->next = 0;
            if (ps)
                  ps->next = p;
            else
                  sample = p;
            ps = p;

            readstr(p->name());
            p->start      = readd();
            p->end        = readd();      // end, loopstart and loopend
            p->loopstart  = readd();      // will be checked and turned into
            p->loopend    = readd();      // offsets in fixup_sample()
            p->samplerate = readd();

            p->origpitch  = readByte();
            p->pitchadj   = readb();
            fskip(2);		            // skip sample link
            p->sampletype = readWord();
            }
      fskip(SFSHDRSIZE);	/* skip terminal shdr */
      }

//---------------------------------------------------------
//   fixup_pgen
//    "fixup" (inst # -> inst ptr) instrument
//    references in preset list
//---------------------------------------------------------

void SFont::fixup_pgen()
      {
      for (Preset* p = preset; p; p = p->next) {
            for (Zone* z = p->zone; z; z = z->next) {
                  int i = z->instsampno;
                  if (i) {			/* load instrument # */
                        Inst* ii = inst;
                        --i;
                        while (i && ii) {
                              ii = ii->next;
                              --i;
                              }
                        if (!ii)
                              longjmp(env, 49);
	                  z->inst = ii;
                        }
                  }
            }
      }

//---------------------------------------------------------
//   fixup_igen
//    "fixup" (sample # -> sample ptr) sample references
//    in instrument list
//---------------------------------------------------------

void SFont::fixup_igen()
      {
      for (Inst* p = inst; p; p = p->next) {
            for (Zone* z = p->zone; z; z = z->next) {
                  int i = z->instsampno;
                  if (i) {			/* load sample # */
                        Sample* ii = sample;
                        --i;
                        while (i && ii) {
                              ii = ii->next;
                              --i;
                              }
                        if (!ii) {
//                              printf("invalid sample %d\n", z->instsampno);
                              longjmp(env, 50);
                              }
	                  z->samp = ii;
                        }
                  }
            }
      }

//---------------------------------------------------------
//   fixup_sample
//    convert sample end, loopstart and loopend to offsets
//    and check if valid
//---------------------------------------------------------

void SFont::fixup_sample()
      {
      for (Sample* sam = sample; sam; sam = sam->next) {
            /* if sample is not a ROM sample and end is over the sample data chunk
               or sam start is greater than 4 less than the end (at least 4 samples) */
            if ((!(sam->sampletype & SF_SAMPLETYPE_ROM) && sam->end
               > unsigned(sdtachunk_size)) || sam->start > (sam->end - 4)) {
                  gerr ("Sample '%s' start/end file positions are invalid,"
      	      " disabling and will not be saved", sam->name());

                  /* disable sample by setting all sample markers to 0 */
                  sam->start = sam->end = sam->loopstart = sam->loopend = 0;
                  return;
                  }
            else if (sam->loopend > sam->end || sam->loopstart >= sam->loopend
               || sam->loopstart <= sam->start) { /* loop is fowled?? (cluck cluck :) */
                  /* can pad loop by 8 samples and ensure at least 4 for loop (2*8+4) */
                  if ((sam->end - sam->start) >= 20) {
                        sam->loopstart = sam->start + 8;
                        sam->loopend = sam->end - 8;
                        }
                  else { /* loop is fowled, sample is tiny (can't pad 8 samples) */
                        sam->loopstart = sam->start + 1;
                        sam->loopend = sam->end - 1;
                        }
                  }
            sam->end -= 1;	/* marks last sample, contrary to SF spec. */
            }
      }

//---------------------------------------------------------
//   load_sampledata
//---------------------------------------------------------

void SFont::load_sampledata()
      {
      if (fseek(fd, samplepos, SEEK_SET) == -1)
            longjmp(env, 53);
      sampledata = new short[samplesize];
      safe_fread(sampledata, samplesize);
      for (Sample* s = sample; s; s = s->next)
            s->setData(sampledata);
      }

//---------------------------------------------------------
//   get_sample
//---------------------------------------------------------

Sample* SFont::get_sample(const char *s) const
      {
      for (Sample* sp = sample; sp; sp = sp->next) {
            if (strcmp(sp->name(), s) == 0)
                  return sp;
            }
      return 0;
      }

//---------------------------------------------------------
//   load
//---------------------------------------------------------

int SFont::load(const char* fn)
      {
      static char* errMsg = "iiwu: load soundfont: %s failed: %s\n";

      file = strdup(fn);

      if (!(fd = fopen (file, "rb"))) {
            fprintf(stderr, errMsg, "open", strerror(errno));
            return 1;
            }
      struct stat buf;
      if (fstat(fileno(fd), &buf) == -1) {
            fprintf(stderr, errMsg, "stat", strerror(errno));
            return 1;
            }
      fsize = buf.st_size;


      int error = setjmp(env);
      if (error == 0) {
            load_body();
            }
      else {
            const char* s = "??";
            switch (error) {
                  case 0:  break;
                  case 1:  s = "Sound font version info chunk has invalid size"; break;
                  case 2:  s = "Sound font version unsupported, convert to version 2.0x"; break;
                  case 3:  s = "ROM version info chunk has invalid size"; break;
                  case 4:  s = "INFO sub chunk has invalid chunk size"; break;
                  case 5:  s = "Invalid chunk id in INFO chunk"; break;
                  case 6:  s = "INFO chunk size mismatch"; break;
                  case 7:  s = "Invalid chunk id in level 0 parse"; break;
                  case 8:  s = "Not a RIFF file"; break;
                  case 9:  s = "Not a sound font file"; break;
                  case 10: s = "Sound font file size mismatch"; break;
                  case 11: s = "Invalid ID found when expecting INFO chunk"; break;
                  case 12: s = "Invalid ID found when expecting SAMPLE chunk"; break;
                  case 13: s = "Invalid ID found when expecting HYDRA chunk"; break;
                  case 14: s = "Expected SMPL chunk found invalid id instead"; break;
                  case 15: s = "SDTA chunk size mismatch"; break;
                  case 16: s = "Unexpected PDTA sub-chunk"; break;
                  case 17: s = "chunk size is not a multiple of xxx bytes"; break;
                  case 18: s = "chunk size exceeds remaining PDTA chunk size"; break;
                  case 19: s = "Preset header chunk size is invalid"; break;
                  case 20: s = "Preset header indices not monotonic"; break;
                  case 21: s = "xx preset zones not referenced, discarding"; break;
                  case 22: s = "Preset header indices not monotonic"; break;
                  case 23: s = "Preset bag chunk size is invalid"; break;
                  case 24: s = "Preset bag chunk size mismatch"; break;
                  case 25: s = "Preset bag generator indices not monotonic"; break;
                  case 26: s = "Preset bag modulator indices not monotonic"; break;
                  case 27: s = "No preset generators and terminal index not 0"; break;
                  case 28: s = "No preset modulators and terminal index not 0"; break;
                  case 29: s = "Preset bag generator indices not monotonic"; break;
                  case 30: s = "Preset bag modulator indices not monotonic"; break;
                  case 31: s = "Preset modulator chunk size mismatch"; break;
                  case 32: s = "Preset modulator chunk size mismatch"; break;
                  case 33: s = "Preset generator chunk size mismatch"; break;
                  case 34: s = "Instrument header has invalid size"; break;
                  case 35: s = "Instrument header indices not monotonic"; break;
                  case 36: s = "Instrument bag chunk size is invalid"; break;
                  case 37: s = "Instrument bag chunk size mismatch"; break;
                  case 38: s = "Instrument generator indices not monotonic"; break;
                  case 39: s = "Instrument modulator indices not monotonic"; break;
                  case 40: s = "Instrument chunk size mismatch"; break;
                  case 41: s = "Instrument generator indices not monotonic"; break;
                  case 42: s = "Instrument modulator indices not monotonic"; break;
                  case 43: s = "Instrument modulator chunk size mismatch"; break;
                  case 44: s = "Instrument modulator chunk size mismatch"; break;
                  case 45: s = "IGEN chunk size mismatch"; break;
                  case 46: s = "Instrument generator chunk size mismatch"; break;
                  case 47: s = "IGEN chunk size mismatch"; break;
                  case 48: s = "Sample header has invalid size"; break;
                  case 49: s = "Preset xx xx: Invalid instrument reference"; break;
                  case 50: s = "Instrument xxx: Invalid sample reference"; break;
                  case 51: s = "File read: EOF"; break;
                  case 52: s = "File read"; break;
                  case 53: s = "seek"; break;
                  }
            fprintf(stderr, errMsg, "", s);
            fclose(fd);
            return true;
            }
//      dump();
      return 0;
      }

//---------------------------------------------------------
//   Inst
//---------------------------------------------------------

Inst::Inst()
      {
      zone        = 0;
      }

Inst::~Inst()
      {
      Zone* z = zone;
      while (z) {
            Zone* zz = z->next;
            delete zz;
            z = zz;
            }
      }

//---------------------------------------------------------
//   get_preset
//---------------------------------------------------------

Preset* SFont::get_preset(char hbank, char lbank, char prog)
      {
      Preset* p = preset;
      int n = 0;
      while (p) {
            if (p->hbank == hbank && p->lbank == lbank && (p->prog == prog)) {
                  return p;
                  }
            ++n;
            p = p->next;
            }
      if (next)
            return next->get_preset(hbank, lbank, prog);
      return 0;
      }

//---------------------------------------------------------
//   append_sfont
//---------------------------------------------------------

void SFont::append_sfont(SFont* sfont2)
      {
      if (sfont2 == 0)
            return;
      if (next)
            next->append_sfont(sfont2);
      else {
            sfont2->next = 0;
            next = sfont2;
            }
      }

//=========================================================
//    DEBUG
//=========================================================

#ifdef DEBUG
void SFont::dump() const
      {
      printf("Sfont <%s> Version %d.%d, size %d\n", file,
         version.major, version.minor, fsize);
      for (Preset* p = preset; p; p = p->next)
            p->dump();
      for (Inst* i = inst; i; i = i->next)
            i->dump();
      }

void Preset::dump() const
      {
      printf("Preset <%s> idx %d sfont %p\n", name, zoneIndex, _sfont);
      for (Zone* z = zone; z; z = z->next) {
            z->dump();
            }
      }
void Zone::dump() const
      {
      printf("  Zone %d-%d %d-%d\n", keylo,keyhi,vello,velhi);
      printf("    inst %p, samp %p, sfgen %p, mod %p\n",
         inst, samp, sfgen, mod);
      for (SFGen* g = sfgen; g; g = g->next)
            g->dump();
      for (SFMod* m = mod; m; m = m->next)
            m->dump();

      }

void Inst::dump() const
      {
      printf("Instrument <%s>\n", name);
      for (Zone* z = zone; z; z = z->next)
            z->dump();
      }

void SFGen::dump() const
      {
      printf("      SFGen: id %d\n", id);
      }

void SFMod::dump() const
      {
      printf("      SFMod: %d %d %d %d %d\n",
         src, dest, amount, amtsrc, trans);
      }
#endif

