//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: sndfile.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//  based on libsndfile:
//  Copyright (C) 1999-2000 Erik de Castro Lopo <erikd@zip.com.au>
//=========================================================

#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <mcheck.h>
#include <math.h>
#include <errno.h>

#include <qprogressdialog.h>

#include "sndfile.h"
#include "config.h"
#include "sfendian.h"

#if (CPU_IS_LITTLE_ENDIAN == 1)
#	define	MAKE_MARKER(a,b,c,d)		((a)|((b)<<8)|((c)<<16)|((d)<<24))
#elif (CPU_IS_BIG_ENDIAN == 1)
#	define	MAKE_MARKER(a,b,c,d)		(((a)<<24)|((b)<<16)|((c)<<8)|(d))
#else
#	error "Cannot determine endian-ness of processor."
#endif

#if 0
struct ErrorStruct {	
      int   error ;
	char	*str ;
      };

static ErrorStruct SndfileErrors [] = {
	{ SFE_NO_ERROR		 , "No Error." },
	{ SFE_BAD_FILE		 , "File does not exist or is not a regular file (possibly a pipe?)." },
	{ SFE_OPEN_FAILED		 , "Could not open file." },
	{ SFE_BAD_OPEN_FORMAT	 , "Bad format specified for file open." },
	{ SFE_BAD_SNDFILE_PTR	 , "Not a valid SNDFILE* pointer." },
	{ SFE_BAD_SF_INFO_PTR	 , "NULL SF_INFO pointer passed to libsndfile." },
	{ SFE_BAD_INT_FD		 , "Bad file descriptor." },
	{ SFE_BAD_INT_PTR		 , "Internal error, Bad pointer." },
	{ SFE_MALLOC_FAILED	 , "Internal malloc () failed." },
	{ SFE_BAD_SEEK		 , "Internal fseek() failed." },
	{ SFE_NOT_SEEKABLE	 , "Seek attempted on unseekable file type." },
	{ SFE_UNIMPLEMENTED	 , "File contains data in an unimplemented format." },
	{ SFE_BAD_READ_ALIGN  	 , "Attempt to read a non-integer number of channels." },
	{ SFE_BAD_WRITE_ALIGN 	 , "Attempt to write a non-integer number of channels." },
	{ SFE_UNKNOWN_FORMAT	 , "File contains data in an unknown format." },
	{ SFE_NOT_READMODE	 , "Read attempted on file currently open for write." },
	{ SFE_NOT_WRITEMODE	 , "Write attempted on file currently open for read." },
	{ SFE_BAD_MODE_RW		 , "This file format does not support read/write mode." },
	{ SFE_BAD_SF_INFO		 , "Internal error : SF_INFO struct incomplete." },

	{ SFE_SHORT_READ		 , "Short read error." },
	{ SFE_SHORT_WRITE		 , "Short write error." },
	{ SFE_INTERNAL		 , "Unspecified internal error." },
	
	{ SFE_WAV_NO_RIFF		 , "Error in WAV file. No 'RIFF' chunk marker." },
	{ SFE_WAV_NO_WAVE		 , "Error in WAV file. No 'WAVE' chunk marker." },
	{ SFE_WAV_NO_FMT		 , "Error in WAV file. No 'fmt ' chunk marker." },
	{ SFE_WAV_FMT_SHORT	 , "Error in WAV file. Short 'fmt ' chunk." },

	{ SFE_WAV_FMT_TOO_BIG	 , "Error in WAV file. 'fmt ' chunk too large." },

	{ SFE_WAV_BAD_FORMAT	 , "Error in WAV file. Errors in 'fmt ' chunk." },
	{ SFE_WAV_BAD_BLOCKALIGN , "Error in WAV file. Block alignment in 'fmt ' chunk is incorrect." },
	{ SFE_WAV_NO_DATA		 , "Error in WAV file. No 'data' chunk marker." },
	{ SFE_WAV_UNKNOWN_CHUNK	 , "Error in WAV file. File contains an unknown chunk marker." },
	
	{ SFE_WAV_ADPCM_NOT4BIT	 , "Error in ADPCM WAV file. Invalid bit width."},
	{ SFE_WAV_ADPCM_CHANNELS , "Error in ADPCM WAV file. Invalid number of channels."},
	{ SFE_WAV_GSM610_FORMAT	 , "Error in GSM610 WAV file. Invalid format chunk."},

	{ SFE_AIFF_NO_FORM	 , "Error in AIFF file, bad 'FORM' marker."},
	{ SFE_AIFF_UNKNOWN_CHUNK , "Error in AIFF file, unknown chunk."},
	{ SFE_COMM_CHUNK_SIZE	 , "Error in AIFF file, bad 'COMM' chunk size."},
	{ SFE_AIFF_NO_SSND	 , "Error in AIFF file, bad 'SSND' chunk."},
	{ SFE_AIFF_NO_DATA	 , "Error in AIFF file, no sound data."},
	
	{ SFE_AU_UNKNOWN_FORMAT	 , "Error in AU file, unknown format."},
	{ SFE_AU_NO_DOTSND	 , "Error in AU file, missing '.snd' or 'dns.' marker."},

	{ SFE_RAW_READ_BAD_SPEC	 , "Error while opening RAW file for read. Must specify format, pcmbitwidth and channels."},
	{ SFE_RAW_BAD_BITWIDTH	 , "Error. RAW file bitwidth must be a multiple of 8."},

	{ SFE_PAF_NO_MARKER	 , "Error in PAF file, no marker."},
	{ SFE_PAF_VERSION		 , "Error in PAF file, bad version."},
	{ SFE_PAF_UNKNOWN_FORMAT , "Error in PAF file, unknown format."},
	{ SFE_PAF_SHORT_HEADER	 , "Error in PAF file. File shorter than minimal header."},
	
	{ SFE_SVX_NO_FORM		 , "Error in 8SVX / 16SV file, no 'FORM' marker."},
	{ SFE_SVX_NO_BODY		 , "Error in 8SVX / 16SV file, no 'BODY' marker."},
	{ SFE_SVX_NO_DATA		 , "Error in 8SVX / 16SV file, no sound data."},
	{ SFE_SVX_BAD_COMP	 , "Error in 8SVX / 16SV file, unsupported compression format."},
	
	{ SFE_NIST_BAD_HEADER	 , "Error in NIST file, bad header."},
	{ SFE_NIST_BAD_ENCODING	 , "Error in NIST file, unsupported compression format."},

	{ SFE_MAX_ERROR		 , "Maximum error number." }
      };
#endif


static int is_au_snd_file (const QString ext)
      {	
	return ext == "au" || ext == "snd";
      }

static int guess_file_type (FILE *file, const QString ext)
      {	
      unsigned int buffer[3];

	fread(buffer, sizeof(buffer), 1, file);
	fseek(file, 0, SEEK_SET);

	if (buffer [0] == MAKE_MARKER ('R','I','F','F') && buffer [2] == MAKE_MARKER ('W','A','V','E'))
		return SF_FORMAT_WAV ;
		
	if (buffer[0] == MAKE_MARKER ('F','O','R','M')) {
            if (buffer [2] == MAKE_MARKER ('A','I','F','F') || buffer [2] == MAKE_MARKER ('A','I','F','C'))
			return SF_FORMAT_AIFF;
		if (buffer [2] == MAKE_MARKER ('8','S','V','X') || buffer [2] == MAKE_MARKER ('1','6','S','V'))
			return SF_FORMAT_SVX;
		return 0;
		}
		
	if ((buffer [0] == MAKE_MARKER ('.','s','n','d') || buffer [0] == MAKE_MARKER ('d','n','s','.')))
		return SF_FORMAT_AU;
		
	if ((buffer [0] == MAKE_MARKER ('f','a','p',' ') || buffer [0] == MAKE_MARKER (' ','p','a','f')))
		return SF_FORMAT_PAF;
	
	if (buffer [0] == MAKE_MARKER ('N','I','S','T'))
		return SF_FORMAT_NIST;
		
	if (is_au_snd_file (ext))
		return SF_FORMAT_AU | SF_FORMAT_ULAW;

	/* Default to header-less RAW PCM file type. */
	return SF_FORMAT_RAW;
      }


//---------------------------------------------------------
//   validateSfinfo
//---------------------------------------------------------

bool SndFile::validateSfinfo()
      {	
      if (!_samplerate)
	      return false;
	if (!_samples)
	      return false;
	if (!_channels)
	      return false;
	if (!_format & SF_FORMAT_TYPEMASK)
	      return false;
	if (!_format & SF_FORMAT_SUBMASK)
	      return false;
	if (!_sections)
	      return false;
	return true;
      }

bool SndFile::validate()
      {	
      if (!_blockwidth)
		return false;	
	if (!_bytewidth)
		return false;	
	if (!_datalength)
		return false;
	if (_blockwidth != _channels * _bytewidth)
		return false;	
	return true;
      }

//---------------------------------------------------------
//   SndFile
//---------------------------------------------------------

SndFile::SndFile(const QString path)
      {
      finfo       = new QFileInfo(path);
      _file       = 0;
      refs        = 0;
      cache       = 0;
      csize       = 0;
      _format     = 0;
      fmt         = 0;
      _samples    = 0;
      _samplerate = 0;
      _channels   = 0;
      _format     = 0;
      }

int SndFile::incRef()
      {
      ++refs;
//      printf("SndFile::incRef %d\n", refs);
      return refs;
      }

int SndFile::decRef()
      {
      --refs;
//      printf("SndFile::decRef %d\n", refs);
      return refs;
      }

SndFile::~SndFile()
      {
      close();
      delete finfo;
      finfo = 0;
      if (cache) {
            for (unsigned i = 0; i < channels(); ++i)
                  delete cache[i];
            delete cache;
            cache = 0;
            }
      }

//---------------------------------------------------------
//   openRead
//    returns true on error
//---------------------------------------------------------

bool SndFile::openRead()
      {
	setErrno(0);
	bool isSeekable = true;
	QString p = finfo->filePath();
	_file = fopen (p.latin1(), "r");
	if (!_file) {	
            setErrno(SFE_OPEN_FAILED);
		close();
		return true;
		}
	_mode = SF_MODE_READ;
      _filelength = 2048;
	if (isSeekable) {	
            fseek(_file, 0, SEEK_END);
		_filelength = ftell(_file) ;
		fseek (_file, 0, SEEK_SET);
		}
	
	int filetype = guess_file_type(_file, finfo->extension());

	switch (filetype) {	
            case  SF_FORMAT_WAV:
		      setErrno(wav_open_read());
			break;

		default:
                  printf("unknown format %d 0x%0x\n", filetype, filetype);
			setErrno(SFE_UNKNOWN_FORMAT);
                  break;
		}
	/* Both the file format and the file must be seekable to enable sf_seek(). */
	_seekable = _seekable && isSeekable;
		
	if (_errno) {	
            printf("sf open error %d\n", _errno);
		close();
		return true;
		}
	if (!validateSfinfo()) {
		_errno = SFE_BAD_SF_INFO;
            printf("sf open error %d\n", _errno);
		close();
		return true;
		};
		
	if (!validate()) {	
		_errno = SFE_INTERNAL;
            printf("sf open error %d\n", _errno);
		close();
		return true;
		}
      QString cacheName = finfo->dirPath(true) + "/" + finfo->baseName() + ".wca";
      csize             = (samples() + cacheMag - 1)/cacheMag;
      cache             = new SampleV*[channels()];
      for (unsigned ch = 0; ch < channels(); ++ch)
            cache[ch] = new SampleV[csize];
      readCache(cacheName);
	return false;
      }

//---------------------------------------------------------
//   search
//---------------------------------------------------------

SndFile* SndFileList::search(const QString& name)
      {
      for (iSndFile i = begin(); i != end(); ++i) {
            if ((*i)->path() == name)
                  return *i;
            }
      return 0;
      }

//---------------------------------------------------------
//   readCache
//---------------------------------------------------------

void SndFile::readCache(const QString& path)
      {
      FILE* cfile = fopen(path.latin1(), "r");
      if (cfile) {
            for (unsigned ch = 0; ch < channels(); ++ch)
                  fread(cache[ch], csize * sizeof(SampleV), 1, cfile);
            fclose(cfile);
            return;
            }

      //---------------------------------------------------
      //  create cache
      //---------------------------------------------------

      QString label("create peakfile for ");
      label += basename();
      QProgressDialog* progress = new QProgressDialog(label, 0, csize, 0, 0, true);
      progress->setMinimumDuration(0);
      progress->show();
      float data[channels()][cacheMag];
      float* fp[channels()];
      for (unsigned k = 0; k < channels(); ++k)
            fp[k] = &data[k][0];
      int interval = csize / 10;
      for (int i = 0; i < csize; i++) {
            if ((i % interval) == 0) {
                  progress->setProgress(i);
                  }
            seek(i * cacheMag, 0);
            read(fp, cacheMag);
            for (unsigned ch = 0; ch < channels(); ++ch) {
                  float rms = 0.0;
                  cache[ch][i].peak = 0;
                  for (int n = 0; n < cacheMag; n++) {
                        float fd = data[ch][n];
                        rms += fd * fd;
                        int idata = int(fd * 255.0);
                        if (idata < 0)
                              idata = -idata;
                        if (cache[ch][i].peak < idata)
                              cache[ch][i].peak = idata;
                        }
                  // amplify rms value +12dB
                  int rmsValue = int((sqrt(rms/cacheMag) * 255.0));
                  if (rmsValue > 255)
                        rmsValue = 255;
                  cache[ch][i].rms = rmsValue;
                  }
            }
      progress->setProgress(csize);
      writeCache(path);
      delete progress;
      }

//---------------------------------------------------------
//   writeCache
//---------------------------------------------------------

void SndFile::writeCache(const QString& path)
      {
      FILE* cfile = fopen(path.latin1(), "w");
      if (cfile == 0)
            return;
      for (unsigned ch = 0; ch < channels(); ++ch)
            fwrite(cache[ch], csize * sizeof(SampleV), 1, cfile);
      fclose(cfile);
      }

//---------------------------------------------------------
//   play
//---------------------------------------------------------

#if 0
int SndFile::play(int fd, int samplepos, int nsamples)
      {
      printf("write wave: not implemented: %d %d %d\n", fd, samplepos, nsamples);
      short* src = _data + headerOffset/2 + samplepos * _channels;
      if (samplepos + nsamples >= _samples)
            nsamples = _samples - samplepos;
      int n = nsamples * _channels * bytes;
      if (::write(fd, src, n) != n) {
            perror("write");
            fprintf(stderr, "fd %d  n %d\n", fd, n);
            return 0;
            }
      return nsamples;
      }
#endif

//---------------------------------------------------------
//   read
//---------------------------------------------------------

void SndFile::read(SampleV* s, int mag, unsigned pos)
      {
      for (unsigned ch = 0; ch < channels(); ++ch) {
            s[ch].peak = 0;
            s[ch].rms = 0;
            }
      if (pos > samples())
            return;

      if (mag < cacheMag) {
            float data[channels()][mag];
            float* fp[channels()];
            for (unsigned i = 0; i < channels(); ++i)
                  fp[i] = &data[i][0];
            seek(pos, 0);
            read(fp, mag);

            for (unsigned ch = 0; ch < channels(); ++ch) {
                  s[ch].peak = 0;
                  float rms = 0.0;
                  for (int i = 0; i < mag; i++) {
                        float fd = data[ch][i];
                        rms += fd;
                        int idata = int(fd * 255.0);
                        if (idata < 0)
                              idata = -idata;
                        if (s[ch].peak < idata)
                              s[ch].peak = idata;
                        }
                  s[ch].rms = 0;    // TODO rms / mag;
                  }
            }
      else {
            mag /= cacheMag;
            int rest = csize - (pos/cacheMag);
            int end  = mag;
            if (rest < mag)
                  end = rest;

            for (unsigned ch = 0; ch < channels(); ++ch) {
                  int rms = 0;
                  int off = pos/cacheMag;
                  for (int offset = off; offset < off+end; offset++) {
                        rms += cache[ch][offset].rms;
                        if (s[ch].peak < cache[ch][offset].peak)
                              s[ch].peak = cache[ch][offset].peak;
                        }
                  s[ch].rms = rms / mag;
                  }
            }
      }

void SndFile::setFormat(int fmt, int ch, int rate, int bits)
      {
      _format      = fmt;
      _channels    = ch;
      _samplerate  = rate;
      _pcmbitwidth = bits;
      }

//---------------------------------------------------------
//   openWrite
//---------------------------------------------------------

bool SndFile::openWrite()
      {
	_errno = 0;

	QString p = finfo->filePath();
	if (!(_file = fopen (p.latin1(), "w"))) {
            _errno = SFE_OPEN_FAILED;
            printf("SndFile openWrite <%s> failed: %s\n",
               p.latin1(), strerror(errno));
		close();
		return true;
		}
	_mode = SF_MODE_WRITE;
	
	_filelength = ftell(_file);
	fseek(_file, 0, SEEK_SET);

	switch (format() & SF_FORMAT_TYPEMASK) {	
            case SF_FORMAT_WAV:
                  if ((_errno = wav_open_write())) {	
                        close();
			      return true;
                        }
		      break;
		default:	
                  _errno = SFE_UNKNOWN_FORMAT;
                  close();
                  return true;
		}
	return false;
      }

//---------------------------------------------------------
//   seek
//---------------------------------------------------------

off_t SndFile::seek(off_t offset, int whence)
      {
	off_t realseek, position;

	if (!_seekable) {	
            _errno = SFE_NOT_SEEKABLE;
            return ((off_t) -1);
		}
	
	if (! (_blockwidth && _datalength && _dataoffset)) {	
            _errno = SFE_BAD_SEEK;
		return ((off_t) -1);
		}
	
	switch (whence) {	
            case SEEK_SET:
                  if (offset < 0 || offset * _blockwidth > _datalength) {	
                        _errno = SFE_BAD_SEEK;
                        return ((off_t) -1);
                        }
                  realseek = _dataoffset + offset * _blockwidth;
                  fseek(_file, realseek, SEEK_SET);
                  position = ftell(_file);
                  break;
				
		case SEEK_CUR:
                  realseek = offset * _blockwidth;
			position = ftell(_file) - _dataoffset;
                  if (position + realseek > (int)_datalength || position + realseek < 0) {
                        _errno = SFE_BAD_SEEK;
                        return ((off_t) -1);
                        }
                  fseek (_file, realseek, SEEK_CUR);
                  position = ftell (_file);
                  break;
				
            case SEEK_END:
                  if (offset > 0) {	
                        _errno = SFE_BAD_SEEK;
                        return ((off_t) -1);
                        }
                  realseek = (_samples + offset) * _blockwidth + _dataoffset;
                  fseek (_file, realseek, SEEK_SET);
                  position = ftell(_file);
                  break ;
				
		default:
                  _errno = SFE_BAD_SEEK;
                  return ((off_t) -1);
		}
      current = (position - _dataoffset) / _blockwidth;
      return current;
      }

#if 0
//---------------------------------------------------------
//   read_raw
//---------------------------------------------------------

size_t SndFile::read_raw(void *ptr, size_t bytes)
      {
	if (_mode != SF_MODE_READ) {	
            _errno = SFE_NOT_READMODE;
		return ((size_t) -1);
		}
	if (current >= _datalength) {	
            memset (ptr, 0, bytes);
		return 0;
		}
	if (bytes % (_channels * bytewidth)) {	
            _errno = SFE_BAD_READ_ALIGN;
		return (size_t) -1;
		}
	size_t count = fread(ptr, 1, bytes, _file);
	if (count < bytes)
		memset (((char*)ptr) + count, 0, bytes - count);
	current += count / _blockwidth;
	return count ;
      }
#endif

//---------------------------------------------------------
//   read
//---------------------------------------------------------

size_t SndFile::read(float** ptr, size_t frames)
      {
	if (_mode != SF_MODE_READ)
		return SFE_NOT_READMODE;
	
	if (current >= _samples) {
            for (unsigned i = 0; i < channels(); ++i)
                  memset(ptr[i], 0, frames * sizeof (float));
		return 0;
		}
	size_t count = fmt->read(ptr, frames);
	if (current + count > samples()) {	
            count        = samples() - current;
		size_t extra = frames - count ;
            for (unsigned i = 0; i < channels(); ++i)
		      memset (ptr[i] + count, 0, extra * sizeof(float));
		current = samples();
		}
	current += count;
	return count;
      }

#if 0
//---------------------------------------------------------
//   write_raw
//---------------------------------------------------------

size_t SndFile::write_raw(void *ptr, size_t len)
      {
	if (_mode != SF_MODE_WRITE) {	
            _errno = SFE_NOT_WRITEMODE;
		return (size_t) -1;
		}
	if (len % (_channels * bytewidth)) {	
            _errno = SFE_BAD_WRITE_ALIGN;
		return (size_t) -1;
		}
	size_t count = fwrite(ptr, 1, len, _file);
	current += count / _blockwidth;
	return count;
      }
#endif

//---------------------------------------------------------
//   write
//---------------------------------------------------------

size_t SndFile::write(float** ptr, size_t frames)
      {
	if (_mode != SF_MODE_WRITE) {	
            _errno = SFE_NOT_WRITEMODE;
printf("SndFile: write: error: not write mode\n");
		return (size_t) -1;
		}
// printf("SndFile: write %d frames\n", frames);
	size_t count = fmt->write(ptr, frames);
	current += count;
	return count;
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void SndFile::close()
      {
      if (_file) {
            if (fmt)
                  fmt->close();
	      fclose(_file);
            _file = 0;
            }
      }

//---------------------------------------------------------
//   unimpl
//---------------------------------------------------------

size_t SndFileFormat::unimpl() const
      {
      sfile->setErrno(SFE_UNIMPLEMENTED);
      return (size_t) -1;
      }

/*-----------------------------------------------------------------------------------------------
 */

void endswap_short_array (short *ptr, int len)
      {
      int k;
	for (k = 0 ; k < len ; k++)
		ptr[k] = ((((ptr[k])>>8)&0xFF)|(((ptr[k])&0xFF)<<8));
      }

void endswap_int_array (int *ptr, int len)
      {	
      int k ;
	for (k = 0 ; k < len ; k++)
		ptr[k] = ((((ptr[k])>>24)&0xFF)|(((ptr[k])>>8)&0xFF00)|
					(((ptr[k])&0xFF00)<<8)|(((ptr[k])&0xFF)<<24));
      }
