//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: alsa9audio.cpp,v 1.4 2002/02/27 11:52:58 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "config.h"

#ifdef AUDIO
#ifdef ALSACVS
#include <alsa/asoundlib.h>
#else
#include <sys/asoundlib.h>
#endif
#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==9)
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <qstring.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include "globals.h"
#include "audiodev.h"

//---------------------------------------------------------
//   AlsaAudioDevice
//---------------------------------------------------------

class AlsaAudioDevice : public AudioDevice {
      int card;
      int dev;
      snd_pcm_t* playbackHandle;
      snd_pcm_t* captureHandle;

   public:
      AlsaAudioDevice(int c, int d, int f, const QString& s)
         : AudioDevice(s) {
            card = c;
            dev  = d;
            _rwFlags = f;
            playbackHandle = 0;
            captureHandle = 0;
            }
      virtual QString open(int);
      virtual void close();
      virtual int read(unsigned char* p, int n);
      virtual int write(const unsigned char* p, int n);
      virtual int selectrfd() const;
      virtual int selectwfd() const;
      virtual void start() const;
      virtual void stop () const;
      };

static int buffer_time = -1;
static int avail_min   = -1;

static size_t bits_per_sample;
static size_t bits_per_frame;
static size_t chunk_bytes;
static int chunk_size = -1;

//---------------------------------------------------------
//   xrun
//    playback write error handler
//---------------------------------------------------------

void xrun(snd_pcm_t* handle)
      {
      snd_pcm_status_t *status;
      int res;

      snd_pcm_status_alloca(&status);
      if ((res = snd_pcm_status(handle, status))<0) {
            printf("status error: %s", snd_strerror(res));
            exit(EXIT_FAILURE);
            }
      int state = snd_pcm_status_get_state(status);
      if (state == SND_PCM_STATE_XRUN) {
            struct timeval now, diff, tstamp;
            gettimeofday(&now, 0);
            snd_pcm_status_get_trigger_tstamp(status, &tstamp);
            timersub(&now, &tstamp, &diff);
            if (debugMsg)
                  fprintf(stderr, "xrun >= %.3f ms\n", diff.tv_sec * 1000 + diff.tv_usec / 1000.0);
            if ((res = snd_pcm_prepare(handle))<0) {
                  printf("xrun: prepare error: %s", snd_strerror(res));
                  exit(EXIT_FAILURE);
                  }
            return;     // ok, data should be accepted again
            }
      else {
            printf("ALSA xrun: read/write error in state %d\n", state);
            }
      }

//---------------------------------------------------------
//   setParams
//---------------------------------------------------------

static void setParams(snd_pcm_t* handle)
      {
      //-----------------------------------------------
      //   HW parameters

      snd_pcm_hw_params_t* hwparams;
      snd_pcm_hw_params_alloca(&hwparams);
      int err = snd_pcm_hw_params_any(handle, hwparams);
      if (err < 0) {
            printf("Broken configuration for this PCM: no configurations available");
            exit(EXIT_FAILURE);
            }

      err = snd_pcm_hw_params_set_access(handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED);
      if (err < 0) {
            printf("ALSA: Access type not available");
            exit(EXIT_FAILURE);
            }

      err = snd_pcm_hw_params_set_format(handle, hwparams, SND_PCM_FORMAT_S16_LE);
      if (err < 0) {
            printf("ALSA: Sample format non available");
            exit(EXIT_FAILURE);
            }
      err = snd_pcm_hw_params_set_channels(handle, hwparams, 2);
      if (err < 0) {
            printf("ALSA: Channels count non available");
            exit(EXIT_FAILURE);
            }

      snd_pcm_hw_params_set_rate_near(handle, hwparams, sampleRate, 0);

      //
      // buffer
      //
// war 4
      buffer_time = snd_pcm_hw_params_set_buffer_size_near(handle, hwparams, segmentSize*3);
      buffer_time = snd_pcm_hw_params_set_period_size_near(handle, hwparams, segmentSize, 0);

      err = snd_pcm_hw_params(handle, hwparams);
      if (err < 0) {
            printf("Unable to install hw params:");
            exit(EXIT_FAILURE);
            }
      chunk_size      = snd_pcm_hw_params_get_period_size(hwparams, 0);
      int buffer_size = snd_pcm_hw_params_get_buffer_size(hwparams);

// printf("segment %d chunk %d, buffer %d\n", segmentSize, chunk_size, buffer_size);

      if (chunk_size == int(buffer_size)) {
            printf("Can't use period equal to buffer size (%u == %u)", chunk_size, buffer_size);
            exit(EXIT_FAILURE);
            }

      //-----------------------------------------------
      //   SW parameters

      snd_pcm_sw_params_t *swparams;
      snd_pcm_sw_params_alloca(&swparams);
      snd_pcm_sw_params_current(handle, swparams);
      int xfer_align = snd_pcm_sw_params_get_xfer_align(swparams);
      err = snd_pcm_sw_params_set_sleep_min(handle, swparams, 0);
      size_t n;
      if (avail_min < 0)
            n = chunk_size;
      else
            n = int (snd_pcm_hw_params_get_rate(hwparams, 0) * (double) avail_min / 1000000);
      snd_pcm_sw_params_set_avail_min(handle, swparams, n);
      snd_pcm_sw_params_set_xfer_align(handle, swparams, xfer_align);
      if (snd_pcm_sw_params(handle, swparams) < 0) {
            printf("unable to install sw params:");
            exit(EXIT_FAILURE);
            }

      bits_per_sample = snd_pcm_format_physical_width(SND_PCM_FORMAT_S16_LE);
      bits_per_frame  = bits_per_sample * 2; // 2=channels
      chunk_bytes     = chunk_size * bits_per_frame / 8;

//      printf("bits/frame %d,  real chunk_size = %i buffer %d, chunk %d\n",
//         bits_per_frame, chunk_size, buffer_size, chunk_size);
      }

//---------------------------------------------------------
//   open
//---------------------------------------------------------

QString AlsaAudioDevice::open(int rw)
      {
      int openFlags = SND_PCM_NONBLOCK;

      //---------------------------------------------------
      //  setup playback stream
      //---------------------------------------------------

      if (rw & 1) {
            snd_pcm_stream_t stream = SND_PCM_STREAM_PLAYBACK;
            int err = snd_pcm_open(&playbackHandle, "default", stream, openFlags);
      	if (err < 0)
	      	return QString("Alsa open: ")+QString(snd_strerror(err));
            setParams(playbackHandle);
            int res = snd_pcm_prepare(playbackHandle);
            if (res <0) {
                  printf("prepare error: %s", snd_strerror(res));
                  exit(EXIT_FAILURE);
                  }
            }
      if (rw & 2) {
            snd_pcm_stream_t stream = SND_PCM_STREAM_CAPTURE;
            int err = snd_pcm_open(&captureHandle, "default", stream, openFlags);
      	if (err < 0)
	      	return QString("Alsa open: ")+QString(snd_strerror(err));
            setParams(captureHandle);
            }
      return QString("OK");
      }

//---------------------------------------------------------
//   read
//---------------------------------------------------------

int AlsaAudioDevice::read(unsigned char* buffer, int frames)
      {
      if (captureHandle == 0)
            return 0;
      int n = 0;
      while (frames > 0) {
            int rv = snd_pcm_readi(captureHandle, buffer, frames);
            if (rv == -EAGAIN || (rv >= 0 && rv < frames))
                  snd_pcm_wait(captureHandle, 1000);
            else if (rv == -EPIPE)
                  xrun(captureHandle);
            else if (rv < 0) {
                  fprintf(stderr, "AlsaAudioDevice: read(0x%x, 0x%x, %d+%d) failed: %s\n",
                    unsigned(captureHandle), unsigned(buffer), frames, n, snd_strerror(rv));
                  return rv;
                  }
            else {
                  n      += rv;
                  frames -= rv;
                  buffer   += rv * bits_per_frame / 8;
                  }
            }
      return n;
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void AlsaAudioDevice::start() const
      {
#if 0
      int err;
      if (captureHandle) {
	      err = snd_pcm_start(captureHandle);
            if (err == -EPIPE)
                  xrun(playbackHandle);
            else if (err < 0)
	            printf("Alsa capture go: %s\n", snd_strerror(err));
            }
      if (playbackHandle) {
	      int err = snd_pcm_start(playbackHandle);
            if (err == -EPIPE)
                  xrun(playbackHandle);
            else if (err < 0) {
	            printf("Alsa playback go: %s\n", snd_strerror(err));
                  }
            }
#endif
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void AlsaAudioDevice::stop() const
      {
      if (captureHandle) {
	      int err = snd_pcm_pause(captureHandle, 1);
            if (err < 0)
	            printf("Alsa capture pause: %s\n", snd_strerror(err));
            }
      if (playbackHandle) {
	      int err = snd_pcm_pause(playbackHandle, 1);
            if (err < 0)
	            printf("Alsa playback pause: %s\n", snd_strerror(err));
            }
      }

//---------------------------------------------------------
//   write
//---------------------------------------------------------

int AlsaAudioDevice::write(const unsigned char* buffer, int n)
      {
      if (playbackHandle == 0)
            return 0;
      int result    = 0;
      int framesize = bits_per_frame/8;
      while (n > 0) {
            int r = snd_pcm_writei(playbackHandle, buffer, n);
            if (r == -EAGAIN) {
                  snd_pcm_wait(playbackHandle, 1000);
                  continue;
                  }
            else if (r == -EPIPE) {
                  xrun(playbackHandle);
                  continue;
                  }
            else if (r < 0) {
                  fprintf(stderr, "AlsaAudioDevice: write(0x%x, 0x%x, %d) failed: %s\n",
                    unsigned(playbackHandle), unsigned(buffer), n, snd_strerror(r));
                  exit(1);
                  }
            result += r;
            buffer += r * framesize;
            n -= r;
            if (n)
                  snd_pcm_wait(playbackHandle, 1000);
            }
      return result;
      }

//---------------------------------------------------------
//   selectwfd
//---------------------------------------------------------

int AlsaAudioDevice::selectwfd() const
      {
      if (playbackHandle) {
            int n = snd_pcm_poll_descriptors_count(playbackHandle);
            if (n != 1) {
                  printf("AlsaAudioDevice:selectwfd: not supported\n");
                  abort();
                  }
            struct pollfd fd[n];
            snd_pcm_poll_descriptors(playbackHandle, fd, n);
            return fd[0].fd;
            }
      return -1;
      }

//---------------------------------------------------------
//   selectrfd
//---------------------------------------------------------

int AlsaAudioDevice::selectrfd() const
      {
      if (captureHandle) {
            int n = snd_pcm_poll_descriptors_count(captureHandle);
            if (n != 1) {
                  printf("AlsaAudioDevice:selectrfd: not supported\n");
                  abort();
                  }
            struct pollfd fd[n];
            snd_pcm_poll_descriptors(captureHandle, fd, n);
            return fd[0].fd;
            }
      return -1;
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void AlsaAudioDevice::close()
      {
      if (captureHandle) {
	      int err = snd_pcm_close(captureHandle);
            if (err < 0)
	            printf("Alsa close capture: %s\n", snd_strerror(err));
            }
      if (playbackHandle) {
	      int err = snd_pcm_close(playbackHandle);
            if (err < 0)
	            printf("Alsa close playback: %s\n", snd_strerror(err));
            }
      captureHandle  = 0;
      playbackHandle = 0;
      }

//---------------------------------------------------------
//   initAlsaAudio
//    collect infos about audio devices
//    return true if no soundcard found
//---------------------------------------------------------

bool initAlsaAudio()
      {
      int card = -1;
      int err;
      if (snd_card_next(&card) < 0 || card < 0) {
            fprintf(stderr, "ALSA: no soundcard found\n");
            return true;
            }
      snd_ctl_card_info_t* info;
      snd_ctl_card_info_alloca(&info);
      snd_pcm_info_t* pcminfo;
      snd_pcm_info_alloca(&pcminfo);
      while (card >= 0) {
            snd_ctl_t* handle;
            char name[32];
            sprintf(name, "hw:%d", card);
/*??*/      if ((err = snd_ctl_open(&handle, name, SND_CTL_NONBLOCK)) < 0) {
                  fprintf(stderr, "ALSA: control open (%d): %s\n",
                     card, snd_strerror(err));
                  continue;
                  }

            if ((err = snd_ctl_card_info(handle, info)) < 0) {
                  fprintf(stderr, "ALSA: control hardware info %d: %s\n",
                     card, snd_strerror(err));
                  snd_ctl_close(handle);
                  continue;
                  }
            int dev = -1;
            for (;;) {
                  if (snd_ctl_pcm_next_device(handle, &dev) < 0)
                        fprintf(stderr, "ALSA: pcm next device: %s\n",
                           snd_strerror(err));
                  if (dev < 0)
                        break;
                  int flags = 0;
                  snd_pcm_info_set_device(pcminfo, dev);
                  snd_pcm_info_set_subdevice(pcminfo, 0);
                  snd_pcm_info_set_stream(pcminfo, SND_PCM_STREAM_PLAYBACK);
                  if (snd_ctl_pcm_info(handle, pcminfo) >= 0)
                        flags |= 1;
                  snd_pcm_info_set_stream(pcminfo, SND_PCM_STREAM_CAPTURE);
                  if (snd_ctl_pcm_info(handle, pcminfo) >= 0)
                        flags |= 2;
                  if (flags) {
                        const char* name = snd_pcm_info_get_name(pcminfo);
                        AudioDevice* d = new AlsaAudioDevice (card, dev, flags, QString(name));
                        audioDevices.add(d);
                        }
                  }
            snd_ctl_close(handle);
            if (snd_card_next(&card) < 0) {
                  fprintf(stderr, "ALSA: snd_card_next: %s\n",
                     snd_strerror(err));
                  break;
                  }
            }
      return false;
      }
#endif
#endif
