//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: alsa5audio.cpp,v 1.4 2002/02/27 11:52:58 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "config.h"

#ifdef AUDIO
#ifdef ALSACVS
#include <alsa/asoundlib.h>
#else
#include <sys/asoundlib.h>
#endif
#if (SND_LIB_MAJOR==0) && (SND_LIB_MINOR==5)
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <qstring.h>
#include <sys/ioctl.h>
#include "globals.h"
#include "audiodev.h"

//---------------------------------------------------------
//   AlsaAudioDevice
//---------------------------------------------------------

class AlsaAudioDevice : public AudioDevice {
      int card;
      int dev;
      snd_pcm_t* handle;
      int openFlags;

      void writeError();
      void readError();

   public:
      AlsaAudioDevice(int c, int d, int f, const QString& s)
         : AudioDevice(s) {
            card = c;
            dev  = d;
            _rwFlags = f;
            handle = 0;
            }
      virtual QString open(int);
      virtual void close();
      virtual int read(unsigned char* p, int n);
      virtual int write(const unsigned char* p, int n);
      virtual int selectrfd() const;
      virtual int selectwfd() const;
      virtual void start() const;
      virtual void stop () const;
      };

//---------------------------------------------------------
//   open
//---------------------------------------------------------

QString AlsaAudioDevice::open(int rw)
      {
      switch (rw) {
            case 3:
                  openFlags = SND_PCM_OPEN_DUPLEX;
                  break;
            case 1:
                  openFlags = SND_PCM_OPEN_PLAYBACK;
                  break;
            case 2:
                  openFlags = SND_PCM_OPEN_CAPTURE;
                  break;
            case 0:
                  return QString("no operation configured");
            default:
                  fprintf(stderr, "illegal rw 0x%02x\n", rw);
                  exit(1);
            }
      int err = snd_pcm_open_subdevice(&handle, card, dev, 0, openFlags | SND_PCM_OPEN_NONBLOCK);
	if (err < 0) {
		return QString("Alsa open: ")+QString(snd_strerror(err));
            }

      snd_pcm_channel_flush(handle, SND_PCM_CHANNEL_PLAYBACK);

      //---------------------------------------------------
      //  setup playback stream
      //---------------------------------------------------

      if (openFlags == SND_PCM_OPEN_PLAYBACK || openFlags == SND_PCM_OPEN_DUPLEX) {
      	snd_pcm_channel_params_t params;

      	memset(&params, 0, sizeof(params));

      	params.channel           = SND_PCM_CHANNEL_PLAYBACK;
      	params.mode              = SND_PCM_MODE_BLOCK;

            params.format.interleave = 1;
            params.format.format     = SND_PCM_SFMT_S16_LE;
            params.format.rate       = 44100;
            params.format.voices     = 2;

      	params.start_mode        = SND_PCM_START_FULL;
      	//params.stop_mode         = SND_PCM_STOP_ROLLOVER;
      	params.stop_mode         = SND_PCM_STOP_STOP;
            params.time              = 0;

            // frag_size in bytes?
      	params.buf.block.frag_size = segmentSize*4;
      	params.buf.block.frags_min = 2;
      	params.buf.block.frags_max = 2;

            snd_pcm_playback_flush(handle);
      	if ((err = snd_pcm_channel_params(handle, &params)) < 0)
      		return QString("Alsa playback stream params: ")+QString(snd_strerror(err));

      	if ((err = snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_PLAYBACK)) < 0)
      		return QString("Alsa stream prepare playback: ")+QString(snd_strerror(err));

            snd_pcm_channel_setup_t  setup;
            setup.mode    = SND_PCM_MODE_BLOCK;
            setup.channel = SND_PCM_CHANNEL_PLAYBACK;

            if ((err = snd_pcm_channel_setup(handle, &setup)) < 0)
                  return QString("Alsa channel setup: ")+QString(snd_strerror(err));
//            alsa_set_frag(handle, segmentSize, 6);
            }

      //---------------------------------------------------
      //  setup capture stream
      //---------------------------------------------------

      if (openFlags == SND_PCM_OPEN_CAPTURE || openFlags == SND_PCM_OPEN_DUPLEX) {
      	snd_pcm_channel_params_t params;
      	memset(&params, 0, sizeof(params));

      	params.channel           = SND_PCM_CHANNEL_CAPTURE;
      	params.mode              = SND_PCM_MODE_BLOCK;

            params.format.interleave = 1;
            params.format.format     = SND_PCM_SFMT_S16_LE;
            params.format.rate       = 44100;
            params.format.voices     = 2;

      	params.start_mode        = SND_PCM_START_FULL;
      	params.stop_mode         = SND_PCM_STOP_ROLLOVER;
            params.time              = 0;

      	params.buf.block.frag_size = segmentSize*4;
      	params.buf.block.frags_min = 2;
      	params.buf.block.frags_max = 6;

      	if ((err = snd_pcm_channel_params(handle, &params)) < 0)
      		return QString("Alsa capture stream params: ")+QString(snd_strerror(err));

      	if ((err = snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_CAPTURE)) < 0)
      		return QString("Alsa stream prepare capture: ")+QString(snd_strerror(err));
            }

      //---------------------------------------------------
      //    read back setup
      //---------------------------------------------------

      snd_pcm_channel_setup_t setup;
	memset(&setup, 0, sizeof(setup));
      if (openFlags == SND_PCM_OPEN_PLAYBACK || openFlags == SND_PCM_OPEN_DUPLEX)
	      setup.channel = SND_PCM_CHANNEL_PLAYBACK;
      else
	      setup.channel = SND_PCM_CHANNEL_CAPTURE;

	if ((err = snd_pcm_channel_setup(handle, &setup)) < 0)
		return QString("Alsa stream setup: ")+QString(snd_strerror(err));

//	bufferSize = setup.buf.block.frag_size;
//      start();
      return QString("OK");
      }

//---------------------------------------------------------
//   read
//    read n frames
//---------------------------------------------------------

int AlsaAudioDevice::read(unsigned char* buffer, int n)
      {
      if (handle == 0)
            return 0;
      int nn = n * 4;
      int rv = snd_pcm_read(handle, buffer, nn);
      if (rv != nn) {
            if (rv < 0)
                  readError();
            else {
                  fprintf(stderr, "AlsaAudioDevice: read(0x%p, 0x%p, %d) returns: %d\n",
                    handle, buffer, nn, rv);
                  }
            }
      return n;
      }

//---------------------------------------------------------
//   readError
//---------------------------------------------------------

void AlsaAudioDevice::readError()
      {
      snd_pcm_channel_status_t status;

      memset(&status, 0, sizeof(status));
      status.channel = SND_PCM_CHANNEL_CAPTURE;

      if (snd_pcm_channel_status(handle, &status)<0) {
            fprintf(stderr, "ALSA: record channel status error\n");
            exit(1);
            }
      if (status.status == SND_PCM_STATUS_RUNNING)
            return;     // everything is ok, but the driver is waiting for data
      if (status.status == SND_PCM_STATUS_OVERRUN) {
            fprintf(stderr, "ALSA: overrun at position %u\n", status.scount);
            if (snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_CAPTURE)<0) {
                  fprintf(stderr, "ALSA: overrun: record channel prepare error\n");
                  exit(1);
                  }
            return;           /* ok, data should be accepted again */
            }
      printf("alsa read(%x,,) failed\n", (unsigned int) handle);
      exit(2);
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void AlsaAudioDevice::start() const
      {
      if (!handle)
            return;
      int err = 0;
      switch (openFlags) {
            case SND_PCM_OPEN_DUPLEX:
	            err = snd_pcm_channel_go(handle, SND_PCM_CHANNEL_CAPTURE);
                  if (err < 0)
	                  printf("Alsa stream go: %s\n", snd_strerror(err));
                  // FALL THROUGH
            case SND_PCM_OPEN_PLAYBACK:
	            err = snd_pcm_channel_go(handle, SND_PCM_CHANNEL_PLAYBACK);
                  break;
            case SND_PCM_OPEN_CAPTURE:
	            err = snd_pcm_channel_go(handle, SND_PCM_CHANNEL_CAPTURE);
                  break;
            }
      if (err < 0)
	      printf("Alsa stream go: %s\n", snd_strerror(err));
      }

//---------------------------------------------------------
//   stop
//---------------------------------------------------------

void AlsaAudioDevice::stop() const
      {
      if (!handle)
            return;
      int err = 0;
      switch (openFlags) {
            case SND_PCM_OPEN_DUPLEX:
	            err = snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_CAPTURE);
	            if (err < 0)
		            printf("Alsa channel prepare capture: %s\n", snd_strerror(err));
            case SND_PCM_OPEN_PLAYBACK:
	            err = snd_pcm_playback_drain(handle);
	            if (err < 0)
		            printf("Alsa playback drain: %s\n", snd_strerror(err));
	            err = snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_PLAYBACK);
                  break;
            case SND_PCM_OPEN_CAPTURE:
	            err = snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_CAPTURE);
                  break;
            }
      if (err < 0)
	      printf("Alsa stream prepare: %s\n", snd_strerror(err));
      }

//---------------------------------------------------------
//   writeError
//---------------------------------------------------------

void AlsaAudioDevice::writeError()
      {
      snd_pcm_channel_status_t status;

      memset(&status, 0, sizeof(status));
      status.channel = SND_PCM_CHANNEL_PLAYBACK;
      if (snd_pcm_channel_status(handle, &status)<0) {
            fprintf(stderr, "ALSA: playback channel status error\n");
            exit(1);
            }
      if (status.status == SND_PCM_STATUS_UNDERRUN) {
            fprintf(stderr, "ALSA: underrun at position %u\n", status.scount);
            if (snd_pcm_channel_prepare(handle, SND_PCM_CHANNEL_PLAYBACK) < 0) {
                  fprintf(stderr, "ALSA: underrun: playback channel prepare error\n");
                  exit(1);
                  }
            return;
            }
      printf("alsa write(%x,,) failed\n", (unsigned int) handle);
      exit(1);
      }

//---------------------------------------------------------
//   write
//    n = segment size
//---------------------------------------------------------

int AlsaAudioDevice::write(const unsigned char* buffer, int n)
      {
      if (handle == 0)
            return n;

      int nn = n * 4;
      while (nn > 0) {
            int rv = snd_pcm_write(handle, buffer, nn);
            if (rv < 0) {
                  if (rv == -EAGAIN)
                        continue;
                  writeError();
                  return 0;
                  }
            nn     -= rv;
            buffer += rv;
            }
      return n;
      }

//---------------------------------------------------------
//   selectwfd
//---------------------------------------------------------

int AlsaAudioDevice::selectwfd() const
      {
      if (handle)
            return snd_pcm_file_descriptor(handle, SND_PCM_CHANNEL_PLAYBACK);
      return -1;
      }

//---------------------------------------------------------
//   selectrfd
//---------------------------------------------------------

int AlsaAudioDevice::selectrfd() const
      {
      if (handle)
            return snd_pcm_file_descriptor(handle, SND_PCM_CHANNEL_CAPTURE);
      return -1;
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void AlsaAudioDevice::close()
      {
//      printf("alsa pcm close\n");
      stop();

      if (handle != 0) {
	      int err = snd_pcm_close(handle);
            if (err < 0)
	            printf("Alsa close : %s\n", snd_strerror(err));
            }
      handle = 0;
      }

//---------------------------------------------------------
//   initAlsaAudio
//    collect infos about audio devices
//    return true if no soundcard found
//---------------------------------------------------------

bool initAlsaAudio()
      {
      int err;
      unsigned int mask = snd_cards_mask();
      if (!mask) {
            fprintf(stderr, "ALSA: no soundcard found\n");
            return true;
            }
      for (int card = 0; card < SND_CARDS; ++card) {
            if (!(mask & (1 << card)))
                  continue;
            snd_ctl_t* handle;
            if ((err = snd_ctl_open(&handle, card)) < 0) {
                  fprintf(stderr, "ALSA: control open: %s\n",
                     snd_strerror(err));
                  continue;
                  }
            struct snd_ctl_hw_info info;
            if ((err = snd_ctl_hw_info(handle, &info)) < 0) {
                  fprintf(stderr, "ALSA: control hardware info: %s\n",
                     snd_strerror(err));
                  snd_ctl_close(handle);
                  continue;
                  }
            snd_pcm_info_t pcminfo;
            for (unsigned dev = 0; dev < info.pcmdevs; ++dev) {
                  if ((err = snd_ctl_pcm_info(handle, dev, &pcminfo)) < 0) {
                        fprintf(stderr, "ALSA: pcm info: %s\n",
                           snd_strerror(err));
                        continue;
                        }
                  if (pcminfo.flags & (SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE | SND_PCM_INFO_DUPLEX)) {
                        int flags = 0;
                        if (pcminfo.flags & SND_PCM_INFO_PLAYBACK)
                              flags |= 1;
                        if (pcminfo.flags & SND_PCM_INFO_CAPTURE)
                              flags |= 2;
                        if (pcminfo.flags & SND_PCM_INFO_DUPLEX)
                              flags |= 3;
                        AudioDevice* d = new AlsaAudioDevice(card, dev, flags, QString((char*)(pcminfo.name)));
                        audioDevices.add(d);
                        }
                  }
            snd_ctl_close(handle);
            }
      return false;
      }
#endif
#endif

