//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: tlist.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//  (C) Copyright 1999 Werner Schweer (ws@seh.de)
//=========================================================

#include <qlineedit.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>

#include "icons.h"
#include "seq.h"
#include "scrollscale.h"
#include "tlist.h"
#include "xml.h"
#include "device.h"
#include "midiport.h"
#include "audioport.h"
#include "comment.h"
#include "track.h"
#include "song.h"
#include "header.h"
#include "midithread.h"

//---------------------------------------------------------
//   THeaderTip::maybeTip
//---------------------------------------------------------

void THeaderTip::maybeTip(const QPoint &pos)
      {
      Header* w  = (Header*)parentWidget();
      int section = w->sectionAt(pos.x());
      if (section == -1)
            return;
      QRect r(w->sectionPos(section), 0, w->sectionSize(section),
         w->height());
      QString p;
      switch(section) {
            case COL_RECORD:   p = w->tr("Enable Recording"); break;
            case COL_ACTIVITY: p = w->tr("Track Activity"); break;
            case COL_MUTE:     p = w->tr("Mute Indicator"); break;
            case COL_CLASS:    p = w->tr("Track Type"); break;
            case COL_NAME:     p = w->tr("Track Name"); break;
            case COL_OCHANNEL: p = w->tr("Output Channel Number"); break;
            case COL_OPORT:    p = w->tr("Output Port"); break;
            case COL_TIMELOCK: p = w->tr("Time Lock"); break;
            default: return;
            }
      tip(r, p);
      }

//---------------------------------------------------------
//   TList
//---------------------------------------------------------

TList::TList(Header* hdr, QWidget* parent, int ymag)
   : View(parent, 1, ymag)
      {
      header    = hdr;
      setBg(white);
      setFocusPolicy(QWidget::StrongFocus);
      scroll    = 0;
      editTrack = 0;
      editor    = 0;
      drag = NORMAL;
      selectedTrackColor = yellow;
      activityColor = black;

      connect(song, SIGNAL(heartBeat()), SLOT(updateActivity()));
      }

//---------------------------------------------------------
//   drawCenteredPixmap
//    small helper function for "draw()" below
//---------------------------------------------------------

static void drawCenteredPixmap(QPainter& p, const QPixmap* pm, const QRect& r)
      {
      p.save();
      p.setWorldXForm(false);
      p.drawPixmap(r.x() + (r.width() - pm->width())/2, r.y() + (r.height() - pm->height())/2, *pm);
      p.restore();
      }

//---------------------------------------------------------
//   drawCenteredText
//---------------------------------------------------------

static void drawCenteredText(QPainter& p, const QString& s, const QRect& r)
      {
      p.save();
      p.setWorldXForm(false);
      p.drawText(r, Qt::AlignVCenter|Qt::AlignLeft, s);
      p.restore();
      }

//---------------------------------------------------------
//   draw
//---------------------------------------------------------

void TList::draw(QPainter& p, const QRect&rect)
      {
      int x = rect.x();
      int y = rect.y();
      int w = rect.width();
      int h = rect.height();

      //---------------------------------------------------
      //    Tracks
      //---------------------------------------------------

      p.setPen(black);
      TrackList* l = song->tracks();
      int idx = 0;
      for (iTrack i = l->begin(); i != l->end(); ++i, ++idx) {
            int yy = idx * TH;
            if (yy >= y + h)
                  break;
            if (yy + TH < y)
                  continue;
            //
            // clear one row
            //
            if ((*i)->selected()) {
                  p.fillRect(x, yy, w, TH, selectedTrackColor);
//                  p.fillRect(header->sectionPos(COL_ACTIVITY) + header->sectionSize(COL_ACTIVITY) + 1, yy, w, TH, selectedTrackColor);
                  }
            else
                  p.eraseRect(x, yy, w, TH);
            Track::TrackType type = (*i)->type();

            int x = 0;
            for (int index = 0; index < header->count(); ++index) {
                  int section = header->mapToSection(index);
                  int w   = header->sectionSize(section);
                  QRect r = p.xForm(QRect(x+2, yy, w-4, TH));

                  switch (section) {
                        case COL_RECORD:
                              if ((*i)->recordFlag())
                                    drawCenteredPixmap(p, record1Icon, r);
                              break;
                        case COL_CLASS:
                              {
                              const QPixmap* pm = 0;
                              switch((*i)->type()) {
                                    case Track::MIDI: pm = noteIcon;  break;
                                    case Track::DRUM: pm = stickIcon; break;
                                    case Track::WAVE: pm = waveIcon;  break;
                                    }
                              drawCenteredPixmap(p, pm, r);
                              }
                              break;
                        case COL_MUTE:
                              if (song->mute(dynamic_cast<SoundSource*>(*i))) {
                                    drawCenteredPixmap(p, muteIcon, r);
                                    }
                              break;
                        case COL_TIMELOCK:
                              if ((type == Track::MIDI || type == Track::DRUM)
                                 && (*i)->locked()) {
                                    drawCenteredPixmap(p, lockIcon, r);
                                    }
                              break;
                        case COL_NAME:
                              drawCenteredText(p, (*i)->tname(), r);
                              break;
                        case COL_OCHANNEL:
                              {
                              QString s;
                              s.setNum((*i)->outChannel() + 1);
                              drawCenteredText(p, s, r);
                              }
                              break;
                        case COL_OPORT:
                              {
                              int outport = (*i)->outPort() + 1;
                              QString s;
                              s.sprintf("%d(none)", outport);
                              switch((*i)->type()) {
                                    case Track::WAVE:
                                          {
                                          AudioNode* node = ((WaveTrack*)(*i))->route();
                                          if (node == song->master())
                                                s.sprintf("master");
                                          else if (node == song->group(0))
                                                s.sprintf("group0");
                                          else if (node == song->group(1))
                                                s.sprintf("group1");
                                          else if (node == song->group(2))
                                                s.sprintf("group2");
                                          else if (node == song->group(3))
                                                s.sprintf("group3");
                                          }
                                          break;
                                    case Track::MIDI:
                                    case Track::DRUM:
                                          {
                                          Device* dev = (*i)->outDevice();
                                          if (dev)
                                                s.sprintf("%d(%s)", outport,
                                                    dev->name().latin1());
                                          }
                                          break;
                                    }
                              drawCenteredText(p, s, r);
                              }
                              break;
                        default:
                              break;
                        }
                  x += header->sectionSize(section);
                  }
            }
      //---------------------------------------------------
      //    horizontal lines
      //---------------------------------------------------

      p.setPen(gray);
      int yy  = (y / TH) * TH;
      for (; yy < y + h; yy += TH) {
            p.drawLine(x, yy, x + w, yy);
            }
      if (drag == DRAG) {
            int y  = (startY/TH) * TH;
            int dy = startY - y;
            int yy = curY - dy;
            p.setPen(green);
            p.drawLine(x, yy, x + w, yy);
            p.drawLine(x, yy+TH, x + w, yy+TH);
            p.setPen(gray);
            }

      //---------------------------------------------------
      //    vertical Lines
      //---------------------------------------------------

      p.setWorldXForm(false);
      int n = header->count();
      int xpos = 0;
      for (int index = 0; index < n-1; index++) {
            int section = header->mapToSection(index);
            xpos += header->sectionSize(section);
            p.drawLine(xpos, 0, xpos, height());
            }
      p.setWorldXForm(true);
      }

//---------------------------------------------------------
//   returnPressed
//---------------------------------------------------------

void TList::returnPressed()
      {
      editor->hide();
      if (editor->text() != editTrack->tname()) {
            Track* track = editTrack->clone();
            track->setTName(editor->text());
            midiThread->msgChangeTrack(editTrack, track);
            }
      editTrack = 0;
      setFocus();
      }

//---------------------------------------------------------
//   viewMouseDoubleClickEvent
//---------------------------------------------------------

void TList::viewMouseDoubleClickEvent(QMouseEvent* ev)
      {
      int x = ev->x();
      bool shift = ev->state() & ShiftButton;

      int section = header->sectionAt(x);
      if (section == -1)
            return;

      unsigned track = ev->y() / TH;
      if (track >= song->tracks()->size()) {
            Track* t = song->addTrack();
            if (!shift)
                  song->tracks()->deselect();
            t->setSelected(true);
            emit selectionChanged();
            //
            //   evtl Scrollbar justieren
            //
            unsigned ntracks = mapyDev(height()) / TH;
            if (song->tracks()->size() + 2 > ntracks) {
                  scroll->setRange(0, (song->tracks()->size() + 2)*TH);
                  }
            }
      else {
            Track* t = song->tracks()->findIdx(track);

            int colx = mapx(header->sectionPos(section));
            int colw = rmapx(header->sectionSize(section));
            int coly = mapy(track * TH);
            int colh = rmapy(TH);

            if (section == COL_NAME) {
                  editTrack = t;
                  if (editor == 0) {
                        editor = new QLineEdit(this);
                        connect(editor, SIGNAL(returnPressed()),
                           SLOT(returnPressed()));
                        editor->setFrame(true);
                        }
                  editor->setText(editTrack->tname());
                  editor->end(false);
                  editor->setGeometry(colx, coly, colw, colh);
                  editor->show();
                  }
            else
                  viewMousePressEvent(ev);
            }
      }

//---------------------------------------------------------
//   portsPopupMenu
//---------------------------------------------------------

void TList::portsPopupMenu(Track* t, int x, int y)
      {
      switch(t->type()) {
            case Track::MIDI:
            case Track::DRUM:
                  {
                  MidiTrack* track = (MidiTrack*)t;
                  QPopupMenu* p = midiPortsPopup(0);
                  int n = p->exec(mapToGlobal(QPoint(x, y)), 0);
                  if (n != -1) {
                        track->setOutPort(n);
                        song->update();
                        }
                  delete p;
                  }
                  break;
            case Track::WAVE:
                  break;
            }
      }

//---------------------------------------------------------
//   oportPropertyPopupMenu
//---------------------------------------------------------

void TList::oportPropertyPopupMenu(Track* t, int x, int y)
      {
      if (t->type() != Track::MIDI && t->type() != Track::DRUM)
            return;
      int oPort      = t->outPort();
      MidiPort* port = &midiPorts[oPort];
// printf("Channel %d %d %d\n", oChannel, port->hasGui(), port->guiVisible());

      QPopupMenu* p = new QPopupMenu(this);
      p->setCheckable(true);
      p->insertItem("Show Gui", 0);

      p->setItemEnabled(0, port->hasGui());
      p->setItemChecked(0, port->guiVisible());

      int n = p->exec(mapToGlobal(QPoint(x, y)), 0);
      if (n == 0) {
            port->showGui(!port->guiVisible());
            }
      delete p;
      }


//---------------------------------------------------------
//   tracklistChanged
//---------------------------------------------------------

void TList::tracklistChanged()
      {
      redraw();
      }

//---------------------------------------------------------
//   keyPressEvent
//---------------------------------------------------------

void TList::keyPressEvent(QKeyEvent* e)
      {
      int key = e->key();
      switch (key) {
            case Key_Up:
                  moveSelection(-1);
                  break;
            case Key_Down:
                  moveSelection(1);
                  break;
            default:
                  e->ignore();
                  break;
            }
      }

//---------------------------------------------------------
//   moveSelection
//---------------------------------------------------------

void TList::moveSelection(int n)
      {
      TrackList* tracks = song->tracks();

      // check for single selection
      int nselect = 0;
      for (iTrack t = tracks->begin(); t != tracks->end(); ++t)
            if ((*t)->selected())
                  ++nselect;
      if (nselect != 1)
            return;
      for (iTrack t = tracks->begin(); t != tracks->end(); ++t) {
            iTrack s = t;
            if ((*t)->selected()) {
                  if (n > 0) {
                        while (n--) {
                              ++t;
                              if (t == tracks->end()) {
                                    --t;
                                    break;
                                    }
                              }
                        }
                  else {
                        while (n++ != 0) {
                              if (t == tracks->begin())
                                    break;
                              --t;
                              }
                        }
                  (*s)->setSelected(false);
                  (*t)->setSelected(true);
                  if (editTrack && editTrack != *t)
                        returnPressed();
                  redraw();
                  break;
                  }
            }
      emit selectionChanged();
      }

//---------------------------------------------------------
//   viewMousePressEvent
//---------------------------------------------------------

void TList::viewMousePressEvent(QMouseEvent* ev)
      {
      int x             = ev->x();
      int y             = ev->y();
      int button        = ev->button();
      bool shift        = ev->state() & ShiftButton;
      unsigned track    = y / TH;
      TrackColumn col   = TrackColumn(header->sectionAt(x));
      TrackList* tracks = song->tracks();
      if (track >= tracks->size())
            return;

      startY   = y;
      sTrack   = track;
      drag     = START_DRAG;
      Track* t = tracks->findIdx(track);

      switch (col) {
            case COL_RECORD:
                  if (button == QMouseEvent::LeftButton)
                        t->setRecordFlag(!t->recordFlag());
                  break;
            case COL_NONE:
            case COL_ACTIVITY:
                  break;
            case COL_CLASS:
                  if (button == QMouseEvent::LeftButton)
                        classesPopupMenu(track, mapx(x), mapy(track * TH));
                  break;
            case COL_OPORT:
                  if (button == QMouseEvent::LeftButton)
                        portsPopupMenu(t, mapx(x), mapy(track * TH));
                  else if (button == QMouseEvent::RightButton)
                        oportPropertyPopupMenu(t, mapx(x), mapy(track*TH));
                  break;
            case COL_MUTE:
                  if (button == QMouseEvent::LeftButton) {
                        SoundSource* ss = dynamic_cast<SoundSource*>(t);
                        song->setMute(ss, !song->mute(ss));
                        }
                  break;

            case COL_NAME:
                  if (button == QMouseEvent::LeftButton) {
                        if (!shift) {
                              tracks->deselect();
                              t->setSelected(true);
                              }
                        else
                              t->setSelected(!t->selected());
                        if (editTrack && editTrack != t)
                              returnPressed();
                        emit selectionChanged();
                        }
                  else if (button == QMouseEvent::RightButton) {
                        QPopupMenu* p = new QPopupMenu(this);
                        p->clear();
                        p->insertItem(*deleteIcon, tr("Delete Track"), 0);
                        p->insertItem(tr("Track Comment"), 1);
                        int n = p->exec(ev->globalPos(), 0);
                        if (n != -1) {
                              switch (n) {
                                    case 0:     // delete track
                                          for (iTrack it = tracks->begin(); it != tracks->end(); ++it)
                                                (*it)->setSelected(false);
                                          t->setSelected(true);
                                          midiThread->msgRemoveTracks();
                                          break;

                                    case 1:     // show track comment
                                          {
                                          TrackComment* tc = new TrackComment(t, 0);
                                          tc->show();
                                          }
                                          break;

                                    default:
                                          printf("action %d\n", n);
                                          break;
                                    }

                              }
                        delete p;
                        }
                  break;

            case COL_TIMELOCK:
                  if (button == QMouseEvent::LeftButton)
                        t->setLocked(!t->locked());
                  break;

            case COL_OCHANNEL:
                  {
                  MidiTrack* mt = dynamic_cast<MidiTrack*>(t);
                  if (mt == 0)
                        break;
                  int channel = mt->outChannel();
                  if (button == QMouseEvent::RightButton) {
                        if (channel < 15)
                              ++channel;
                        }
                  else if (button == QMouseEvent::MidButton) {
                        if (channel > 0)
                              --channel;
                        }
                  if (channel != t->outChannel()) {
                        mt->setOutChannel(channel);
                        song->update();
                        }
                  }
            }
      redraw();
      }

//---------------------------------------------------------
//   viewMouseMoveEvent
//---------------------------------------------------------

void TList::viewMouseMoveEvent(QMouseEvent* ev)
      {
      curY = ev->y();
      int delta = curY - startY;
      switch (drag) {
            case START_DRAG:
                  if (delta < 0)
                        delta = -delta;
                  if (delta <= 2)
                        return;
                  drag = DRAG;
                  setCursor(QCursor(sizeVerCursor));
                  redraw();
                  break;
            case NORMAL:
                  break;
            case DRAG:
                  redraw();
                  break;
            }
      }

//---------------------------------------------------------
//   viewMouseReleaseEvent
//---------------------------------------------------------

void TList::viewMouseReleaseEvent(QMouseEvent* ev)
      {
      if (drag == DRAG) {
            int y = ev->y();
            unsigned dTrack = y / TH;
            midiThread->msgMoveTrack(sTrack, dTrack);
            setCursor(QCursor(arrowCursor));
            }
      drag = NORMAL;
      redraw();
      if (editTrack)
            editor->setFocus();
      }

//---------------------------------------------------------
//   updateActivity
//---------------------------------------------------------

extern "C" double log2(double);

void TList::updateActivity()
      {
      TrackList* tl = song->tracks();
      int idx = 0;
      QPainter p(this);
      setPainter(p);
      for (iTrack i = tl->begin(); i != tl->end(); ++i, ++idx) {
            // normal cast does not work!
            SoundSource* ss = dynamic_cast<SoundSource*>(*i);
            int y = idx * TH;
            int w = header->sectionSize(COL_ACTIVITY);
            int x = header->sectionPos(COL_ACTIVITY);

            int val  = ss->activity();
            int cval = ss->curActivity();

            int diff = val - cval;
            if (diff == 0)
                  continue;
            diff /= 4;
            cval = diff ? cval + diff : val;
            ss->setCurActivity(cval);

            double ko = double(w) / 10.0;
            int nval  = int(log2(double(cval)) * ko);
            if (nval < 0)
                  nval = 0;
            if (nval > w)
                  nval = w;

            if (activityMode == 0) {
                  p.fillRect(x+1, y+1, nval, TH-1, activityColor);
                  }
            else {
	            for (int count = 1; count < nval; count++) {
	                  int percent = (count * 0xff) / w;
	                  int r = (percent<0x7f) ? percent * 2 : 0xff;
	                  int g = (percent>0x7f) ? (0x1ff-(percent*2)) : 0xff;
	                  int b = 0;
	                  if (!(count % 2))
	                        r = g = b = 0xff;
	                  p.fillRect(count+x, y+1, 1, TH-1, QBrush(QColor(r, g, b)));
	                  }
	            }
            p.fillRect(x+nval+1, y+1, w-nval-1, TH-1, white);
            }
      }

//---------------------------------------------------------
//   checkEmpty
//---------------------------------------------------------

static bool checkEmpty(Track* t, QWidget* parent)
      {
      if (t->parts()->size()) {
            QMessageBox::information(parent, "MusE",
               parent->tr("Cannot transform non empty track"));
            return true;
            }
      return false;
      }

//---------------------------------------------------------
//   classesPopupMenu
//---------------------------------------------------------

void TList::classesPopupMenu(int track, int x, int y)
      {
      TrackList* tracks = song->tracks();
      Track* t = tracks->findIdx(track);
      QPopupMenu p(this);
      p.clear();
      p.insertItem(*noteIcon, tr("Midi"), Track::MIDI);
      p.insertItem(*stickIcon, tr("Drum"), Track::DRUM);
      p.insertItem(*waveIcon, tr("Wave"), Track::WAVE);
      int n = p.exec(mapToGlobal(QPoint(x, y)), 0);

      if (n == -1)
            return;

      QString name  = t->tname();
      bool selected = t->selected();

      switch (Track::TrackType(n)) {
            case Track::MIDI:
                  if (t->type() == Track::DRUM)
                        t->setType(Track::MIDI);
                  else if (t->type() == Track::WAVE) {
                        if (checkEmpty(t, this))
                              break;
                        MidiTrack* mt = new MidiTrack(name);
                        mt->setSn(t->sn());
                        mt->setSelected(selected);
                        mt->setMute(((WaveTrack*)t)->mute());
                        mt->setSolo(((WaveTrack*)t)->soloMute());
                        midiThread->msgChangeTrack(t, mt);
                        }
                  break;
            case Track::DRUM:
                  if (t->type() == Track::MIDI)
                        t->setType(Track::DRUM);
                  else if (t->type() == Track::WAVE) {
                        if (checkEmpty(t, this))
                              break;
                        MidiTrack* mt = new MidiTrack(name);
                        mt->setSn(t->sn());
                        mt->setType(Track::DRUM);
                        mt->setSelected(selected);
                        mt->setMute(((WaveTrack*)t)->mute());
                        mt->setSolo(((WaveTrack*)t)->soloMute());
                        midiThread->msgChangeTrack(t, mt);
                        }
                  break;
            case Track::WAVE:
                  if (t->type() == Track::MIDI || t->type() == Track::DRUM) {
                        if (checkEmpty(t, this))
                              break;
                        MidiTrack* mt = (MidiTrack*) t;
                        // MUTABOR!
                        bool mute     = mt->mute();
                        bool soloMute = mt->soloMute();

                        WaveTrack* wt = new WaveTrack(name);

                        wt->setSn(t->sn());
                        wt->setSelected(selected);
                        wt->setMute(mute);
                        wt->setSolo(soloMute);
                        midiThread->msgChangeTrack(t, wt);

                        if (selected)
                              emit selectionChanged();
                        }
                  break;
            }
      }

//---------------------------------------------------------
//   writeStatus
//---------------------------------------------------------

void TList::writeStatus(int level, Xml& xml, const char* name) const
      {
      xml.tag(level++, name);
      header->writeStatus(level, xml);
      xml.etag(level, name);
      }

//---------------------------------------------------------
//   readStatus
//---------------------------------------------------------

void TList::readStatus(Xml& xml, const char* name)
      {
      for (;;) {
            Xml::Token token(xml.parse());
            const QString& tag(xml.s1());
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == header->name())
                              header->readStatus(xml);
                        else
                              xml.unknown("Tlist");
                        break;
                  case Xml::TagEnd:
                        if (tag == name)
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   setActivityMode
//---------------------------------------------------------

void TList::setActivityMode(int i)
      {
      activityMode = i;
      }

//---------------------------------------------------------
//   setActivityColor
//---------------------------------------------------------

void TList::setActivityColor(QColor c)
      {
      activityColor = c;
      }

//---------------------------------------------------------
//   setSelectedTrackColor
//---------------------------------------------------------

void TList::setSelectedTrackColor(QColor c)
      {
      selectedTrackColor = c;
      redraw();
      }

QColor TList::getSelectedTrackColor() const
      {
      return selectedTrackColor;
      }

