//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: arranger.cpp,v 1.5 2002/02/27 08:48:09 muse Exp $
//  (C) Copyright 1999 Werner Schweer (ws@seh.de)
//=========================================================

#include "config.h"

#include <stdio.h>
#include "arranger.h"
#include <qlayout.h>
#include <qsplitter.h>
#include <qcombobox.h>
#include <qlistbox.h>
#include <qtoolbutton.h>
#include <qbuttongroup.h>
#include <qsizegrip.h>
#include <qlabel.h>
#include <qaccel.h>
#include <qfontmetrics.h>
#include <qcombobox.h>
#include <qwhatsthis.h>
#include <qtoolbar.h>
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qhbox.h>
#include <qstringlist.h>
#include <qfiledialog.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qstyle.h>

#include "song.h"
#include "mtscale.h"
#include "scrollscale.h"
#include "pcanvas.h"
#include "poslabel.h"
#include "intlabel.h"
#include "seq.h"
#include "xml.h"
#include "splitter.h"
#include "lcombo.h"
#include "midiport.h"
#include "../driver/mididev.h"
#include "utils.h"
#include "globals.h"
#include "midictrl.h"
#include "tlist.h"
#include "minstrument.h"
#include "icons.h"
#include "../sf/sndfile.h"
#include "trackinfobase.h"
#include "header.h"
#include "utils.h"
#include "midithread.h"

//---------------------------------------------------------
//   TWhatsThis::text
//---------------------------------------------------------

QString TWhatsThis::text(const QPoint& pos)
      {
      int section = header->sectionAt(pos.x());
      if (section == -1)
            return 0;
      switch(section) {
            case COL_RECORD:   return header->tr("Enable Recording"); break;
            case COL_ACTIVITY: return header->tr("Track Activity"); break;
            case COL_MUTE:     return header->tr("Mute Indicator"); break;
            case COL_CLASS:    return header->tr("Track Type"); break;
            case COL_NAME:     return header->tr("Track Name"); break;
            case COL_OCHANNEL: return header->tr("Output Channel Number"); break;
            case COL_OPORT:    return header->tr("Output Port"); break;
            case COL_TIMELOCK: return header->tr("Time Lock"); break;
            default: break;
            }
      return 0;
      }

//---------------------------------------------------------
//   ColorListItem
//---------------------------------------------------------

class ColorListItem : public QListBoxItem {
      int r, g, b;
      int h;
      int fontheight;
      QString label;
      virtual int height(const QListBox*) const { return h; }
      virtual int width(const QListBox*) const  { return 80; }
      virtual void paint(QPainter* p) {
            p->fillRect(5, 2, h-4, h-4, QBrush(QColor(r,g,b)));
            p->drawText(5 + h - 4 + 3, fontheight * 0.75, label);
            }
   public:
      ColorListItem(int _r, int _g, int _b, int _h,  int _fh, const char* txt) {
            r = _r;
            g = _g;
            b = _b;
            label = txt;
            h = _h;
	    fontheight = _fh;
            }
      QString text() const { return QString("PartColor"); }
      };

//---------------------------------------------------------
//   Arranger
//---------------------------------------------------------

Arranger::Arranger(MainWindow* parent, const char* name = 0)
   : QWidget(parent, name)
      {
      _raster  = 0;      // measure
      selected = 0;
      setMinimumSize(600, 50);
      showTrackinfo = false;

      trackInfoSize = 12 * fontMetrics().width('m');

      //---------------------------------------------------
      //    ToolBar
      //---------------------------------------------------

      QToolBar* toolbar = new QToolBar("Arranger", parent);

      solo = new QToolButton(toolbar);
      solo->setToggleButton(true);
      solo->setText(tr(" Solo "));
      connect(solo, SIGNAL(toggled(bool)), SLOT(soloChanged(bool)));
      connect(song, SIGNAL(soloChanged(SoundSource*)), SLOT(soloChanged(SoundSource*)));

      QLabel* label = new QLabel(tr("Cursor"), toolbar, "Cursor");
      label->setAlignment(AlignRight|AlignVCenter);
      label->setIndent(3);
      cursorPos = new PosLabel(toolbar);
      cursorPos->setEnabled(false);

      const char* rastval[] = {
            "Off", "Bar", "1/2", "1/4", "1/8", "1/16"
            };
      label = new QLabel(tr("Snap"), toolbar, "Snap");
      label->setAlignment(AlignRight|AlignVCenter);
      label->setIndent(3);
      QComboBox* raster = new QComboBox(toolbar);
      for (int i = 0; i < 6; i++)
            raster->insertItem(rastval[i], i);
      raster->setCurrentItem(1);
      connect(raster, SIGNAL(activated(int)), SLOT(_setRaster(int)));

      // Song len
      label = new QLabel(tr("Len"), toolbar, "Len");
      label->setAlignment(AlignRight|AlignVCenter);
      label->setIndent(3);
      lenEntry = new QSpinBox(0, 50000000, 1, toolbar);
      lenEntry->setValue(song->len());
      connect(lenEntry, SIGNAL(valueChanged(int)), SLOT(songlenChanged(int)));

      QComboBox* partColor = new QComboBox(false, toolbar);
      connect(partColor, SIGNAL(activated(int)), SLOT(setPartColor(int)));
      QListBox* colorList = new QListBox(toolbar);
      colorList->setColumnMode(1);
      partColor->setListBox(colorList);

      // part color selection
      const QFontMetrics& fm = colorList->fontMetrics();
      int h = fm.lineSpacing();

      PartColor* p = partColors;
      for (int i = 0; i < NUM_PARTCOLORS; ++i, ++p) {
            ColorListItem* item = new ColorListItem(p->r, p->g, p->b, h, fontMetrics().height(), p->name);
            colorList->insertItem(item, i);
            }
      colorList->setCurrentItem(0);

      typeBox = new LabelCombo(tr("Type"), toolbar);
      typeBox->insertItem(tr("NO"), 0);
      typeBox->insertItem("GM", 1);
      typeBox->insertItem("GS", 2);
      typeBox->insertItem("XG", 3);
      typeBox->setCurrentItem(0);
      connect(typeBox, SIGNAL(activated(int)), SLOT(modeChange(int)));

      QVBoxLayout* box  = new QVBoxLayout(this);
      box->addWidget(hLine(this), 0);

      //---------------------------------------------------
      //  Tracklist
      //---------------------------------------------------

      int xscale = -100;
      int yscale = 1;

      split  = new Splitter(Horizontal, this, "split");

      box->addWidget(split, 100);

      QWidget* tracklist = new QWidget(split);
      tracklist->setMinimumSize(1, 20);
      split->setResizeMode(tracklist, QSplitter::KeepSize);

      //---------------------------------------------------
      //    Track Info
      //---------------------------------------------------

      genTrackInfo(tracklist);

      // Track-Info Button
      QWidget* ttools    = new QWidget(tracklist);
      QHBoxLayout* ttbox = new QHBoxLayout(ttools);
      ib                 = new QToolButton(ttools);
      ib->setText(tr("TrackInfo"));

//      ttools->setFixedHeight(18);

      ib->setToggleButton(true);
      ib->setOn(showTrackinfo);

      ttbox->addWidget(ib, 0);
      ttbox->addStretch(100);
      connect(ib, SIGNAL(toggled(bool)), SLOT(showInspector(bool)));

      header = new Header(tracklist, "header");
      header->setFixedHeight(30);

      QFontMetrics fm1(header->font());
      int fw = style().pixelMetric(QStyle::PM_DefaultFrameWidth, header);
      fw = fw * 4 + 2;

      header->addLabel("R", fm1.width('R')+fw);
      header->addLabel("A", fm1.width('A')+fw);
      header->addLabel("M", fm1.width('M')+fw);
      header->addLabel("C", fm1.width('C')+fw);
      header->addLabel(tr("Track"), 100);
      header->addLabel(tr("O-Port"), 60);
      header->addLabel("Ch", 30);
      header->addLabel("T", fm1.width('T')+fw);
      header->setResizeEnabled(false, COL_RECORD);
      header->setResizeEnabled(false, COL_MUTE);
      header->setResizeEnabled(false, COL_CLASS);
      header->setResizeEnabled(false, COL_OCHANNEL);
      header->setResizeEnabled(false, COL_TIMELOCK);

      header->setStretchEnabled(true, COL_NAME);
      header->setTracking(true);

      new THeaderTip(header);
      new TWhatsThis(header, header);

      list = new TList(header, tracklist, yscale);

      connect(list, SIGNAL(selectionChanged()), SLOT(trackSelectionChanged()));
      connect(header, SIGNAL(sizeChange(int,int,int)), list, SLOT(redraw()));
      connect(header, SIGNAL(moved(int,int)), list, SLOT(redraw()));
      connect(header, SIGNAL(moved(int,int)), this, SLOT(headerMoved()));

      //         0         1         2
      //   +-----------+--------+---------+
      //   | Trackinfo | vline  | Header  | 0
      //   |           |        +---------+
      //   |           |        | TList   | 1
      //   +-----------+--------+---------+
      //   |             hline            | 2
      //   +------------------------------+
      //   | ttools                       | 3
      //   +------------------------------+

      QGridLayout* tgrid = new QGridLayout(tracklist);
      tgrid->addMultiCellWidget(midiTrackInfo,    0, 1, 0, 0, AlignTop);
      tgrid->addMultiCellWidget(waveTrackInfo,    0, 1, 0, 0, AlignTop);
      tgrid->addMultiCellWidget(noTrackInfo,      0, 1, 0, 0, AlignTop);
      tgrid->addMultiCellWidget(vLine(tracklist), 0, 1, 1, 1);
      tgrid->addWidget(header,                       0,    2);
      tgrid->addWidget(list,                         1,    2);
      tgrid->addMultiCellWidget(hLine(tracklist), 2, 2, 0, 2, AlignTop);
      tgrid->addMultiCellWidget(ttools,           3, 3, 0, 3);
      tgrid->setRowStretch(1, 100);
      tgrid->setColStretch(2, 100);

      //---------------------------------------------------
      //    Editor
      //---------------------------------------------------

      QWidget* editor = new QWidget(split);
      int offset = sigmap.ticksMeasure(0);
      hscroll = new ScrollScale(-1000, -10, xscale, song->len(), Horizontal, editor, -offset);
      vscroll = new ScrollScale(1, 5, yscale, 20*TH, Vertical, editor);
      list->setScroll(vscroll);

      QGridLayout* egrid  = new QGridLayout(editor);
      egrid->setColStretch(0, 100);
      egrid->setRowStretch(2, 100);

      time = new MTScale(&_raster, editor, xscale);
      time->setOrigin(-offset, 0);
      canvas = new PartCanvas(&_raster, editor, xscale, yscale);
      canvas->setOrigin(-offset, 0);

      egrid->addMultiCellWidget(time,           0, 0, 0, 1);
      egrid->addMultiCellWidget(hLine(editor),  1, 1, 0, 1);
      egrid->addWidget(canvas,  2, 0);
      egrid->addWidget(vscroll, 2, 1);
      egrid->addWidget(hscroll, 3, 0);

      connect(vscroll, SIGNAL(scrollChanged(int)), canvas, SLOT(setYPos(int)));
      connect(vscroll, SIGNAL(scaleChanged(int)),  canvas, SLOT(setYMag(int)));
      connect(hscroll, SIGNAL(scrollChanged(int)), canvas, SLOT(setXPos(int)));
      connect(hscroll, SIGNAL(scaleChanged(int)),  canvas, SLOT(setXMag(int)));
      connect(vscroll, SIGNAL(scrollChanged(int)), list,   SLOT(setYPos(int)));
      connect(vscroll, SIGNAL(scaleChanged(int)),  list,   SLOT(setYMag(int)));
      connect(hscroll, SIGNAL(scrollChanged(int)), time,   SLOT(setXPos(int)));
      connect(hscroll, SIGNAL(scaleChanged(int)),  time,   SLOT(setXMag(int)));

      connect(canvas,  SIGNAL(timeChanged(int)),   SLOT(setTime(int)));
      connect(time,    SIGNAL(timeChanged(int)),   SLOT(setTime(int)));

      connect(canvas, SIGNAL(tracklistChanged()), list, SLOT(tracklistChanged()));
      connect(canvas, SIGNAL(dclickPart(Track*)), SIGNAL(editPart(Track*)));
      connect(canvas, SIGNAL(startEditor(PartList*,int)),   SIGNAL(startEditor(PartList*, int)));

      connect(partColor, SIGNAL(activated(int)), canvas, SLOT(setCurColorIndex(int)));

      QValueList<int> sizes;
      sizes.append(200);
      split->setSizes(sizes);

      connect(song, SIGNAL(songChanged(int)), SLOT(songChanged(int)));
      connect(song, SIGNAL(soloChanged(SoundSource*)), list, SLOT(redraw()));
      connect(song, SIGNAL(muteChanged(SoundSource*)), list, SLOT(redraw()));
      connect(canvas, SIGNAL(followEvent(int)), hscroll, SLOT(setOffset(int)));
      connect(canvas, SIGNAL(selectionChanged()), SIGNAL(selectionChanged()));
      connect(canvas, SIGNAL(dropFile(const QString&)), SIGNAL(dropFile(const QString&)));
      }

//---------------------------------------------------------
//   headerMoved
//---------------------------------------------------------

void Arranger::headerMoved()
      {
      header->setStretchEnabled(true, COL_NAME);
      }

//---------------------------------------------------------
//   setTime
//---------------------------------------------------------

void Arranger::setTime(int tick)
      {
      cursorPos->setEnabled(tick != -1);
      cursorPos->setValue(tick);
      time->setPos(3, tick, false);
      }

//---------------------------------------------------------
//   toolChange
//---------------------------------------------------------

void Arranger::setTool(int t)
      {
      canvas->setTool(t);
      }

//---------------------------------------------------------
//   dclickPart
//---------------------------------------------------------

void Arranger::dclickPart(Track* t)
      {
      emit editPart(t);
      }

//---------------------------------------------------------
//   setBgPixmap
//---------------------------------------------------------

void Arranger::setBgPixmap(const QString& bg)
      {
      _bgPixmap = bg;
      QPixmap pm(bg);
      canvas->setBg(pm);
      }

//---------------------------------------------------------
//   songlenChanged
//---------------------------------------------------------

void Arranger::songlenChanged(int n)
      {
      int newLen = sigmap.bar2tick(n, 0, 0);
      song->setLen(newLen);
      }

//---------------------------------------------------------
//   songChanged
//---------------------------------------------------------

void Arranger::songChanged(int type)
      {
      vscroll->setRange(0, TH * (song->tracks()->size()+2));
      selected = 0;
      int endTick = song->len();
      int offset  = sigmap.ticksMeasure(0);
      hscroll->setRange(-offset, endTick + offset);  //DEBUG
      canvas->setOrigin(-offset, 0);
      time->setOrigin(-offset, 0);

      int bar, beat, tick;
      sigmap.tickValues(endTick, &bar, &beat, &tick);
      if (tick || beat)
            ++bar;
      lenEntry->setValue(bar);

      trackSelectionChanged();
      if (!selected && !song->tracks()->empty()) {
            song->tracks()->front()->setSelected(true);
            trackSelectionChanged();
            }
      canvas->partsChanged();
      list->redraw();
      typeBox->setCurrentItem(int(song->mtype()));
      if (type & SC_SIG)
            time->redraw();
//      if (type & SC_TRACK_MODIFIED)
//            updateInspector();
      }

//---------------------------------------------------------
//   showInspector
//---------------------------------------------------------

void Arranger::showInspector(bool flag)
      {
      if (flag) {
            showTrackinfo = true;
            if (selected == 0) {
                  updateNoTrackInfo();
                  return;
                  }
            switch(selected->type()) {
                  case Track::MIDI:
                  case Track::DRUM:
                        updateMidiTrackInfo();
                        break;
                  case Track::WAVE:
                        updateWaveTrackInfo();
                        break;
                  }
            }
      else  {
            showTrackinfo = false;
            midiTrackInfo->hide();
            waveTrackInfo->hide();
            noTrackInfo->hide();
            }
      }

//---------------------------------------------------------
//   selectionChanged
//---------------------------------------------------------

void Arranger::trackSelectionChanged()
      {
      TrackList* tracks = song->tracks();
      Track* track = 0;
      for (iTrack t = tracks->begin(); t != tracks->end(); ++t) {
            if ((*t)->selected()) {
                  track = *t;
                  break;
                  }
            }
      if (track == 0) {
            selected = 0;
            updateInspector();
            return;
            }
      if (track == selected)
            return;
      selected = track;
      updateInspector();
      }

//---------------------------------------------------------
//   updateInspector
//---------------------------------------------------------

void Arranger::updateInspector()
      {
      if (!showTrackinfo)
            return;
      if (!selected) {
            updateNoTrackInfo();
            return;
            }
      switch(selected->type()) {
            case Track::MIDI:
            case Track::DRUM:
                  updateMidiTrackInfo();
                  break;
            case Track::WAVE:
                  updateWaveTrackInfo();
                  break;
            }
      }

//---------------------------------------------------------
//   updateNoTrackInfo
//---------------------------------------------------------

void Arranger::updateNoTrackInfo()
      {
      waveTrackInfo->hide();
      midiTrackInfo->hide();
      noTrackInfo->show();
      }

//---------------------------------------------------------
//   iNameChanged
//---------------------------------------------------------

void Arranger::iNameChanged()
      {
      QString txt = midiTrackInfo->iName->text();
      nameChanged(txt);
      }
#if 0
//---------------------------------------------------------
//   iwNameChanged
//---------------------------------------------------------

void Arranger::iwNameChanged()
      {
      QString txt = iwName->text();
      nameChanged(txt);
      }
#endif

//---------------------------------------------------------
//   nameChanged
//---------------------------------------------------------

void Arranger::nameChanged(const QString& txt)
      {
      if (txt == selected->tname())
            return;
      Track* track = selected->clone();
      track->setTName(txt);
      midiThread->msgChangeTrack(selected, track);
      }

//---------------------------------------------------------
//   iOutputChannelChanged
//---------------------------------------------------------

void Arranger::iOutputChannelChanged(int channel)
      {
      --channel;
      MidiTrack* track = (MidiTrack*)selected;
      if (channel != track->outChannel()) {
            track->setOutChannel(channel);
            list->redraw();
            }
      }

//---------------------------------------------------------
//   iKanalChanged
//---------------------------------------------------------

void Arranger::iInputChannelChanged(const QString& s)
      {
      MidiTrack* track = (MidiTrack*)selected;
      int val = string2bitmap(s);
      if (val != track->inChannelMask()) {
            track->setInChannelMask(val);
            list->redraw();
            }
      }

//---------------------------------------------------------
//   iOutputPortChanged
//---------------------------------------------------------

void Arranger::iOutputPortChanged(int index)
      {
      MidiTrack* track = (MidiTrack*)selected;
      if (index == track->outPort())
            return;
      track->setOutPort(index);
      list->redraw();
      }

//---------------------------------------------------------
//   iInputPortChanged
//---------------------------------------------------------

void Arranger::iInputPortChanged(const QString& s)
      {
      int val = string2bitmap(s);
      MidiTrack* track = (MidiTrack*)selected;
      if (val == track->inPortMask())
            return;
      track->setInPortMask(val);
      list->redraw();
      }

//---------------------------------------------------------
//   iHBankChanged
//---------------------------------------------------------

void Arranger::iHBankChanged(int hbank)
      {
      MidiPort* port = &midiPorts[selected->outPort()];
      int channel = ((MidiTrack*)selected)->outChannel();

      int prg   = port->iState(channel)->program;
      int lbank = port->iState(channel)->controller[CTRL_LBANK];
      port->programChange(channel, hbank, lbank, prg);
      updateInspector();
      }

//---------------------------------------------------------
//   iLBankChanged
//---------------------------------------------------------

void Arranger::iLBankChanged(int lbank)
      {
      MidiPort* port = &midiPorts[selected->outPort()];
      int channel = ((MidiTrack*)selected)->outChannel();

      int prg   = port->iState(channel)->program;
      int hbank = port->iState(channel)->controller[CTRL_HBANK];
      port->programChange(channel, hbank, lbank, prg);
      updateInspector();
      }

//---------------------------------------------------------
//   iProgramChanged
//---------------------------------------------------------

void Arranger::iProgramChanged(int prg)
      {
      prg -= 1;

      MidiPort* port = &midiPorts[selected->outPort()];
      int channel = selected->outChannel();

      int lbank = port->iState(channel)->controller[CTRL_LBANK];
      int hbank = port->iState(channel)->controller[CTRL_HBANK];
      port->programChange(channel, hbank, lbank, prg);
      updateInspector();
      }

//---------------------------------------------------------
//   iLautstChanged
//---------------------------------------------------------

void Arranger::iLautstChanged(int val)
      {
      MidiPort* port = &midiPorts[selected->outPort()];
      int channel = ((MidiTrack*)selected)->outChannel();
      port->iState(channel)->controller[CTRL_VOLUME] = val;

      // Realtime Change:
      port->iState(channel)->controller[CTRL_VOLUME] = val;
      port->setCtrl(channel, CTRL_VOLUME, val);
      updateInspector();
      }

//---------------------------------------------------------
//   iTranspChanged
//---------------------------------------------------------

void Arranger::iTranspChanged(int val)
      {
      MidiTrack* track = (MidiTrack*)selected;
      track->transposition = val;
      }

//---------------------------------------------------------
//   iAnschlChanged
//---------------------------------------------------------

void Arranger::iAnschlChanged(int val)
      {
      MidiTrack* track = (MidiTrack*)selected;
      track->velocity = val;
      }

//---------------------------------------------------------
//   iVerzChanged
//---------------------------------------------------------

void Arranger::iVerzChanged(int val)
      {
      MidiTrack* track = (MidiTrack*)selected;
      track->delay = val;
      }

//---------------------------------------------------------
//   iLenChanged
//---------------------------------------------------------

void Arranger::iLenChanged(int val)
      {
      MidiTrack* track = (MidiTrack*)selected;
      track->len = val;
      }

//---------------------------------------------------------
//   iKomprChanged
//---------------------------------------------------------

void Arranger::iKomprChanged(int val)
      {
      MidiTrack* track = (MidiTrack*)selected;
      track->compression = val;
      }

//---------------------------------------------------------
//   iPanChanged
//---------------------------------------------------------

void Arranger::iPanChanged(int val)
      {
      MidiPort* port = &midiPorts[selected->outPort()];
      int channel = selected->outChannel();
      port->iState(channel)->controller[CTRL_PANPOT] = val;

      // Realtime Change:
      port->setCtrl(channel, CTRL_PANPOT, val);
      }

//---------------------------------------------------------
//   setPartCOlor
//---------------------------------------------------------

void Arranger::setPartColor(int color)
      {
      TrackList* tl = song->tracks();
      for (iTrack t = tl->begin(); t != tl->end(); ++t) {
            PartList* pl = (*t)->parts();
            for (iPart p = pl->begin(); p != pl->end(); ++p)
                  if (p->second->selected())
                        p->second->setColorIndex(color);
            }
      canvas->redraw();
      }

//---------------------------------------------------------
//   soloChanged
//---------------------------------------------------------

void Arranger::soloChanged(bool flag)
      {
      if (selected)
            song->setSolo(flag ? dynamic_cast<SoundSource*>(selected) : 0);
      }

//---------------------------------------------------------
//   soloChanged
//---------------------------------------------------------

void Arranger::soloChanged(SoundSource* s)
      {
      disconnect(solo, SIGNAL(toggled(bool)), this, SLOT(soloChanged(bool)));
      solo->setOn(s == dynamic_cast<SoundSource*>(selected));
      connect(solo, SIGNAL(toggled(bool)), SLOT(soloChanged(bool)));
      }

//---------------------------------------------------------
//   modeChange
//---------------------------------------------------------

void Arranger::modeChange(int mode)
      {
      song->setMType(MType(mode));
      updateInspector();
      }

//---------------------------------------------------------
//   setMode
//---------------------------------------------------------

void Arranger::setMode(int mode)
      {
      typeBox->setCurrentItem(mode);
      }

//---------------------------------------------------------
//   TrackInfo
//---------------------------------------------------------

class MidiTrackInfo : public TrackInfoBase {
   public:
      QSize sizeHint() const {
            return QSize(10*fontMetrics().width('m'), 5000);
            }
      MidiTrackInfo(QWidget* parent) : TrackInfoBase(parent) {}
      };

//---------------------------------------------------------
//   genTrackInfo
//---------------------------------------------------------

void Arranger::genTrackInfo(QWidget* parent)
      {
      noTrackInfo   = new QWidget(parent);
      midiTrackInfo = new MidiTrackInfo(parent);
      waveTrackInfo = new QWidget(parent);

      noTrackInfo->setFixedWidth(trackInfoSize);
      waveTrackInfo->setFixedWidth(12*fontMetrics().width('m'));

      midiTrackInfo->hide();
      waveTrackInfo->hide();
      if (!showTrackinfo)
            noTrackInfo->hide();
      genWaveTrackInfo();
      genMidiTrackInfo();

      noTrackInfo->setMinimumHeight(50);
      midiTrackInfo->setMinimumHeight(50);
      waveTrackInfo->setMinimumHeight(50);
      }

//---------------------------------------------------------
//   writeStatus
//---------------------------------------------------------

void Arranger::writeStatus(int level, Xml& xml)
      {
      xml.tag(level++, "arranger");
      xml.intTag(level, "info", ib->isOn());
      split->writeStatus(level, xml);
      list->writeStatus(level, xml, "list");

      xml.intTag(level, "part_type",  canvas->showPartType());
      xml.intTag(level, "show_events", canvas->showPartEvent());
      xml.strTag(level, "image", _bgPixmap.latin1());
      xml.intTag(level, "xpos", hscroll->pos());
      xml.intTag(level, "xmag", hscroll->mag());
      xml.intTag(level, "ypos", vscroll->pos());
      xml.intTag(level, "ymag", vscroll->mag());
      xml.intTag(level, "grid", hasGrid()?1:0);
      xml.etag(level, "arranger");
      }

//---------------------------------------------------------
//   readStatus
//---------------------------------------------------------

void Arranger::readStatus(Xml& xml)
      {
      for (;;) {
            Xml::Token token(xml.parse());
            const QString& tag(xml.s1());
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "info")
                              showTrackinfo = xml.parseInt();
                        else if (tag == split->name())
                              split->readStatus(xml);
                        else if (tag == "part_type")
                              canvas->setShowPartType(xml.parseInt());
                        else if (tag == "show_events")
                              canvas->setShowPartEvent(xml.parseInt());
                        else if (tag == "list")
                              list->readStatus(xml, "list");
                        else if (tag == "image")
                              setBgPixmap(xml.parse1());
                        else if (tag == "xmag")
                              hscroll->setMag(xml.parseInt());
                        else if (tag == "xpos") {
                              int hpos = xml.parseInt();
                              hscroll->setPos(hpos);
                              }
                        else if (tag == "ymag")
                              vscroll->setMag(xml.parseInt());
                        else if (tag == "ypos")
                              vscroll->setPos(xml.parseInt());
                        else if (tag == "grid")
                              setGrid((xml.parseInt() == 1)?true:false);
                        else
                              xml.unknown("Arranger");
                        break;
                  case Xml::TagEnd:
                        if (tag == "arranger") {
                              ib->setOn(showTrackinfo);
                              return;
                              }
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   setRaster
//---------------------------------------------------------

void Arranger::_setRaster(int index)
      {
      static int rasterTable[] = {
            1, 0, 768, 384, 192, 96
            };
      _raster = rasterTable[index];
      canvas->redraw();
      }

//---------------------------------------------------------
//   instrPopup
//---------------------------------------------------------

void Arranger::instrPopup()
      {
      int channel = ((MidiTrack*)selected)->outChannel();
      int port = selected->outPort();
      MidiInstrument* instr = midiPorts[port].instrument();
      instr->populatePatchPopup(pop, channel, song->mtype());

      int rv = pop->exec(midiTrackInfo->iPatch->mapToGlobal(QPoint(10,5)));
      if (rv != -1) {
            //
            // TODO: songtype bercksichtigen; Drumtrack?
            //       bei GM hbank und lbank nicht setzen
            //       bei GS lbank nicht setzen

            int hb   = (rv >> 16) & 0xff;
            int lb   = (rv >> 8)  & 0xff;
            int prog = rv & 0xff;

            MidiPort* mp = &midiPorts[port];

            mp->programChange(channel, hb, lb, prog);
            updateInspector();
            }
      }

//---------------------------------------------------------
//   genMidiTrackInfo
//---------------------------------------------------------

void Arranger::genMidiTrackInfo()
      {
      QWhatsThis::add(midiTrackInfo->iOutput, tr("Midi output port midi events from this track are send to.\n"
        "Midi ports are associated to a specific midi device in config->midiPorts menu"));
      QWhatsThis::add(midiTrackInfo->iInput, tr("Midi input port midi events are recorded from.\n"
        "Midi ports are associated to a specific midi device in config->midiPorts menu"));

      connect(midiTrackInfo->midiThru, SIGNAL(toggled(bool)), SLOT(midiThruChanged(bool)));

      connect(midiTrackInfo->iPatch, SIGNAL(released()), SLOT(instrPopup()));

      pop = new QPopupMenu(midiTrackInfo->iPatch);

      connect(midiTrackInfo->iName, SIGNAL(returnPressed()), SLOT(iNameChanged()));
      connect(midiTrackInfo->iOutputChannel, SIGNAL(valueChanged(int)), SLOT(iOutputChannelChanged(int)));
      connect(midiTrackInfo->iInputChannel, SIGNAL(textChanged(const QString&)), SLOT(iInputChannelChanged(const QString&)));
      connect(midiTrackInfo->iHBank, SIGNAL(valueChanged(int)), SLOT(iHBankChanged(int)));
      connect(midiTrackInfo->iLBank, SIGNAL(valueChanged(int)), SLOT(iLBankChanged(int)));
      connect(midiTrackInfo->iProgram, SIGNAL(valueChanged(int)), SLOT(iProgramChanged(int)));
      connect(midiTrackInfo->iLautst, SIGNAL(valueChanged(int)), SLOT(iLautstChanged(int)));
      connect(midiTrackInfo->iTransp, SIGNAL(valueChanged(int)), SLOT(iTranspChanged(int)));
      connect(midiTrackInfo->iAnschl, SIGNAL(valueChanged(int)), SLOT(iAnschlChanged(int)));
      connect(midiTrackInfo->iVerz, SIGNAL(valueChanged(int)), SLOT(iVerzChanged(int)));
      connect(midiTrackInfo->iLen, SIGNAL(valueChanged(int)), SLOT(iLenChanged(int)));
      connect(midiTrackInfo->iKompr, SIGNAL(valueChanged(int)), SLOT(iKomprChanged(int)));
      connect(midiTrackInfo->iPan, SIGNAL(valueChanged(int)), SLOT(iPanChanged(int)));
      connect(midiTrackInfo->iOutput, SIGNAL(activated(int)), SLOT(iOutputPortChanged(int)));
      connect(midiTrackInfo->iInput, SIGNAL(textChanged(const QString&)), SLOT(iInputPortChanged(const QString&)));
      }

//---------------------------------------------------------
//   midiThruChanged
//---------------------------------------------------------

void Arranger::midiThruChanged(bool flag)
      {
      ((MidiTrack*)selected)->setMidiThruFlag(flag);
      }

//---------------------------------------------------------
//   updateMidiTrackInfo
//---------------------------------------------------------

void Arranger::updateMidiTrackInfo()
      {
      waveTrackInfo->hide();
      noTrackInfo->hide();
      midiTrackInfo->show();

      int outChannel = selected->outChannel();
      int inChannel  = selected->inChannelMask();
      int outPort    = selected->outPort();
      int inPort     = selected->inPortMask();

      midiTrackInfo->iInput->clear();
      midiTrackInfo->iOutput->clear();

      midiTrackInfo->midiThru->setChecked(((MidiTrack*)selected)->midiThruFlag());

      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiPort* port  = &midiPorts[i];
            MidiDevice* dev = port->device();
            QString name;
            name.sprintf("%d(%s)", i+1, dev ? dev->name().latin1() : "none");
            midiTrackInfo->iOutput->insertItem(name, i);
            if (i == outPort)
                  midiTrackInfo->iOutput->setCurrentItem(i);
            if (!(port->rwFlags() & 0x1)) {
                  // disable entry
                  }
            }
      midiTrackInfo->iInput->setText(bitmap2String(inPort));
      midiTrackInfo->iInputChannel->setText(bitmap2String(inChannel));

      ChannelState* istate = midiPorts[outPort].iState(outChannel);

      if (midiTrackInfo->iName->text() != selected->tname()) {
            midiTrackInfo->iName->setText(selected->tname());
            midiTrackInfo->iName->home(false);
            }

      midiTrackInfo->iOutputChannel->setValue(outChannel+1);
      midiTrackInfo->iInputChannel->setText(bitmap2String(inChannel));

      MidiInstrument* instr = midiPorts[outPort].instrument();

      midiTrackInfo->iPatch->setText(instr->getPatchName(outChannel, istate->controller[CTRL_HBANK],
         istate->controller[CTRL_LBANK], istate->program, song->mtype()));

      midiTrackInfo->iHBank->blockSignals(true);
      midiTrackInfo->iLBank->blockSignals(true);
      midiTrackInfo->iProgram->blockSignals(true);
      midiTrackInfo->iLautst->blockSignals(true);
      midiTrackInfo->iPan->blockSignals(true);

      midiTrackInfo->iHBank->setValue(istate->controller[CTRL_HBANK]);
      midiTrackInfo->iLBank->setValue(istate->controller[CTRL_LBANK]);
      midiTrackInfo->iProgram->setValue(istate->program + 1);
      midiTrackInfo->iLautst->setValue(istate->controller[CTRL_VOLUME]);
      midiTrackInfo->iPan->setValue(istate->controller[CTRL_PANPOT]);

      midiTrackInfo->iHBank->blockSignals(false);
      midiTrackInfo->iLBank->blockSignals(false);
      midiTrackInfo->iProgram->blockSignals(false);
      midiTrackInfo->iLautst->blockSignals(false);
      midiTrackInfo->iPan->blockSignals(false);

      midiTrackInfo->iTransp->setValue(((MidiTrack*)selected)->transposition);
      midiTrackInfo->iAnschl->setValue(((MidiTrack*)selected)->velocity);
      midiTrackInfo->iVerz->setValue(((MidiTrack*)selected)->delay);
      midiTrackInfo->iLen->setValue(((MidiTrack*)selected)->len);
      midiTrackInfo->iKompr->setValue(((MidiTrack*)selected)->compression);
      }

//---------------------------------------------------------
//   reset
//---------------------------------------------------------

void Arranger::reset()
      {
      canvas->setXPos(0);
      canvas->setYPos(0);
      hscroll->setPos(0);
      vscroll->setPos(0);
      time->setXPos(0);
      time->setYPos(0);
      }

//---------------------------------------------------------
//   genWaveTrackInfo
//---------------------------------------------------------

void Arranger::genWaveTrackInfo()
      {
      QVBoxLayout* ibox = new QVBoxLayout(waveTrackInfo);

      QLabel* iLabel = new QLabel(tr("Track Info"), waveTrackInfo);
      iLabel->setAlignment(AlignCenter);
      iLabel->setFrameStyle(QFrame::Panel|QFrame::Raised);
      iLabel->setLineWidth(1);
//      iLabel->setFixedHeight(20);

      iwName = new QLineEdit(waveTrackInfo);
//      iwName->setFont(font3);

      iChannels = new QComboBox(false, waveTrackInfo);
//      iChannels->setFixedHeight(20);
      iChannels->insertItem("Mono", 0);
      iChannels->insertItem("Stereo", 1);

      QHBoxLayout* hbox = new QHBoxLayout;
      QLabel* pLabel    = new QLabel(tr("Record File:"), waveTrackInfo);
      pLabel->setIndent(5);
      QToolButton* buttonPath = new QToolButton(waveTrackInfo);
      buttonPath->setPixmap(*openIconS);
      connect(buttonPath, SIGNAL(clicked()), SLOT(recFileDialog()));
      hbox->addWidget(pLabel);
      hbox->addWidget(buttonPath);
      hbox->setStretchFactor(pLabel, 100);

      ifName = new QLabel(waveTrackInfo);
//      ifName->setFont(font3);
      ifName->setFrameStyle(QFrame::Panel|QFrame::Sunken);
      ifName->setLineWidth(1);
      ibox->addWidget(iLabel);
      ibox->addWidget(iwName);
      ibox->addWidget(iChannels);
      ibox->addLayout(hbox);
      ibox->addWidget(ifName);

      connect(iChannels, SIGNAL(activated(int)), SLOT(iChannelsChanged(int)));
      }

//---------------------------------------------------------
//   recFileDialog
//---------------------------------------------------------

void Arranger::recFileDialog()
      {
      WaveTrack* track = (WaveTrack*)selected;
      SndFile* sf = track->recFile();
      SndFile* nsf = getSndFile(sf,  this, "recFileDialog");
      if (nsf) {
            if (nsf->openWrite()) {
//TODO: popup message
                  fprintf(stderr, "snd file <%s> open error\n",
                     nsf->name().latin1());
                  delete nsf;
                  return;
                  }
            track->setRecFile(nsf);
            ifName->setText(nsf->basename());
            }
      }

//---------------------------------------------------------
//   iChannelsChanged
//---------------------------------------------------------

void Arranger::iChannelsChanged(int i)
      {
      int ch = 0;
      switch(i) {
            case 0:    ch = 1; break;  // mono
            case 1:    ch = 2; break;  // stereo
            }
      WaveTrack* track = (WaveTrack*)selected;
      track->setChannels(ch);
      }

//---------------------------------------------------------
//   updateWaveTrackInfo
//---------------------------------------------------------

void Arranger::updateWaveTrackInfo()
      {
      noTrackInfo->hide();
      midiTrackInfo->hide();
      waveTrackInfo->show();
      WaveTrack* track = (WaveTrack*)selected;
      if (iwName->text() != track->name()) {
            iwName->setText(track->name());
            iwName->home(false);
            }
      switch(track->channels()) {
            case 1:     iChannels->setCurrentItem(0);
                        break;
            case 2:     iChannels->setCurrentItem(1);
                        break;
            default:    fprintf(stderr, "internal error: illegal channel number %d\n",
                           track->channels());
                        break;
            }
//      iFormat->setCurrentItem(0);
      SndFile* sf = track->recFile();
      if (sf)
            ifName->setText(sf->basename());
      }

//---------------------------------------------------------
//   cmd
//---------------------------------------------------------

void Arranger::cmd(int cmd)
      {
      int ncmd;
      switch (cmd) {
            case CMD_CUT_PART:
                  ncmd = PartCanvas::CMD_CUT_PART;
                  break;
            case CMD_COPY_PART:
                  ncmd = PartCanvas::CMD_COPY_PART;
                  break;
            case CMD_PASTE_PART:
                  ncmd = PartCanvas::CMD_PASTE_PART;
                  break;
            default:
                  return;
            }
      canvas->cmd(ncmd);
      }

int Arranger::showPartType() const
      {
      return canvas->showPartType();
      }
int Arranger::showPartEvent() const
      {
      return canvas->showPartEvent();
      }
void Arranger::setShowPartType(int val)
      {
      canvas->setShowPartType(val);
      }
void Arranger::setShowPartEvent(int val)
      {
      canvas->setShowPartEvent(val);
      }
void Arranger::setActivityMode(int i) {
	list->setActivityMode(i);
      }

void Arranger::setActivityColor(QColor c) {
	list->setActivityColor(c);
      }

void Arranger::setSelectedTrackColor(QColor c)
      {
	list->setSelectedTrackColor(c);
      }

int Arranger::getActivityMode() const
      {
      return list->getActivityMode();
      }
QColor Arranger::getActivityColor() const
      {
      return list->getActivityColor();
      }
QColor Arranger::getSelectedTrackColor() const
      {
      return list->getSelectedTrackColor();
      }

//---------------------------------------------------------
//   setGrid
//---------------------------------------------------------

void Arranger::setGrid(bool b)
      {
      canvas->setGrid(b);
      }

//---------------------------------------------------------
//   hasGrid
//---------------------------------------------------------

bool Arranger::hasGrid()
      {
      return canvas->hasGrid();
      }
