/******************************************************************************
** 
**  $Id: p11x_session.c,v 1.8 2009-06-24 16:21:36 rousseau Exp $
**
**  Package: PKCS-11
**  Author : Chris Osgood <oznet@mac.com>
**  License: Copyright (C) 2002 Schlumberger Network Solutions
**           <http://www.slb.com/sns>
**  Purpose: Session & object management functions
** 
******************************************************************************/

#include "cryptoki.h"

/******************************************************************************
** Function: session_LookupSession
**
** Looks up the session in the hash table, given its handle.
**
** Parameters:
**  handle - handle of the session
**
** Returns:
**  Pointer to the session with the given handle, if found.
**  NULL if no session is found with the given handle.
*******************************************************************************/
P11_Session*
session_LookupSession(CK_SESSION_HANDLE handle)
{
    P11_Session *cur;

    cur = st.session_hash[handle % NUM_SESSION_HASH_SLOTS];
    for( ; cur ; cur = cur->hnext) {
        if( cur->handle == handle ) {
            return cur;
        }
    }
    return NULL;
}

/******************************************************************************
** Function: session_AddSession
**
** Adds a new session 
**
** Parameters:
**  phandle - Returns the handle for the new session
**  psession - Returns a pointer to the new session
**
** Returns:
**  CKR_HOST_MEMORY if memory alloc failed
**  CKR_OK
*******************************************************************************/
CK_RV session_AddSession(CK_SESSION_HANDLE *phandle, P11_Session **psession)
{
    CK_RV rv = CKR_OK;
    unsigned int idx;

    if (st.sessions)
    {
        st.sessions->prev = (P11_Session *)calloc(1, sizeof(P11_Session));
		if (!st.sessions->prev) {
            rv = CKR_HOST_MEMORY;
			goto finish;
        }

		st.sessions->prev->next = st.sessions;
        st.sessions = st.sessions->prev;
        st.sessions->check = st.sessions;

    }
    else
    {
        st.sessions = (P11_Session *)calloc(1, sizeof(P11_Session));
		if (!st.sessions) {
            rv = CKR_HOST_MEMORY;
			goto finish;    
		}

		st.sessions->check = st.sessions;

    }

	*psession = st.sessions;

    /*
     * Find an unused session handle
     */
    do {
        st.last_session_handle++;
    } while( (st.last_session_handle == CK_INVALID_HANDLE) ||
             session_LookupSession(st.last_session_handle)    );
    *phandle = st.last_session_handle;
    (*psession)->handle = st.last_session_handle;

    /*
     * Put this session in the session hashtable
     */
    idx = (*psession)->handle % NUM_SESSION_HASH_SLOTS;
    (*psession)->hnext = st.session_hash[idx];
    st.session_hash[idx] = *psession;

finish:
    return rv;
}

/******************************************************************************
** Function: session_FreeSession
**
** Deletes/removes a session
**
** Parameters:
**  session - Session
**
** Returns:
**  CKR_SESSION_HANDLE_INVALID if session handle is invalid
**  CKR_OK
*******************************************************************************/
CK_RV session_FreeSession(P11_Session *session)
{
    CK_RV rv = CKR_OK;
	unsigned int idx;


      log_Log(LOG_LOW, "Removing session: %lX", session->handle);

	  /*
	  * take the session out of the session hashtable
      */
    idx = session->handle % NUM_SESSION_HASH_SLOTS;
    if( st.session_hash[idx] == session ) {
        /* first session in the list */
        st.session_hash[idx] = session->hnext;
    } else {
        P11_Session *prev = st.session_hash[idx];
        while( prev->hnext != session ) {
            prev = prev->hnext;
        }
        prev->hnext = prev->hnext->hnext;
    }

    /*
     * Take the session out of the session list
     */

    if (session->prev) /* Fixme: check for head of list? st.sessions */
    {
        session->prev->next = session->next;
        if (session == st.sessions) /* Fixme: Is this needed? */
            st.sessions = session->prev;
    }

    if (session->next)
    {
        session->next->prev = session->prev;

        if (session == st.sessions)
            st.sessions = session->next;
    }

    if (!session->prev && !session->next)
        st.sessions = 0x00;

    if (session->search_attrib)
        free(session->search_attrib);

        
    /* Clear memory, just to be safe */
    memset(session, 0x00, sizeof(P11_Session));
    
    free(session);

    return rv;
}

