/*
** Copyright (C) 1998 Steve Borho <steve@borho.myip.org>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "mount.h"

#include "XPM/mount.xpm"
#include "XPM/tile.xpm"
#include "XPM/button.xpm"

#define NAME "mount"
#define CLASS "mountapp"

/* How many tics (50ms) with no button events before starting to cycle */
#define STARTUP_TICS 100
/* Once started, how many tics between flips? */
#define CYCLE_TICS   50

/* Defaults. Altered by command line options. */
char            bckcol[STR] = BACKCOLOR;
char            txtdpy[STR] = "";

/* GLOBAL X Resources */
Atom            _XA_GNUSTEP_WM_FUNC;
Atom            deleteWin;

Display        *dpy;
Window          Win[2];
Window          Root;
XWMHints       *hints;
GC              WinGC;
int             activeWin;

Pixmap          back;
Pixmap          tile;
Pixmap          disp;
Pixmap          mask;
Pixmap          buttons;

Font            gcf1, gcf2;
unsigned long   gcc1, gcc2;
int             txtx1, txty1, txtx2 = 8, txty2 = 40, txty3 = 0;

bool            withdrawn = TRUE;
bool            sigHUP = FALSE;
struct allInfo  *aiPtr = 0;

int             main(int argc, char *argv[])
{
    int             i, mountTics = 0;
    XEvent          event;
    bool            finished = FALSE;
    int             idleTics = 0;

    aiPtr = (struct allInfo *)malloc(sizeof(allInfo));
    if (aiPtr == NULL)
    {
        fprintf(stderr, "Unable to allocate memory\n");
        exit(-1);
    }

    bzero((void *)aiPtr, sizeof(allInfo));
    aiPtr->newbieMessages = TRUE;

    scanArgs(argc, argv);
    initXWin(argc, argv);

    if (getuid() == 0)
        userOnly = FALSE;
    else
        userOnly = TRUE;

    if (userOnly && firstTimeUser)
        printf("Initially ignoring all non-user mountable devices\n");

    parseFstab(aiPtr);

    readProperties(aiPtr);

    if (firstTimeUser)
        showHelpPage();

    /* Any mount points to monitor? */
    for (i = 0; i < aiPtr->numMounts; i++)
        if (aiPtr->mpi[i]->ignored == FALSE)
        {
            aiPtr->curMount = i;
            break;
        }
    if (i >= aiPtr->numMounts)
    {
        fprintf(stderr,
"mount.app defaults to not showing non-user mountable devices.\n"
"Currently, all mount points are marked as 'Do not show'. Which\n"
"probably means that none of your devices are user mountable.\n"
"I'm 'un-ignoring' %s temporarily just so I have something to show\n",
                aiPtr->mpi[aiPtr->numMounts-1]->name);
        aiPtr->mpi[aiPtr->numMounts-1]->ignored = FALSE;
        aiPtr->curMount = aiPtr->numMounts - 1;
    }
    /*
     * If initialMnt is defined, then try to override the initial
     * curMount
     */
    if (*initialMnt)
        for (i = 0; i < aiPtr->numMounts; i++)
            if ((aiPtr->mpi[i]->ignored == FALSE) &&
                (strcmp(aiPtr->mpi[i]->mountStr, initialMnt) == 0))
            {
                aiPtr->curMount = i;
                break;
            }

    /* Assign default icons if they haven't been overwritten */
    for (i = 0; i < MAX_ICONS; i++)
        if (aiPtr->icoIsLoaded[i] == 0)
            setIcon(i, iconDefaultName[i], aiPtr);

    /* Set up signal handlers */
    signal(SIGCHLD, noZombie);
    signal(SIGHUP, sigHUPHandler);

    initializeAppIcon();

    checkMount(aiPtr, TRUE);

    XSelectInput(dpy, Win[activeWin],
        ExposureMask | ButtonPressMask | ButtonReleaseMask);
    XMapWindow(dpy, Win[0]);

    /* Main Loop */
    while (!finished)
    {
        /* Were we signalled by an external app to reread configs? */
        if (sigHUP)
        {
            sigHUP = FALSE;

            for (i = 0; i<aiPtr->numMounts; i++)
                free(aiPtr->mpi[i]);

            bzero((void *)aiPtr->mpi, sizeof(aiPtr->mpi));
            aiPtr->numMounts = 0;
            aiPtr->btnState = 0;
            aiPtr->btnTimer = 0;
            aiPtr->dblTimer = 0;
            aiPtr->errorOccured = 0;

            parseFstab(aiPtr);
            readProperties(aiPtr);

            /* Any mount points to monitor? */
            for (i = 0; i < aiPtr->numMounts; i++)
                if (aiPtr->mpi[i]->ignored == FALSE)
                {
                    aiPtr->curMount = i;
                    break;
                }
            if (i >= aiPtr->numMounts)
            {
                fprintf(stderr, "No mount points available to monitor\n");
                exit(0);
            }

            /*
             * If initialMnt is defined, then try to override curMount
             */
            if (*initialMnt)
                for (i = 0; i < aiPtr->numMounts; i++)
                    if ((aiPtr->mpi[i]->ignored == FALSE) &&
                            (strcmp(aiPtr->mpi[i]->mountStr, initialMnt) == 0))
                    {
                        aiPtr->curMount = i;
                        break;
                    }

            idleTics = 0;
            checkMount(aiPtr, TRUE);
        }

        /* Collect the remains of any children that have died */
        shovelDeadChildren();

        /* Did any of our children leave us a nasty note? */
        if (aiPtr->errorOccured)
        {
            aiPtr->errorOccured = FALSE;
           
            for (i = 0; i<aiPtr->numMounts; i++)
            {
                if (aiPtr->mpi[i]->childStatus)
                {
                    handleReturnCode(aiPtr->mpi[i]);
                    aiPtr->mpi[i]->childStatus = 0;
                    if (i == aiPtr->curMount)
                    {
                        drawBtns(BTN_MOUNT, aiPtr);
                        update(aiPtr);
                        repaint();
                    }
                }
            }
        }

        while (XPending(dpy))
        {
            XNextEvent(dpy, &event);
            switch (event.type)
            {
            case Expose:
                repaint();
                break;
            case ButtonPress:
                idleTics = 0;
                pressEvent(&event.xbutton, aiPtr);
                break;
            case ButtonRelease:
                idleTics = 0;
                releaseEvent(&event.xbutton, aiPtr);
                break;
            case ClientMessage:
                if (event.xclient.data.l[0] == deleteWin)
                    finished = TRUE;
                break;
            }
        }

        /* Double-click timeout */
        if (aiPtr->dblTimer > 0)
            aiPtr->dblTimer--;

        /* Repeat next/prev */
        if (aiPtr->btnState & (BTN_PREV | BTN_NEXT))
        {
            aiPtr->btnTimer++;
            if (aiPtr->btnTimer >= REPEAT_TIMER)
            {
                do
                {
                    if (aiPtr->btnState & BTN_NEXT)
                    {
                        aiPtr->curMount++;
                        if (aiPtr->curMount >= aiPtr->numMounts)
                            aiPtr->curMount = 0;
                    }
                    else
                    {
                        aiPtr->curMount--;
                        if (aiPtr->curMount < 0)
                            aiPtr->curMount = aiPtr->numMounts - 1;
                    }
                }
                while (aiPtr->mpi[aiPtr->curMount]->ignored);

                /* forced because of changed mount point */
                checkMount(aiPtr, TRUE);

                aiPtr->btnTimer = 0;
            }
        }
        else if (--mountTics <= 0 || aiPtr->currentMountChanged)
        {
            /* Check mounts every few seconds, and update as needed */
            mountTics = CHK_MNT_TIMER;
            checkMount(aiPtr, aiPtr->currentMountChanged);
            aiPtr->currentMountChanged = FALSE;
        }
        else if (idleTics++ > STARTUP_TICS && aiPtr->idleCycle &&
            (idleTics % CYCLE_TICS == 0))
        {
            do
            {
                aiPtr->curMount++;
                if (aiPtr->curMount >= aiPtr->numMounts)
                    aiPtr->curMount = 0;
            }
            while (aiPtr->mpi[aiPtr->curMount]->ignored);

            /* forced because of changed mount point */
            checkMount(aiPtr, TRUE);
        }

        XFlush(dpy);
        usleep(50000L);
    }

    /* Cleanup */
    while (aiPtr->numMounts > 0)
    {
        free(aiPtr->mpi[aiPtr->numMounts - 1]);
        aiPtr->numMounts--;
    }
    while (aiPtr->numIcons > 0)
    {
        XFreePixmap(dpy, aiPtr->icons[aiPtr->numIcons - 1]);
        aiPtr->numIcons--;
    }
    XFreeGC(dpy, WinGC);
    XUnloadFont(dpy, gcf1);
    XUnloadFont(dpy, gcf2);
    XFreePixmap(dpy, disp);
    XFreePixmap(dpy, mask);
    XFreePixmap(dpy, tile);
    XFreePixmap(dpy, back);
    XFreePixmap(dpy, buttons);
    freeXWin();
    return 0;
}

void            initXWin(int argc, char *argv[])
{
    XWMHints        hints;
    XSizeHints      shints;

    if ((dpy = XOpenDisplay(txtdpy)) == NULL)
    {
        fprintf(stderr, "Unable to connect to X display: %s\n", txtdpy);
        exit(1);
    }
    _XA_GNUSTEP_WM_FUNC = XInternAtom(dpy, "_GNUSTEP_WM_FUNCTION", False);
    deleteWin = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    Root = DefaultRootWindow(dpy);
    createWin(&Win[0]);
    createWin(&Win[1]);

    hints.window_group = Win[0];
    shints.min_width = 64;
    shints.min_height = 64;
    shints.max_width = 64;
    shints.max_height = 64;
    shints.x = 0;
    shints.y = 0;
    hints.initial_state = WithdrawnState;
    hints.icon_window = Win[1];
    hints.flags = WindowGroupHint | StateHint | IconWindowHint;
    shints.flags = PMinSize | PMaxSize | PPosition;

    if (withdrawn)
    {
        shints.flags |= USPosition;
        hints.initial_state = WithdrawnState;
        hints.flags = WindowGroupHint | StateHint | IconWindowHint;
        activeWin = 1;
    }
    else
    {
        hints.initial_state = NormalState;
        hints.flags = WindowGroupHint | StateHint;
        activeWin = 0;
    }
    XSetWMHints(dpy, Win[0], &hints);
    XSetWMNormalHints(dpy, Win[0], &shints);
    XSetCommand(dpy, Win[0], argv, argc);
    XStoreName(dpy, Win[0], NAME);
    XSetIconName(dpy, Win[0], NAME);
    XSetWMProtocols(dpy, Win[activeWin], &deleteWin, 1);
}

void            initializeAppIcon()
{
    XGCValues       gcv;
    unsigned long   gcm;
    XpmAttributes   pixatt;
    XFontStruct    *xfs;
    XpmColorSymbol  ledcols[1] =
    {
        {"back_color", NULL, 0}
    };

    gcm = GCForeground | GCBackground | GCGraphicsExposures;
    gcv.graphics_exposures = False;
    WinGC = XCreateGC(dpy, Root, gcm, &gcv);

    ledcols[0].pixel = getColor(bckcol, 1.00);
    pixatt.numsymbols = 1;
    pixatt.colorsymbols = ledcols;
    pixatt.exactColors = FALSE;
    pixatt.closeness = 40000;
    pixatt.valuemask = XpmColorSymbols | XpmExactColors | XpmCloseness;
    XpmCreatePixmapFromData(dpy, Root, mount_xpm, &back, &mask, &pixatt);
    XpmCreatePixmapFromData(dpy, Root, tile_xpm, &tile, NULL, &pixatt);
    XpmCreatePixmapFromData(dpy, Root, button_xpm, &buttons, NULL, &pixatt);
    disp = XCreatePixmap(dpy, Root, 64, 64, DefaultDepth(dpy, DefaultScreen(dpy)));

    /* Install mask or copy background tile */
    if (withdrawn)
        XShapeCombineMask(dpy, Win[activeWin], ShapeBounding, 0, 0, mask,
                ShapeSet);
    else
        XCopyArea(dpy, tile, disp, WinGC, 0, 0, 64, 64, 0, 0);


    /* Copy background */
    XSetClipMask(dpy, WinGC, mask);
    XCopyArea(dpy, back, disp, WinGC, 0, 0, 64, 64, 0, 0);
    XSetClipMask(dpy, WinGC, None);

    xfs = XLoadQueryFont(dpy, txtfont);
    if (xfs == 0)
    {
        fprintf(stderr, "Unable to load text font\n");
        exit(1);
    }
    gcf1 = xfs->fid;
    gcc1 = getColor(txtcolor, 1);
    if (gcc1 == 0)
    {
        fprintf(stderr, "Unable to load text color\n");
        exit(1);
    }
    txtx1 = 6 - xfs->min_bounds.lbearing;
    txty1 = 5 + xfs->ascent;
    txty2 = txty1 + xfs->descent;
   
    xfs = XLoadQueryFont(dpy, capfont);
    if (xfs == 0)
    {
        fprintf(stderr, "Unable to load cap font\n");
        exit(1);
    }
    gcf2 = xfs->fid;
    gcc2 = getColor(capcolor, 1);
    if (gcc1 == 0)
    {
        fprintf(stderr, "Unable to load cap color\n");
        exit(1);
    }
    txtx2 = 6 - xfs->min_bounds.lbearing;
    txty2 = txty2 + xfs->ascent;
    txty3 = txty2 + xfs->descent + xfs->ascent;
}

void            freeXWin()
{
    XDestroyWindow(dpy, Win[0]);
    XDestroyWindow(dpy, Win[1]);
    XCloseDisplay(dpy);
}

void            createWin(Window * win)
{
    XClassHint      classHint;

    *win = XCreateSimpleWindow(dpy, Root, 10, 10, 64, 64, 0, 0, 0);
    classHint.res_name = NAME;
    classHint.res_class = CLASS;
    XSetClassHint(dpy, *win, &classHint);
}

unsigned long   getColor(char *colorName, float dim)
{
    XColor          Color;
    XWindowAttributes Attributes;

    XGetWindowAttributes(dpy, Root, &Attributes);
    Color.pixel = 0;

    XParseColor(dpy, Attributes.colormap, colorName, &Color);
    Color.red = (unsigned short)(Color.red / dim);
    Color.blue = (unsigned short)(Color.blue / dim);
    Color.green = (unsigned short)(Color.green / dim);
    Color.flags = DoRed | DoGreen | DoBlue;
    XAllocColor(dpy, Attributes.colormap, &Color);

    return Color.pixel;
}

void            setIcon(int num, char *filename, allInfo * ai)
{
    char            pathBuf[256];
    XpmAttributes   pixatt;

    sprintf(pathBuf, "%s/%s", icondir, filename);
    pixatt.exactColors = FALSE;
    pixatt.closeness = 40000;
    pixatt.valuemask = XpmExactColors | XpmCloseness;

    safeCopy(ai->iconFileName[num], pathBuf, STR);

    if (access(pathBuf, R_OK) == 0)
    {
        if (XpmReadFileToPixmap(dpy, Root, pathBuf, &ai->icons[num],
                NULL, &pixatt) == 0)
            ai->icoIsLoaded[num] = TRUE;
    }
    else
    {
        fprintf(stderr, "Icon not found: %s\n", pathBuf);
        exit(-1);
    }
}

void            printHelp(int argc, char *argv[])
{
    fprintf(stderr, "mount.app %s - the WindowMaker universal mount point\n",
            VERSION);
    fprintf(stderr, "Copyright (C) 1999  Steve Borho <steve@borho.myip.org>\n");
    fprintf(stderr, "This software comes with ABSOLUTELY NO WARRANTY\n");
    fprintf(stderr, "This software is free software, and you are welcome "
        "to redistribute it\n");
    fprintf(stderr, "under certain conditions\n");
    fprintf(stderr, "See the README file for a more complete notice.\n\n");
    fprintf(stderr, "usage:\n\n   %s [options] [-display <disp>]\n", argv[0]);
    fprintf(stderr, "\noptions:\n\n");
    fprintf(stderr, "   -h | -help             display this help screen\n");
    fprintf(stderr, "   -i                     ignore NFS mounts\n");
    fprintf(stderr, "   -n                     wm other than Window Maker\n");
    fprintf(stderr, "   -display display       select target display"
        " (see X manual pages)\n\n");
    exit(0);
}

void            scanArgs(int argc, char *argv[])
{
    int             optchar;
    int             disp_set = 0;

    while ((optchar = getopt(argc, argv, "hind:")) != EOF)
    {
        disp_set = 0;
        switch (optchar)
        {
        case 'n':
            withdrawn = FALSE;
            break;
        case 'i':
            ignoreNFS = TRUE;
            break;
        case 'd':
            if (!strcmp(optarg, "isplay"))
                disp_set++;
            break;
        case 'h':
        default:
            printHelp(argc, argv);
            break;
        }
    }

    /* if the last argument was -display, this is the server name */
    if (disp_set && (optind < argc))
        safeCopy(txtdpy, argv[optind++], sizeof(txtdpy) - 1);
}

void            checkMount(allInfo * ai, bool forced)
{
    int             nowMounted = FALSE;
    int             newCapacity = 0;

    dev_t           prvdev;
    dev_t           dirdev;

    struct MPInfo  *mp = ai->mpi[ai->curMount];

    /* I think I can assume that / is mounted. */
    if (strcmp(mp->mountStr, "/") == 0)
        nowMounted = TRUE;
    else
    {
        /* Check if the directory above the mount is on a different device. */
        struct stat     st;
        char           *mpstr, *parent_ptr;

        mpstr = strdup(mp->mountStr);

        if (!mpstr)
            return;

        if (stat(mpstr, &st) == -1)
        {
            /*
             * The mount point directory most likely doesn't exist.
             * What the hell do I do now?  For now, I'll fake some
             * data.  Maybe we should color code bad mount points in
             * the future.
             */
            mp->totalSize = (double)100;
            mp->availSize = (double)0;
            mp->freeSize = (double)0;
            mp->currentCap = 0;
            update(ai);
            return;
        }
        dirdev = st.st_dev;
        parent_ptr = strrchr(mpstr, '/');
        if (parent_ptr && parent_ptr != mpstr)
            *parent_ptr = 0;
        else
            strcpy(mpstr, "/");
        if (stat(mpstr, &st) == -1)
        {
            perror("Stat of parent dir failed");
            return;
        }
        prvdev = st.st_dev;
        if (prvdev != dirdev)
            nowMounted = TRUE;
        free(mpstr);
    }

    /* If we dont want it, don't waste time getting it. */
    if (nowMounted && (mp->showCap != CAPACITY_NONE))
    {
        struct statfs   sfs;

        statfs(mp->mountStr, &sfs);
        newCapacity = (int)((sfs.f_blocks - sfs.f_bfree) * 100.0 /
            (sfs.f_blocks - sfs.f_bfree + sfs.f_bavail) + .5);

        /* Save these statistics for later when showing capacity */
        mp->totalSize = (double)(sfs.f_blocks) * (sfs.f_bsize);
        mp->availSize = (double)(sfs.f_bavail) * (sfs.f_bsize);
        mp->freeSize = (double)(sfs.f_bfree) * (sfs.f_bsize);
    }

    if (nowMounted)
        ai->btnState |= BTN_MOUNT;
    else
        ai->btnState &= ~BTN_MOUNT;

    if ((nowMounted != mp->isMounted) ||
        (nowMounted && newCapacity != mp->currentCap &&
            (mp->showCap != CAPACITY_NONE)) || forced)
    {
        if (nowMounted != mp->isMounted || forced)
        {
            mp->isMounted = nowMounted;
            drawBtns(BTN_MOUNT, ai);
            if ((mp->showCap != CAPACITY_NONE) || forced)
            {
                mp->currentCap = newCapacity;
            }
            update(ai);
        }
        else
        {
            if (nowMounted && newCapacity != mp->currentCap &&
                (mp->showCap != CAPACITY_NONE))
            {
                mp->currentCap = newCapacity;
                update(ai);
            }
        }
        repaint();
    }
}

void            pressEvent(XButtonEvent * xev, allInfo * ai)
{
    int             x = xev->x;
    int             y = xev->y;
    struct MPInfo  *mp;

    if (x >= 46 && y >= 47 && x <= 58 && y <= 57)
    {
        do
        {
            ai->curMount++;
            if (ai->curMount >= ai->numMounts)
                ai->curMount = 0;
        }
        while (ai->mpi[ai->curMount]->ignored);

        ai->btnState |= BTN_NEXT;
        ai->btnTimer = 0;
        drawBtns(BTN_NEXT, ai);
        /* forced because of changed mount point */
        checkMount(ai, TRUE);
        return;
    }
    if (x >= 33 && y >= 47 && x <= 45 && y <= 57)
    {
        do
        {
            ai->curMount--;
            if (ai->curMount < 0)
                ai->curMount = ai->numMounts - 1;
        }
        while (ai->mpi[ai->curMount]->ignored);

        ai->btnState |= BTN_PREV;
        ai->btnTimer = 0;
        drawBtns(BTN_PREV, ai);
        /* forced because of changed mount point */
        checkMount(ai, TRUE);
        return;
    }

    mp = ai->mpi[ai->curMount];
    if (x >= 5 && y >= 47 && x <= 32 && y <= 57)
    {
        if (mountcmd[0] == 0)
        {
            if (mp->isMounted)
            {
                mp->performEject = mp->ejectFlag && (xev->button == Button1);
                internalUmount(mp);
            }
            else
                internalMount(mp);
        }
        else
        {
            char            cmd[256];
            int             n = (sizeof(cmd) / sizeof(cmd[0]));

            if (mp->isMounted)
            {
                mp->performEject = mp->ejectFlag && (xev->button == Button1);
                snprintf(cmd, n, umountcmd, mp->mountStr);
                cmd[n - 1] = '\0';
                newSystem(cmd, mp);
            }
            else
            {
                snprintf(cmd, n, mountcmd, mp->mountStr);
                cmd[n - 1] = '\0';
                newSystem(cmd, mp);
            }
        }
        drawBtns(BTN_MOUNT, aiPtr);
    }
    else if (x >= 25 && y >= 30 && x <= 59 && y <= 42)
    {
        if (mp->isMounted)
        {
            /* First or second click of double-click? */
            if (ai->dblTimer > 0)
            {
                char            cmd[256];
                int             n = (sizeof(cmd) / sizeof(cmd[0]));

                snprintf(cmd, n, opencmd, mp->mountStr);
                cmd[n - 1] = '\0';
                newSystem(cmd, mp);
                ai->dblTimer = 0;
            }
            else
                ai->dblTimer = DBL_CLICK_TIMER;
        }
    }
    else if (ai->dblTimer > 0)
    {
        callConfigApp(ai);
        ai->dblTimer = 0;
    }
    else
        ai->dblTimer = DBL_CLICK_TIMER;
}

void            releaseEvent(XButtonEvent * xev, allInfo * ai)
{
    ai->btnState &= ~(BTN_PREV | BTN_NEXT);
    drawBtns(BTN_PREV | BTN_NEXT, ai);
    repaint();
}

void            repaint()
{
    XEvent          xev;

    XCopyArea(dpy, disp, Win[activeWin], WinGC, 0, 0, 64, 64, 0, 0);
    while (XCheckTypedEvent(dpy, Expose, &xev)) ;
}

/* Helper function for update()
 * Turns a size (in byte) into something human-readable */
void		format_size(char *s, double size_in_byte)
{
   const char *unit;
   double size = (double) size_in_byte;
   
   if (size >= 1024 * 1024 * 1024)
     {
	size /= (1024 * 1024 * 1024);
	unit = "GB";
     }
   else if (size >= 1024 * 1024)
     {
	size /= (1024 * 1024);
	unit = "MB";
     }
   else if (size >= 1024)
     {
	size /= 1024;
	unit = "kB";
     }
   else
     {
	unit = "Byte";
     }  
   
   if ((size < 10) && (unit != "Byte"))
     {
	sprintf (s, "%.1f%s", size, unit);
     }
   else
     {
	sprintf (s, "%.0f%s", size, unit);
     }
}

/* Needs to be called if the current mountpoint or capacity change */
void            update(allInfo * ai)
{
    struct MPInfo  *mp;

    mp = ai->mpi[ai->curMount];

    /* Copy part of background */
    XCopyArea(dpy, back, disp, WinGC, 4, 4, 56, 39, 4, 4);

    /* Copy icon */
    if (ai->icoIsLoaded[mp->icon])
        XCopyArea(dpy, ai->icons[mp->icon], disp, WinGC,
            0, 0, 32, 10, 25, 30);

    if (mp->isMounted && (mp->showCap != CAPACITY_NONE))
    {
        /* Draw capacity text */
        char            capstr[20];
        char            sizestr[18];

        switch (mp->showCap)
        {
        case CAPACITY_PFULL:
            sprintf(capstr, "%d%%", mp->currentCap);
            break;
        case CAPACITY_PEMPTY:
            sprintf(capstr, "%d%%", 100 - mp->currentCap);
            break;
        case CAPACITY_MFULL:
	    format_size(sizestr, (mp->totalSize - mp->freeSize));
            if (ai->descriptChar)
                sprintf(capstr, "> %s", sizestr);
            else
                sprintf(capstr, "%s", sizestr);
            break;
        case CAPACITY_MEMPTY:
	    format_size(sizestr, (mp->availSize));
	    if (ai->descriptChar)
                sprintf(capstr, "< %s", sizestr);
            else
                sprintf(capstr, "%s", sizestr);
            break;
        }

        XSetFont(dpy, WinGC, gcf2);
        XSetForeground(dpy, WinGC, gcc2);
        XDrawString(dpy, disp, WinGC, txtx2, txty2, (char *)capstr,
            strlen(capstr));
    }

    /* Draw mountpoint name text */
    XSetFont(dpy, WinGC, gcf1);
    XSetForeground(dpy, WinGC, gcc1);
    XDrawString(dpy, disp, WinGC, txtx1, txty1, (char *)mp->name,
        strlen(mp->name));
   
    if (ai->showErrorChars)
        XDrawString(dpy, disp, WinGC, txtx1, txty3, &mp->childStatusChar, 1);
}

void            drawBtns(int btns, allInfo * ai)
{
    if (btns & BTN_PREV)
    {
        XCopyArea(dpy, buttons, disp, WinGC, 30, (ai->btnState & BTN_PREV) ? 11 : 0, 13, 11, 33, 47);
    }
    if (btns & BTN_NEXT)
    {
        XCopyArea(dpy, buttons, disp, WinGC, 43, (ai->btnState & BTN_NEXT) ? 11 : 0, 13, 11, 46, 47);
    }
    if (btns & BTN_MOUNT)
    {
        if (ai->mpi[ai->curMount]->childPid)
        {
            /* Working: ORANGE */
            XCopyArea(dpy, buttons, disp, WinGC, 0, 11, 28, 11, 5, 47);
        }
        else if (ai->mpi[ai->curMount]->isMounted)
        {
            /* Mounted: GREEN */
            XCopyArea(dpy, buttons, disp, WinGC, 0, 22, 28, 11, 5, 47); 
        }
        else
        {
            /* Unmounted: BLACK*/
            XCopyArea(dpy, buttons, disp, WinGC, 0, 0, 28, 11, 5, 47);
        }
    }
}
