/* 
 *   Creation Date: <2002/10/02 22:24:24 samuel>
 *   Time-stamp: <2004/03/21 21:39:29 samuel>
 *   
 *	<main.c>
 *	
 *	
 *   
 *   Copyright (C) 2002, 2004 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "of.h"
#include "ofmem.h"
#include "fs.h"
#include "prom.h"
#include "elfload.h"
#include "ablk_sh.h"
#include "osi_calls.h"
#include "boothelper_sh.h"
#include "pseudofs_sh.h"
#include "molversion.h"

static void	patch_newworld_rom( char *start, size_t size );
static void	newworld_timer_hack( char *start, size_t size );


static void
transfer_control_to_elf( ulong entry )
{
	extern void call_elf( ulong entry );
	printm("Starting ELF boot loader\n");
	call_elf( entry );

	fatal_error("call_elf returned unexpectedly\n");
}

static int
load_elf_rom( ulong *entry, fs_ops_t *fs, file_desc_t fd )
{
	int i, lszz_offs, elf_offs;
	char buf[128], buf2[128], *addr;
	Elf32_Ehdr ehdr;
	Elf32_Phdr *phdr;
	size_t s;

	fs->get_path( fd, buf, sizeof(buf) );
	printm("Loading '%s' from '%s'\n", buf, fs_vol_name(fs,buf2,sizeof(buf2)) );

	/* the ELF-image (usually) starts at offset 0x4000 */
	if( (elf_offs=find_elf32(fs, fd)) < 0 ) {
		printm("----> %s is not an ELF image\n", buf );
		exit(1);
	}
	if( !(phdr = elf32_readhdrs(fs, fd, elf_offs, &ehdr)) )
		fatal_error("elf32_readhdrs failed\n");

	*entry = ehdr.e_entry;
	
	/* load segments. Compressed ROM-image assumed to be located immediately after the last segment */
	lszz_offs = elf_offs;
	for( i=0; i<ehdr.e_phnum; i++ ) {
		/* p_memsz, p_flags */
		s = MIN( phdr[i].p_filesz, phdr[i].p_memsz );
		fs->lseek( fd, elf_offs + phdr[i].p_offset, SEEK_SET );

		/* printm("filesz: %08lX memsz: %08lX p_offset: %08lX p_vaddr %08lX\n", 
		   phdr[i].p_filesz, phdr[i].p_memsz, phdr[i].p_offset, phdr[i].p_vaddr ); */

		if( phdr[i].p_vaddr != phdr[i].p_paddr )
			printm("WARNING: ELF segment virtual addr != physical addr\n");
		lszz_offs = MAX( lszz_offs, elf_offs + phdr[i].p_offset + phdr[i].p_filesz );
		if( !s )
			continue;
		if( ofmem_claim( phdr[i].p_vaddr, phdr[i].p_memsz, 0 ) == -1 )
			fatal_error("Claim failed!\n");

		addr = (char*)phdr[i].p_vaddr;
		if( fs->read( fd, addr, s) != s )
			fatal_error("read failed\n");

		/* patch CODE segment */
		if( *entry >= phdr[i].p_vaddr && *entry < phdr[i].p_vaddr + s ) {
			patch_newworld_rom( (char*)phdr[i].p_vaddr, s );
			newworld_timer_hack( (char*)phdr[i].p_vaddr, s );
		}
		flush_icache_range( addr, addr+s );

		/* printm("ELF ROM-section loaded at %08lX (size %08lX)\n", 
		   (ulong)phdr[i].p_vaddr, (ulong)phdr[i].p_memsz );*/
	}
	free( phdr );
	return lszz_offs;
}


/************************************************************************/
/*	newworld ROM loading						*/
/************************************************************************/

#define ROM_BASE	0x1100000		/* where we decide to put things */

/* fix bug present in the 2.4 and the 3.0 Apple ROM */
static void
patch_newworld_rom( char *start, size_t size )
{
	int s;
	ulong mark[] = { 0x7c7d1b78, 		/* mr r29,r3 */
			 0x7c9c2378,		/* mr r28,r4 */
			 0x7cc33378,		/* mr r3,r6 */
			 0x7c864214,		/* add r4,r6,r8   <------ BUG -- */
			 0x80b10000,		/* lwz r5,0(r17) */
			 0x38a500e8 };		/* addi r5,r5,232 */

	/* Correcting add r4,r6,r8  ---->  addi r4,r6,8 */
	for( s=0; s<size-sizeof(mark); s+=4 )
		if( memcmp( start+s, mark, sizeof(mark)) == 0 ) {
			printm("FIXING ROM BUG @ %X!\n", s+12);
			((ulong*)(start+s))[3] = 0x38860008;	/* addi r4,r6,8 */
		}
}

/* This hack is only needed on machines with a timebase slower than 12.5 MHz
 * (50 MHz bus frequency). Typically only old, accelerated machines fall
 * into this category. The cause of the problem is an overflow in Apple's
 * calibration routine.
 */
static void
newworld_timer_hack( char *start, size_t size )
{
	int s;
	ulong mark[] = { 0x7d0000a6, 0x5507045e, 0x7ce00124, 0x4c00012c,
			 0x38e00000, 0x3c80000f, 0x6084ffff, 0x98830c00,
			 0x7c0006ac, 0x98830a00, 0x7c0006ac, 0x7c9603a6,
			 0x4c00012c, 0x7cb602a6, 0x2c050000, 0x4181fff8,
			 0x7c0004ac, 0x88830a00, 0x7c0006ac, 0x88a30800,
			 0x7c0006ac, 0x88c30a00, 0x7c0006ac, 0x7c043040,
			 0x40a2ffe4, 0x5085442e, 0x7ca500d0, 0x54a5043e,
			 0x7c053840, 0x7ca72b78, 0x4082ff9c, 0x7ca32b78,
			 0x7d000124, 0x4c00012c, 0x4e800020
	};

	/* return #via ticks corresponding to 0xfffff DEC ticks (VIA frequency == 47/60 MHz) */
	for( s=0; s < size-sizeof(mark); s+=4 ) {
		if( !memcmp( start+s, mark, sizeof(mark)) ) {
			extern char timer_calib_start[], timer_calib_end[];
			extern ulong nw_dec_calibration;
			int hz = OSI_UsecsToMticks(1000);
			nw_dec_calibration = OSI_MticksToUsecs(0xfffff*47)/60;
			memcpy( start + s, timer_calib_start, timer_calib_end - timer_calib_start );

			printm("Timer calibration fix: %d.%02d MHz [%ld]\n", 
			       hz/1000, (hz/10)%100, nw_dec_calibration );
			break;
		}
	}
}

static ulong
load_newworld_rom( fs_ops_t *fs, file_desc_t fd )
{
	mol_phandle_t ph;
	int lszz_offs, lszz_size;
	ulong entry, data[2];
	
	lszz_offs = load_elf_rom( &entry, fs, fd );
	lszz_size = fs->lseek( fd, 0, SEEK_END ) - lszz_offs;
	fs->lseek( fd, lszz_offs, SEEK_SET );

	/* printm("Compressed ROM image: offset %08X, size %08X loaded at %08x\n", 
	   lszz_offs, lszz_size, ROM_BASE ); */

	if( ofmem_claim( ROM_BASE, lszz_size, 0 ) == -1 )
		fatal_error("Claim failure (lszz)!\n");
	
	fs->read( fd, (char*)ROM_BASE, lszz_size );
	
	/* Fix the /rom/macos/AAPL,toolbox-image,lzss property (phys, size) */
	if( (ph=prom_create_node( "/rom/macos/" )) == -1 )
		fatal_error("Failed creating /rom/macos/");

	data[0] = ROM_BASE;
	data[1] = lszz_size;
	prom_set_prop( ph, "AAPL,toolbox-image,lzss", (char*)data, sizeof(data) );

	/* The 7.8 rom (MacOS 9.2) uses AAPL,toolbox-parcels instead of 
	 * AAPL,toolbox-image,lzss. It probably doesn't hurt to have it
	 * always present (we don't have an easy way to determine ROM version...)
	 */
	prom_set_prop( ph, "AAPL,toolbox-parcels", (char*)data, sizeof(data) );
	return entry;
}

static file_desc_t
newworld_disk_bootable( fs_ops_t *fs, int fast )
{
	file_desc_t fd = 0;
	char *s, buf[128];

	if( fast ) {
		int ind;
		for( ind=0; !fd && (s=BootHGetStrResInd("macos_rompath", buf, sizeof(buf), ind++, 0)) ; )
			fd = fs_open_path( fs, s );
		for( ind=0; !fd && (s=BootHGetStrResInd("macos_rompath_", buf, sizeof(buf), ind++, 0)) ; )
			fd = fs_open_path( fs, s );
	} else {
		printm("Searching %s for a 'Mac OS ROM' file\n", fs_vol_name( fs, buf, sizeof(buf)) );
		if( (fd=fs_search_rom(fs)) ) {
			printm(" \n**** HINT ***************************************************\n");
			printm("*  The booting can be speeded up by adding the line\n");
			printm("*      macos_rompath: '%s'\n", fs->get_path( fd, buf, sizeof(buf)) );
			printm("*  to the /etc/mol/molrc.macos (recommended).\n");
			printm("*************************************************************\n \n");
		}
	}
	return fd;
}


static char *
newworld_load( const char *path, const char *spec, int do_search )
{
	fs_ops_t *fs;
	file_desc_t fd;
	int hfsplus=0;
	char *entry, buf[80];
	
	if( (fs=fs_open( FS_HFS_PLUS, spec )) )
		hfsplus=1;
	else if( !(fs=fs_open( FS_HFS, spec )) )
		return NULL;
				
	/* try predefined paths first */
	if( !(fd=newworld_disk_bootable(fs, do_search)) ) {
		fs_close(fs);
		return NULL;
	}
	printm("Boot Disk: %s [HFS%s]\n", spec, hfsplus ? "+":"" );

	entry = (char*)load_newworld_rom( fs, fd );
	fs->close( fd );
	fs_close( fs );

#if 0
	strcat( buf, ",\\\\:tbxi" ); 
	prom_set_prop( prom_find_device("/chosen"), "bootpath", buf, strlen(buf)+1 );
#else
	sprintf( buf, "%s/disk@0:0", path );
	prom_set_prop( prom_find_device("/chosen"), "bootpath", buf, strlen(buf)+1 );
#endif
	return entry;
}

static void
newworld_startup( void )
{
	mol_phandle_t ph = prom_find_device("/pci/pci-bridge/mol-blk");
	ablk_disk_info_t info;
	char spec[80], path[80];
	int i, j, type;
	file_desc_t fd;
	fs_ops_t *fs;
	char *entry = NULL;

	if( prom_package_to_path( ph, path, sizeof(path) ) == -1 )
		fatal_error("MOLBlockDriver node not found\n");
	
	/* user specified nwrom takes precedence */
	fs = fs_pseudo_open();
	if( (fd=fs_open_path(fs, "nwrom")) ) {
		entry = (char*)load_newworld_rom( fs, fd );		
		fs->close(fd);
	}
	fs_close(fs);

	/* determine boot volume. It is assumed boot-flagged disks have lower index */
	for( type=0; !entry && type<3 ; type++ ) {
		for( i=0; !entry && !OSI_ABlkDiskInfo(0, i, &info) ; i++ ) {
			if( type<=1 && !(info.flags & ABLK_BOOT_HINT) )
				continue;
			if( type>1 && (info.flags & ABLK_BOOT_HINT) )
				continue;

			for( j=0; !entry && j<32; j++ ) {
				sprintf( spec, "%s/disk@%x:%d", path, i, j );
				entry = newworld_load( path, spec, (!type || type==2) );

				/* tell mol which disk we boot from */
				if( entry )
					OSI_ABlkBlessDisk( 0 /*channel*/, i );
			}
		}
	}

	if( entry ) {
		transfer_control_to_elf( (ulong)entry );
		/* won't come here */
		return;
	}

	printm("\n--- No bootable disk was found! -----------------------------\n");
	printm("If this is an oldworld machine, try booting from the MacOS\n");
	printm("install CD and install MacOS from within MOL.\n");
	printm("-------------------------------------------------------------\n");
	exit(1);
}

/************************************************************************/
/*	yaboot booting							*/
/************************************************************************/

static void
yaboot_startup( void )
{
	fs_ops_t *fs = fs_pseudo_open();
	file_desc_t fd;

	if( (fd=fs_open_path(fs, "ofclient")) || (fd=fs_open_path(fs, "yaboot")) ) {
		ulong entry = load_newworld_rom( fs, fd );		
		fs->close(fd);
		fs_close(fs);
		transfer_control_to_elf( entry );
		/* won't come here */
	}
	printm("*** Boot failure! No secondary bootloader specified ***\n");
	exit(1);
}


/************************************************************************/
/*	entry								*/
/************************************************************************/

/* asm exports */
extern void unexpected_excep( int vector );
extern void entry( void );

void
unexpected_excep( int vector )
{
	printm("MOL panic: Unexpected exception %x\n", vector );
	for( ;; )
		;
}

void
entry( void )
{
	printm("\n");
	printm("=============================================================\n");
	printm("Mac-on-Linux OpenFirmware %s\n", MOL_RELEASE );

	of_startup();
	init_video();

	if( prom_find_device("/mol-platform") != -1 )
		yaboot_startup();
	else
		newworld_startup();
}
