/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: template.c,v 1.25 2004/03/18 02:27:09 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "mconfig.h"

/*#define MTIMER_ENABLED*/

#ifdef MTIMER_ENABLED
#include "misc.h"
#endif

#include "plugin_config.h"
#include "template.h"

/**
 * get a \n terminated line from the stringbuffer
 *
 * a string buffer is a linear string (tmpl_string) with variable str_pos
 * locator which is moved from \n to \n
 *
 * The line is copied to the buffer of the tmpl_string for further processing
 *
 * @param conf
 * @return -1 error (conf == NULL), 0 end of string, 1 new string at
 * @see tmpl_string
 */

int tmpl_get_line_from_string (tmpl_string *conf) {
	const char *s = conf->string + conf->str_pos;
	int i = 0;

	if (!conf) return -1;

	if (!*s) return 0;

	for (i = 0; *(s + i) != '\n' && *(s + i) != '\0'; i++) ;

	if (*(s + i) == '\n') i++;
	
	buffer_copy_string_len(conf->buffer, s, i);

	conf->str_pos += i;

	return 1;
}

/**
 * get a \n terminated line from a file
 *
 * the buffer element of 'conf' contains the extracted new line
 *
 * @param conf
 * @return -1 conf is invalid (NULL), 0 end of file, 1 a new line
 *
 */

int tmpl_get_line_from_file (tmpl_file *conf) {
	int newline = 1;

	if (!conf) return -1;

	if (!fgets(conf->buffer, conf->buf_len-1, conf->f)) {
		newline = 0;
	}

	while (newline && conf->buffer[strlen(conf->buffer)-1] != '\n') {
		conf->buffer = realloc(conf->buffer, (conf->buf_len+conf->buf_inc+1) * sizeof(char));

		if (!fgets(conf->buffer+strlen(conf->buffer), conf->buf_inc-1, conf->f)) {
			newline = 0;
		}

		conf->buf_len += conf->buf_inc;
	}

	return newline;
}

/**
 * insert a key into the variable array
 *
 */

int tmpl_insert_key(tmpl_main *tmpl, const char *s, const char *def_val) {
	int i;

	if (!tmpl) return -1;

	/* init the tmpl_keys structure at the first run */
	if (tmpl->tmpl_keys == NULL) {
		tmpl->tmpl_keys_size = 16;
		tmpl->tmpl_keys_pos = 0;
		tmpl->tmpl_keys = malloc(sizeof(tmpl_key *) * tmpl->tmpl_keys_size);

		for (i = 0; i < tmpl->tmpl_keys_size; i++) {
			tmpl->tmpl_keys[i] = malloc(sizeof(tmpl_key));
			memset(tmpl->tmpl_keys[i], 0, sizeof(tmpl_key));
			
			tmpl->tmpl_keys[i]->value = buffer_init();
		}
	}

	/* increase the array of keys if neccesary */
	if (tmpl->tmpl_keys_size == tmpl->tmpl_keys_pos) {
		tmpl->tmpl_keys_size += 16;
		tmpl->tmpl_keys = realloc(tmpl->tmpl_keys, sizeof(tmpl_key *) * tmpl->tmpl_keys_size);

		for (i = tmpl->tmpl_keys_pos; i < tmpl->tmpl_keys_size; i++) {
			tmpl->tmpl_keys[i] = malloc(sizeof(tmpl_key));
			memset(tmpl->tmpl_keys[i], 0, sizeof(tmpl_key));
			
			tmpl->tmpl_keys[i]->value = buffer_init();
		}
	}

	/* search for the key */
	for (i = 0; i < tmpl->tmpl_keys_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_keys[i]->name, s)) {
			break;
		}
	}

	/* if key is not found insert it in the array it the last position */
	if (i == tmpl->tmpl_keys_pos) {
		tmpl->tmpl_keys[tmpl->tmpl_keys_pos]->name = strdup(s);

		if (def_val) {
			tmpl->tmpl_keys[tmpl->tmpl_keys_pos]->defvalue = strdup(def_val);
		}

		tmpl->tmpl_keys_pos++;
	}

	return 0;
}

#define TMPL_DEF_BLOCK "_default"

/**
 * append a string to the current block
 *
 * @param tmpl template handle
 * @param s string
 * @return -1 hard error, 0 no error
 */

int tmpl_current_block_append(tmpl_main *tmpl, const char *s) {
	int i;
	char *block;

	if (!tmpl) return -1;

	/* init blocks at the first run */
	if (tmpl->tmpl_blocks == NULL) {
		tmpl->tmpl_blocks_size = 16;
		tmpl->tmpl_blocks_pos = 0;
		tmpl->tmpl_blocks = malloc(sizeof(tmpl_block *) * tmpl->tmpl_blocks_size);

		for (i = 0; i < tmpl->tmpl_blocks_size; i++) {
			tmpl->tmpl_blocks[i] = malloc(sizeof(tmpl_block));
			memset(tmpl->tmpl_blocks[i], 0, sizeof(tmpl_block));
			
			tmpl->tmpl_blocks[i]->value = buffer_init();
		}
	}

	/* increase array if neccesary */
	if (tmpl->tmpl_blocks_size == tmpl->tmpl_blocks_pos) {
		tmpl->tmpl_blocks_size += 16;
		tmpl->tmpl_blocks = realloc(tmpl->tmpl_blocks, sizeof(tmpl_block *) * tmpl->tmpl_blocks_size);

		for (i = tmpl->tmpl_blocks_pos; i < tmpl->tmpl_blocks_size; i++) {
			tmpl->tmpl_blocks[i] = malloc(sizeof(tmpl_block));
			memset(tmpl->tmpl_blocks[i], 0, sizeof(tmpl_block));
			
			tmpl->tmpl_blocks[i]->value = buffer_init();
		}
	}

	/* if no 'current block' is set */
	block = (tmpl->tmpl_current_block) ? tmpl->tmpl_current_block : TMPL_DEF_BLOCK;
#if 0
	printf("appending '%s' to block '%s'\n", s, block);
#endif

	/* search for the block and add the string 's' to the block */
	for (i = 0; i < tmpl->tmpl_blocks_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_blocks[i]->name, block)) {
			
			buffer_append_string(tmpl->tmpl_blocks[i]->value, s);
			
			break;
		}
	}

	/* if no block is found by this name create a new block and add the string */
	if (i == tmpl->tmpl_blocks_pos) {
		tmpl->tmpl_blocks[i]->name = strdup(block);
		
		buffer_copy_string(tmpl->tmpl_blocks[i]->value, s);
		
		tmpl->tmpl_blocks_pos++;
	}
#if 0
	printf("append -- '%s' -> '%s'\n",
	       tmpl->tmpl_blocks[i]->name,
	       tmpl->tmpl_blocks[i]->value);
#endif
	return 0;
}

/**
 * set current block
 *
 * @param tmpl template handle
 * @param key name of the current block
 * @return -1 hard error, 0 no error
 */

int tmpl_set_current_block(tmpl_main *tmpl, const char *key) {
	if (!tmpl) return -1;

	if (tmpl->tmpl_current_block) free(tmpl->tmpl_current_block);

	if (key) {
		tmpl->tmpl_current_block = strdup(key);
	} else {
		tmpl->tmpl_current_block = NULL;
	}

	return 0;
}

/**
 * extract the template from a file
 *
 * @param tmpl template handle
 * @param filename filename which should be parsed
 *
 * @return 0 on success
 */

int tmpl_load_template(tmpl_main *tmpl, const char *filename) {
#define N 20 + 1
	int ovector[3 * N], i;
	tmpl_file t;
#define TMPL_DEPTH_MAX 16
	char * tmpl_block_depth_names[TMPL_DEPTH_MAX];
	int tmpl_block_depth = 0;
	int lc = 0;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif

	if (!tmpl) return -1;

	if (!filename) {
		M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_GENERATION, M_DEBUG_LEVEL_ERRORS,
			 "no template file specified\n");
		return -1;
	}

	if (!(t.f = fopen(filename, "r"))) {
		M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_GENERATION, M_DEBUG_LEVEL_ERRORS,
			 "can't open template file '%s': %s\n",
			 filename,
			 strerror(errno));
		return -1;
	}

	t.buf_len = 128;
	t.buf_inc = 128;

	t.buffer = malloc(sizeof(char) * t.buf_len);

	for (i = 0; i < TMPL_DEPTH_MAX; i++)
		tmpl_block_depth_names[i] = NULL;

	while (tmpl_get_line_from_file(&t)) {
		int offset = 0;
		int n;

		int slen;
		char *s;

		lc++;
#if 0
		printf("%s.%d: %s", __FILE__, __LINE__, t.buffer);
#endif
		while ((n = pcre_exec(tmpl->match_tag_block, NULL, t.buffer, strlen(t.buffer), offset, 0, ovector, 3 * N)) == 4 || n == 6 || n == 3) {
			slen = ovector[0] - offset;
			s = malloc(slen + 1);
			strncpy(s, t.buffer + offset, slen);
			s[slen] = '\0';

			tmpl_current_block_append(tmpl, s);

			free(s);

			if (n == 4 || n == 3) { /* a key */
				char *defval = NULL;
				/* the key */
				slen = ovector[5] - ovector[4];

				s = malloc(slen + 1);
				strncpy(s, t.buffer + ovector[4], slen);
				s[slen] = '\0';

				/* handle default value */
				if (n == 4) {
					slen = ovector[7] - ovector[6];

					defval = malloc(slen + 1);
					strncpy(defval, t.buffer + ovector[6], slen);
					defval[slen] = '\0';
				}

				tmpl_insert_key(tmpl, s, defval);
				if (defval) free(defval);

				tmpl_current_block_append(tmpl, "{");
				tmpl_current_block_append(tmpl, s);
				tmpl_current_block_append(tmpl, "}");

				free(s);
			} else { /* a block */
				int slen = ovector[11] - ovector[10];
				char *s;

				s = malloc(slen + 1);
				strncpy(s, t.buffer + ovector[10], slen);
				s[slen] = '\0';

				if (*(t.buffer + ovector[8]) == 'B') {
					char *block;
					/* add {<blockname>} to the parent */
					tmpl_current_block_append(tmpl, "{");
					tmpl_current_block_append(tmpl, s);
					tmpl_current_block_append(tmpl, "}");

					block = tmpl->tmpl_current_block ? tmpl->tmpl_current_block : TMPL_DEF_BLOCK;

					if (tmpl_block_depth >= TMPL_DEPTH_MAX) {
						M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "line %d: max. depth (%d) of blocks reached\n",
							 lc,
							 TMPL_DEPTH_MAX);
						free(t.buffer);
						return -1;
					}

					tmpl_block_depth_names[tmpl_block_depth] = strdup(block);
					tmpl_block_depth++;

					/* BEGIN */
					tmpl_set_current_block(tmpl, s);
				} else {
					/* END */

					if (tmpl_block_depth > 0) {
						if (0 == strcmp(tmpl->tmpl_current_block, s)) {
							tmpl_block_depth--;

							tmpl_set_current_block(tmpl, tmpl_block_depth_names[tmpl_block_depth]);
							free(tmpl_block_depth_names[tmpl_block_depth]);
							tmpl_block_depth_names[tmpl_block_depth] = NULL;
						} else {
							M_DEBUG3(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
								 "line %d: wrong BEGIN (%s)/END (%s) combination\n",
								 lc,
								 tmpl->tmpl_current_block,
								 s);
							free(t.buffer);
							return -1;
						}
					} else {
						M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "line %d: more END (%s) then BEGIN\n",
							 lc,
							 s);
						free(t.buffer);
						return -1;
					}
				}

				free(s);
			}
			offset = ovector[1];
		}

		if (n < 0 && n != PCRE_ERROR_NOMATCH) {
			M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			free(t.buffer);
			return M_RECORD_HARD_ERROR;
		}

		/* the rest */
		slen = strlen(t.buffer) - offset;
		s = malloc(slen + 1);
		strncpy(s, t.buffer + offset, slen);
		s[slen] = '\0';

		tmpl_current_block_append(tmpl, s);

		free(s);
	}

	if (tmpl_block_depth > 0) {
		M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "line %d: missing END tag for %s\n",
			 lc,
			 tmpl_block_depth_names[tmpl_block_depth]);
		free(t.buffer);
		return -1;
	}

	fclose(t.f);
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif

	free(t.buffer);
	return 0;
}


/**
 * extract the template from a string
 *
 * @param tmpl template handle
 * @param filename filename which should be parsed
 *
 * @return 0 on success
 */

int tmpl_load_string(tmpl_main *tmpl, const char *string) {
#define N 20 + 1
	int ovector[3 * N], i;
	tmpl_string t;
#define TMPL_DEPTH_MAX 16
	char * tmpl_block_depth_names[TMPL_DEPTH_MAX];
	int tmpl_block_depth = 0;
	int lc = 0;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif

	if (!tmpl) return -1;

	if (!string) {
		M_DEBUG0(tmpl->debug_level, M_DEBUG_SECTION_GENERATION, M_DEBUG_LEVEL_ERRORS,
			 "no template string specified\n");
		return -1;
	}

	t.buffer = buffer_init();
	t.string = string;
	t.str_pos = 0;

	for (i = 0; i < TMPL_DEPTH_MAX; i++)
		tmpl_block_depth_names[i] = NULL;

	while (tmpl_get_line_from_string(&t)) {
		int offset = 0;
		int n;

		int slen;
		char *s;

		lc++;
#if 0
		printf("%s.%d: %s", __FILE__, __LINE__, t.buffer);
#endif
		while ((n = pcre_exec(tmpl->match_tag_block, NULL, t.buffer->ptr, t.buffer->used - 1, offset, 0, ovector, 3 * N)) == 4 || n == 6 || n == 3) {
			slen = ovector[0] - offset;
			s = malloc(slen + 1);
			strncpy(s, t.buffer->ptr + offset, slen);
			s[slen] = '\0';

			tmpl_current_block_append(tmpl, s);

			free(s);

			if (n == 4 || n == 3) { /* a key */
				char *defval = NULL;
				/* the key */
				slen = ovector[5] - ovector[4];

				s = malloc(slen + 1);
				strncpy(s, t.buffer->ptr + ovector[4], slen);
				s[slen] = '\0';

				/* handle default value */
				if (n == 4) {
					slen = ovector[7] - ovector[6];

					defval = malloc(slen + 1);
					strncpy(defval, t.buffer->ptr + ovector[6], slen);
					defval[slen] = '\0';
				}

				tmpl_insert_key(tmpl, s, defval);

				tmpl_current_block_append(tmpl, "{");
				tmpl_current_block_append(tmpl, s);
				tmpl_current_block_append(tmpl, "}");

				free(s);
			} else { /* a block */
				int slen = ovector[11] - ovector[10];
				char *s;

				s = malloc(slen + 1);
				strncpy(s, t.buffer->ptr + ovector[10], slen);
				s[slen] = '\0';

				if (*(t.buffer->ptr + ovector[8]) == 'B') {
					char *block;
					/* add {<blockname>} to the parent */
					tmpl_current_block_append(tmpl, "{");
					tmpl_current_block_append(tmpl, s);
					tmpl_current_block_append(tmpl, "}");

					block = tmpl->tmpl_current_block ? tmpl->tmpl_current_block : TMPL_DEF_BLOCK;

					if (tmpl_block_depth >= TMPL_DEPTH_MAX) {
						M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "line %d: max. depth (%d) of blocks reached\n",
							 lc,
							 TMPL_DEPTH_MAX);
						buffer_free(t.buffer);
						return -1;
					}

					tmpl_block_depth_names[tmpl_block_depth] = strdup(block);
					tmpl_block_depth++;

					/* BEGIN */
					tmpl_set_current_block(tmpl, s);
				} else {
					/* END */

					if (tmpl_block_depth > 0) {
						if (0 == strcmp(tmpl->tmpl_current_block, s)) {
							tmpl_block_depth--;

							tmpl_set_current_block(tmpl, tmpl_block_depth_names[tmpl_block_depth]);
							free(tmpl_block_depth_names[tmpl_block_depth]);
							tmpl_block_depth_names[tmpl_block_depth] = NULL;
						} else {
							M_DEBUG3(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
								 "line %d: wrong BEGIN (%s)/END (%s) combination\n",
								 lc,
								 tmpl->tmpl_current_block,
								 s);
							buffer_free(t.buffer);
							return -1;
						}
					} else {
						M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
							 "line %d: more END (%s) then BEGIN\n",
							 lc,
							 s);
						buffer_free(t.buffer);
						return -1;
					}
				}

				free(s);
			}
			offset = ovector[1];
		}

		if (n < 0 && n != PCRE_ERROR_NOMATCH) {
			M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
				 "execution error while matching: %d\n", n);
			buffer_free(t.buffer);
			return M_RECORD_HARD_ERROR;
		}

		/* the rest */
		slen = t.buffer->used - offset;
		s = malloc(slen + 1);
		strncpy(s, t.buffer->ptr + offset, slen);
		s[slen] = '\0';

		tmpl_current_block_append(tmpl, s);

		free(s);
	}

	if (tmpl_block_depth > 0) {
		M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "line %d: missing END tag for %s\n",
			 lc,
			 tmpl_block_depth_names[tmpl_block_depth]);
		buffer_free(t.buffer);
		return -1;
	}

#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	buffer_free(t.buffer);
	return 0;
}

/**
 * transform a block of the template into the resulting string
 *
 * @return transformed string
 */

int tmpl_replace_block(tmpl_main *tmpl, const char *block, buffer *replaced_block) {
#define N 20 + 1
	int i;
	tmpl_string t;
	buffer *str;
#ifdef MTIMER_ENABLED
	static mtimer timer;
#endif
	
	if (!tmpl) return -1;
	
#ifdef MTIMER_ENABLED
	MTIMER_START(timer);
#endif
	for (i = 0; i < tmpl->tmpl_blocks_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_blocks[i]->name, block)) {
			break;
		}
	}

	if (i == tmpl->tmpl_blocks_pos) {
		fprintf(stderr, "%s.%d: block '%s' is unknown\n", 
			__FILE__, __LINE__,
			block);
		return -1;
	}

	t.string = tmpl->tmpl_blocks[i]->value->ptr;
	t.str_pos = 0;

	t.buffer = buffer_init();
	
	str = replaced_block;
	
	str->used = 0;
	
	while (tmpl_get_line_from_string(&t)) {
		char *open, *close, *start;
		
#if 0		
		fprintf(stderr, "%s.%d: %s\n", __FILE__, __LINE__, t.buffer->ptr);
#endif
		/* search for { */
		
		start = t.buffer->ptr;
		while (NULL != (open = strchr(start, '{')) &&
		       NULL != (close = strchr(open, '}')) &&
		       close - open > 1) {

			/* before the { */
			
			buffer_append_string_len(str, start, open - start);

			/* find the key and substitute it */
			for (i = 0; i < tmpl->tmpl_keys_pos; i++) {
				if (0 == strncmp(tmpl->tmpl_keys[i]->name, open + 1, close - open - 1)) {
#if 0
					fprintf(stderr, "%s.%d: %s, %s\n", __FILE__, __LINE__, tmpl->tmpl_keys[i]->name, tmpl->tmpl_keys[i]->value->ptr);
#endif
					if (tmpl->tmpl_keys[i]->value->used) {
						buffer_append_string(str, tmpl->tmpl_keys[i]->value->ptr);
					} else if (tmpl->tmpl_keys[i]->defvalue) {
						buffer_append_string(str, tmpl->tmpl_keys[i]->defvalue);
					} else {
#if 0
						fprintf(stdout, "%s.%d: %s - no value\n",
							__FILE__, __LINE__,
							tmpl->tmpl_keys[i]->name);
#endif
					}
					break;
				}
			}

			if (i == tmpl->tmpl_keys_pos) {
				buffer *b;
					
				b = buffer_init();
				
				buffer_copy_string_len(b, open + 1, close - open - 1);
				/* ok, we have to extract the key for fprintf */
				M_DEBUG2(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
					 "key '%s' not found in block '%s'\n",
					 b->ptr, block);
				buffer_free(b);
			}

			start = close + 1;
			
		}
		
		/* copy the rest to the end of the string */
		buffer_append_string(str, start);
	}
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	buffer_free(t.buffer);
	return 0;
}

/**
 * transform template to a string
 */

int tmpl_replace(tmpl_main *tmpl, buffer *replaced_block) {
	if (!tmpl) return -1;

	return tmpl_replace_block(tmpl, TMPL_DEF_BLOCK, replaced_block);
}

/**
 * set a template variable to a value
 *
 * the value is copied into the internal data store and can be changed externally
 */

int tmpl_set_var (tmpl_main *tmpl, const char *key, const char *value) {
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;
	if (!value) return -1;

	for (i = 0; i < tmpl->tmpl_keys_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_keys[i]->name, key)) {
			buffer_copy_string(tmpl->tmpl_keys[i]->value, value);
			
			break;
		}
	}

	if (i == tmpl->tmpl_keys_pos) {
#if 0
		fprintf(stderr, "%s.%d: variable '%s' not found in block\n",
			__FILE__, __LINE__,
			key);
#endif
	}

#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return (i == tmpl->tmpl_keys_pos) ? -1 : 0;
}

/**
 * append a value to a variable 'key'
 *
 * @param tmpl template handle
 * @param key name of the variable
 * @param value value which shall be appended
 * @return -1 if variable is not found or invalid template handle, 0 no error
 */

int tmpl_append_var (tmpl_main *tmpl, const char *key, const char *value) {
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;

	/* append the value to the specified variable */
	for (i = 0; i < tmpl->tmpl_keys_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_keys[i]->name, key)) {
			buffer_append_string(tmpl->tmpl_keys[i]->value, value);
			
			break;
		}
	}
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	/* return -1 if variable is not found */
	return -(i == tmpl->tmpl_keys_pos);
}

/**
 * remove a variable from the template
 */

int tmpl_clear_var (tmpl_main *tmpl, const char *key) {
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;

	for (i = 0; i < tmpl->tmpl_keys_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_keys[i]->name, key)) {
			buffer_reset(tmpl->tmpl_keys[i]->value);
			
			break;
		}
	}
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return -(i == tmpl->tmpl_keys_pos);
}

/**
 * remove a block from the template
 */

int tmpl_clear_block (tmpl_main *tmpl, const char *key) {
	return tmpl_clear_var(tmpl, key);
}

/**
 * parse the current block and add the transformed block to the main block
 *
 * Sideeffect:
 * sets the current block to the default block after processing
 *
 */

int tmpl_parse_current_block(tmpl_main *tmpl) {
	char *block;
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;

	block = (tmpl->tmpl_current_block) ? tmpl->tmpl_current_block : TMPL_DEF_BLOCK;

	for (i = 0; i < tmpl->tmpl_blocks_pos; i++) {
		if (0 == strcmp(tmpl->tmpl_blocks[i]->name, block)) {
			if (0 == tmpl_replace_block(tmpl, block, tmpl->tmp_buf)) {
				tmpl_insert_key(tmpl, block, NULL);
				tmpl_append_var(tmpl, block, tmpl->tmp_buf->ptr);
			}

			break;
		}
	}

	if (i == tmpl->tmpl_blocks_pos) {
		M_DEBUG1(tmpl->debug_level, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_WARNINGS,
			 "block %s not found\n",
			 block);
	}

	tmpl_set_current_block(tmpl, NULL);
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return 0;
}


/**
 * remove all keys
 *
 */
int tmpl_free_keys(tmpl_main *tmpl) {
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;
	if (!tmpl->tmpl_keys) return -1;

	for (i = 0; i < tmpl->tmpl_keys_size; i++) {
		if (tmpl->tmpl_keys[i]->value) buffer_free(tmpl->tmpl_keys[i]->value);
		if (tmpl->tmpl_keys[i]->defvalue) free(tmpl->tmpl_keys[i]->defvalue);
		if (tmpl->tmpl_keys[i]->name) free(tmpl->tmpl_keys[i]->name);
		free(tmpl->tmpl_keys[i]);
	}

	free(tmpl->tmpl_keys);
	tmpl->tmpl_keys = NULL;
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return 0;
}


/**
 * remove all blocks
 */
int tmpl_free_blocks(tmpl_main *tmpl) {
	int i;
#ifdef MTIMER_ENABLED
	static mtimer timer;

	MTIMER_START(timer);
#endif
	if (!tmpl) return -1;
	if (!tmpl->tmpl_blocks) return -1;

	for (i = 0; i < tmpl->tmpl_blocks_size; i++) {
		if (tmpl->tmpl_blocks[i]->value) buffer_free(tmpl->tmpl_blocks[i]->value);
		if (tmpl->tmpl_blocks[i]->name) free(tmpl->tmpl_blocks[i]->name);
		free(tmpl->tmpl_blocks[i]);
	}

	free(tmpl->tmpl_blocks);
	tmpl->tmpl_blocks = NULL;
#ifdef MTIMER_ENABLED
	MTIMER_STOP(timer);
	MTIMER_CALC(timer);

	fprintf(stderr, "timer %s: %ld msec\n",
		__FUNCTION__, timer.span );
#endif
	return 0;
}

/**
 * init the template handle
 */

tmpl_main *tmpl_init() {
	tmpl_main *tmpl;
	const char *errptr;
	int erroffset = 0;

	tmpl = malloc(sizeof(tmpl_main));
	
	/* set everything to zero */
	memset(tmpl, 0, sizeof(tmpl_main));
	
	if ((tmpl->match_tag_block = pcre_compile(
				      /* tag                         | block */
				      "({([A-Z][A-Z0-9_]*?)(?:=(.*?)|)}|<!-- (BEGIN|END) ([a-z][a-z0-9_]*) -->)",
				      0, &errptr, &erroffset, NULL)) == NULL) {
		
		M_DEBUG1(10, M_DEBUG_SECTION_PARSING, M_DEBUG_LEVEL_ERRORS,
			 "rexexp compilation error at %s\n",
			 errptr);
		return NULL;
	}
	
	tmpl->tmp_buf = buffer_init();
	
	return tmpl;
}

/**
 * cleanup the template
 */
int tmpl_free(tmpl_main *tmpl) {
	tmpl_free_blocks(tmpl);
	tmpl_free_keys(tmpl);
	
	pcre_free(tmpl->match_tag_block);
	
	buffer_free(tmpl->tmp_buf);
	
	free(tmpl);

	return 0;
}
