/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.modlogan.org
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: marray.c,v 1.2 2004/08/27 20:07:37 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <assert.h>

#include "config.h"
#include "mlist.h"
#include "mdatatypes.h"
#include "mconfig.h"
#include "marray.h"

marray *marray_init() {
	marray *a;
	
	a = malloc(sizeof(marray));
	memset(a, 0, sizeof(marray));
	
	a->next_power_of_2 = 1;
	
	return a;
}

void marray_free(marray *a) {
	size_t i;
	if (!a) return;
	
	for (i = 0; i < a->used; i++) {
		if (a->data[i]) mdata_free(a->data[i]);
	}
	
	if (a->data)   free(a->data);
	if (a->sorted) free(a->sorted);
	
	free(a);
}

typedef enum {APPEND, OVERWRITE} insert_t;

int marray_prepare_insert(marray *a) {
	if (a->size == 0) {
		a->size   = 16;
		a->data   = malloc(sizeof(mdata *) * a->size);
		a->sorted = malloc(sizeof(size_t)  * a->size);
	} else if (a->size == a->used) {
		a->size  += 16;
		a->data   = realloc(a->data,   sizeof(mdata *) * a->size);
		a->sorted = realloc(a->sorted, sizeof(size_t)  * a->size);
	}
	
	return 0;
}

int marray_get_index(marray *a, const char *key, size_t *index) {
	int pos, i;
	size_t ndx = a->used + 1;
	
	/* try to find the string */
	for (i = pos = a->next_power_of_2 / 2; ; i >>= 1) {
		int cmp;
		if (pos < 0) {
			pos += i;
		} else if (pos >= a->used) {
			pos -= i;
		} else {
			cmp = strcmp(key, a->data[a->sorted[pos]]->key);
			
			if (cmp == 0) {
				/* found */
				ndx = a->sorted[pos];
				break;
			} else if (cmp < 0) {
				pos -= i;
			} else {
				pos += i;
			}
		}
		if (i == 0) break;
	}
	
	if (ndx > a->used) {
		/* not found */
		return -1;
	} else {
		*index = ndx;
		return 0;
	}
}

int marray_insert_uniq(marray *a, mdata *data, insert_t type) {
	int i, pos = 0;
	size_t ndx;
	
	if (0 == marray_get_index(a, data->key, &ndx)) {
		if (type == APPEND) {
			if (mdata_append(a->data[ndx], data) == M_DATA_APPENDED) {
				/* the memory isn't used anymore */
				mdata_free(data);
				return 0;
			}
			/* otherwise append to the array */
			fprintf(stderr, "**\n");
		} else { /* overwrite */
			mdata_free(a->data[ndx]);
			a->data[ndx] = data;
			return 0;
		}
	}
	
	marray_prepare_insert(a);
	
	/* insert */
	ndx = a->used;
	
	a->data[a->used++] = data;
	
	/* insert here ? */
	if (pos != ndx &&
	    ((pos < 0) || 
	     strcmp(data->key, a->data[a->sorted[pos]]->key) > 0)) {
		pos++;
	} 
	
	/* move everything on step to the right */
	for (i = ndx; i > pos; i--) {
		a->sorted[i] = a->sorted[i - 1];
	}
	
	/* insert */
	a->sorted[pos] = ndx;
	
	if (a->next_power_of_2 == ndx) a->next_power_of_2 <<= 1;
	
	return 0;
}

int marray_insert(marray *a, mdata *data) {
	return marray_insert_uniq(a, data, APPEND);
}

int marray_insert_replace(marray *a, mdata *data) {
	return marray_insert_uniq(a, data, OVERWRITE);
}

size_t *marray_sort_by_count(marray *a) {
	size_t *sorted;
	size_t i;
	
	sorted = malloc(a->used * sizeof(size_t));
	
	for (i = 0; i < a->used; i++) {
		sorted[i] = i;
	}
	
	for (i = 0; i < a->used - 1; i++) {
		size_t j, max = i;
		
		for (j = i + 1; j < a->used; j++) {
			if (mdata_get_count(a->data[sorted[j]]) > mdata_get_count(a->data[sorted[max]])) max = j;
		}
		
		if (max != i) {
			size_t tmp;
			
			tmp = sorted[max];
			sorted[max] = sorted[i];
			sorted[i] = tmp;
		}
	}
	
	return sorted;
}
	
int marray_is_empty(marray *a) {
	if (!a) return 1;

	return (a->used == 0);
}

int marray_append (marray *a, mdata *data) {
	if (!a) return -1;
	if (!data) return -1;
	
	marray_prepare_insert(a);
	
	a->data[a->used++] = data;

	return 0;
}

int marray_write(gzFile *fd, marray *a) {
	size_t i;
	
	for (i = 0; i < a->used; i++) {
		if (a->data[i]) mdata_write(fd, a->data[i]);
	}
	return 0;
}

int marray_count(marray *a) {
	if (!a) return 0;

	return a->used;
}

mdata *marray_get_data(marray *a, const char *str) {
	size_t ndx;
	
	if (0 == marray_get_index(a, str, &ndx)) {
		return a->data[ndx];
	} else {
		return NULL;
	}
}

int marray_in_array(marray *a, const char *str) {
	size_t ndx;
	
	if (0 == marray_get_index(a, str, &ndx)) {
		return 1;
	} else {
		return 0;
	}
}

double marray_sumup(marray *a) {
	double c = 0;
	size_t i;
	if (!a) return 0;

	for (i = 0; i < a->used; i++) {
		c += mdata_get_count(a->data[i]);
	}

	return c;
}

double marray_sumup_vcount(marray *a) {
	double c = 0;
	size_t i;
	if (!a) return 0;

	for (i = 0; i < a->used; i++) {
		c += mdata_get_vcount(a->data[i]);
	}

	return c;
}
