/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: process.c,v 1.34 2001/12/30 22:54:25 ostborn Exp $
*/

#include <libintl.h>
#include <locale.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <ctype.h>

#include <pcre.h>

#include "config.h"
#include "mrecord.h"
#include "mlocale.h"
#include "mconfig.h"
#include "mplugins.h"
#include "mstate.h"
#include "mdatatypes.h"
#include "datatypes/count/datatype.h"
#include "datatypes/visited/datatype.h"
#include "datatypes/state/datatype.h"
#include "datatypes/sublist/datatype.h"
#include "datatypes/visit/datatype.h"
#include "datatypes/brokenlink/datatype.h"
#include "misc.h"

#include "plugin_config.h"
#include "md5_global.h"
#include "md5.h"

#define UNRESOLVED_TLD "unre"

#define M_WEB_IGNORE_HOST	1
#define M_WEB_IGNORE_REQ_URL	2
#define M_WEB_IGNORE_USERAGENT	3
#define M_WEB_IGNORE_HOSTMASK	4

#define M_WEB_HIDE_HOST		1
#define M_WEB_HIDE_REQ_URL	2
#define M_WEB_HIDE_REFERRER	3
#define M_WEB_HIDE_BROKENLINK	4
#define M_WEB_HIDE_EXTENSION	5

#define M_WEB_GROUP_REFERRER	1
#define M_WEB_GROUP_HOST	2
#define M_WEB_GROUP_OS		3
#define M_WEB_GROUP_UA		4
#define M_WEB_GROUP_REQ_URL	5
#define M_WEB_GROUP_BROKENLINK	6
#define M_WEB_GROUP_SEARCHSTRINGS 7
#define M_WEB_GROUP_EXTENSION   8


int is_matched(mlist *l, const char *url) {
	if (!url || !l) return 0;

	while (l) {
		mdata *data = l->data;
		
		if (data == NULL) {
			l = l->next;
			continue;
		}
		if (data->type != M_DATA_TYPE_MATCH) {
			fprintf(stderr, "%s.%d: wrong datatype for a match: %d\n", __FILE__, __LINE__, data->type);
			l = l->next;
			continue;
		}
		
		if (strmatch(data->data.match.match, data->data.match.study, url)) return 1;

		l = l->next;
	}

	return 0;
}

int hostmask_match(const char *hostmask, const char *host) {
	long hm, nm, ip;
	const char *h;
	int i;
	
	/* 0-3 are for the hostmask, 4 is for the netmask (/24) */
	int hm_elem[5];
	
	/* 0-3 are for the host */
	int h_elem[4];
	
	if (!hostmask) return 0;
	if (!host) return 0;
	
	memset(hm_elem, 0, sizeof(hm_elem));
	memset(h_elem, 0, sizeof(h_elem));
	
	i = 0;
	for (h = hostmask; *h; h++) {
		switch(*h) {
		case '.':
			i++;
			/* number of dots in the IP field */
			if (i > 3) {
				fprintf(stderr, "%s.%d: too much dots in hostmask: '%s'\n",
					__FILE__, __LINE__,
					hostmask
					);
				return 0;
			}
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			hm_elem[i] = hm_elem[i] * 10 + (*h - '0');
			
			/* overflow */
			if (hm_elem[i] > 255) {
				fprintf(stderr, "%s.%d: value is too high '%d' in ip: '%s'\n",
					__FILE__, __LINE__,
					h_elem[i],
					hostmask
					);
				return 0;
			}
			break;
		case '/':
			/* we need 3 dot before our netmask */
			if (i != 3) {
				fprintf(stderr, "%s.%d: not enough dots in hostmask: '%s'\n",
					__FILE__, __LINE__,
					hostmask
					);
				return 0;
			}
			
			i++;
			break;
		default:
			/* every after character is invalid */
			fprintf(stderr, "%s.%d: invalid character '%c' in hostmask: '%s'\n",
				__FILE__, __LINE__,
				*h,
				hostmask
				);
			return 0;
		}
	}
	
	/* everythink set ? */
	if (i != 4) return 0;
	
	/* generate hostmask */
	hm = hm_elem[0] << 24 |
		hm_elem[1] << 16 |
		hm_elem[2] <<  8 |
		hm_elem[3] <<  0;
	
	/* generate netmask */
	nm = 0;
	for (i = 0; i < hm_elem[4]; i++) {
		nm |= 1 << (31 - i);
	}
	
	i = 0;
	for (h = host; *h; h++) {
		switch(*h) {
		case '.':
			i++;
			/* number of dots in the IP field */
			if (i > 3) {
				fprintf(stderr, "%s.%d: too much dots in ip: '%s'\n",
					__FILE__, __LINE__,
					host
					);
				return 0;
			}
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
			h_elem[i] = h_elem[i] * 10 + (*h - '0');
			
			/* overflow */
			if (h_elem[i] > 255) {
				fprintf(stderr, "%s.%d: value is too high '%d' in ip: '%s'\n",
					__FILE__, __LINE__,
					h_elem[i],
					host
					);
				return 0;
			}
			break;
		default:
			/* every after character is invalid */
#if 0
			fprintf(stderr, "%s.%d: invalid character '%c' in ip: '%s'\n",
				__FILE__, __LINE__,
				*h,
				host
				);
#endif
			return 0;
		}
	}
	
	/* everythink set ? */
	if (i != 3) return 0;
	
	/* generate host-ip */
	ip = h_elem[0] << 24 |
		h_elem[1] << 16 |
		h_elem[2] <<  8 |
		h_elem[3] <<  0;
#if 0	
	fprintf(stderr, "-> %08lx & %08lx =? %08lx -> %d\n", ip, nm, hm, (ip & nm) == hm);
#endif	
	return ((ip & nm) == hm ? 1 : 0);
}

int is_matched_hostmask(mlist *l, const char *url) {
	if (!url || !l) return 0;

	while (l) {
		mdata *data = l->data;
		
		if (data == NULL) {
			l = l->next;
			continue;
		}
		if (data->type != M_DATA_TYPE_COUNT) {
			fprintf(stderr, "%s.%d: wrong datatype for a match_hostmask: %d\n", __FILE__, __LINE__, data->type);
			l = l->next;
			continue;
		}
		
		if (hostmask_match(data->key, url)) return 1;
		
		l = l->next;
	}

	return 0;
}




int hide_field(mconfig *ext_conf, const char *url, int field) {
	config_processor *conf = ext_conf->plugin_conf;
	mlist *l = NULL;
	
	switch (field) {
	case M_WEB_HIDE_REFERRER:
		l = conf->hide_referrer;
		break;
	case M_WEB_HIDE_BROKENLINK:
		l = conf->hide_brokenlinks;
		break;
	case M_WEB_HIDE_REQ_URL:
		l = conf->hide_url;
		break;
	case M_WEB_HIDE_HOST:
		l = conf->hide_host;
		break;
	case M_WEB_HIDE_EXTENSION:
		l = conf->hide_extension;
		break;
	default:
		fprintf(stderr, "%s.%d: Unknown hide field: %d\n", __FILE__, __LINE__, field);
		break;
	}
	
	
	
	if (!url || !l) return 0;
	
	return is_matched(l, url);
}

int ignore_field(mconfig *ext_conf, const char *url, int field) {
	config_processor *conf = ext_conf->plugin_conf;
	mlist *l = NULL;
	int ret;

	switch (field) {
	case M_WEB_IGNORE_REQ_URL:
		l = conf->ignore_url;
		break;
	case M_WEB_IGNORE_HOST:
		l = conf->ignore_host;
		break;
	case M_WEB_IGNORE_USERAGENT:
		l = conf->ignore_ua;
		break;
	case M_WEB_IGNORE_HOSTMASK:
		l = conf->ignore_hostmask;
		break;
	default:
		fprintf(stderr, "%s.%d: Unknown ignore field: %d\n", __FILE__, __LINE__, field);
		break;
	}
	
	if (!url || !l) return 0;
	
	if (field != M_WEB_IGNORE_HOSTMASK) {
		ret = is_matched(l, url);
	} else {
		ret = is_matched_hostmask(l, url);
	}
	
	return ret;
}

char * is_grouped (mconfig *ext_conf, mlist *l, const char *str) {	
	char *r = NULL;
	
	if (!str || !l) return NULL;
	
	while (l) {
		mdata *data = l->data;
		
		if (data == NULL) {
			l = l->next;
			continue;
		}
		if (data->type != M_DATA_TYPE_MATCH) {
			fprintf(stderr, "%s.%d: wrong datatype for a match: %d\n", __FILE__, __LINE__, data->type);
			l = l->next;
			continue;
		}
		if (strmatch(data->data.match.match, data->data.match.study, str)) {
			r = substitute(ext_conf, data->data.match.match, data->data.match.study, data->key, str);
			
			if (r == NULL) 
				fprintf(stderr, "%s.%d: substitute failed: %p - %s - %s\n", 
					__FILE__, __LINE__,
					data->data.match.match, data->key, str);
			
			break;
		}
		
		l = l->next;
	}
	
	return r;
}



char * group_field (mconfig *ext_conf, const char *str, int field) {
	config_processor *conf = ext_conf->plugin_conf;
	mlist *l = NULL;
	
	switch (field) {
	case M_WEB_GROUP_REFERRER:
		l = conf->group_referrer;
		break;
	case M_WEB_GROUP_HOST:
		l = conf->group_hosts;
		break;
	case M_WEB_GROUP_OS:
		l = conf->group_os;
		break;
	case M_WEB_GROUP_UA:
		l = conf->group_ua;
		break;
	case M_WEB_GROUP_REQ_URL:
		l = conf->group_url;
		break;
	case M_WEB_GROUP_BROKENLINK:
		l = conf->group_brokenlinks;
		break;
	case M_WEB_GROUP_SEARCHSTRINGS:
		l = conf->group_searchstrings;
		break;
	case M_WEB_GROUP_EXTENSION:
		l = conf->group_extension;
		break;
	default:
		fprintf(stderr, "%s.%d: Unknown group field: %d\n", __FILE__, __LINE__, field);
		break;
	}
	
	if (!str || !l) return NULL;
	
	return is_grouped (ext_conf, l, str);
		
}

int is_file(mlogrec_web *record) {
	return record->req_status == 200;
}

int is_page(mconfig *ext_conf, mlogrec_web *record) {	
	config_processor *conf = ext_conf->plugin_conf;
	mlist *l = conf->page_type;
	char *url = record->req_url;
	
	if (!url || !l) return 0;

#if 0
	if (!is_file(record)) return 0;
#endif 
	
	while (l) {
		mdata *data = l->data;
		
		if (data && strmatch(data->data.match.match, data->data.match.study, url)) {
#if 0
			fprintf(stderr, "%s.%d: is a page: %s\n", __FILE__, __LINE__, url);
#endif
			return 1;
		}
		
		l = l->next;
	}
	
	return 0;
}

int is_robot(const char *url) {
	
	if (!url) return 0;
	
	return !strcmp(url, "/robots.txt");
}

int insert_view_to_views(mconfig *ext_conf, mstate *state, mlogrec *record, mdata *visit, int is_hit) {
	config_processor *conf = ext_conf->plugin_conf;
	mlist*	hlist = NULL;
	mlogrec_web *recweb = record->ext;
	mlogrec_web_extclf *recext = NULL;
	mstate_web *staweb = state->ext;
	int debug_me = conf->debug_visits;

	if (recweb == NULL)  return -1;
	if (recweb->req_url == NULL)  return -1;
	if (recweb->ext_type != M_RECORD_TYPE_WEB_EXTCLF) return -1;
	
	/* can be NULL, but that's handled below */
	recext = recweb->ext;

	/* lookup the last hit which this duration applies */
	for (hlist = visit->data.visit.hits; hlist->next && hlist->next->data; hlist = hlist->next)
		;
	
	if (hlist->data) {
		char* last_url = hlist->data->key;
		time_t	duration = 0;

		/* use input generated timediff or calculate it for page views */
		if (visit->data.visit.timediff) {
			duration = visit->data.visit.timediff;
		} else {
			duration = record->timestamp - visit->data.visit.timestamp;
			if (duration >= conf->visit_timeout) {
				duration = 5; /* XXXST create config variable or average over time */
			}
		}

		/* check to see if this hit should be recorded as a page view */
		if (!hide_field(ext_conf, last_url, M_WEB_HIDE_REQ_URL)) {
			char* 	grouped = group_field(ext_conf, last_url, M_WEB_GROUP_REQ_URL);
			mdata*	data = NULL;

			if (debug_me) {
				fprintf(stderr, "process.is_visit: ** %20s,  time: %ld - %ld\n",
					last_url, 
					duration, 
					recext->duration);
			}
		
			if (grouped) {
				data = mdata_Visited_create(grouped, duration, M_DATA_STATE_GROUPED, is_hit ? 1 : 0);
				free(grouped);
			} else {
				data = mdata_Visited_create(last_url, duration, M_DATA_STATE_PLAIN, is_hit ? 1 : 0);
			}
			mhash_insert_sorted(staweb->views, data);
		}
	} else {
		if (debug_me) {
			fprintf(stderr, "process.is_visit: No data for last hit!!\n");
		}
	}

	return 1;
}

int append_hit_to_visit(mconfig *ext_conf, mstate *state, mlogrec *record, mdata *visit) {
	mlogrec_web *recweb = record->ext;
	mlogrec_web_extclf *recext = NULL;
	mstate_web *staweb = state->ext;

	if (recweb == NULL)  return -1;
	if (recweb->req_url == NULL)  return -1;
	if (recweb->ext_type != M_RECORD_TYPE_WEB_EXTCLF) return -1;
	
	/* can be NULL, but that's handled below */
	recext = recweb->ext;

	/* set last visited page */
	if (!hide_field(ext_conf,recweb->req_url, M_WEB_HIDE_REQ_URL)) {
		mdata *hit;
		if (visit->data.visit.type == M_DATA_VISIT_ROBOT) {
			mdata *_data;
			
			_data = mdata_Count_create(recweb->req_url, 1, M_DATA_STATE_PLAIN);
			mhash_insert_sorted(staweb->indexed_pages, _data);
		}
		
		hit = mdata_BrokenLink_create(recweb->req_url, 1, M_DATA_STATE_PLAIN, record->timestamp, NULL);
		
		mlist_append(visit->data.visit.hits, hit);
		
		visit->data.visit.count++;
	}

	/* save the last record's stats */
	visit->data.visit.timediff = recext ? recext->duration : 0;
	visit->data.visit.timestamp = record->timestamp;

	return 1;
}

int is_visit(mconfig *ext_conf, mstate *state,mlogrec *record) {
	config_processor *conf = ext_conf->plugin_conf;
	mstate_web *staweb = state->ext;
	mlist *l = staweb->visit_list;
	int visit_timeout = conf->visit_timeout;
	int visited = 0;

	int new_entry = 1;
	int debug_me = conf->debug_visits;

	mlogrec_web *recweb = record->ext;
	mlogrec_web_extclf *recext = NULL;

	if (recweb == NULL)  return -1;
	if (recweb->req_url == NULL)  return -1;
	if (recweb->ext && recweb->ext_type != M_RECORD_TYPE_WEB_EXTCLF) return -1;
	
	/* can be NULL, but that's handled below */
	recext = recweb->ext;
	
	if (!is_page(ext_conf, recweb)) return 0;
	
	while(l) {
		mdata *data = l->data;
		
		if (!data) break;
		
		/* remove finished visits */
		if (record->timestamp - data->data.visit.timestamp > visit_timeout) {
			mlist *act;
			mlist *vlist;
			mdata *d;
			int i;
			/* md5 */
			char md5str[33];
			MD5_CTX context;
			unsigned char digest[16];
			char *r;
			
			if (debug_me) {
				fprintf(stderr, "process.is_visit: <- %20s (%20s), time: %ld - %ld\n",
					data->key, 
					data->data.visit.useragent, 
					data->data.visit.timestamp, 
					record->timestamp - data->data.visit.timestamp);
			}
			
			/* record the last hit as a view */
			insert_view_to_views(ext_conf, state, record, data, 1);
			
			/* extract the visit list */
			vlist = data->data.visit.hits;
			data->data.visit.hits = NULL;
			
			/* build key */
			
			act = vlist;
			
			md5str[0] = '\0';
			MD5Init(&context);
			
			while (act) {
				if (act->data) {
					MD5Update(&context, act->data->key, strlen(act->data->key));
				}
				act = act->next;
			}
			
			MD5Final(digest, &context);
			for (i = 0, r = md5str; i < 16; i++, r += 2) {
				sprintf(r, "%02x", digest[i]);
			}
			*r = '\0';
#if 0			
			fprintf(stderr, "%s.%d: -%d- %s\n", __FILE__, __LINE__, mdata_get_count(staweb->visits), md5str);
#endif			

			d = mdata_SubList_create(md5str, vlist);
			mhash_insert_sorted(staweb->visits, d);
			
			act = l;
			
			if (l->next) {
				l = l->next;
				
				if (act->next)
					act->next->prev = act->prev;
				if (act->prev) {
					act->prev->next = act->next;
				} else {
					staweb->visit_list = l;
				}
				
				mlist_free_entry(act);
			} else {
				mdata_free(data);
				
				l->data = NULL;
				
				break;
			}
		} else if (!strcmp(recweb->req_host, data->key) && 
			   (!recext || !(recext->req_useragent) || !(data->data.visit.useragent) || 
			    !strcmp(recext->req_useragent, data->data.visit.useragent))) {

			if (debug_me) {
				fprintf(stderr, "process.is_visit: -- %20s (%20s), time: %ld - %ld\n",
					data->key, 
					data->data.visit.useragent, 
					record->timestamp, 
					record->timestamp - data->data.visit.timestamp);
			}
	
			insert_view_to_views(ext_conf, state, record, data, 1);
			append_hit_to_visit(ext_conf, state, record, data);

			new_entry = 0;
			break;

		/* another hit for the same visit */
		} else {
			if (!l->next) break;
			
			l = l->next;
		}
	}
	
	/* insert the a new visit if the URL should not be hidden from the user */
	if (new_entry && !hide_field(ext_conf,recweb->req_url, M_WEB_HIDE_REQ_URL)) {
		int type = is_robot(recweb->req_url);
		mdata *data;
		
		visited = 1;
		
		if (debug_me) {
			fprintf(stderr, "process.is_visit: -> %20s (%20s), time: %ld\n",
				recweb->req_host, 
				recext ? recext->req_useragent : "", 
				record->timestamp);
		}
		
		data = mdata_Visit_create(recweb->req_host, 
					  NULL,  /* append the hit and other goodies in append_hit... */
					  recext ? recext->req_useragent : NULL, 
					  1, record->timestamp, 
					  0, type);
		
		append_hit_to_visit(ext_conf, state, record, data);

		if (l->data) {
			mlist *n = mlist_init();

			n->data = data;
			
			n->prev = l;
			l->next = n;
		} else {
			l->data = data;
		}
	}

	return visited;
}

int is_searchengine(mconfig *ext_conf, mstate *state, mlogrec_web_extclf *record) {
	config_processor *conf = ext_conf->plugin_conf;
	int site_found = 0;
	int key_found = 0;
	char *found_key = NULL;
#define N 20 + 1
	
	mlist *l = conf->searchengines;
	mstate_web *staweb = state->ext;
	
	if (!l) return 0;
	
	if (!(record->ref_getvars && record->ref_url)) return 0;
	
	while (l && !site_found) {
		char *search;
		char *c1, *c2, c3;
		int ovector[3 * N];
		
		if (!(l->data)) {
			l = l->next;
			continue;
		}
		
		search = l->data->key;
		
		c1 = record->ref_getvars;
	/* check every /&(.*?)&/ part for a known string */		
		while (!site_found && (c2 = strchr(c1, '&'))) {
			c3 = *c2;
			*c2 = '\0';
			
			if (!strncmp(c1, search, strlen(search))) {
				mdata *data;
				/* sublist */
				mlist *p = l->data->data.sublist.sublist;
				char *c4;
				
				if ((c4 = strchr(c1, '='))) {
					c4++;
				} else {
					c4 = c1;
				}
				
				key_found = 1;
				found_key = search;
				
				/* we got the searchkey, let's check the site now */
				
				while (p) {
					mdata *_data;
					int n;
				
					_data = p->data;
				
					n = pcre_exec(_data->data.match.match, NULL, record->ref_url, strlen(record->ref_url), 0, 0, ovector, 3 * N);
					
					if (n < 0) {
						if (n != PCRE_ERROR_NOMATCH) {
							fprintf(stderr,"%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
						}
					} else {
						break;
					}
					p = p->next;
				}

				if (p) {
					mdata *_data = p->data;
				
					if (_data && _data->key) {
						int key = strtol(_data->key,NULL,10);
						
						if (strlen(_data->key) > 2 && 
							_data->key[0] == '"' && 
							_data->key[strlen(_data->key)-1] == '"') {
					/* grouping of the urls */
					
							char *s = malloc(strlen(_data->key));
							char *grouped;
							
							strncpy(s, _data->key+1, strlen(_data->key)-1);
							s[strlen(_data->key)-2] = '\0';
						
							
							if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
								data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
								mhash_insert_sorted(staweb->searchstring, data);
								free(grouped);
							} else {
								data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
								mhash_insert_sorted(staweb->searchstring, data);
							}
						
							data = mdata_Count_create(s, 1, M_DATA_STATE_GROUPED);
							mhash_insert_sorted(staweb->searchsite, data);
						
							free(s);
						} else if (key >= 0) {
					/* take the url and be happy */
							char *grouped;
							if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
								data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
								mhash_insert_sorted(staweb->searchstring, data);
								free(grouped);
							} else {
								data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
								mhash_insert_sorted(staweb->searchstring, data);
							}

							data = mdata_Count_create(record->ref_url, 1, M_DATA_STATE_PLAIN);
							mhash_insert_sorted(staweb->searchsite, data);
						} else if (key < 0) {
					/* this is FALSE detection, just ignore it */
					
						} else {
							fprintf(stderr, "%s.%d: don't know how the handle searchstring-definition-action: %s\n",__FILE__, __LINE__, _data->key);
						}
					} else {
						char *grouped;
						if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
							data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
							mhash_insert_sorted(staweb->searchstring, data);
							free(grouped);
						} else {
							data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
							mhash_insert_sorted(staweb->searchstring, data);
						}
					
						data = mdata_Count_create(record->ref_url, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(staweb->searchsite, data);
					}
					site_found = 1;
				} 
			}
			*c2 = c3;
			c1 = ++c2;
		}
	/* check the last string */
		if (!site_found && !strncmp(c1, search, strlen(search))) {
			mdata *data;
			char *c4;
			mlist *p = l->data->data.sublist.sublist;
					
			if ((c4 = strchr(c1, '='))) {
				c4++;
			} else {
				c4 = c1;
			}
			
			key_found = 1;
			found_key = search;
			
			/* we got the searchkey, let's check the site now */
				
			while (p) {
				int n;
				
				n = pcre_exec(p->data->data.match.match, NULL, record->ref_url, strlen(record->ref_url), 0, 0, ovector, 3 * N);
				
				if (n < 0) {
					if (n != PCRE_ERROR_NOMATCH) {
						fprintf(stderr,"%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
					}
				} else {
					break;
				}
				
				p = p->next;
			}

			if (p) {
				mdata *_data = p->data;
				
				if (_data && _data->key) {
					int key = strtol(_data->key,NULL,10);
					
					if (strlen(_data->key) > 2 && 
						_data->key[0] == '"' && 
						_data->key[strlen(_data->key)-1] == '"') {
				/* grouping of the urls */
						char *grouped;
						char *s = malloc(strlen(_data->key));
						
						strncpy(s, _data->key+1, strlen(_data->key)-1);
						s[strlen(_data->key)-2] = '\0';
					
						if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
							data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
							mhash_insert_sorted(staweb->searchstring, data);
							free(grouped);
						} else {
							data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
							mhash_insert_sorted(staweb->searchstring, data);
						}
					
						data = mdata_Count_create(s, 1, M_DATA_STATE_GROUPED);
						mhash_insert_sorted(staweb->searchsite, data);
					
						free(s);
					} else if (key >= 0) {
				/* take the url and be happy */
						char *grouped;
						if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
							data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
							mhash_insert_sorted(staweb->searchstring, data);
							free(grouped);
						} else {
							data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
							mhash_insert_sorted(staweb->searchstring, data);
						}
							
						data = mdata_Count_create(record->ref_url, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(staweb->searchsite, data);
					} else if (key < 0) {
				/* this is FALSE detection, just ignore it */
				
					} else {
						fprintf(stderr, "%s.%d: don't know how the handle searchstring-definition-action: %s\n",__FILE__, __LINE__, _data->key);
					}
				} else {
					char *grouped;
					if ((grouped = group_field(ext_conf, urlescape(c4), M_WEB_GROUP_SEARCHSTRINGS))) {
						data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
						mhash_insert_sorted(staweb->searchstring, data);
						free(grouped);
					} else {
						data = mdata_Count_create(urlescape(c4), 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(staweb->searchstring, data);
					}
				
					data = mdata_Count_create(record->ref_url, 1, M_DATA_STATE_PLAIN);
					mhash_insert_sorted(staweb->searchsite, data);
				}
				site_found = 1;
			}
		}
		
		l = l->next;
	}
	
	if (conf->debug_searchengines > 0 && !site_found && key_found) {
		char *k = malloc(strlen(found_key));
		FILE * f;
		char * fn = NULL;
		
		fn = malloc(strlen(ext_conf->outputdir) + strlen("/searchstrings") + 1);
		sprintf(fn, "%s/searchstrings", ext_conf->outputdir);
		k[strlen(found_key)-1] = '\0';
		strncpy(k, found_key, strlen(found_key)-1);
		if ((f = fopen(fn, "a+"))) {
			fprintf(f, "#+[%s]\t%s || SK: %s->%s\n", 
				k, record->ref_url,
				record->ref_url, record->ref_getvars);
			fclose(f);
		}
		free(fn);
		free(k);
	} else if (conf->debug_searchengines > 1 && !site_found && !key_found) {
		char * fn = NULL;
		FILE *f;
		
		fn = malloc(strlen(ext_conf->outputdir) + strlen("/searchstrings.checkme") + 1);
		sprintf(fn, "%s/searchstrings.checkme", ext_conf->outputdir);
		if ((f = fopen(fn, "a+"))) {
			fprintf(f, "#+[%s]\t%s || SE: %s -> %s\n", 
				found_key, record->ref_url,
				record->ref_url, record->ref_getvars);
			fclose(f);
		}
		free(fn);
	}
#undef N
	return site_found;
}

char *urltolower(char *str) {
	char *s;
	
	if (!str) return NULL;
	
	if (!(strncasecmp(str, "http://", strlen("http://")))) {
		s = str;
		
		while (*s && *s != '/') 
			*s++ = tolower(*s);
		
		s = str + strlen("http://");
	} else if (!(strncasecmp(str, "https://", strlen("https://")))) {
		s = str;
		
		while (*s && *s != '/') 
			*s++ = tolower(*s);
		
		s = str + strlen("https://");
	} else if (!(strncasecmp(str, "ftp://", strlen("ftp://")))) {
		s = str;
		
		while (*s && *s != '/') 
			*s++ = tolower(*s);
		
		s = str + strlen("ftp://");
	} else {
		s = str;
	}
	
	while (*s && *s != '/') 
		*s++ = tolower(*s);
		
	return str;
}

mstate *splitter(mconfig *ext_conf, mlist *state_list, mlogrec *record) {
	config_processor *conf = ext_conf->plugin_conf;
	char *name = NULL;	/* name if the state -> directory-name */
	mstate *state = NULL;
	int split_enable = 0;
	
	/* record extension */
	mlogrec_web *recweb = NULL;
	
	/* record web extensions */
	mlogrec_web_extclf *recext = NULL;
	mlogrec_web_squid *recsquid = NULL;
	mlogrec_web_ftp *recftp = NULL;
	
	recweb = record->ext;
	
	switch (recweb->ext_type) {
	case M_RECORD_TYPE_WEB_EXTCLF:
		recext = recweb->ext; break;
	case M_RECORD_TYPE_WEB_FTP:
		recftp = recweb->ext; break;
	case M_RECORD_TYPE_WEB_SQUID:
		recsquid = recweb->ext; break;
	}
	
	if (conf->split_def) {
		mlist *l = conf->split_def;
		
		while (l) {
			mdata *data = l->data;
			char *str = NULL;
			
			if (!data) break;
			
			split_enable = 1;
		
		/* decide which field we shall look at */
			switch(data->data.split.fieldtype) {
			case M_SPLIT_FIELD_REQURL:
				str = recweb->req_url;
				break;
			case M_SPLIT_FIELD_REQUSER:
				str = recweb->req_user;
				break;
			case M_SPLIT_FIELD_SRVHOST:
				if (recext)
					str = recext->srv_host;
				break;
			case M_SPLIT_FIELD_SRVPORT:
				if (recext)
					str = recext->srv_port;
				break;
			case M_SPLIT_FIELD_REQHOST:
				str = recweb->req_host;
				break;
			case M_SPLIT_FIELD_REFURL:
				if (recext)
					str = recext->ref_url;
				break;
			case M_SPLIT_FIELD_DEFAULT:
				break;
			default:
				fprintf(stderr, "%s.%d: unknown type: %d\n", __FILE__, __LINE__, data->type);
			}
			
			if (ext_conf->debug_level > 2)
				fprintf(stderr, "%s.%d: -1- type: %d - %s\n", __FILE__, __LINE__, data->type, str);
			
			if (str != NULL) {
		/* do the test on the string */
				name = substitute(ext_conf, data->data.split.match, NULL, data->key, str);
#if 0
				/* this should happen whenever a split doesn't apply. */
				if (name == NULL) 
					fprintf(stderr, "%s.%d: substitute failed\n", __FILE__, __LINE__);
#endif
				
			} else if (data->data.split.fieldtype == M_SPLIT_FIELD_DEFAULT) {
		/* if a default is specified it is used when it occures */
				name = malloc(strlen(data->key)+1);
				strcpy(name, data->key);
				
				if (ext_conf->debug_level > 2)
					fprintf(stderr, "%s.%d: (def) state-name: %s\n", __FILE__, __LINE__, name);
			}
			
			if (name) 
				break;
			
			l = l->next;
		}
	} 
	
	if (split_enable == 0) {
		/* splitter isn't enabled, take a default name */
		name = malloc(1);
		*name = '\0';
	} 
	
	if (name) {
		/* we've got a name. try to find the list entry with this name */
		mlist *l = state_list;
		
		while (l) {
			mdata *data = l->data;
			if (!data) break;
			
			if (!strcmp(name, data->key)) {
				state = data->data.state.state;
				break;
			}
			
			l = l->next;
		}
		
		if (!state) {
			mdata *data = mdata_State_create(name, NULL, NULL);
			mlist_insert_sorted(state_list, data);
			
			state = data->data.state.state;
		}
		
		free(name);
	} else {
		fprintf(stderr, "%s.%d: no match found by the splitter. isn't there a default ??\n", __FILE__, __LINE__);
	}
	return state;
}

int mplugins_processor_web_insert_record(mconfig *ext_conf, mlist *state_list, mlogrec *record) {
	config_processor *conf = ext_conf->plugin_conf;
	struct tm *tm;
	int isvisit = 0, isfile = 0, ispage = 0, i;
#define TIMERS 6
	
	static mtimer t[TIMERS];
	static int init = 0;
/*	double l = 0;*/
	
	/* record extension */
	mlogrec_web *recweb = NULL;
	
	/* record web extensions */
	mlogrec_web_extclf *recext = NULL;
	mlogrec_web_squid *recsquid = NULL;
	mlogrec_web_ftp *recftp = NULL;
	
	mstate_web *staweb = NULL;
	mstate *state = NULL;
	
	mdata *data = NULL;

	if (record->ext_type != M_RECORD_TYPE_WEB) return -1;
	
	if (record->ext == NULL) return -1;

	recweb = record->ext;
	state = splitter(ext_conf, state_list, record);

	if (state == NULL) return -1;
	
	switch (recweb->ext_type) {
	case M_RECORD_TYPE_WEB_EXTCLF:
		recext = recweb->ext; break;
	case M_RECORD_TYPE_WEB_FTP:
		recftp = recweb->ext; break;
	case M_RECORD_TYPE_WEB_SQUID:
		recsquid = recweb->ext; break;
	}

	if (state->ext) {
		switch(state->ext_type) {
		case M_STATE_TYPE_WEB:
			staweb = state->ext; break;
		default:
			fprintf(stderr, "%s.%d: unsupport state subtype\n", __FILE__, __LINE__);
			return -1;
		}
	} else {
		state->ext = mstate_init_web();
		state->ext_type = M_STATE_TYPE_WEB;
		
		staweb = state->ext;
	}

	if (!init) {
		for (i = 0; i < TIMERS; i++) {
			MTIMER_RESET(t[i]);
		}
		init = 1;
	}

	MTIMER_START(t[0]);

	urltolower(recweb->req_url);
	urltolower(recweb->req_host);
	
	if (recext != NULL) urltolower(recext->ref_url);

/* skip ignored records */
	
	if (	ignore_field(ext_conf, recweb->req_url, M_WEB_IGNORE_REQ_URL) || 
		ignore_field(ext_conf, recweb->req_host, M_WEB_IGNORE_HOST )  ||
		ignore_field(ext_conf, recweb->req_host, M_WEB_IGNORE_HOSTMASK )  ||
		( recext != NULL &&
			ignore_field(ext_conf, recext->req_useragent, M_WEB_IGNORE_USERAGENT ))
		) {
		return 0;
	}
	
	MTIMER_START(t[1]);

/* hourly/daily stats */
	if ((tm = localtime(&(record->timestamp)))) {

		/* perhaps we have created a new state */
		if (!state->timestamp) {
			state->year	= tm->tm_year+1900;
			state->month	= tm->tm_mon+1;
		}
		state->timestamp = record->timestamp;
		
		staweb->hours[tm->tm_hour].xfersize += recweb->xfersize;
		staweb->days[tm->tm_mday-1].xfersize += recweb->xfersize;
		staweb->hours[tm->tm_hour].hits++;
		staweb->days[tm->tm_mday-1].hits++;
		if (is_page(ext_conf, recweb)) {
			staweb->hours[tm->tm_hour].pages++;
			staweb->days[tm->tm_mday-1].pages++;
			ispage = 1;
		}

		if (is_file(recweb) || recweb->ext_type == M_RECORD_TYPE_WEB_FTP) {
			staweb->hours[tm->tm_hour].files++;
			staweb->days[tm->tm_mday-1].files++;
			isfile = 1;
		}
		if (is_visit(ext_conf, state, record)) {
			staweb->hours[tm->tm_hour].visits++;
			staweb->days[tm->tm_mday-1].visits++;
			isvisit = 1;
		}
	}
	
	MTIMER_STOP(t[1]);
	MTIMER_CALC(t[1]);
	
/* rewrite the urls */
	


/* Used Protocol for this query*/	
	
	MTIMER_START(t[2]);
	if (recweb->req_protocol) {
		data = mdata_Count_create(recweb->req_protocol, 1, M_DATA_STATE_PLAIN);
		mhash_insert_sorted(staweb->req_prot_hash, data);
	}

/* User Method for this query (GET, POST, PUT, HEAD, OPTIONS) */	
	if (recweb->req_method) {
		data = mdata_Count_create(recweb->req_method, 1, M_DATA_STATE_PLAIN);
		mhash_insert_sorted(staweb->req_meth_hash, data);
	}
	
	if (recweb->req_status && recweb->req_url) {
		char buf[4];
		
		sprintf(buf, "%3d", recweb->req_status);
		
		data = mdata_Count_create(buf, 1, M_DATA_STATE_PLAIN);
		mhash_insert_sorted(staweb->status_hash, data);

/* FIXME: specific to HTTP */
		switch (recweb->req_status) {
			case 404: 
				if (!hide_field(ext_conf, recweb->req_url, M_WEB_HIDE_BROKENLINK)) {
					char *grouped;
					mdata *link;
					if ((grouped = group_field(ext_conf, recweb->req_url, M_WEB_GROUP_BROKENLINK))) {
						link = mdata_BrokenLink_create(grouped, 1, M_DATA_STATE_GROUPED, record->timestamp, recext != NULL ? recext->ref_url : NULL);
						mhash_insert_sorted(staweb->status_missing_file, link);

						free(grouped);
					} else {
						link = mdata_BrokenLink_create(recweb->req_url, 1, M_DATA_STATE_PLAIN, record->timestamp, recext != NULL ? recext->ref_url : NULL);
						mhash_insert_sorted(staweb->status_missing_file, link);
					}
				}
				break;
			case 500:
			{
				mdata *link;
				
				link = mdata_BrokenLink_create(recweb->req_url, 1, M_DATA_STATE_PLAIN, record->timestamp, recext != NULL ? recext->ref_url : NULL);
				mhash_insert_sorted(staweb->status_internal_error, link);
				break;
			}
		} 
	}
	
	MTIMER_STOP(t[2]);
	MTIMER_CALC(t[2]);
	
	MTIMER_START(t[3]);

	if (recweb->req_host && !hide_field(ext_conf, recweb->req_host, M_WEB_HIDE_HOST)) {
		char *grouped;
		char *req_hostname = recweb->req_host;

		/* try to resolve the ip */
		if ((grouped = strrchr(recweb->req_host, '.'))) {
			/* check if we have to resolve the IP */
			if (isondx(grouped+1) == M_RESOLV_UNRESOLVED) {
				/* we have to ask he resolver for an FQDN */
#ifdef HAVE_LIBADNS
				adns_answer *answer = NULL;
				mdata *query;
				
				if (conf->debug_resolver)
					fprintf(stderr, "%s.%d: resolve %-15s -- ", __FILE__, __LINE__, recweb->req_host);

				if (ext_conf->enable_resolver) {
					if ((query = mhash_get_data(ext_conf->query_hash, recweb->req_host))) {
						if (!query->data.query.hostname) {
							adns_wait(*(ext_conf->adns), query->data.query.query, &answer, NULL);
				
							if (answer->status == adns_s_ok) {
								query->data.query.hostname = malloc(strlen(*answer->rrs.str)+1);
								strcpy(query->data.query.hostname, *answer->rrs.str);
							
								if (conf->debug_resolver)
									fprintf(stderr, "cache miss ");
							} else {
								if (conf->debug_resolver)
									fprintf(stderr, "error      ");
								
								query->data.query.hostname = malloc(strlen(recweb->req_host)+1);
								strcpy(query->data.query.hostname, recweb->req_host);
							}
							free(answer);
						} else {
							if (conf->debug_resolver)
								fprintf(stderr, "cache hit  ");
						}
					
						if (conf->debug_resolver)
							fprintf(stderr, "--> %s\n", query->data.query.hostname);

						req_hostname = query->data.query.hostname;
					}
				}
#endif
			}
		}

		/*
		 * NOTE: (FIXME)
		 *   the following grouping affects the "number of hosts" as most of the 
		 *   hosts are grouped. Perhaps we should lists of hosts which are in a 
		 *   specific group
		 */
		
		/* check the cache for the hostname */
		grouped = NULL;
		for (i = 0; i < conf->host_cache_max; i++) {
			if (conf->host_cache->entry[i]->key && 0 == strcmp(req_hostname, conf->host_cache->entry[i]->key)) {
				grouped = conf->host_cache->entry[i]->value;
				break;
			}
		}
		
		if (grouped) {
			data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, isvisit);
		} else if ((grouped = group_field(ext_conf, req_hostname, M_WEB_GROUP_HOST))) {
			int oldest = -1;
			int oldest_tstmp = conf->host_cache->last_tstmp;
			data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, isvisit);
			
			/* insert into the cache */
			for (i = 0; i < conf->host_cache_max; i++) {
				if (conf->host_cache->entry[i]->tstmp < oldest_tstmp) {
					oldest = i;
					oldest_tstmp = conf->host_cache->entry[i]->tstmp;
				}
			}
			
			/* free the oldest entry and insert */
			if (oldest != -1) {
				if (conf->host_cache->entry[oldest]->key) free(conf->host_cache->entry[oldest]->key);
				if (conf->host_cache->entry[oldest]->value) free(conf->host_cache->entry[oldest]->value);
				conf->host_cache->entry[oldest]->key = malloc(strlen(req_hostname) + 1);
				strcpy(conf->host_cache->entry[oldest]->key, req_hostname);
				
				conf->host_cache->entry[oldest]->value = grouped;
				conf->host_cache->entry[oldest]->tstmp = conf->host_cache->last_tstmp++;
			} else {
				free(grouped);
			}
		} else {
			data = mdata_Visited_create(req_hostname, 1, M_DATA_STATE_PLAIN, isvisit);
		}
		mhash_insert_sorted(staweb->host_hash, data);

		/* splitting the TLD from the FQDN */
		if ((grouped = strrchr(req_hostname, '.'))) {
			if (misoname(grouped+1)) {
				data = mdata_Visited_create(isondx(grouped+1) != M_RESOLV_UNRESOLVED ?
							    grouped+1 : UNRESOLVED_TLD,
							    1, 
							    M_DATA_STATE_PLAIN, 
							    isvisit);
				
				mhash_insert_sorted(staweb->country_hash, data);
			}
		}
	}
	MTIMER_STOP(t[3]);
	MTIMER_CALC(t[3]);
	
	MTIMER_START(t[4]);
	
	if (recweb->req_url) {
		mdata *data;
		char *c1;
		
		/*
		 * NOTE: (FIXME)
		 *   favicon.ico is requested by other browser then MSIE too
		 *   They are not using it exclusivly for bookmarking. 
		 */

		if ((c1 = strstr(recweb->req_url, "favicon.ico")) ) {
			char c2 = *c1;
			*c1 = '\0';
			data = mdata_Count_create(recweb->req_url, 1, M_DATA_STATE_PLAIN);
			mhash_insert_sorted(staweb->bookmarks, data);
			*c1 = c2;
		} else if (is_robot(recweb->req_url)) {
			if (recext == NULL || recext->req_useragent == NULL) {
				/* worth nothing */
				if (ext_conf->debug_level > 10)
					fprintf(stderr, "%s requested w/o useragent set - ignored.\n",recweb->req_url);
			} else {
				data = mdata_Count_create(recext->req_useragent, 1, M_DATA_STATE_PLAIN);
				mhash_insert_sorted(staweb->robots, data);
			}
		}
		
		/* file extensions */
		if (recweb->req_status != 404 && 
		    !hide_field(ext_conf, recweb->req_url, M_WEB_HIDE_EXTENSION)) {
			char *grouped;
			if ((grouped = group_field(ext_conf, recweb->req_url, M_WEB_GROUP_EXTENSION))) {
				data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, recweb->xfersize);
				mhash_insert_sorted(staweb->extension, data);
				free(grouped);
			} else {
				if (ext_conf->debug_level > 5)
					fprintf(stderr,"%s.%d: the default rule for groupextension is missing for '%s'!!\n",
						__FILE__, __LINE__, recweb->req_url);
			}
		}
		
		/* hide url */
		if (!hide_field(ext_conf, recweb->req_url, M_WEB_HIDE_REQ_URL)) {
			char *grouped;
			if ((grouped = group_field(ext_conf, recweb->req_url, M_WEB_GROUP_REQ_URL))) {
				data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, recweb->xfersize);
				mhash_insert_sorted(staweb->req_url_hash, data);
				free(grouped);
			} else {
				data = mdata_Visited_create(recweb->req_url, 1, M_DATA_STATE_PLAIN, recweb->xfersize);
				mhash_insert_sorted(staweb->req_url_hash, data);
			}
		}
	}
	
	MTIMER_STOP(t[4]);
	MTIMER_CALC(t[4]);

/*
** Extensions
*/

	MTIMER_START(t[5]);
/* User Operating System */
	if (recext != NULL) {
		
		if (recext->req_useros) {
			char *grouped;
			if ((grouped = group_field(ext_conf, recext->req_useros, M_WEB_GROUP_OS))) {
				data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, isvisit);
				mhash_insert_sorted(staweb->os_hash, data);
				free(grouped);
			} else {
				data = mdata_Visited_create(recext->req_useros, 1, M_DATA_STATE_PLAIN, isvisit);
				mhash_insert_sorted(staweb->os_hash, data);
			}
		}

/* User Agent */
		if (recext->req_useragent) {
			char *grouped;
			if ((grouped = group_field(ext_conf, recext->req_useragent, M_WEB_GROUP_UA))) {
				data = mdata_Visited_create(grouped, 1, M_DATA_STATE_GROUPED, isvisit);
				mhash_insert_sorted(staweb->ua_hash, data);
				free(grouped);
			} else {
				data = mdata_Visited_create(recext->req_useragent, 1, M_DATA_STATE_PLAIN, isvisit);
				mhash_insert_sorted(staweb->ua_hash, data);
			}
		}
	
		if (recext->ref_url) {
			if (!hide_field(ext_conf, recext->ref_url, M_WEB_HIDE_REFERRER)) {
				if (!is_searchengine(ext_conf, state, recext)) {
					char *grouped;
					if ((grouped = group_field(ext_conf, recext->ref_url,M_WEB_GROUP_REFERRER))) {
						data = mdata_Count_create(grouped, 1, M_DATA_STATE_GROUPED);
						mhash_insert_sorted(staweb->ref_url_hash, data);
						free(grouped);
					} else {
						data = mdata_Count_create(recext->ref_url, 1, M_DATA_STATE_PLAIN);
						mhash_insert_sorted(staweb->ref_url_hash, data);
					}
				}
			}
		}
		
		if (recext->srv_host) {
			data = mdata_Visited_create(recext->srv_host, 1, M_DATA_STATE_PLAIN, isvisit);
			mhash_insert_sorted(staweb->vhost_hash, data);
		}
	}

	MTIMER_STOP(t[5]);
	MTIMER_CALC(t[5]);
	
	MTIMER_STOP(t[0]);
	MTIMER_CALC(t[0]);
	
	if (conf->debug_timing) {
		fprintf(stderr, "--> ");
		for (i = 0; i < TIMERS; i++) {
			fprintf(stderr, "%ld ", MTIMER_GET_MSEC(t[i]));
		}
		fprintf(stderr, "\n");
	}
#undef TIMERS
	return 0;
}



