/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: mrecord.h,v 1.22 2002/01/03 10:31:14 ostborn Exp $
*/

#ifndef _M_RECORD_H_
#define _M_RECORD_H_

#include <time.h>
#include "config.h"

#define M_RECORD_NO_ERROR	0
#define M_RECORD_EOF		-1
#define M_RECORD_SKIPPED	1
#define M_RECORD_CORRUPT        2
#define M_RECORD_IGNORED        3
#define M_RECORD_HARD_ERROR     4


/* a record
** --------
** a record is chain if different struct's 
** mlogrec -> mlogrec_web -> mlogrec_clf_extended
**                        -> mlogrec_ftp
**                        -> mlogrec_squid
**
** each '->' in this picture is name 'ext' in the struct
** the type of each dereferenced struct is carried in 'ext_type' as an int
** which is one of the M_RECORD_TYPE's below.
*/

#define M_RECORD_TYPE_UNSET		0
#define M_RECORD_TYPE_WEB		1
#define M_RECORD_TYPE_TELECOM		2
#define M_RECORD_TYPE_TRAFFIC		3
#define M_RECORD_TYPE_MAIL              4

#define M_RECORD_TYPE_WEB_UNSET		0
#define M_RECORD_TYPE_WEB_FTP		1
#define M_RECORD_TYPE_WEB_EXTCLF	2
#define M_RECORD_TYPE_WEB_SQUID		3

#define M_RECORD_TYPE_TELECOM_UNSET	0
#define M_RECORD_TYPE_TELECOM_INTERNAL	1

#define M_RECORD_TYPE_TRAFFIC_UNSET	0
#define M_RECORD_TYPE_TRAFFIC_FLOW	1
#define M_RECORD_TYPE_TRAFFIC_IPCHAINS	2

#define M_RECORD_TYPE_MAIL_UNSET        0
#define M_RECORD_TYPE_MAIL_QMAIL_STATUS 1
#define M_RECORD_TYPE_MAIL_VIRUS        2

typedef struct {
/* required fields (required by the main loop) */
	time_t	timestamp;		/**< (req) timestamp */ 
	
	int	ext_type; 		/**< (int) */
	void	*ext; 			/**< (int) */
} mlogrec;
	
typedef struct {
	char	*req_host;		/**< (req) requesting host */ 
	char	*req_user;		/**< (opt) ... user */ 
	char	*req_protocol;		/**< (req) protocol used for this request*/ 
	char	*req_url;		/**< (req) ... url w/o http get vars*/ 
	int	req_status;		/**< (req) status of the request */ 
	double	xfersize;		/**< (req) transfered bytes */ 
	
	char	*req_method;		/**< (opt) */
	char	*req_getvars;		/**< (opt) */
	
	int	ext_type; 		/**< (int) */
	void	*ext;			/**< (int) */
} mlogrec_web;

typedef struct {
	char	*ref_url;		/**< where the user was before */
	char	*ref_getvars;

	char	*req_useragent;		/**< what the user is using */
	char	*req_useros;	
	
	char	*srv_host;		/**< which server is serving this request */
	char	*srv_port;

	time_t	duration;		/**< duration of transfer (mostly for streaming) */
} mlogrec_web_extclf;

enum { M_RECORD_FTP_COMMAND_UNSET, M_RECORD_FTP_COMMAND_PUT, M_RECORD_FTP_COMMAND_GET,
		M_RECORD_FTP_COMMAND_DELETE, M_RECORD_FTP_COMMAND_MKDIR,
		M_RECORD_FTP_COMMAND_RMDIR 
};
		

#define M_RECORD_FTP_MODE_UNSET		0
#define M_RECORD_FTP_MODE_ASCII		1
#define M_RECORD_FTP_MODE_BINARY	2
typedef struct {
	char	*req_group;
	int	trans_command;
	time_t	trans_duration;
	int	trans_mode;
} mlogrec_web_ftp;

#define M_RECORD_SQUID_LOG_UNSET		0
#define M_RECORD_SQUID_LOG_TCP_HIT		1
#define M_RECORD_SQUID_LOG_TCP_MISS		2
#define M_RECORD_SQUID_LOG_TCP_REFRESH_HIT	3
#define M_RECORD_SQUID_LOG_TCP_REF_FAIL_HIT	4
#define M_RECORD_SQUID_LOG_TCP_REFRESH_MISS	5
#define M_RECORD_SQUID_LOG_TCP_CLIENT_REFRESH	6
#define M_RECORD_SQUID_LOG_TCP_IMS_HIT		7
#define M_RECORD_SQUID_LOG_TCP_IMS_MISS		8
#define M_RECORD_SQUID_LOG_TCP_SWAPFAIL		9
#define M_RECORD_SQUID_LOG_TCP_DENIED		10
#define M_RECORD_SQUID_LOG_UDP_HIT		11
#define M_RECORD_SQUID_LOG_UDP_HIT_OBJ		12
#define M_RECORD_SQUID_LOG_UDP_MISS		13
#define M_RECORD_SQUID_LOG_UDP_DENIED		14
#define M_RECORD_SQUID_LOG_UDP_INVALID		15
#define M_RECORD_SQUID_LOG_UDP_RELOADING	16

#define M_RECORD_SQUID_DATA_UNSET		0
#define M_RECORD_SQUID_DATA_DIRECT		1
#define M_RECORD_SQUID_DATA_FIREWALL_IP_DIRECT	2
#define M_RECORD_SQUID_DATA_FIRST_PARENT_MISS	3
#define M_RECORD_SQUID_DATA_FIRST_UP_PARENT	4
#define M_RECORD_SQUID_DATA_LOCAL_IP_DIRECT	5
#define M_RECORD_SQUID_DATA_SIBLING_HIT		6
#define M_RECORD_SQUID_DATA_NO_DIRECT_FAIL	7
#define M_RECORD_SQUID_DATA_NO_PARENT_DIRECT	8
#define M_RECORD_SQUID_DATA_PARENT_HIT		9
#define M_RECORD_SQUID_DATA_SINGLE_PARENT	10
#define M_RECORD_SQUID_DATA_SOURCE_FASTEST	11
#define M_RECORD_SQUID_DATA_PARENT_UDP_HIT_OBJ	12
#define M_RECORD_SQUID_DATA_SIBLING_UDP_HIT_OBJ	13
#define M_RECORD_SQUID_DATA_PASSTHROUGH_PARENT	14
#define M_RECORD_SQUID_DATA_SSL_PARENT_MISS	15
#define M_RECORD_SQUID_DATA_DEFAULT_PARENT	16
#define M_RECORD_SQUID_DATA_ROUNDROBIN_PARENT	17
#define M_RECORD_SQUID_DATA_CLOSEST_PARENT_MISS	18
#define M_RECORD_SQUID_DATA_CLOSEST_DIRECT	19

typedef struct {
	int	log_tag;
	int	data_tag;
} mlogrec_web_squid;

#define M_RECORD_TELECOM_DIRECTION_UNSET	0
#define M_RECORD_TELECOM_DIRECTION_IN		1
#define M_RECORD_TELECOM_DIRECTION_OUT		2

typedef struct {
	char 	*called_number;		/**< destination */
	char	*calling_number;	/**< source */
	int 	direction;		/**< incomming/outgoing call */
	time_t	duration;		/**< duration of the call */
	
	int	ext_type;
	void	*ext;
} mlogrec_telecom;

typedef struct {
	char	*user_id;		/**< for pin protected phones */
	char	*provider;		/**< used provider */
	int	units_to_pay;		/**< the german telekom varies the time/unit not the price/unit */
} mlogrec_telecom_internal;

/* taken from /etc/protocols */
#define M_RECORD_TRAFFIC_PROTOCOL_UNSET	        0
#define M_RECORD_TRAFFIC_PROTOCOL_ICMP		1
#define M_RECORD_TRAFFIC_PROTOCOL_IGMP		2
#define M_RECORD_TRAFFIC_PROTOCOL_GGP		3
#define M_RECORD_TRAFFIC_PROTOCOL_TCP		6
#define M_RECORD_TRAFFIC_PROTOCOL_EGP		8
#define M_RECORD_TRAFFIC_PROTOCOL_PARP		12
#define M_RECORD_TRAFFIC_PROTOCOL_UDP		17
#define M_RECORD_TRAFFIC_PROTOCOL_HMP		20
#define M_RECORD_TRAFFIC_PROTOCOL_IDP		22
#define M_RECORD_TRAFFIC_PROTOCOL_RDP		27


typedef struct {
	char *src;
	char *dst;
	unsigned long xfer_incomming;
	unsigned long xfer_outgoing;
	
	int	ext_type;
	void	*ext;
} mlogrec_traffic;

typedef struct {
	int	protocol;
	unsigned int	packets;
	int     src_port;
	int     dst_port;
	int     src_as;
	int     dst_as;
} mlogrec_traffic_flow;

#define M_RECORD_IPCHAINS_ACTION_UNSET	0
#define M_RECORD_IPCHAINS_ACTION_ACCEPT	1
#define M_RECORD_IPCHAINS_ACTION_DENY	2
#define M_RECORD_IPCHAINS_ACTION_REJECT	3
#define M_RECORD_IPCHAINS_ACTION_MASQ	4

typedef struct {
	int	protocol;
	char	*device;
	char	*host;
	char	*chain;
	int	action;
	int	src_port;
	int	dst_port;
	int	rule;
} mlogrec_traffic_ipchains;

typedef struct {
	char	*receipient;		/**< (req) receipient */ 
	char	*sender;		/**< (req) sender */ 
	time_t	duration;		/**< (req) durection */ 
	long	bytes_in;		/**< (req) traffic incomming */ 
	long	bytes_out;		/**< (req) traffic outgoing */ 
	
	int status_dsn;
	int status_smtp;
	char *status_text;
	
	int	ext_type; 		/**< (int) */
	void	*ext;			/**< (int) */
} mlogrec_mail;

typedef struct {
	int local_cur;
	int local_max;
	
	int remote_cur;
	int remote_max;
	
	int deliver_cur;
	int queue_cur;
} mlogrec_mail_qmail_status;

typedef struct {
	char *scanner;
	char *virus;
	char *subject;
} mlogrec_mail_virus;

/* init */
mlogrec			*mrecord_init();

mlogrec_web		*mrecord_init_web();
mlogrec_web_extclf	*mrecord_init_web_extclf();
mlogrec_web_ftp		*mrecord_init_web_ftp();
mlogrec_web_squid	*mrecord_init_web_squid();

mlogrec_telecom		*mrecord_init_telecom();
mlogrec_telecom_internal *mrecord_init_telecom_internal();

mlogrec_traffic		*mrecord_init_traffic();
mlogrec_traffic_flow	*mrecord_init_traffic_flow();
mlogrec_traffic_ipchains *mrecord_init_traffic_ipchains();

mlogrec_mail            *mrecord_init_mail();
mlogrec_mail_qmail_status *mrecord_init_mail_qmail_status();
mlogrec_mail_virus *mrecord_init_mail_virus();

/* free */
void mrecord_free(mlogrec *record);

void mrecord_free_web(mlogrec_web *record);
void mrecord_free_web_extclf(mlogrec_web_extclf *record);
void mrecord_free_web_ftp(mlogrec_web_ftp *record);
void mrecord_free_web_squid(mlogrec_web_squid *record);

void mrecord_free_telecom(mlogrec_telecom *record);
void mrecord_free_telecom_internal(mlogrec_telecom_internal *record);

void mrecord_free_traffic(mlogrec_traffic *record);
void mrecord_free_traffic_flow(mlogrec_traffic_flow *record);
void mrecord_free_traffic_ipchains(mlogrec_traffic_ipchains *record);

void mrecord_free_mail(mlogrec_mail * record);
void mrecord_free_mail_qmail_status(mlogrec_mail_qmail_status * record);
void mrecord_free_mail_virus(mlogrec_mail_virus * record);

/* reset */
void mrecord_reset(mlogrec *record);

/* copy */

int mrecord_copy_web_extclf(mlogrec_web_extclf *dst, mlogrec_web_extclf *src);
int mrecord_copy_web_squid(mlogrec_web_squid *dst, mlogrec_web_squid *src);
int mrecord_copy_web(mlogrec_web *dst, mlogrec_web *src);
int mrecord_copy(mlogrec *dst, mlogrec *src);

int mrecord_move(mlogrec *dst, mlogrec *src);

#endif
