/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: plugin_config.c,v 1.24 2002/01/01 11:43:00 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

int mplugins_input_clf_dlinit(mconfig *ext_conf) {
	config_input *conf = NULL;
	const char *errptr;
	int erroffset = 0;
	
	conf = malloc(sizeof(config_input));
	memset(conf, 0, sizeof(config_input));
	
	conf->match_os = mlist_init();
	conf->match_ua = mlist_init();
	conf->record_list = mlist_init();
	conf->inputfilename = NULL;

	/* will be replaced by setdefaults if we have a real inputfilename */
	conf->inputfile = stdin;
	conf->buf_len = 256;
	conf->buf_inc = 128;
	conf->buffer = malloc(conf->buf_len * sizeof(char));
	
	if ((conf->match_url = pcre_compile(
		/* method     url               protocol */
		"^([A-Za-z]+) (.+?(?:\\?(.*?))?)(?: +(.*?)|)$", 
		0, &errptr, &erroffset, NULL)) == NULL) {
		
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);
		return -1;
	} 
	
	if ((conf->match_referrer = pcre_compile(
		/*page     get-vars */
		"^(.*?)(\\?(.*?))*$", 
		0, &errptr, &erroffset, NULL)) == NULL) {
		
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);
		return -1;
	} 
	
	if ((conf->match_timestamp = pcre_compile(
		"^([ 0-9]{2})/([a-zA-Z]{3})/([0-9]{4}):([0-9]{2}):([0-9]{2}):([0-9]{2})", 
		0, &errptr, &erroffset, NULL)) == NULL) {
		
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);
		return -1;
	} 
	
	conf->match_timestamp_extra = pcre_study(conf->match_timestamp, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);
		return -1;
	}
	conf->match_url_extra = pcre_study(conf->match_url, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);
		return -1;
	}
	
	conf->match_clf = NULL;
	conf->match_clf_squid = NULL;
	conf->match_clf_extended = NULL;
	conf->match_clf_extra = NULL;
	conf->match_clf_squid_extra = NULL;
	conf->match_clf_extended_extra = NULL;
	conf->match_referrer_extra = NULL;
	
	ext_conf->plugin_conf = conf;
	
	return 0;
}

int mplugins_input_clf_dlclose(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;
	
	if (conf->inputfilename) 
		free(conf->inputfilename);
	
	if (conf->inputfile && conf->inputfile != stdin) 
		fclose(conf->inputfile);
	
	mlist_free(conf->match_os);
	mlist_free(conf->match_ua);
	mlist_free(conf->record_list);
	
	if (conf->buffer) free(conf->buffer);
	if (conf->format) free(conf->format);
	
	if (conf->match_url) free(conf->match_url);
	if (conf->match_referrer) free(conf->match_referrer);
	if (conf->match_timestamp) free(conf->match_timestamp);
	
	free(ext_conf->plugin_conf);
	ext_conf->plugin_conf = NULL;
	
	return 0;
}

int mplugins_input_clf_parse_config(mconfig *ext_conf, const char *filename, const char *section) {
	config_input *conf = ext_conf->plugin_conf;
	
	const mconfig_values config_values[] = {
		{"matchua",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->match_ua)},
		{"matchos",	M_CONFIG_TYPE_MATCH,	M_CONFIG_VALUE_APPEND, &(conf->match_os)},
		{"inputfile",	M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->inputfilename)},
		{"readaheadlimit",	M_CONFIG_TYPE_INT,	M_CONFIG_VALUE_OVERWRITE, &(conf->read_ahead_limit)},
		{"format",	M_CONFIG_TYPE_STRING,	M_CONFIG_VALUE_OVERWRITE, &(conf->format)},
		
		{NULL, M_CONFIG_TYPE_INT, 0, NULL}
	};
	
	return mconfig_parse_section(ext_conf, filename, section, config_values);
}

int parse_clf_field_info(mconfig *ext_conf, const char *_buffer) {
	config_input *conf = ext_conf->plugin_conf;
	char *buf, *pa, *pe;
	int pos = 0, i;
	char *match_buf;
	const char *errptr;
	int erroffset = 0;
	
	if (_buffer == NULL) return -1;
	
	if ((buf = malloc(strlen(_buffer)+1)) == NULL) {
		return -1;
	}
	
	strcpy(buf, _buffer);
	
	pa = buf;
	pe = NULL;
	
	while ((pe = strchr(pa, ' ')) != NULL) {
		*pe = '\0';
		
		for (i = 0; def[i].field != NULL; i++) {
			if (strcmp(def[i].field, pa) == 0) {
				break;
			}
		}
		
		if (def[i].field != NULL) {
			if (pos >= M_CLF_MAX_FIELDS) return -1;
			
			conf->trans_fields[pos++] = i;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "Unknown fieldtype: %s\n", pa);
			free(buf);
			return -1;
		}
		
		pa = pe + 1;
	}
	
	/* don't forget the last param */
	if (*pa) {
		for (i = 0; def[i].field != NULL; i++) {
			if (strcmp(def[i].field, pa) == 0) {
				break;
			}
		}
	
		if (def[i].field != NULL) {
			if (pos >= M_CLF_MAX_FIELDS) return -1;
			conf->trans_fields[pos++] = i;
		} else {
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "Unknown fieldtype: %s\n", pa);
			free(buf);
			return -1;
		}
	}
	
	free(buf);
	
	match_buf = malloc(1024);
	
	*match_buf = '\0';
	
	for (i = 0; i < pos; i++) {
		if (*match_buf == '\0') {
			match_buf = strcat(match_buf, "^");
		} else {
			match_buf = strcat(match_buf, " ");
		}
		match_buf = strcat(match_buf, def[conf->trans_fields[i]].match);
	}
	
	match_buf = strcat(match_buf, "\\s*$");
	
	M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
		"match = %s\n", match_buf);
	
	if ((conf->match_clf = pcre_compile(match_buf,
		0, &errptr, &erroffset, NULL)) == NULL) {
		
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp compilation error at %s\n", errptr);

		free(match_buf);
		return -1;
	} 
	free(match_buf);
	
	conf->match_clf_extra = pcre_study(conf->match_clf, 0, &errptr);
	if (errptr != NULL) {
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "regexp studying error at %s\n", errptr);
		return -1;
	}
	
	return 0;
}


int mplugins_input_clf_set_defaults(mconfig *ext_conf) {
	config_input *conf = ext_conf->plugin_conf;
	
	if (conf->inputfilename && strcmp(conf->inputfilename, "-") != 0) {
		if (!(conf->inputfile = fopen(conf->inputfilename, "r"))) {
			M_DEBUG2(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "%s: %s\n", conf->inputfilename, strerror(errno));
			return -1;
		}
		M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(clf) using %s as inputfile\n", conf->inputfilename);
	} else {
		M_DEBUG0(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_VERBOSE,
			 "(clf) using (stdin) as inputfile\n");
	}
	
	if (conf->read_ahead_limit < 1) {
		conf->read_ahead_limit = 1;
	}
	
	if (!conf->format) {
		const char *errptr;
		int erroffset = 0;
		if ((conf->match_clf = pcre_compile(
			"^(.*?) (.*?) (.*?) \\[(.*?)\\] \"(.*?)\" ([-0-9]{1,3}) ([-0-9]+)( \"(.*?)\" \"(.*?)\"| ([A-Z:_]+?)|)\\s*$", 
			0, &errptr, &erroffset, NULL)) == NULL) {
		
			M_DEBUG1(ext_conf->debug_level, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "regexp compilation error at %s\n", errptr);
			return -1;
		} 
	} else {
		if (parse_clf_field_info(ext_conf, conf->format)) {
			return -1;
		}
	}
	
	return 0;
}

int mplugins_init(mplugin *func) {
	func->dlinit = mplugins_input_clf_dlinit;
	func->dlclose = mplugins_input_clf_dlclose;
	func->parse_config = mplugins_input_clf_parse_config;
	func->set_defaults = mplugins_input_clf_set_defaults;
	func->get_next_record = mplugins_input_clf_get_next_record;
	func->insert_record = NULL;
	func->gen_report = NULL;
        func->gen_history = NULL;
	
	return 0;
}
