/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: datatype.c,v 1.14 2001/12/23 00:12:22 ostborn Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <zlib.h>
#include <libxml/tree.h>

#include "config.h"
#include "mdatatypes.h"
#include "datatype.h"
#include "misc.h"

/* begin of BrokenLink */

int mdata_BrokenLink_to_xml(gzFile *fd, mdata *data) {
#define WR(x,f) \
	gzprintf(fd, "<%s>%"#f"</%s>\n", #x, data->data.brokenlink.x, #x);
#define WRS(x,f) \
	if (data->data.brokenlink.x != NULL) { \
		char *s = url_encode(data->data.brokenlink.x); \
		gzprintf(fd, "<%s>%"#f"</%s>\n", #x, s, #x); \
		free(s); \
	} else { \
		gzprintf(fd, "<%s />", #x); \
	}
	
	WR(count, d);
	WR(grouped, d);
	WR(timestamp, ld);
	WRS(referrer, s);
#undef WRS
#undef WR
	
	return 0;
}

int mdata_BrokenLink_free(mdata *data) {
	if (!data) return -1;
	if (data->type != M_DATA_TYPE_BROKENLINK) return -1;
	
	if (data->data.brokenlink.referrer)	free(data->data.brokenlink.referrer);
	
	return 0;
}

int mdata_BrokenLink_setdata(mdata *data, const char *str, int count, int type,
	time_t timestamp, const char *referrer) {
	
	assert(str);
	
	data->key	= malloc(strlen(str)+1);
	assert(data->key);
	strcpy(data->key, str);
	
	if (data->type == M_DATA_TYPE_UNSET) 
		data->type		= M_DATA_TYPE_BROKENLINK;
	
	data->data.brokenlink.count	= count;
	data->data.brokenlink.grouped	= type;
	data->data.brokenlink.timestamp = timestamp;
	
	
	if (referrer) {
		data->data.brokenlink.referrer = malloc(strlen(referrer) + 1);
		assert(data->data.brokenlink.referrer);
		strcpy(data->data.brokenlink.referrer, referrer);
	} else {
		data->data.brokenlink.referrer = NULL;
	}
	
	return 0;
}

int mdata_BrokenLink_from_xml(mstate_stack *m, int tagtype, mdata *data, int type, const xmlChar *value, const xmlChar **attrs) {
	char *referrer = NULL;
	int i;
	const mdata_values data_values[] = {
		{ "grouped", M_DATA_FIELDTYPE_LONG, 	&(data->data.brokenlink.grouped) },
		{ "count", M_DATA_FIELDTYPE_LONG, 	&(data->data.brokenlink.count) },
		{ "timestamp", M_DATA_FIELDTYPE_LONG, 	&(data->data.brokenlink.timestamp) },
		{ "referrer", M_DATA_FIELDTYPE_STRING, 	&(referrer) },
		
		{ NULL, M_DATA_FIELDTYPE_UNSET,  NULL}
	};
	
#if 0	
	M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
		 "|--> %s - %d\n", 
		 value, tagtype);
#endif 
	switch(tagtype) {
	case M_TAG_BEGIN:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
			return -1;
		} else {
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			if (data_values[i].string != NULL) {
				m->st_depth_max++;
				m->st[++m->st_depth].id = i + 1;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_END:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
			return -1;
		} else {
			for (i = 0; data_values[i].string != NULL && 0 != strcmp(data_values[i].string, value); i++)
				;
			if (data_values[i].string != NULL) {
				data->data.brokenlink.referrer = url_decode(referrer);
	
				/* set data */
				if (data->type == M_DATA_TYPE_UNSET) 
					data->type		= M_DATA_TYPE_BROKENLINK;
				
				m->st[m->st_depth].id = 0;
				m->st_depth_max--;
			} else {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "unknown tag '%s'\n", 
					 value);
				return -1;
			}
		}
		break;
	case M_TAG_TEXT:
		if (m->st_depth != m->st_depth_max) {
			M_DEBUG3(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
				 "cur(depth) != max(depth) [%d - %d]- not my job (%s)\n",
				 m->st_depth, 
				 m->st_depth_max,
				 value);
			return -1;
		} else {
			if (mdata_insert_value(m, tagtype, 
					       data_values[m->st[m->st_depth].id - 1].dest, 
					       data_values[m->st[m->st_depth].id - 1].type, 
					       value, attrs)) {
				M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "insert failed for '%s'\n", 
					 value);
				return -1;
			} else {
#if 0
				M_DEBUG2(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
					 "inserting '%s' for '%s'\n", 
					 value, data_values[m->st[m->st_depth] - 1].string);
#endif
			}
		}
		break;
	default:
		M_DEBUG1(M_DEBUG_LEVEL_ERRORS, M_DEBUG_SECTION_INIT, M_DEBUG_LEVEL_ERRORS,
			 "can't handle tagtype '%d'\n", 
			 type);
		return -1;
	}

	return 0;
}



int mdata_BrokenLink_append(mdata *dst, mdata *src) {
	dst->data.brokenlink.count += src->data.brokenlink.count;
	return M_DATA_APPENDED;
}

mdata *mdata_BrokenLink_init() {
	mdata *data = mdata_init();
	assert(data);
	
	data->key		= NULL;
	data->type		= M_DATA_TYPE_BROKENLINK;
	
#ifdef DEBUG_DATATYPES
	fprintf(stderr, "%s.%d: got %ld (%d)\n", __FILE__, __LINE__, data->id, data->type);
#endif
/* init specifics */	
	data->data.brokenlink.count	= 0;
	data->data.brokenlink.grouped	= M_DATA_STATE_PLAIN;
	data->data.brokenlink.timestamp	= 0;
	data->data.brokenlink.referrer	= NULL;
	
	
	return data;
}

int mdata_BrokenLink_show(const mdata *data) {
	if (!data) return -1;
	if (!data->type == M_DATA_TYPE_BROKENLINK) return -1;
	
	        fprintf(stderr, "* datatype: brokenlink\n");
	
	if (data->key == NULL) {
		fprintf(stderr, "* key     : (unset)\n");
	} else {
		fprintf(stderr, "* key     : %s\n", data->key);
		fprintf(stderr, "* count   : %d\n", data->data.brokenlink.count);
		fprintf(stderr, "* grouped : %d\n", data->data.brokenlink.grouped);
		fprintf(stderr, "* tstamp  : %ld\n", data->data.brokenlink.timestamp);
		fprintf(stderr, "* referrer: %s\n", data->data.brokenlink.referrer);
	}
	
	return 0;
}

mdata *mdata_BrokenLink_copy(mdata *src) {
	mdata *data = mdata_BrokenLink_init();
	assert(data);
	mdata_BrokenLink_setdata(data, src->key, 
		src->data.brokenlink.count, 
		src->data.brokenlink.grouped,
		src->data.brokenlink.timestamp,
		src->data.brokenlink.referrer
	);
	
	return data;
}

mdata *mdata_BrokenLink_create(const char *str, 
	int count, int grouped,
	time_t timestamp, const char *referrer) {
	mdata *data = mdata_BrokenLink_init();
	assert(data);
	mdata_BrokenLink_setdata(data, str, count, grouped, timestamp, referrer);
	
	return data;
}

/* end of BrokenLink */
