#include "modeline.h"

/* the structure containing all the global variables */
ml_sess ml;

/* constants needed to calculate virtual screen size for framebuffer */
const int possible_virtual_width [] = { 640, 800, 1024, 1152, 1280, 1600, 1920, 2048 };
const int rounding = 128;

/* if something fails, we come here to die */
void Die (const char *fmt, ...)
{
  va_list ap;
  
  fflush (stdout);
  va_start (ap, fmt);
  vfprintf (stderr, fmt, ap);
  va_end (ap);
  exit (1);
}

/* print a warning when we start */
void Warning (void)
{
  printf ("WARNING!!!\n\n");
  printf ("This utility is distributed with NO WARRANTY!\n");
  printf ("Use it at your OWN RISK. The use of incorrect monitor timings might cause\n");
  printf ("damage to your monitor, or maybe even your eyes. The Authors of this software\n");
  printf ("are in NO WAY responsible for any damage it may cause.\n\n");
  
  printf ("Call %s with --nowarn to disable this message\n", PACKAGE);
  printf ("Exit now with CTRL-C or press any key to continue\n");
  
  getchar ();
}

/*
 * get a bunch of characters
 * from stdin (max 15 chars)
 * and convert them to integers
 */
int getint (void)
{
  int i = 0;
  int c;
  char temp [16];
  
  while (i < 16)
  {
    temp [i] = getchar ();
    if (temp [i] == '\n') break;
    i++;
  }
  
  c = strtol (temp, NULL, 10);
  return c;
}

/* find a usable virtual resolution for the framebuffer */
int GetVirtX (int xres)
{
  int i;
  int test;
  
  for (i = 0; i < 8; i++)
  {
    test = possible_virtual_width [i] - (rounding * (possible_virtual_width[i] / rounding));
    if ((possible_virtual_width [i] >= xres) && (test == 0))
      return possible_virtual_width [i];
  }
  
  return -1;
}

int main (int argc, char **argv)
{
  int virtual_width = 0;    /* virtual width for fb.modes (in pixels) */
  int virtual_heigth = 0;   /* virtual heigth for fb.modes (int pixels) */
  
  double dotclock = 0.0;    /* dotclock (in MHz) */
  double pixclock = 0.0;    /* pixclock (in picoseconds) */
  double calcdot = 2.0;     /* calculated dotclock (in MHz) */
  
  int upper = 0;    /* upper margin (in pixels) */
  int lower = 0;    /* lower margin (in pixels) */
  int right = 0;    /* right margin (in pixels) */
  int left = 0;     /* left margin (in pixels) */
  int vsync = 0;    /* vertical sync (in pixels) */
  int hsync = 0;    /* horisontal sync (in pixels) */
  
  int xupper = 0;   /* upper margin (in pixels) [ml.heigth + upper] */
  int xlower = 0;   /* lower margin (in pixels) [xvsync + lower] */
  int xright = 0;   /* right margin (in pixels) [ml.width +right]*/
  int xleft = 0;    /* left margin (in pixels) [xhsync + left] */
  int xvsync = 0;   /* vertical sync (in pixels) [xupper + vsync] */
  int xhsync = 0;   /* horisontal sync (in pixels) [xright + hsync] */
  int vert = 0;     /* [xleft / calcdot] */
  
  double hf = 0.0;  /* [rigth / calcdot] */
  double hb = 0.0;  /* [left / calcdot] */
  double hw = 0.0;  /* [hsync / calcdot] */
  int vf = 0;       /* [upper * vert] */
  int vb = 0;       /* [lower * vert] */
  int vh = 0;       /* [vsync * vert] */
  
  char *outfile = "XF86Config";  /* output filename */
  char *outtext [] = { "XFree86", "FrameBuffer", "Matrox FrameBuffer" };
  
  /* set the defaults before reading args */
  ml.width = 0;     /* screen width (in pixels) */
  ml.heigth = 0;    /* screen heigth (in pixels) */
  ml.refresh = 0;   /* refreshrate (in Hz) */
  ml.depth = 8;     /* colordepth (in bits, for framebuffer output) */
  
  ml.aspect_x = 4;  /* horisontal aspect ratio */
  ml.aspect_y = 3;  /* verticatl aspect ratio */
  
  ml.output = 1;    /* output: XFree (1), fb.modes (2) or matrox lilo (3) */
  
#ifdef DEBUG
  ml.flags.verbose = 1;     /* verbose */
#else
  ml.flags.verbose = 0;     /* not verbose */
#endif
  ml.flags.quiet = 0;       /* not quiet */
  ml.flags.guess = 1;       /* try to guess */
  ml.flags.widescreen = 0;  /* no widescreen */
  ml.flags.doublescan = 0;  /* no doublescan */
  ml.flags.interlaced = 0;  /* no interlace */
  ml.flags.warn = 1;        /* display warning */
  ml.flags.gotmem = 0;      /* flag is set to 1 when memory is parsed */
  
  ml.File = "mode.conf";    /* first configfile to search for */
  
  if (CheckArgs (argc, argv)) exit (0);
  
  /* quiet and verbose doesn't mix :) */
  if (ml.flags.quiet && ml.flags.verbose)
  {
    printf ("How am I gonna manage being both quiet AND verbose???\n");
    Die ("cannot use both --quiet and --verbose\n");
  }
  
  /* doublescan & interlaced mode unavaliable for matroxfb */
  if (ml.output == 3 && (ml.flags.doublescan || ml.flags.interlaced))
    Die ("Matrox FB does not allow to specify %s mode as kernel params.\n",
         ml.flags.doublescan ? "a doublescan" : "an interlaced");
  
  /* display logo */
  if (!ml.flags.quiet) Logo ();
  
  /* display warning */
  if (ml.flags.warn) Warning ();
  
  if (ml.flags.verbose)
  {
    printf ("Output: %s", outtext [ml.output - 1]);
    if (ml.output > 1) printf (" (depth: %i)", ml.depth);
    printf ("\n\n");
    
    if (ml.flags.widescreen && !ml.flags.guess)
      printf ("widescreen not avaliable when guessing is disabled...\n");
    
    if (ml.aspect_x && !ml.flags.guess)
      printf ("aspect ratio not needed when guessing is disabled...\n");
    
    if (ml.flags.widescreen && ml.aspect_x)
    {
      printf ("widescreen disregarded when aspect ratio is given...\n");
      ml.flags.widescreen = 0;
    }
    
  }
  
  /* try to find and read a config file */
  if (ReadFile (ml.File)) Die ("\nUnable to find configfile...\n\n");
  
  /*
   * set default values for those not found in config
   */
  if (ml.h.sw_max == 0 && ml.h.sw == 0 && ml.h.sw_min == 0) ml.h.sw = 3.5;
  if (ml.h.sw_max == 0 && ml.h.sw == 0) ml.h.sw = ml.h.sw_min;
  if (ml.h.sw == 0 && ml.h.sw_min == 0) ml.h.sw = ml.h.sw_max;
  
  if (ml.v.sw_max == 0 && ml.v.sw == 0 && ml.v.sw_min == 0) ml.v.sw = 100;
  if (ml.v.sw_max == 0 && ml.v.sw == 0) ml.v.sw = ml.v.sw_min;
  if (ml.v.sw == 0 && ml.v.sw_min == 0) ml.v.sw = ml.v.sw_max;
  
  if (ml.h.fp == 0 && ml.max_dot != 0) ml.h.fp = (32 / ml.max_dot);
  if (ml.h.bp == 0 && ml.max_dot != 0) ml.h.bp = (32 / ml.max_dot);
  
  if (ml.v.fp == 0) ml.v.fp = 12;
  if (ml.v.bp == 0) ml.v.bp = 500;
  
  if (ml.flags.verbose)
  {
    printf ("\nMemory size: %d", ml.Memory);
    
    printf ("\nHorizontal:\nFront: %f, Back: %f", ml.h.fp, ml.h.bp);
    if (ml.h.sw)
      printf (", Sync: %f\n", ml.h.sw);
    else
      printf (", SyncMax: %f, SyncMin: %f\n", ml.h.sw_max, ml.h.sw_min);
    
    printf ("\nVertical:\nFront: %f, Back: %f", ml.v.fp, ml.v.bp);
    if (ml.v.sw)
      printf (", Sync: %f\n\n", ml.v.sw);
    else
      printf (", SyncMax: %f, SyncMin: %f\n\n", ml.v.sw_max, ml.v.sw_min);
    
    if (ml.flags.guess)
      printf ("Aspect ratio %i:%i will be used when guessing\n\n", ml.aspect_x, ml.aspect_y);
  }
  
  if (!ml.width)
  {
    if (ml.flags.guess && ml.heigth)
    {
      if (ml.aspect_x)
      {
        ml.width = (ml.heigth / ml.aspect_y) * ml.aspect_x;
        if (!ml.flags.quiet)
          printf ("Width: %i (guessing using aspect ratio %i:%i)\n", ml.width, ml.aspect_x, ml.aspect_y);
      }
      
      else if (ml.flags.widescreen)
      {
        ml.width = (ml.heigth / 9) * 16;
        if (!ml.flags.quiet)
          printf ("Width: %i (guessing using aspect ratio 16:9)\n", ml.width);
      }
      
      else
      {
        ml.width = (ml.heigth / 3) * 4;
        if (!ml.flags.quiet)
          printf ("Width: %i (guessing using aspect ratio 4:3)\n", ml.width);
      }
      
      if (!ml.flags.quiet) printf ("Heigth: %i\n", ml.heigth);
    }
    
    else
    {
      printf ("Width: ");
      ml.width = getint ();
      if (ml.heigth) printf ("Heigth: %i\n", ml.heigth);
    }
    
  }
  
  else if (!ml.flags.quiet) printf ("Width: %i\n", ml.width);
  
  if (!ml.heigth)
  {
    if (ml.flags.guess && ml.width)
    {
      if (ml.aspect_x)
      {
        ml.heigth = (ml.width / ml.aspect_x) * ml.aspect_y;
        if (!ml.flags.quiet)
          printf ("Heigth: %i (guessing using aspect ratio %i:%i)\n", ml.heigth, ml.aspect_x, ml.aspect_y);
      }
      
      else if (ml.flags.widescreen)
      {
        ml.heigth = (ml.width / 16) * 9;
        if (!ml.flags.quiet) printf ("Heigth: %i (guessing using aspect ratio 16:9)\n", ml.heigth);
      }
      
      else
      {
        ml.heigth = (ml.width / 4) * 3;
        if (!ml.flags.quiet)
          printf ("Heigth: %i (guessing using aspect ratio 4:3)\n", ml.heigth);
      }
      
    }
    
    else
    {
      printf ("Heigth: ");
      ml.heigth = getint ();
    }
    
  }
  
  else if (!ml.flags.quiet) printf ("Heigth: %i\n", ml.heigth);
  
  if(!ml.refresh)
  {
    printf ("Hertz: ");
    ml.refresh = getint ();
  }
  
  else if (!ml.flags.quiet) printf ("Hertz: %i\n", ml.refresh);
  if (!ml.flags.quiet || ml.output == 2) printf ("\n");
  
  /*
   * enter the calculation loop
   * exit only when the dotclock is correctly calculated
   */
  while (calcdot != dotclock)
  {
    while (((double) right / calcdot) < ml.h.fp) right += 8;
    
    if (ml.h.sw) while (((double) hsync / calcdot) < ml.h.sw) hsync += 8;
    else
    {
      while (((double) hsync / calcdot) < ml.h.sw_min) hsync += 8;
      if (((double) hsync / calcdot) > ml.h.sw_max) hsync -= 8;
    }
    
    while (((double) left / calcdot) < ml.h.bp) left += 8;
    hf = (double) right / calcdot;
    hw = (double) hsync / calcdot;
    hb = (double) left / calcdot;
    
    xright = ml.width + right;
    xhsync = xright + hsync;
    xleft = xhsync + left;
    
    vert = (xleft / (int) calcdot);
    while ((upper * vert) < ml.v.fp) upper++;
    if (ml.v.sw) while ((vsync * vert) < ml.v.sw) vsync++;
    else
    {
      while ((vsync * vert) < ml.v.sw_min) vsync++;
      if ((vsync * vert) > ml.v.sw_max) vsync--;
    }
    
    while ((lower * vert) < ml.v.bp) lower++;
    vf = upper * vert;
    vh = vsync * vert;
    vb = lower * vert;
    
    xupper = ml.heigth + upper;
    xvsync = xupper + vsync;
    xlower = xvsync + lower;
    
    /* Video timings HOWTO precise that we must have an odd total nb of
       vertical lines in interlaced mode.
       What to do if this is not the case ? */
    if (ml.flags.interlaced && (xlower & 1) == 0) ++xlower;
    
    /* Compute the dot clock (MHz) from the total pixels in one line by
       total lines in a frame, by the vertical refresh rate. */
    dotclock = (double) (xleft * xlower * ml.refresh) / 1000000.00;
    
    /* Adjust the dotclock depending on double scan or interlaced mode:
       In interlaced mode we divide by 2 the effective dotclock, as hw draws
       only half of the display per frame, while in double scan mode we
       multiply the dotclock by 2, as graphic card hardware draws the display
       twice. */
    if(ml.flags.doublescan) dotclock *= 2.0;
    if(ml.flags.interlaced) dotclock /= 2.0;
    
    pixclock = 1000000.00 / dotclock;
    
    if (calcdot < dotclock)
      calcdot += 0.001;
    
    else if (calcdot > dotclock)
    {
      calcdot -= 0.001;
      break;
    }
    
  }
  
  if (ml.output > 1)
  {
    if (ml.Memory <= 0) Die ("Memory is %d, location of error unknown\n", ml.Memory);
    ml.Memory -= 16384;
    virtual_width = GetVirtX (ml.width);
    virtual_heigth = (int) ml.Memory / virtual_width;
  }
  
  if (ml.output == 1) outfile = "XF86Config";
  else if (ml.output == 2) outfile = "fb.modes";
  else outfile = "lilo.conf";
  
  if (!ml.flags.quiet) printf ("\nPut the following into your %s file\n\n", outfile);
  if (!ml.flags.quiet) printf ("#---------# begin clip #---------#\n");
  
  if (ml.output == 1)
  {
    printf ("ModeLine \"%ix%i@%i\" ", ml.width, ml.heigth, ml.refresh);
    printf (" %.3f  %i  %i  %i  %i  %i  %i  %i  %i  %s\n", dotclock, ml.width, xright, 
            xhsync, xleft, ml.heigth, xupper, xvsync, xlower,
            ml.flags.doublescan ? "Doublescan" : ml.flags.interlaced ? "Interlace" : "");
  }
  
  else if (ml.output == 2)
  {
    printf ("mode \"%ix%i@%i\"\n", ml.width, ml.heigth, ml.refresh);
    printf ("  geometry   %i %i   %i %i   %i\n", ml.width, ml.heigth, virtual_width, virtual_heigth, ml.depth);
    printf ("  timings    %.0f   %i %i   %i %i   %i %i\n", pixclock, left, right, upper, lower, hsync, vsync);
    if (ml.flags.doublescan) printf ("  double true\n");
    if (ml.flags.interlaced) printf ("  laced true\n");
    printf ("endmode\n");
  }
  
  else
  {
    if (ml.flags.doublescan) Die ("doublescan i matrox outputmode\n(this message should never appear)");
    printf ("append = \"video=matrox:xres:%i,yres:%i,depth:%i,left:%i,right:%i,hslen:%i,",
            ml.width, ml.heigth, ml.depth, left, right, hsync);
    printf ("upper:%i,lower:%i,vslen:%i,pixclock:%.0f\"\n", lower, upper, vsync, pixclock);
  }
  
  if (!ml.flags.quiet) printf ("#----------# end clip #----------#\n");
  
  if (dotclock > ml.max_dot)
  {
    printf ("\n\nWARNING!! WARNING!! The dot clock has a higher value   WARNING!! WARNING!!\n");
    printf ("WARNING!! WARNING!! than specified in the config-file! WARNING!! WARNING!!\n");
    printf ("WARNING!! WARNING!! Try lower the Hertz, until this    WARNING!! WARNING!!\n"); 
    printf ("WARNING!! WARNING!! message disapear.                  WARNING!! WARNING!!\n\n");
    if (ml.flags.verbose) printf ("Max dot-clock: %.3f\n", ml.max_dot);
  }
  
  if(ml.flags.verbose)
  {
    printf ("\nin dot: %.3f  out dot: %.3f\n", calcdot, dotclock);
    
    printf ("\npixels:\n");
    printf ("h: front = %4i  width  = %4i  back = %4i\n", right, hsync, left);
    printf ("v: front = %4i  heigth = %4i  back = %4i\n", upper, vsync, lower);
    
    printf ("\nHorizontal:\nFront: %.3f, Back: %.3f, Sync: %.3f\n", hf, hb, hw);
    printf ("\nVertical:\nFront: %4i,  Back: %4i,  Sync: %4i\n\n", vf, vb, vh);
  }
  
  if (!ml.flags.quiet && ml.output == 3) printf ("\nRemember to run /sbin/lilo afterwards\n");
  
  exit (0);
}
