#ifndef GAMEINTERFACE_H
#define GAMEINTERFACE_H

#include "Tools.h"
#include "SDLTools.h"
#include "ObjectFwd.h"


//----------------------------------------------------------------------------
/**
 * Since all decoration and moving objects don't know anything about
 * the PlayGround or the GameControlBase, this interface must be used
 * to set or get information from libMoaggGame.
 */
class GameInterface
{
  public:
    //------------------------------------------------------------------------
    GameInterface() {}
    virtual ~GameInterface() {}

    //------------------------------------------------------------------------
    /**
     * @return The number of tiles of the playground in x-direction.
     */
    virtual Uint16 getPlayGroundXTiles() const = 0;

    /**
     * @return The number of tiles of the playground in x-direction.
     */
    virtual Uint16 getPlayGroundYTiles() const = 0;

    /**
     * Adds the given x gravity-increment to the gravity matrix
     * at the given position.
     */
    virtual void addXGravityToPlayGround(Uint16 x, Uint16 y,
                                         int increment) = 0;

    /**
     * Adds the given y gravity-increment to the gravity matrix
     * at the given position.
     */
    virtual void addYGravityToPlayGround(Uint16 x, Uint16 y,
                                         int increment) = 0;

    virtual int getXGravity(const SDL_Rect &r) const = 0;
    virtual int getYGravity(const SDL_Rect &r) const = 0;

    virtual int getXVelocity(const SDL_Rect &r) const = 0;
    virtual int getYVelocity(const SDL_Rect &r) const = 0;

    virtual unsigned getFriction(const SDL_Rect &r) const = 0;

    virtual bool isBackgroundBetween(Uint16 x1, Uint16 y1,
                                     Uint16 x2, Uint16 y2) const = 0;

    virtual const Platform *isInLandingZone(const Ship *s) const = 0;

    virtual const Ship *getPlayerShip() const = 0;

    virtual void onShipTakeoff(const Ship *s) = 0;
    virtual void onShipLanded(Ship *s, const Platform *platform) = 0;
    virtual void onShipFiring(const Ship *s) = 0;
    virtual void onShipOutOfFuel(const Ship *s) = 0;

    //------------------------------------------------------------------------
    static void setInstance(GameInterface *instance)
    {
        sm_instance = instance;
    }

    static GameInterface *getInstance()
    {
        return sm_instance;
    }

  private:

    //------------------------------------------------------------------------
    static GameInterface *sm_instance;
};


//----------------------------------------------------------------------------
class NullGameInterface : public GameInterface
{
    SINGLETON_OBJECT(NullGameInterface);

  public:
    //------------------------------------------------------------------------
    NullGameInterface() {}
    ~NullGameInterface() {}

    //------------------------------------------------------------------------
    Uint16 getPlayGroundXTiles() const { return 0; }
    Uint16 getPlayGroundYTiles() const { return 0; }

    void addXGravityToPlayGround(Uint16 x, Uint16 y, int increment) {}
    void addYGravityToPlayGround(Uint16 x, Uint16 y, int increment) {}

    int getXGravity(const SDL_Rect &r) const { return 0; }
    int getYGravity(const SDL_Rect &r) const { return 0; }

    int getXVelocity(const SDL_Rect &r) const { return 0; }
    int getYVelocity(const SDL_Rect &r) const { return 0; }

    unsigned getFriction(const SDL_Rect &r) const { return 0; }

    bool isBackgroundBetween(Uint16 x1, Uint16 y1, Uint16 x2, Uint16 y2) const
    { return true; }

    const Platform *isInLandingZone(const Ship *s) const { return NULL; }

    const Ship *getPlayerShip() const { return NULL; }

    void onShipTakeoff(const Ship *s) {}
    void onShipLanded(Ship *s, const Platform *platform) {}
    void onShipFiring(const Ship *s) {}
    void onShipOutOfFuel(const Ship *s) {}
};

#endif //GAMEINTERFACE_H
