#ifndef OBJECTREPOSITORY_H
#define OBJECTREPOSITORY_H

#include "Tools.h"
#include "ObjectBase.h"

//----------------------------------------------------------------------------
/**
 * This class is used to store all objects of a level
 * using their object ids as key.
 * An object id may have the following value:
 * <ul>
 *   <li>0: Reserved as "uninitialized" value.
 *          No fully initialized object must have a zero object id.</li>
 *   <li>1-1023: Reserved manually assigned values for objects defined
 *               in the level file (e.g. crates, turrets, platforms).</li>
 *   <li>1024-65535: Reserved dynamically assigned values for objects
 *                   defined in the level file with omitted object id
 *                   (e.g. black holes, platforms not relevant for the
 *                   game flow) or objects that are created in-game
 *                   (e.g. projectiles, particles).</li>
 * </ul>
 */
class ObjectRepository
{
    SINGLETON_OBJECT(ObjectRepository);

  public:

    //------------------------------------------------------------------------
    ObjectRepository();
    ~ObjectRepository();

    //------------------------------------------------------------------------
    size_t size() const;

    //------------------------------------------------------------------------
    void clear();

    //------------------------------------------------------------------------
    bool doesExist(const unsigned id) const;

    //------------------------------------------------------------------------
    unsigned getDynamicObjectId() const;

    //------------------------------------------------------------------------
    /**
     * @param o A pointer to the object to add.
     */
    void addObject(ObjectBase *o);

    /**
     * @param o A pointer to the object (existing in m_id2object) to show.
     */
    void showObject(ObjectBase *o);

    /**
     * @param id The object id to find the object in m_id2object to show.
     */
    void showObjectId(const unsigned id);

    //------------------------------------------------------------------------
    /**
     * Sets the toRemove and autoDelete flag of the matching object
     * in m_objectList.
     *
     * @param o A pointer to the object to mark for removal.
     * @param autoDelete true, if the object shall be deleted, else false.
     */
    void markObjectToRemove(ObjectBase *o, bool autoDelete = true);

    /**
     * Removes all objects that were marked for removal.
     * Further they will be deleted, if their autoDelete flag is set.
     */
    void removeObjectsToRemove();

    //------------------------------------------------------------------------
    ObjectBase *getObject(const unsigned id);
    const ObjectBase *getObject(const unsigned id) const;

    //------------------------------------------------------------------------
    void acceptParticles(ObjectVisitor &v);
    void acceptParticles(ObjectConstVisitor &v) const;
    
    //------------------------------------------------------------------------
    void acceptObjects(ObjectVisitor &v);
    void acceptObjects(ObjectConstVisitor &v) const;

    //------------------------------------------------------------------------
    inline void acceptAll(ObjectVisitor &v)
    {
        acceptObjects(v);
        acceptParticles(v);
    }

    inline void acceptAll(ObjectConstVisitor &v) const
    {
        acceptObjects(v);
        acceptParticles(v);
    }

  private:

    //------------------------------------------------------------------------
    class PImpl;
    class PImpl *m_pImpl;
};

#endif //OBJECTREPOSITORY_H
