""" Builds menus from action, group and menu extensions. """


# Enthought library imports.
from enthought.envisage.action.api import MenuBuilder 
from enthought.envisage.workbench import Window
from enthought.pyface.action.api import Group, MenuManager
from enthought.traits.api import Instance

# Local imports.
from workbench_action_factory import WorkbenchActionFactory


class MenuBuilder(MenuBuilder):
    """ Builds menus from action, group and menu extensions. """

    #### 'MenuBuilder' interface ##############################################

    # The workbench window that we are building menus for.
    window = Instance(Window)

    ###########################################################################
    # Protected 'MenuBuilder' interface.
    ###########################################################################

    def _create_action(self, extension):
        """ Creates an action implementation from an action extension. """

        factory = WorkbenchActionFactory()

        return factory.create_action(self.window, extension)

    def _create_group(self, extension):
        """ Creates a group implementation from a group extension. """

        if len(extension.class_name) > 0:
            klass = self._import_class(extension)

        else:
            klass = Group

        return klass(id=extension.id, separator=extension.separator)

    def _create_menu_manager(self, extension):
        """ Creates a menu manager implementation from a menu extension. """

        if len(extension.class_name) > 0:
            klass = self._import_class(extension)

        else:
            klass = MenuManager

        menu_manager = klass(id=extension.id, name=extension.name)
        for group in extension.groups:
            menu_manager.append(self._create_group(group))

        return menu_manager

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _import_class(self, extension):
        """ Imports an extension implementation class. """

        application = self.window.application

        # The plugin definition that the extension was defined in.
        plugin_definition = extension._action_set_._definition_

        # We are about to actually import the menu manager class, so make
        # sure that the plugin that contributed it has been started.
        application.start_plugin(plugin_definition.id)

        # Import the implementation class
        return application.import_symbol(extension.class_name)

#### EOF ######################################################################
