#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

""" A base class for actions that switch perspectives.
"""

# Standard library imports.
import logging

# Enthought library imports
from enthought.envisage.ui import UIPlugin
from enthought.pyface.action.api import Action
from enthought.traits.api import Dict, Range, Str, Tuple


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class PerspectiveAction(Action):
    """ A base class for actions that switch perspectives.
    """

    ###########################################################################
    # Traits
    ###########################################################################

    ### 'PerspectiveAction' interface #########################################

    # A map defining the views this perspective contains.  The key is the id
    # of the view and the value is the position the view should be in.
    contained_views = Dict(key_trait=Str, value_trait=Str)

    # A map defining the sizes of the view regions within this perspective.
    # The keys are the valid Envisage view positions and the values are a
    # tuple of form (width, height) where each dimension is a float value
    # between 0 and 1 representing the fraction of the full workbench window
    # dimensions.
    #
    # FIXME: We should be able to control size of the actual views we want
    # rather than the size of the view regions.
    view_sizes = Dict(
        key_trait   = Str,
        value_trait = Tuple(Range(0.0, 1.0), Range(0.0, 1.0)),
        )


    ###########################################################################
    # Public 'Action' interface.
    ###########################################################################

    def perform(self, event):
        """ Performs the action.
        """
        # NOTE: The size changes must be done after the visibility is updated
        # because Envisage resizes positions when views are added or removed.
        self._update_visibility()
        self._update_position_sizes()


    def _update_position_sizes(self):
        """ Updates the sizing of the view positions to those defined within
            this perspective.
        """

        # Update all the position sizes requested.
        active_window = UIPlugin.instance.active_window
        window_size = active_window.control.GetClientSize()
        for key in self.view_sizes:
            # Don't stop trying to update all of the requested view positions
            # because a single one had problems.
            try:
                # Convert the size percentages to actual pixel counts
                size = [-1, -1]
                width, height = self.view_sizes[key]
                if width != -1:
                    size[0] = int(window_size.width * width)
                if height != -1:
                    size[1] = int(window_size.height * height)

                # Update the region size
                active_window.page_layout.set_position_size(key, size)
            except Exception, e:
                logger.exception("Unable to set size on poistion '" + key +
                    "' due to exception: " + str(e))

        # Update the layout of the window
        active_window.page_layout.update_layout()


    def _update_visibility(self):
        """ Updates the visibility of all views within the current active
            window to show only the views that are part of this perspective.

        """
        # Iterate through all known views in the active window
        active_window = UIPlugin.instance.active_window
        for v in active_window.views:
            # Don't stop trying to update the whole perspective because a single
            # view has problems.
            try:
                # Determine the view's visibility and position according to the
                # definition of this perspective.
                make_visible = False
                position = None
                if v.id and v.id in self.contained_views:
                    make_visible = True
                    position = self.contained_views[v.id]

                # Update the current view's visibility and position.
                #
                # Because of issues with Envisage, we only call a method to change
                # visibility if we're really changing the visible state of the view.
                if make_visible and not v.opened == True:
                    v.open()
                elif not make_visible and not v.opened == False:
                    v.close()

                # Only update the position of visible views, and only if they need
                # it.
                if v.opened and not position == v.position:
                    v.position = position
            except Exception, e:
                logger.exception("Unable to control view with id '" + v.id +
                    "' due to an exception: " + str(e))


#### EOF ######################################################################
