#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005-2007 by Enthought, Inc.
#  All rights reserved.
#
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

"""
A base class for actions that determine their enabled status based on
the state of the current project.

"""

# Standard library imports
import logging

# Enthought library imports
from enthought.envisage import get_application
from enthought.envisage.workbench.action import ObjectAction
from enthought.traits.api import Instance

# Local imports.
from enthought.envisage.single_project.services import IPROJECT_MODEL


# Setup a logger for this module.
logger = logging.getLogger(__name__)


class ProjectAction(ObjectAction):
    """
    A base class for actions that determine their enabled status based on
    the state of the current project.

    This class is architected such that implementors can override the
    'refresh' method without losing the default logic that determines
    the enabled state as a function of the current project.

    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'ProjectAction' interface ####################################

    # The project model service we refresh our state against.
    model_service = Instance('enthought.envisage.single_project.' + \
        'model_service.ModelService')


    ##########################################################################
    # 'object' interface
    ##########################################################################

    #### operator methods ####################################################

    def __init__(self, *args, **kws):
        """
        Constructor.

        Extended to setup a listener on the project model service so
        that we can refresh whenever the project changes.

        """

        # Retrieve the project model service and register ourself to listen
        # for project state changes.  This is done first to avoid errors
        # during any refresh calls triggered by further initialization.
        if 'model_service' in kws:
            self.model_service = kws['model_service']
            del kws['model_service']
        else:
            self.model_service = get_application().get_service(IPROJECT_MODEL)
        self._update_model_service_listeners(remove=False)

        super(ProjectAction, self).__init__(*args, **kws)

        return


    ##########################################################################
    # 'Action' interface
    ##########################################################################

    #### public interface ####################################################

    def destroy(self):
        """
        Destroy this action.

        Overridden here to remove our project model service listeners.

        """

        self._update_model_service_listeners(remove=True)

        return

    ##########################################################################
    # 'ProjectAction' interface
    ##########################################################################

    #### public interface ####################################################

    def refresh(self):
        """
        Refresh the enabled state of this action.

        This default implementation enables the action only when there is a
        current project.

        """

        self.enabled = self._refresh_project_exists()

        return


    #### protected interface #################################################

    def _refresh_project_exists(self):
        """
        Return the refresh state according to whether the model service has a
        current project.

        Returns True if this action should be enabled.  False otherwise.

        """

        enabled = False
        if self.model_service is not None \
            and self.model_service.project is not None:
            enabled = True

        return enabled


    def _update_model_service_listeners(self, remove=False):
        """
        Update our listeners on the project model service.

        These are done as individual listener methods so that derived
        classes can change the behavior when a single event occurs.

        """

        logger.debug( (remove and 'Removing ' or 'Adding ') + \
            'listeners on project model service for ProjectAction [%s]', self)

        self.model_service.on_trait_change(self._on_project_changed,
            'project', remove=remove)
        self.model_service.on_trait_change(self._on_project_selection_changed,
            'selection', remove=remove)

        return


    #### trait handlers ######################################################

    def _on_project_changed(self, obj, trait_name, old, new):
        """
        Handle changes to the value of the current project.

        """

        self.refresh()

        return


    def _on_project_selection_changed(self, obj, trait_name, old, new):
        """
        Handle changes to the selection value within the current project.

        """

        self.refresh()

        return


#### EOF #####################################################################
