#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, 2006 by Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

"""
An action to save the current project to a new location.  This is
only enabled when there is a current project.

"""

# Standard library imports.
import logging

# Enthought library imports
from enthought.envisage.single_project.project_action import ProjectAction


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class SaveAsAction(ProjectAction):
    """
    An action to save the current project to a new location.  This is
    only enabled when there is a current project.

    """

    ##########################################################################
    # Traits
    ##########################################################################

    #### public 'ObjectAction' interface #####################################

    # The universal object locator (UOL).
    uol = 'service://enthought.envisage.single_project.UiService'

    # The name of the method to invoke on the object.
    method_name = 'save_as'


    ##########################################################################
    # 'ProjectAction' interface
    ##########################################################################

    #### public interface ####################################################

    def refresh(self):
        """
        Refresh the enabled state of this action.

        This implementation enables the action when there is a current project
        which is marked as being allowed to do a 'save as' operation.

        """

        self.enabled = self._refresh_project_exists() and \
            self._refresh_is_save_as_allowed()

        return


    #### trait handlers ######################################################

    def _on_project_changed(self, obj, trait_name, old, new):
        """
        Handle changes to the value of the current project.

        Extended to ensure that we listen for changes to the is_save_as_allowed
        flag on the current project.

        """

        if old is not None:
            self._update_project_listeners(old, remove=True)
        if new is not None:
            self._update_project_listeners(new, remove=False)

        super(SaveAsAction, self)._on_project_changed(obj, trait_name, old, new)


    ##########################################################################
    # 'SaveAsAction' interface
    ##########################################################################

    #### protected interface #################################################

    def _refresh_is_save_as_allowed(self):
        """
        Return the refresh state according to whether the current project is
        marked as being capable of doing a 'save as'.

        Returns True if the action should be enabled and False otherwise.

        """

        return self.model_service.project.is_save_as_allowed


    def _update_project_listeners(self, project, remove):
        """
        Update listeners on the specified project.

        """

        logger.debug( (remove and 'Removing ' or 'Adding ') + \
            'listeners on project [%s] for SaveAsAction [%s]', project, self)

        project.on_trait_change(self._on_is_save_as_allowed,
            'is_save_as_allowed', remove=remove)

        return


    #### trait handlers ######################################################

    def _on_is_save_as_allowed(self, obj, trait_name, old, new):
        """
        Handle changes to the value of the current project's is_save_as_allowed.

        """

        self.refresh()

        return


#### EOF #####################################################################
