""" A Pythonic namespace. """


# Standard library imports.
import inspect


class Namespace(object):
    """ A Pythonic namespace.

    This class does NOT inherit from 'HasTraits' because we want to control
    attribute access via '__getattr__' and '__setattr__'.

    """

    ###########################################################################
    # 'object' interface.
    ###########################################################################
    
    def __init__(self, **kw):
        """ Creates a new namespace.

        All keyword arguments are bound in the namespace.

        """

        # Bind all keyword arguments into the namespace.
        self.__dict__['_namespace'] = kw.copy()

        return

    def __str__(self):
        """ Returns an informal string representation of the object. """

        return str(self._namespace)
    
    def __getattr__(self, name):
        """ Looks up a name in the namespace. """

        return self.__dict__['_namespace'][name]
        
    def __setattr__(self, name, value):
        """ Binds a name in the namespace. """

        self.__dict__['_namespace'][name] = value

        return

    ###########################################################################
    # 'Namespace' interface.
    ###########################################################################

    def bind(self, name, value):
        """ Binds a name to a value in the target namespace.

        This will create intermediate namespaces if they do not exist.

        """

        components = name.split('.')

        # Find the target namespace, creating intermediate namespaces if they
        # do not exist.
        target = self
        for atom in components[:-1]:
            # Lookup the next namespace.
            namespace = target._namespace.get(atom)

            # If the namespace does not exist then create one.
            if namespace is None:
                namespace = Namespace()
                setattr(target, atom, namespace)

            target = namespace

        # Bind the value in the target namespace.
        setattr(target, components[-1], value)
        
        return


class ScriptingNamespace(Namespace):
    """ A scripting namespace. """

    ###########################################################################
    # 'object' interface.
    ###########################################################################
    
    def __init__(self, **kw):
        """ Creates a new namespace.

        All keyword arguments are bound in the namespace.

        """

        # Super class constructor.
        super(ScriptingNamespace, self).__init__(**kw)
        
        # Command pre and post hooks.
        self.__dict__['_pre_hooks'] = []
        self.__dict__['_post_hooks'] = []

        return

    ###########################################################################
    # 'ScriptingNamespace' interface.
    ###########################################################################

    def add_post_hook(self, hook):
        """ Adds a post hook. """

        self._post_hooks.append(hook)

        return
   
    def add_pre_hook(self, hook):
        """ Adds a pre hook. """

        self._pre_hooks.append(hook)

        return

    def remove_post_hook(self, hook):
        """ Removes a post hook. """

        self._post_hooks.remove(hook)

        return

    def remove_pre_hook(self, hook):
        """ Removes a pre hook. """
        
        self._pre_hooks.remove(hook)
        
        return

    def bind_class(self, name, klass):
        """ Binds all methods in the class to a namespace named 'name'! """

        # Create a namespace containing all of the class' class methods (we
        # ignore any methods that start with an underscore).
        namespace = Namespace()

        for attr in klass.__dict__:
            if not attr.startswith('_'):
                value = klass.__dict__[attr]
                if type(value) is classmethod:
                    # fixme: We use 'getattr' here instead of just 'value'
                    # because 'value' is of type 'classmethod' which is *not*
                    # callable. If we use 'getattr' we get an 'instancemethod'
                    # which *is* callable. I think I must be missing something
                    # here 'cos it doesn't make sense... why are 'classmethods'
                    # not callable? Why *does* getattr get us something
                    # different than getting the attribute vis the class
                    # dictionary? Was it a bit of a hack to get class methods
                    # into Python?!?
                    callable_name = '%s.%s' % (name, attr)
                    callable = getattr(klass, attr)
                    wrapper = self._wrap_callable(callable_name, callable)
                    
                    namespace.bind(attr, wrapper)

        # Bind the class's namespace.
        self.bind(name, namespace)

        return

    def bind_function(self, name, function):
        """ Binds a function to the specified name. """

        self.bind(name, self._wrap_callable(name, function))

        return

    def bind_instance(self, name, instance):
        """ Binds all methods in an instance to a namespace named 'name'! """

        # Create a namespace containing all of the instance's bound methods
        # (we ignore any methods that start with an underscore).
        namespace = Namespace()

        # fixme: This shoudl in fact be a complete delegate to the instance -
        # we should wrap all methods AND traits!
        for attr in dir(instance):
            if not attr.startswith('_'):
                value = getattr(instance, attr)
                if inspect.ismethod(value):
                    namespace.bind(
                        attr,
                        self._wrap_callable('%s.%s' % (name, attr),value)
                    )

        # Bind the instance's namespace.
        self.bind(name, namespace)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _wrap_callable(self, name, callable):
        """ Wraps a callable with a function that we can hook! """

        def wrapper(*args, **kw):
            """ A hookable function wrapper. """

            # Pre hooks.
            for hook in self._pre_hooks:
                hook(name, callable, *args, **kw)

            # Call the wrapped function!
            result = callable(*args, **kw)

            # Post hooks. Note that the post hooks have access to and can
            # modify the return value.
            for hook in self._post_hooks:
                result = hook(name, callable, result, *args, **kw)
            
            return result

        return wrapper


# Testing.
if __name__ == '__main__':

    # A scripting namespace.
    scripting = ScriptingNamespace()

    def pre_hook(name, callable, *args, **kw):
        print 'Pre hook', name, callable, args, kw
        return
    
    def post_hook(name, callable, result, *args, **kw):
        print 'Post hook', name, callable, args, kw
        return result

    scripting.add_pre_hook(pre_hook)
    scripting.add_post_hook(post_hook)
    
    #### Function ####

    def open_file(filename):
        print 'open_file', filename

    scripting.bind_function('stuff.file.open_file', open_file)
    print 'Result', scripting.stuff.file.open_file('C:\\tmp\\foogle.py')

    #### Instance ####

    from enthought.traits.api import HasTraits
    
    class FileStuff(HasTraits):
        def open_file(self, filename):
            print 'OpenFile.open_file', filename
            return

        def close_file(self, filename):
            print 'OpenFile.close_file', filename
            return

    scripting.bind_instance('foo.bar.baz', FileStuff())
    
    print 'Result', scripting.foo.bar.baz.open_file('C:\\tmp\\foogle.py')
    print 'Result', scripting.foo.bar.baz.close_file('C:\\tmp\\foogle.py')

    #### Class ####

    from enthought.traits.api import HasTraits
    
    class FileStuff(HasTraits):
        def open_file(cls, filename):
            print 'OpenFile.open_file', filename
            return

        open_file = classmethod(open_file)
        
        def close_file(self, filename):
            print 'OpenFile.close_file', filename
            return

        close_file = classmethod(close_file)
        
    scripting.bind_class('foo.bar.baz', FileStuff)
    
    print 'Result', scripting.foo.bar.baz.open_file('C:\\tmp\\foogle.py')
    print 'Result', scripting.foo.bar.baz.close_file('C:\\tmp\\foogle.py')
    
#### EOF ######################################################################
