#------------------------------------------------------------------------------
# Copyright (c) 2005, Enthought, Inc.
# All rights reserved.
# 
# This software is provided without warranty under the terms of the BSD
# license included in enthought/LICENSE.txt and may be redistributed only
# under the conditions described in the aforementioned license.  The license
# is also available online at http://www.enthought.com/licenses/BSD.txt
# Thanks for using Enthought open source!
# 
# Author: Enthought, Inc.
# Description: <Enthought naming package component>
#------------------------------------------------------------------------------
""" A Python object explorer. """


# Major package imports.
import wx

# Enthought library imports.
from enthought.pyface.api import GUI, LayeredPanel
from enthought.pyface.api import PythonShell, SplitApplicationWindow, SplitPanel
from enthought.traits.api import Any

# Local imports.
from explorer_tree import ExplorerTree


# Factory function for exploring a Python namespace.
def explore(obj):
    """ View a Python object as a naming context. """

    explorer = Explorer(obj=obj, size=(1200, 800))
    explorer.open()
    
    return


class Explorer(SplitApplicationWindow):
    """ The main application window. """

    #### 'Window' interface ###################################################

    title = 'Object Explorer'
    
    #### 'SplitApplicationWindow' interface ###################################

    # The direction in which the panel is split.
    direction = 'vertical'

    # The ratio of the size of the left/top pane to the right/bottom pane.
    ratio = 0.4

    #### 'Explorer' interface #################################################
    
    # The object that we are exploring.
    obj = Any
    
    ###########################################################################
    # Protected 'SplitApplicationWindow' interface.
    ###########################################################################

    def _create_lhs(self, parent):
        """ Creates the left hand side or top depending on the style. """

        split_panel = SplitPanel(
            parent,
            lhs = self._create_tree,
            rhs = self._create_details,
            direction = 'horizontal',
            ratio     = 0.75
        )

        return split_panel.control

    def _create_rhs(self, parent):
        """ Creates the right hand side or bottom depending on the style. """

        return self._create_python_shell(parent)

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_tree(self, parent):
        """ Creates the explorer tree. """

        self._tree = tree = ExplorerTree(parent, obj=self.obj)
        tree.on_trait_change(self._on_selection_changed, 'selection')
        
        return tree.control

    def _create_details(self, parent):
        """ Creates the details panel. """

        sizer = wx.BoxSizer(wx.VERTICAL)
        panel = wx.Panel(parent, -1, style=wx.CLIP_CHILDREN)
        panel.SetSizer(sizer)
        panel.SetAutoLayout(True)

        self._details = LayeredPanel(panel)
        sizer.Add(self._details.control, 1, wx.EXPAND)

        # Resize the splitter to fit the sizer's minimum size.
        sizer.Fit(panel)

        return panel

    def _Xcreate_details(self, parent):
        """ Creates the details panel. """

        self._details = LayeredPanel(parent)
        
        return self._details.control
    
    def _create_python_shell(self, parent):
        """ Creates the Python shell. """

        python_shell = PythonShell(parent)

        # Bind useful names.
        python_shell.bind('widget', self._tree)
        python_shell.bind('w', self._tree)
        python_shell.bind('window', self)
        python_shell.bind('rm', self._tree.resource_manager)
        python_shell.bind('explore', explore)
        
        # Execute useful commands to bind useful names ;^)
        python_shell.execute_command('from enthought.naming.api import *')
        
        return python_shell.control

    def _create_detail_layer(self, parent, binding):
        """ Creates the detail layer for an object. """

        from enthought.traits.api import HasTraits, Str
        from enthought.traits.ui.api import View, Group, Item
        
        class Details(HasTraits):
            """ Details of the selected object. """

            # The string representation of the objects value.
            value = Str

            # The resource type.
            resource_type = Any

            # The node type.
            node_type = Any


            view = View(
                Group(
                    'value',
                    'resource_type',
                    'node_type'
                )
            )

                    
        obj = binding.obj
        resource_manager = self._tree.resource_manager
        resource_type    = resource_manager.get_type_of(obj)
        node_type        = resource_type.node_type
        
        details = Details(
            value         = str(obj),
            resource_type = resource_type.__class__.__name__,
            node_type     = node_type.__class__.__name__
        )

        ui = details.edit_traits(parent=parent, kind='subpanel')
        
        return ui.control
    
    #### Trait event handlers #################################################

    def _on_selection_changed(self, selection):
        """ Called then the tree selection has changed. """

        if len(selection) > 0:
            obj  = selection[0]
            name = str(id(obj))
            
            if not self._details.has_layer(name):
                self._details.add_layer(
                    name,
                    self._create_detail_layer(self._details.control, obj)
                )

            self._details.show_layer(name)
                
        return
    
##### EOF #####################################################################
