#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#  Author: Dave Peterson <dpeterson@enthought.com>
#
#-----------------------------------------------------------------------------

""" A factory for generating a Traits editor that uses a ResourceTree as
    its content control.
"""

# Standard library imports.

# Major packages.

# Enthought library imports
from enthought.traits.api import Any, Dict, Str
from enthought.traits.ui.wx.editor_factory import EditorFactory

# Application specific imports.

# Local imports.
from resource_tree_editor import ResourceTreeEditor


##############################################################################
# class ResourceTreeEditorFactory
##############################################################################

class ResourceTreeEditorFactory(EditorFactory):
    """ A factory for generating a Traits editor that uses a ResourceTree as
        its content control.

        This class can be initialized with a dictionary as the first positional
        parameter and this dictionary will be used as the keywords when
        constructing the ResourceTree control.  This allows you to override
        the default configuration of the ResourceTree to, for example, change
        the selection model.

        Additionally, if you haven't provided a keyword specification for the
        resource_manager trait of the ResourceTree control, the second
        positional argument will specify the id of the Envisage resource
        manager service that the ResourceTree will use.  The editor(s) will
        retrieve the service from the Envisage application themselves using
        this id.  This architecture allows them to always use the current
        resource manager (should it ever change during the life of an
        application.)

        Note that if neither the keyword argument nor the id is provided, we
        fall back to retrieving the resource manager by using the id used by
        the enthought.envisage.resource plugin to register its resource
        manager.
    """

    ##########################################################################
    # Traits
    ##########################################################################

    ### public interface #####################################################

    # The ID of the resource manager service.  The default value matches what
    # the enthought.envisage.resource plugin uses for its resource manager.
    # This will be overridden by any resource_manager keyword argument.
    id = Str('enthought.envisage.resource.IResourceManager')

    # The keyword args to be used to instantiate the ResourceTree
    kwargs = Dict(Str, Any)

    # Function to call when a node is selected.
    on_select = Any

    # Function to call when a node is double-clicked.
    on_dclick = Any


    ##########################################################################
    # 'EditorFactory' interface
    ##########################################################################

    ### private interface ####################################################

    def __init__ (self, *args, **traits):
        if len(args) >= 1:
            self.kwargs = args[0]
        if len(args) >= 2:
            self.id = args[1]
        super(ResourceTreeEditorFactory, self).__init__( **traits )


    ### public interface #####################################################

    def simple_editor(self, ui, object, name, description, parent):
        return ResourceTreeEditor(
            parent,
            factory     = self,
            ui          = ui,
            object      = object,
            name        = name,
            description = description,
            )

    custom_editor = simple_editor
    text_editor = simple_editor
    readonly_editor = simple_editor


#### EOF #####################################################################
