""" The service registry. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.traits.api import Dict, HasTraits, Int, Property


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ServiceRegistry(HasTraits):
    """ The service registry. """

    #### 'ServiceRegistry' interface ##########################################

    # All registered services.
    #
    # { object interface : { object service : dict properties } }
    services = Property(Dict)

    ####  Private interface ###################################################

    # Shadow for the 'services' trait.
    #
    # { object interface : { object service : dict properties } }
    _services = Dict

    # The next service Id (service Ids are never persisted between process
    # invocations so this is simply an ever increasing integer!).
    _service_id = Int

    ###########################################################################
    # 'ServiceRegistry' interface.
    ###########################################################################

    #### Properties ###########################################################

    def _get_services(self):
        """ Returns a dictionary containing all registered services. """

        return self._services.copy()

    #### Methods ##############################################################

    def register_service(self, interface, obj, properties=None):
        """ Registers a service. """

        # Get the next service id.
        service_id = self._next_service_id()

        if properties is None:
            properties = {}

        services_for_interface = self._services.setdefault(interface, {})
        services_for_interface[obj] = (service_id, properties)

        logger.debug('service %s registered (id %d)' % (interface, service_id))

        return service_id

    def unregister_service(self, service_id):
        """ Unregisters a service. """

        for interface, services in self._services.items():
            for service, (sid, properties) in services.items():
                if service_id == sid:
                    del services[service]
                    logger.debug(
                        'service %s unregistered (id %d)' \
                        % (interface, service_id)
                    )
                    break

        return

    def get_service(self, interface, query=None):
        """ Returns at most one service that matches the specified query.

        Returns None if no such service is found.

        NOTE: Don't try to guess *which* one it will return -- the algorithm
        used by this method is subject to change without notice!

        """

        services_for_interface = self._services.get(interface, {})

        if len(services_for_interface) > 0:
            if query is not None:
                for service, (sid, props) in services_for_interface.items():
                    if self._evaluate_query(query, service, props):
                        break

                else:
                    service = None

            else:
                service = services_for_interface.keys()[0]

        else:
            service = None

        return service

    def get_services(self, interface, query=None):
        """ Returns all services that match the specified query.

        If no services match the query, then an empty list is returned.

        """

        services_for_interface = self._services.get(interface, {})

        if query is not None:
            services = []
            for service, (id, properties) in services_for_interface.items():
                if self._evaluate_query(query, service, properties):
                    services.append(service)

        else:
            services = services_for_interface.keys()

        return services

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _next_service_id(self):
        """ Returns the next service ID. """

        self._service_id += 1

        return self._service_id

    def _evaluate_query(self, query, service, properties):
        """ Evaluates a query over a single service.

        Returns True if the service matches the query; otherwise returns False.

        """

        namespace = self._create_namespace(service, properties)
        try:
            result = eval(query, namespace)

        except:
            result = False

        return result

    def _create_namespace(self, service, properties):
        """ Creates a namespace in which to evaluate a query. """

        namespace = {}
        namespace.update(service.__dict__)
        namespace.update(properties)

        for name in service.trait_names():
            if not name.startswith('trait'):
                namespace[name] = getattr(service, name)

        return namespace

    ###########################################################################
    # Debugging interface.
    ###########################################################################

    def dump(self):
        """ Dumps the state of the registry to STDOUT. """

        for interface, services in self._services.items():
            print 'Interface:', interface
            for service, properties in services.items():
                print '  Service:', service, 'Properties:', properties

        return

#### EOF ######################################################################
