""" A plugin definition. """


# Standard library imports.
import inspect, os, sys

# Enthought library imports.
from enthought.traits.api import Bool, HasPrivateTraits, List, Str

# Local imports.
from extension_point import ExtensionPoint
from util import get_module_name


class PluginDefinition(HasPrivateTraits):
    """ A plugin definition. 
    
    This class is declarative; subclasses must not define any methods.
    """

    #### 'PluginDefinition' interface #########################################
    
    # Should the plugin *always* be started?  By default, plugins are started
    # on demand. If this is '''True''' then the plugin is started immediately.
    autostart = Bool(False)

    # The name of the class that implements the plugin.
    class_name = Str

    # Is the plugin enabled? If the plugin is *not* enabled, then none of its
    # extension points or extensions will be loaded and it will not be
    # started.
    enabled = Bool(True)

    # The contributions that this plugin makes to extension points offered by
    # itself or other plugins.
    extensions = List(ExtensionPoint)

    # The extension points offered by this plugin to allow other plugins to
    # contribute to it.
    #
    # fixme: This is a list of *classes* that derive from '''ExtensionPoint'''.
    # I have no idea how we describe the trait!
    extension_points = List

    # The plugin's *globally* unique identifier.
    id = Str

    # The plugin's name.
    name = Str

    # Who wrote it?
    provider_name = Str

    # A URL for contacting the provider.
    provider_url = Str

    # The IDs of the plugins that this plugin requires.
    requires = List(Str)

    # The plugin's version number (in the form '1.0.0').
    version = Str

    #### The following traits are set by the framework ########################
    
    # The absolute path of the file that the plugin definition was defined in.
    # This attribute is set by the framework.
    location = Str
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################
    
    def __init__(self, **traits):
        """ Creates a new plugin definition. """

        # Base class constructor.
        super(PluginDefinition, self).__init__(**traits)

        # The most common usage is for plugin definitions to be loaded from
        # plugin definition modules when an Envisage application is starting
        # up.  In this situation we need a way to discover the plugin
        # definitions as they are imported.  We also have to do this in such a
        # way as to allow multiple applications to use the same plugin
        # definitions even though each definition file is imported exactly
        # once.
        #
        # Currently, we achieve this by simply(ish) adding the definition to
        # a dictionary of definitions that we attach to the module that the
        # definition is defined in.  The slightly scary part is that to find
        # that module we have to do a bit of stack-crawling ;^)
        #
        # Add the plugin definition to the module it is defined in.
        self._add_to_module()

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _add_to_module(self):
        """ Adds the plugin definition to the module it is defined in. """

        # Get the module that the plugin definition is defined in.
        module = self._get_module()

        if not hasattr(module, '__plugin_definitions__'):
            module.__plugin_definitions__ = {}

        module.__plugin_definitions__[self.id] = self

        return
    
    def _get_module(self):
        """ Returns the module that the plugin definition is defined in. """

        # 1 level up gives us the '_add_to_module' frame...
        # 2 levels up gives us our constructor frame...
        # 3 levels up gives us the frame that called the constructor!
        frame = sys._getframe(3)
        
        # This returns a tuple containing the last 5 elements of the frame
        # record which are:
        #
        # - the filename
        # - the line number
        # - the function name
        # - the list of lines of context from the source code
        # - the index of the current line within that list
        filename, lineno, funcname, source, index = inspect.getframeinfo(frame)

        # The plugin definition's location is the directory containing the
        # module that it is defined in.
        self.location = os.path.dirname(os.path.abspath(filename))
                
        # We can't use 'inspect.getmodulename' here as it gets confused because
        # of our import hook.
        #
        # e.g. for the core plugin definition:
        #
        # using inspect -> 'core_plugin_definition'
        # using ours    -> 'enthought.envisage.core.core_plugin_definition'
        module_name = get_module_name(filename)
        
        # If the module name starts with '<' then it is (probably) either
        # '<stdin>' which will be the case if the definition is defined in the
        # standard interactive Python shell, or '<input>' if the definition is
        # defined in the wx Python shell.
        if module_name.startswith('<'):
            module_name = '__main__'
            
        return sys.modules[module_name]

#### EOF ######################################################################
