""" The plugin definition for the Envisage core plugin. """


# Enthought library imports.
from enthought.traits.api import Any, Bool, Dict, Enum, File, HasTraits, List
from enthought.traits.api import Float, Str

# Local imports.
#
# fixme: This seemed like a good idea at the time (to insulate plugins from
# changes to the package structure), but now I'm not so sure. It seems cleaner
# to import ExtensionItem, ExtensionPoint and PluginDefinition from Envisage
# itself so that there is no dependency on the core plugin (i.e. you could
# write your own core plugin if required).
#
# To sum up(!), don't use these anymore, instead use:
#
# from enthought.envisage import ExtensionItem,ExtensionPoint,PluginDefinition
from enthought.envisage.core.extension_item import ExtensionItem
from enthought.envisage.core.extension_point import ExtensionPoint
from enthought.envisage.core.plugin_definition import PluginDefinition


# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = "enthought.envisage.core"

# fixme: Magic strings are generally acceptable in plugin definitions since
# they use Python as a slightly worse XML.
ADDITIONAL_PLUGINS = 'additional_plugins'

###############################################################################
# Extension points.
###############################################################################

class ApplicationObject(ExtensionPoint):
    """ A 'well-known' or 'published' object. """

    # The class name of a factory class that creates the application object.
    #
    # You don't have to specify a factory class name as long as you provide the
    # class name of the application object you wish to create
    # (see the **class_name** attribute).
    factory_class_name = Str

    # The name of the application object class.
    #
    # If you specify this instead of a factory class name, then an instance of
    # the default factory will be created for you.
    class_name = Str

    # Positional arguments to pass to the factory.
    args = List

    # Keyword arguments to pass to the factory.
    kw = Dict

    # The mechanism by which the object should be published.
    #
    # The UOL (Universal Object Locator) can currently be:
    #
    # 1) ``'service://a_service_identifier'``
    # 2) ``'name://a/path/through/the/naming/service'``
    uol = Str

    # Arbitrary properties to publish with the object (if the publishing
    # mechanism supports properties).
    properties = Dict


class Synchronizer(ExtensionPoint):
    """ Synchronizes traits on 'well-known' or 'published' objects. """

    # The UOL (Universal Object Locator) of the first object to synchronize.
    uol1 = Str

    # The name of the trait on the first object to synchronize.
    name1 = Str

    # The UOL of the second object to synchronize.
    uol2 = Str

    # The name of the trait on the second object to synchronize (If omitted,
    # the default is the same value as **name1**).
    name2 = Str

    # Is the synchronization two-way (True) or just from 'uol1.name1' to
    # 'uol2.name2' (False)?
    mutual = Bool(True)


class AdapterFactory(ExtensionItem):
    """ A factory that generates adapters from a target class to an adaptee
    class.
    """

    # The name of the class that implements the factory.
    class_name = Str

    # The name of the class of objects that the factory can adapt.
    adaptee_class_name = Str

    # The name of the adapter class (the class that adapts the adaptee to the
    # target class).
    #
    # This is optional. If you specify this and not **class_name**, an
    # instance of the default adapter factory will be created (i.e., you can
    # get away with not having to create the factory class!).
    adapter_class_name = Str

    # The name of the target class (the class that the factory can adapt
    # objects to).
    target_class_name = Str


class Category(ExtensionItem):
    """ A class that extends another class definition.

    Similar to "categories" in Traits and in Cocoa.
    """

    # The name of the class that implements the category.
    class_name = Str

    # The name of the class that the category extends.
    target_class_name = Str


class Hook(ExtensionItem):
    """ A method hook that allows a callable item to be called before or after
    a particular method is invoked. """

    # The name of the callable that implements the hook.
    callable_name = Str

    # The name of the class to hook.
    target_class_name = Str

    # The name of the method to hook.
    target_method_name = Str

    # Whether the hook is pre-method invocation or post-method invocation.
    type = Enum('pre', 'post')


class TypeManager(ExtensionPoint):
    """ A manager for type extensions (adapters, categories, hooks, etc.). """

    # The ID of the type manager to create or extend.
    id = Str('enthought.envisage.core')

    # Adapter factories known to this type manager.
    adapter_factories = List(AdapterFactory)

    # Categories known to this type manager.
    categories = List(Category)

    # The name of the class that implements the type manager.
    #
    # If this is not set, an instance of
    # **enthought.type_manager.api.TypeManager** is created.
    class_name = Str

    # Hooks known to this type manager.
    hooks = List(Hook)

    # The ID of the parent type manager.
    parent = Str


# DEPRECATED
Types = TypeManager


class PreferenceValue(ExtensionItem):
    """ Class for default preference values using a priority"""

    # Value assigned by a preference
    value = Any

    # Priority placed on the value in the preference
    priority = Float


class Preferences(ExtensionPoint):
    """ The set of preferences defined by a plugin. """

    # The default values for a plugin's preferences.
    # Use PreferenceValue, instead of Any for default values that need
    # to be prioritized for use.
    defaults = Dict(Str, Any)

    # target plugin ID
    target = Str


class Runnable(ExtensionPoint):
    """ An entry point for an application. """

    # The name of the class that implements the **enthought.envisage.Runnable**
    # interface.
    class_name = Str

###############################################################################
# Extensions.
###############################################################################

# Core plugin preferences.
preferences = Preferences(
    defaults = {
        ADDITIONAL_PLUGINS : [],
    }
)

# The default type manager.
type_manager = TypeManager()

# fixme: This looks startlingly like an implementation class (i.e., it is
# neither an extension point nor an extension). It should probably be in its
# own file - and definitely not in here. This is a plugin *definition* ;^)
class UserAddedPlugin(HasTraits):
    """ DEPRECATED: Not for general use.
    """
    # The path to the plugin
    path = File()

    # Load this plugin automatically?
    enabled = Bool(False)

###############################################################################

class CorePluginDefinition(PluginDefinition):
    """ The definition of the core plugin. """

    # The plugin's globally unique identifier.
    id = ID

    # The name of the class that implements the plugin.
    class_name = ID + ".core_plugin.CorePlugin"

    # General information about the plugin.
    name          = "Envisage Core Plugin"
    version       = "1.0.0"
    provider_name = "Enthought Inc"
    provider_url  = "www.enthought.com"
    autostart     = True

    # The IDs of the plugins that this plugin requires to be started before
    # it is started.
    requires = []

    # The extension points offered by this plugin.
    extension_points = [
        ApplicationObject, Preferences, Runnable, TypeManager, Types
    ]

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = [preferences, type_manager]

#### EOF ######################################################################
