""" Builds menus from action, group, and menu extensions. """


# Enthought library imports.
from enthought.traits.api import Any, HasTraits


class MenuBuilder(HasTraits):
    """ Builds menus from action, group, and menu extensions. """

    #### 'MenuBuilder' interface ##############################################

    # A factory that produces action implementations from action extensions.
    action_factory = Any

    # A factory that produces group implementations from group extensions.
    group_factory = Any

    # A factory that produces menu manager implementations from menu
    # extensions.
    menu_factory = Any

    ###########################################################################
    # 'MenuBuilder' interface.
    ###########################################################################

    def initialize_menu_manager(self, menu_manager, action_set_manager, root):
        """ Initializes a menu manager from an action set manager. """

        # Create the menu and group structure.
        self._create_menus_and_groups(menu_manager, action_set_manager, root)

        # Add all of the actions.
        self._add_actions(menu_manager, action_set_manager, root)

        return

    ###########################################################################
    # Protected 'MenuBuilder' interface.
    ###########################################################################

    def _create_action(self, action_extension):
        """ Creates an action implementation from an action extension. """

        if self.action_factory is not None:
            action = self.action_factory.create_action(action_extension)

        else:
            raise NotImplementedError

        return action

    def _create_group(self, group_extension):
        """ Creates a group implementation from a group extension. """

        if self.group_factory is not None:
            group = self.group_factory.create_group(group_extension)

        else:
            raise NotImplementedError

        return group

    def _create_menu(self, menu_extension):
        """ Creates a menu manager implementation from a menu extension. """

        if self.menu_factory is not None:
            menu = self.menu_factory.create_menu(menu_extension)
            for group_extension in menu_extension.groups:
                menu.append(self._create_group(group_extension))

        else:
            raise NotImplementedError

        return menu

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_menus_and_groups(self, menu_manager, action_set_manager, root):
        """ Creates the menu and group structure. """

        # Get all of the menus and groups.
        menus_and_groups = action_set_manager.get_menus(root)
        menus_and_groups.extend(action_set_manager.get_groups(root))

        while len(menus_and_groups) > 0:
            start = len(menus_and_groups)

            for item in menus_and_groups[:]:
                # Attempt to place a group.
                #
                # fixme: We should probably use a nicer way to differentiate
                # between menus and groups ;^)
                if not hasattr(item, 'groups'):
                    path = item.location.path

                    target = self._find_menu_manager(menu_manager, root, path)
                    if target is not None:
                        if self._add_group(target, item):
                            menus_and_groups.remove(item)

                # Attempt to place a menu.
                else:
                    # The last component of the path is the group that the menu
                    # is to be added to.
                    components = item.location.path.split('/')
                    path       = '/'.join(components[:-1])

                    target = self._find_menu_manager(menu_manager, root, path)
                    if target is not None:
                        group = target.find_group(components[-1])
                        if group is not None:
                            if self._add_menu(group, item):
                                menus_and_groups.remove(item)

            end = len(menus_and_groups)

            # If we didn't succeed in placing *any* menus or groups then we
            # must have a circular reference!
            if start == end:
                ids = [item.id for item in menus_and_groups]
                raise ValueError('Could not place %s with ids [%s]' % \
                    (str(menus_and_groups), ids))

        return

    def _add_group(self, menu_manager, group):
        """ Adds a group to a menu manager.

        Returns **True** only if the group could be added.

        """

        if len(group.location.before) > 0:
            item = menu_manager.find_group(group.location.before)
            if item is None:
                return False

            index = menu_manager.groups.index(item)

        elif len(group.location.after) > 0:
            item = menu_manager.find_group(group.location.after)
            if item is None:
                return False

            index = menu_manager.groups.index(item) + 1

        else:
            index = -1

        menu_manager.insert(index, self._create_group(group))

        return True

    def _add_menu(self, group, menu):
        """ Adds a menu manager to a group.

        Returns **True** only if the menu could be added.

        """

        if len(menu.location.before) > 0:
            item = group.find(menu.location.before)
            if item is None:
                return False

            index = group.items.index(item)

        elif len(menu.location.after) > 0:
            item = group.find(menu.location.after)
            if item is None:
                return False

            index = group.items.index(item) + 1

        else:
            index = -1

        group.insert(index, self._create_menu_manager(menu))

        return True

    def _find_menu_manager(self, menu_manager, root, path):
        """ Returns the menu manager at the specified path.

        Returns **None** if the menu manager cannot be found.

        """

        components = path.split('/')
        if len(components) == 1:
            menu_manager = menu_manager

        else:
            components = path.split('/')
            menu_manager = menu_manager.find_item('/'.join(components[1:]))

        return menu_manager

    def _add_actions(self, menu_manager, action_set_manager, root):
        """ Adds all of the actions to the menu manager. """

        actions = action_set_manager.get_actions(root)

        while len(actions) > 0:
            start = len(actions)

            for action in actions[:]:
                for location in action.locations:
                    location_root = self._get_location_root(
                        location, action._action_set_.aliases
                    )
                    path = location.path
                    if location_root == root:
                        # The last component of the path is the group that the
                        # menu is to be added to.
                        components = path.split('/')
                        path       = '/'.join(components[:-1])

                        target = self._find_menu_manager(
                            menu_manager, root, path
                        )
                        if target is not None:
                            group = target.find_group(components[-1])
                            if group is not None:
                                if self._add_action(group, action, location):
                                    actions.remove(action)

            end = len(actions)

            # If we didn't succeed in placing *any* actions then we must have a
            # circular reference!
            if start == end:
                msg = ''
                for action in actions:
                    msg += 'Action [%s] with id [%s],' % (action, action.id)
                raise ValueError('Could not place actions: %s' % msg)

        return

    def _add_action(self, group, action, location):
        """ Adds an action to a group.

        Returns **True** only if the action could be added.

        """

        if len(location.before) > 0:
            item = group.find(location.before)
            if item is None:
                return False

            index = group.items.index(item)

        elif len(location.after) > 0:
            item = group.find(location.after)
            if item is None:
                return False

            index = group.items.index(item) + 1

        else:
            index = -1

        group.insert(index, self._create_action(action))

        return True

    def _get_location_root(self, location, aliases):
        """ Returns the effective root for a location. """

        components = location.path.split('/')
        if components[0] in aliases:
            location_root = aliases[components[0]]

        else:
            location_root = components[0]

        return location_root

#### EOF ######################################################################
