/*!***************************************************************************

  module      : XMLMerge.cpp

  -------------------------------------------------------------------------

  responsible : PeterG (Goertzp)

  special area: IDL to Code Generator   
  description : Merge XML files
				
  
  last changed: 2004-09-21
  see also    : 

  -------------------------------------------------------------------------

  copyright:    Copyright by SAP AG, 2000



    ========== licence begin LGPL
    Copyright (C) 2000 SAP AG

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    ========== licence end

*****************************************************************************/


#include <stdio.h>

#if defined(_WIN32) && (_MSC_VER >= 1300)
#include <iostream>
#include <fstream>
#else
#include <iostream.h>
#include <string.h>
#include <fstream.h>
#endif

#include "idl/XMLMerge/XMLMergeErrorHandling.hpp"


const int maxNameLength = 250;
typedef char fileName[maxNameLength+6];
/* 250 characters, inclusive extension ".xml"
   +5: extension "__mgd" for output file
   +1: NULL terminator "\0" */

const int maxLineLength = 1024;
typedef char XMLLine[maxLineLength];


#define exitRoutine fclose(fpImp); \
                    rc = 1;        \
                    return rc


int lineLen(XMLLine line) {

  int strLen = strlen(line), termLen = 0;

  if (strLen > 0) {
    if ( (line[strLen-1] == 10) || (line[strLen-1] == 13) ) {
      termLen++;
    }
  }
  if (strLen > 1) {
    if ( (line[strLen-2] == 10) || (line[strLen-2] == 13) ) {
      termLen++;
    }
  }
  
  return (strLen - termLen);

}


int getFirstOccurrence( XMLLine line,
                        char * str,
                        int offset = 0) {

  int len = lineLen(line), len2 = strlen(str), i;

  for ( i = offset; i < len-len2+1; i++ ) {
    if (!strncmp( &line[i], str, len2 )) {
      return i;
    }
  }

  return -1;

}


bool onlyLeadingBlanks( XMLLine line,
                        int pos ) {

  bool rc = false;
  int pos2 = lineLen(line), i;
  
  if ( (0 <= pos) && (pos <= pos2) ) {
    rc = true;
    for ( i = 0; i < pos; i++ ) {
      if (!(' ' == line[i])) {
        rc = false;
        break;
      }
    }
  }

  return rc;

}


bool onlyTrailingBlanks( XMLLine line,
                         int pos ) {

  bool rc = false;
  int pos2 = lineLen(line), i;
  
  if ( (-1 <= pos) && (pos <= pos2-1) ) {
    rc = true;
    for ( i = pos+1; i < pos2; i++ ) {
      if (!(' ' == line[i])) {
        rc = false;
        break;
      }
    }
  }

  return rc;

}


bool emptyLine(XMLLine line) {

  bool rc = true;
  int len = lineLen(line), i;

  for ( i = 0; i < len; i++ ) {
    if (!(' ' == line[i])) {
      rc = false;
      break;
    }
  }

  return rc;

}


void initFileName(fileName fName) {

  for ( int i = 0; i < maxNameLength+6; i++ ) {
    fName[i] = '\0';
  }

}


int checkXMLFile(char * name) {

  int rc = 0;
  int len = strlen(name);

  if (len < 4) {
    rc = 1;
  }
  else if ( !(name[len-4] == '.') ||
            !(name[len-3] == 'x') ||
            !(name[len-2] == 'm') ||
            !(name[len-1] == 'l') ) {
    rc = 1;
  }
  if (rc == 1) {
    XMLMerge_ErrorMsg(C_NO_XML_FILE);
  }

  return rc;

}


int getFileName( XMLLine line,
                 int & sp,
                 int & ep,
                 fileName currFile ) {

  int rc = 0;
  int len = lineLen(line), i;

  /* get start and end position */
  sp = ep = 0;
  for ( i = 0; i < len; i++ ) {
    if ( line[i] == '"' ) {
      sp = i + 1;
      break;
    }
  }
  for ( i = i + 1; i < len; i++ ) {
    if ( line[i] == '"' ) {
      ep = i - 1;
      break;
    }
  }
  if (!( (0 < sp) && (sp < ep) && (ep < len) )) {
    XMLMerge_ErrorMsg( C_FILENAME_NOT_FOUND, currFile);
    rc = 1;
    return rc;
  }

  /* check for additional double quotes */
  for ( i = i + 1; i < len; i++ ) {
    if ( line[i] == '"' ) {
      XMLMerge_ErrorMsg( C_TOO_MANY_DOUBLE_QUOTES, currFile );
      rc = 1;
      return rc;
    }
  }

  /* check for blanks in name */
  for ( i = sp; i < ep + 1; i++ ) {
    if ( line[i] == ' ' ) {
      XMLMerge_ErrorMsg( C_BLANK_IN_FILENAME, currFile );
      rc = 1;
      return rc;
    }
  }

  /* check name length */
  if ((ep-sp+1) > maxNameLength) {
    XMLMerge_ErrorMsg( C_FILENAME_TOO_LONG, currFile );
    rc = 1;
    return rc;
  }

  return rc;

}


int processInput( int argC,
                  char * argV[],
                  fileName inFile,
                  fileName outFile ){

  int rc = 0;
  int len, argN, i;

  if (argC < 2) {
    XMLMerge_ErrorMsg(C_TOO_FEW_INPUT_PARAMS);
    rc = 1;
    return rc;
  }

  len = strlen(argV[1]);

//  if (argV[1][0] == '-') {
//    XMLMerge_ErrorMsg(C_XML_FILE_FIRST);
//    rc = 1;
//    return rc;
//  }

  if (len < 2) {
    rc = 1;
  }
  else if //(!( (argV[1][0] == '-') && (argV[1][1] == 'F') )) {
          (strncmp( &argV[1][0], "-F", 2)) {
    rc = 1;
  }
  if (1 == rc) {
    XMLMerge_ErrorMsg(C_XML_FILE_FIRST);
    return rc;
  }

  if (len-2 > maxNameLength) {
    XMLMerge_ErrorMsg(C_FILENAME_TOO_LONG);
    rc = 1;
    return rc;
  }

  rc = checkXMLFile(&argV[1][2]);
  if (rc) {
    return rc;
  }

  if (!strncmp( &argV[1][len-9], "__mgd", 5)) {
    XMLMerge_ErrorMsg(C_BAD_INPUT_FILENAME);
    rc = 1;
    return rc;
  }

  for ( i = 2; i < len; i++) {
    if (argV[1][i] == ' ') {
      XMLMerge_ErrorMsg(C_BLANK_IN_INPUT_FILENAME);
      rc = 1;
      return rc;
    }
  }

  if (argC > 2) {
    for ( argN = 2; argN < argC; argN++ ) {
      if (strlen(&argV[argN][0]) < 2) {
        rc = 1;
      }
      else if (strncmp( &argV[argN][0], "-I", 2)) {
        rc = 1;
      }
      if (rc == 1) {
        XMLMerge_ErrorMsg(C_BAD_INPUT);
        return rc;
      }
    }
  }

  strcpy( &inFile[0], &argV[1][2] );
  strncpy( &outFile[0], &argV[1][2], len-2-4 );
  strcpy( &outFile[len-2-4], "__mgd.xml" );

  return rc;

}


FILE * openFile( fileName fName,
                 int argC,
                 char * argV[] ) {

  FILE * fp = NULL;
  fileName fName2;
  int argN, len, len2, addSlash;

  /* try to open file directly */
  fp = fopen( fName, "r" );

  /* in case of failure, try to open file by adding leading search path */
  if ( (NULL == fp) && (argC > 2) ) {
    for ( argN = 2; argN < argC; argN++ ) {
      addSlash = 0;
      initFileName(fName2);
      len = strlen(&argV[argN][0]);
      if (!( ('/' == argV[argN][len-1]) || ('\\' == argV[argN][len-1]) )) {
        addSlash = 1;
      }
      len2 = len-2 + addSlash + strlen(fName);
      if (len2 > maxNameLength) {
        XMLMerge_WarnMsg(C_PATH_TOO_LONG);
        continue;
      }
      strncpy( &fName2[0], &argV[argN][2], len-2 );
      if (addSlash) {
        fName2[len-2] = '/';
      }
      strcpy( &fName2[len-1], fName );
      fp = fopen( fName2, "r" );
      if (fp) {
        break;
      }
    }
  }

  /* return file pointer */
  return fp;

}


bool importStartTagOK( XMLLine line,
                       fileName fName ) {

  bool rc = false;
  int pos1 = -1, pos2 = -1;

  /* line must contain import start tag */
  pos1 = getFirstOccurrence( line, "<IMPORT FILE=" );
  if (!(pos1 > -1)) {
    XMLMerge_ErrorMsg(C_BAD_CALL_OF_IMPORTSTARTTAGOK);
    return rc;
  }

  /* only leading blanks allowed */
  if (!onlyLeadingBlanks( line, pos1 )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_IMPORTSTARTLAB, fName );
    return rc;
  }

  /* no additional '<' may occur in line */
  if (getFirstOccurrence( line, "<", pos1+1 ) > -1) {
    XMLMerge_ErrorMsg( C_ADDLAB_IN_IMPORTSTARTLINE, fName );
    return rc;
  }

  /* '>' must occur in same line */
  pos2 = getFirstOccurrence( line, ">", pos1+1 );
  if (!(pos2 > -1)) {
    XMLMerge_ErrorMsg( C_IMPORTSTARTRAB_MISSING, fName );
    return rc;
  }

  /* no additional '>' may occur in line */
  if (getFirstOccurrence( line, ">", pos2+1 ) > -1) {
    XMLMerge_ErrorMsg( C_ADDRAB_IN_IMPORTSTARTLINE, fName );
    return rc;
  }

  /* '>' may not have '/' in front */
  if (getFirstOccurrence( line, "/>" ) == pos2-1) {
    XMLMerge_ErrorMsg( C_BAD_IMPORTSTARTRAB, fName );
    return rc;
  }

  /* only trailing blanks allowed */
  if (!onlyTrailingBlanks( line, pos2 )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEHIND_IMPORTSTARTRAB, fName );
    return rc;
  }

  /* ok */
  rc = true;
  return rc;

}


bool importEndTagOK( XMLLine line,
                     fileName fName ) {

  bool rc = false;
  int pos = -1;

  /* line must contain import end tag */
  pos = getFirstOccurrence( line, "</IMPORT>" );
  if (!(pos > -1)) {
    XMLMerge_ErrorMsg(C_BAD_CALL_OF_IMPORTENDTAGOK);
    return rc;
  }

  /* only leading blanks allowed */
  if (!onlyLeadingBlanks( line, pos )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_IMPORTENDLAB, fName );
    return rc;
  }
  
  /* only trailing blanks allowed */
  if (!onlyTrailingBlanks( line, pos+8 )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEHIND_IMPORTENDRAB, fName );
    return rc;
  }

  /* ok */
  rc = true;
  return rc;

}


int importFile( FILE * fpOut,
                XMLLine lineIn,
                int argC,
                char * argV[],
                fileName currFile ) {

  int rc = 0,
      i = 0, sp = 0, ep = 0,
      pos1 = -1, pos2 = -1;
  XMLLine line;
  fileName impFile;
  FILE * fpImp = NULL;
  bool xmlDeclStartFound = false, xmlDeclEndFound = false,
       idl2xmlStartFound = false, idl2xmlEndFound = false,
       idl2xmlEndTagFound = false,
       inImportMode = false;

  /* determine name of file to be imported */
  rc = getFileName( lineIn, sp, ep, currFile );
  if (rc) {
    return rc;
  }

  /* open file to be imported */
  memcpy( impFile, &lineIn[sp], ep-sp+1 );
  impFile[ep-sp+1] = '\0';
  rc = checkXMLFile(impFile);
  if (rc) {
    return rc;
  }
  fpImp = openFile( impFile, argC, argV );
  if (NULL == fpImp) {
    XMLMerge_ErrorMsg( C_FILE_OPEN_ERROR, impFile );
    rc = 1;
    return rc;
  }

  /* skip header lines */
  /* XML declaration tag must be first tag */
  while (!( xmlDeclStartFound && xmlDeclEndFound )) {
    if (!fgets( line, maxLineLength, fpImp )) {
      XMLMerge_ErrorMsg( C_FGETS_FAILED, impFile );
      break;
    }
    /* skip empty line */
    if (emptyLine(line)) {
      continue;
    }
    /* line is not empty */
    pos1 = pos2 = -1;
    if (!xmlDeclStartFound) {
      pos1 = getFirstOccurrence( line, "<" );
      if (pos1 > -1) {
        pos2 = getFirstOccurrence( line, "<?xml" );
        /* '<' must be followed directly by '?xml' */
        if (pos1 != pos2) {
          XMLMerge_ErrorMsg( C_FIRSTLAB_NOT_XMLDECLLAB, impFile );
          exitRoutine;
        }
        /* only leading blanks before '<?xml' allowed */
        else if (!onlyLeadingBlanks( line, pos2 )) {
          XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_XMLDECLLAB, impFile );
          exitRoutine;
        }
        /* behind '<?xml' no further '<' allowed */
        else if (getFirstOccurrence( line, "<", pos2+1 ) > -1) {
          XMLMerge_ErrorMsg( C_ADDLAB_IN_XMLDECLLAB_LINE, impFile );
          exitRoutine;
        }
        /* ok */
        else {
          xmlDeclStartFound = true;
        }
      }
      else {
        XMLMerge_ErrorMsg( C_NONEMPTYLINE_BEFORE_XMLDECLSTART, impFile );
        exitRoutine;
      }
    }
    if (xmlDeclStartFound) {
      /* no '<' allowed, if no more in start line of tag */
      if (pos1 == -1) {
        if (getFirstOccurrence( line, "<" ) > -1) {
          XMLMerge_ErrorMsg( C_ADDLAB_INSIDE_XMLDECL, impFile);
          exitRoutine;
        }
      }
      /* no '<' occurs, or still in start line of tag */
      pos1 = getFirstOccurrence( line, ">" );
      if (pos1 > -1) {
        pos2 = getFirstOccurrence( line, "?>" );
        /* '?' must be directly in front of '>' */
        if (!( (-1 < pos2) && (pos2 == pos1-1) )) {
          XMLMerge_ErrorMsg( C_FIRSTRAB_NOT_XMLDECLRAB, impFile );
          exitRoutine;
        }
        /* only trailing blanks behind '?>' allowed */
        else if (!onlyTrailingBlanks( line, pos2+1 )) {
          XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEHIND_XMLDECLRAB, impFile );
          exitRoutine;
        }
        /* ok */
        else {
          xmlDeclEndFound = true;
        }
      }
    }
  }
  if (!( xmlDeclStartFound && xmlDeclEndFound )) {
    XMLMerge_ErrorMsg( C_XMLDECLTAG_NOT_FOUND, impFile );
    exitRoutine;
  }
  /* IDL2XML tag must be second tag */
  while (!( idl2xmlStartFound && idl2xmlEndFound )) {
    if (!fgets( line, maxLineLength, fpImp )) {
      XMLMerge_ErrorMsg( C_FGETS_FAILED, impFile );
      break;
    }
    /* skip empty line */
    if (emptyLine(line)) {
      continue;
    }
    /* line is not empty */
    pos1 = pos2 = -1;
    if (!idl2xmlStartFound) {
      pos1 = getFirstOccurrence( line, "<" );
      if (pos1 > -1) {
        pos2 = getFirstOccurrence( line, "<IDL2XML" );
        /* '<' must be followed directly by 'IDL2XML' */
        if (pos1 != pos2) {
          XMLMerge_ErrorMsg( C_SECONDLAB_NOT_IDL2XMLSTARTLAB, impFile );
          exitRoutine;
        }
        /* only leading blanks before '<IDL2XML' allowed */
        else if (!onlyLeadingBlanks( line, pos2 )) {
          XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_IDL2XMLSTARTLAB, impFile );
          exitRoutine;
        }
        /* behind '<IDL2XML' no further '<' allowed */
        else if (getFirstOccurrence( line, "<", pos2+1 ) > -1) {
          XMLMerge_ErrorMsg( C_ADDLAB_IN_IDL2XMLSTART_LABLINE, impFile );
          exitRoutine;
        }
        /* ok */
        else {
          idl2xmlStartFound = true;
        }
      }
      else {
        XMLMerge_ErrorMsg( C_NOTONLY_EMPTYLINES_BETWEEN_XMLDECLEND_AND_IDL2XMLSTART, impFile );
        exitRoutine;
      }
    }
    if (idl2xmlStartFound) {
      /* no '<' allowed, if no more in start line of tag */
      if (pos1 == -1) {
        if (getFirstOccurrence( line, "<" ) > -1) {
          XMLMerge_ErrorMsg( C_ADDLAB_INSIDE_IDL2XMLSTART, impFile);
          exitRoutine;
        }
      }
      /* no '<' occurs, or still in start line of tag */
      pos1 = getFirstOccurrence( line, ">" );
      if (pos1 > -1) {
        if (strncmp( impFile, "livecachetypes.xml", 18 )) {
          pos2 = getFirstOccurrence( line, "\">" );
        }
        else {
          pos2 = getFirstOccurrence( line, "L>" );
        }
        /* '"' resp. "L" must be directly in front of '>' */
        if (!( (-1 < pos2) && (pos2 == pos1-1) )) {
          XMLMerge_ErrorMsg( C_SECONDRAB_NOT_IDL2XMLSTARTRAB, impFile );
          exitRoutine;
        }
        /* only trailing blanks behind '">' allowed */
        else if (!onlyTrailingBlanks( line, pos2+1 )) {
          XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEHIND_IDL2XMLSTARTRAB, impFile );
          exitRoutine;
        }
        /* ok */
        else {
          idl2xmlEndFound = true;
        }
      }
    }
  }
  if (!( idl2xmlStartFound && idl2xmlEndFound )) {
    XMLMerge_ErrorMsg( C_IDL2XMLSTARTTAG_NOT_FOUND, impFile );
    exitRoutine;
  }

  /* import relevant content */
  while ( fgets( line, maxLineLength, fpImp ) && (!idl2xmlEndTagFound) ) {
    pos1 = pos2 = -1;
    /* in import mode only empty line or end tag allowed */
    if (inImportMode) {
      if (emptyLine(line)) {
        continue;
      }
      else {
        pos1 = getFirstOccurrence( line, "</IMPORT>" );
        if (pos1 > -1) {
          if (importEndTagOK( line, impFile )) {
            /* write import end tag and continue */
            fputs( line, fpOut );
            inImportMode = false;
            continue;
          }
          else {
            exitRoutine;
          }
        }
        else {
          XMLMerge_ErrorMsg( C_NOTONLY_EMPTYLINES_BETWEEN_IMPORT_START_AND_END, impFile );
          exitRoutine;
        }
      }
    }
    /* check for illegal INCLUDE tag */
    if (getFirstOccurrence( line, "<INCLUDE FILE=" ) > -1) {
      XMLMerge_ErrorMsg( C_BAD_INCLUDE_TAG, impFile);
      exitRoutine;
    }
    /* IDL2XML end tag may not be copied */
    pos1 = getFirstOccurrence( line, "</IDL2XML>" );
    if (pos1 > -1) {
      if (!onlyLeadingBlanks( line, pos1) ) {
        XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_IDL2XMLENDLAB, impFile );
        exitRoutine;
      }
      idl2xmlEndTagFound = true;
      break;
    }
    /* import */
    fputs( line, fpOut );
    pos1 = getFirstOccurrence( line, "<IMPORT FILE=" );
    if (pos1 > -1) {
      if(importStartTagOK( line, impFile )) {
        rc = importFile( fpOut, line, argC, argV, impFile );
        if (0 == rc) {
          inImportMode = true;
        }
        else {
          exitRoutine;
        }
      }
      else {
        exitRoutine;
      }
    }
  }
  
  /* check if </IDL2XML> was found */
  if (!idl2xmlEndTagFound) {
    XMLMerge_ErrorMsg( C_IDL2XMLEND_NOT_FOUND, impFile );
    exitRoutine;
  }

  /* finish subroutine */
  fclose(fpImp);
  return rc;

}


bool includeTagOK( XMLLine line,
                   fileName fName ) {

  bool rc = false;
  int pos1 = -1, pos2 = -1;

  /* line must contain import start tag */
  pos1 = getFirstOccurrence( line, "<INCLUDE FILE=" );
  if (!(pos1 > -1)) {
    XMLMerge_ErrorMsg(C_BAD_CALL_OF_INCLUDETAGK);
    return rc;
  }

  /* only leading blanks allowed */
  if (!onlyLeadingBlanks( line, pos1 )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEFORE_INCLUDELAB, fName );
    return rc;
  }

  /* no additional '<' may occur in line */
  if (getFirstOccurrence( line, "<", pos1+1 ) > -1) {
    XMLMerge_ErrorMsg( C_ADDLAB_IN_INCLUDELINE, fName );
    return rc;
  }

  /* '/>' must occur in same line */
  pos2 = getFirstOccurrence( line, "/>", pos1+1 );
  if (!(pos2 > -1)) {
    XMLMerge_ErrorMsg( C_INCLUDERAB_MISSING, fName );
    return rc;
  }

  /* only trailing blanks allowed */
  if (!onlyTrailingBlanks( line, pos2+1 )) {
    XMLMerge_ErrorMsg( C_NOTONLY_BLANKS_BEHIND_INCLUDERAB, fName );
    return rc;
  }

  /* ok */
  rc = true;
  return rc;

}


int includeFile( FILE * fpOut,
                 XMLLine lineIn,
                 int argC,
                 char * argV[],
                 fileName currFile ) {

  int rc = 0,
      i = 0, sp = 0, ep = 0;
  XMLLine line;
  fileName incFile;
  FILE * fpInc = NULL;

  /* determine name of file to be included */
  rc = getFileName( lineIn, sp, ep, currFile );
  if (rc) {
    return rc;
  }

  /* open file to be included */
  memcpy( incFile, &lineIn[sp], ep-sp+1 );
  incFile[ep-sp+1] = '\0';
  rc = checkXMLFile(incFile);
  if (rc) {
    return rc;
  }
  fpInc = openFile( incFile, argC, argV );
  if (NULL == fpInc) {
    XMLMerge_ErrorMsg( C_FILE_OPEN_ERROR, incFile );
    rc = 1;
    return rc;
  }

  /* include content */
  while (fgets( line, maxLineLength, fpInc )) {
    /* check for illegal IMPORT tag */
    if (getFirstOccurrence( line, "<IMPORT FILE=" ) > -1 ) {
      XMLMerge_ErrorMsg( C_BAD_IMPORT_TAG, incFile );
      rc = 1;
      break;
    }
    /* include */
    if (getFirstOccurrence( line, "<INCLUDE FILE=" ) > -1 ) {
      if(includeTagOK( line, incFile )) {
        rc = includeFile( fpOut, line, argC, argV, incFile );
        if (1 == rc) {
          break;
        }
      }
      else {
        rc = 1;
        break;
      }
    }
    else {
      fputs( line, fpOut );
    }
  }

  /* finish subroutine */
  fclose(fpInc);
  return rc;

}


int mergeFiles( FILE * fpIn,
                FILE * fpOut,
                fileName inFile,
                int argC,
                char * argV[]) {

  int rc = 0, pos = -1;
  XMLLine line;
  bool inImportMode = false;

  while (fgets( line, maxLineLength, fpIn )) {
    pos = -1;
    /* in import mode */
    if (inImportMode) {
      if (emptyLine(line)) {
        continue;
      }
      else {
        pos = getFirstOccurrence( line, "</IMPORT>" );
        if (pos > -1) {
          if (importEndTagOK( line, inFile )) {
            /* write import end tag and continue */
            fputs( line, fpOut );
            inImportMode = false;
            continue;
          }
          else {
            rc = 1;
            break;
          }
        }
        else {
          XMLMerge_ErrorMsg( C_NOTONLY_EMPTYLINES_BETWEEN_IMPORT_START_AND_END, inFile );
          rc = 1;
          break;
        }
      }
    }
    /* check if import mode to be started */
    pos = getFirstOccurrence( line, "<IMPORT FILE=" );
    if (pos > -1) {
      if (importStartTagOK( line, inFile )) {
        /* write import tag and import file */
        fputs( line, fpOut );
        rc = importFile( fpOut, line, argC, argV, inFile );
        /* import was successful */
        if (0 == rc) {
          inImportMode = true;
          continue;
        }
        /* import was not successful */
        else {
          rc = 1;
          break;
        }
      }
      else {
        rc = 1;
        break;
      }
    }
    /* check if file to be included */
    pos = getFirstOccurrence( line, "<INCLUDE FILE=" );
    if (pos > -1) {
      if (includeTagOK( line, inFile )) {
        /* include file */
        rc = includeFile( fpOut, line, argC, argV, inFile );
        /* include was successful */
        if ( 0 == rc) {
          continue;
        }
        /* include was not successful */
        else {
          rc = 1;
          break;
        }
      }
      else {
        rc = 1;
        break;
      }
    }
    /* copy line */
    fputs( line, fpOut );
  }

  return rc;

}


//extern "C" __stdcall DebugBreak();

int main( int argC , char * argV[] ) {

//  DebugBreak();

  int rc, rc2, rc3;

  fileName inFile;
  fileName outFile;
  FILE * fpIn  = NULL;
  FILE * fpOut = NULL;

  /* process input */
  rc = processInput( argC, argV, inFile, outFile );
  if (rc) {
    return rc;
  }  

  /* open files */
  fpIn = fopen( inFile, "r" );
  if (NULL == fpIn) {
    XMLMerge_ErrorMsg( C_FILE_OPEN_ERROR, inFile );
    rc = 1;
    return rc;
  }
  fpOut = fopen( outFile, "w" );
  if (NULL == fpOut) {
    fclose(fpIn);
    XMLMerge_ErrorMsg( C_FILE_OPEN_ERROR, outFile);
    rc = 1;
    return rc;
  }

  /* process input file */
  rc = mergeFiles( fpIn, fpOut, inFile, argC, argV );

  /* close files */
  rc2 = fclose(fpIn);
  rc3 = fclose(fpOut);
  if (rc2) {
    XMLMerge_ErrorMsg( C_FILE_CLOSE_ERROR, inFile );
    rc = 1;
  }
  if (rc3) {
    XMLMerge_ErrorMsg( C_FILE_CLOSE_ERROR, outFile );
    rc = 1;
  }

  return rc;

}

