#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/types.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "mpfio.h"
#include "editor.h"
#include "editorcb.h"
#include "editorop.h"
#include "editorfio.h"

#include "manedit.h"


static int EditorFileLoadProgressCB(
	long cur, long max, void *data
);
static int EditorFileSaveProgressCB(
        long cur, long max, void *data
);

static void EditorFileLoadParseTitleHeading(
        editor_struct *editor, const char *line_ptr,
        editor_item_struct *item_ptr
);
static void EditorFileLoadParseHeader(
        editor_struct *editor, editor_item_struct *item_ptr
);

static void EditorFileSaveFormatHeader(
	editor_struct *editor, editor_item_struct *item_ptr,
	mp_header_struct *mp_header
);
static void EditorFileSaveFormatSection(
        editor_struct *editor, editor_item_struct *item_ptr,
        mp_section_struct *mp_section
);


int EditorFileLoad(
	editor_struct *editor, const char *filename,
	GtkCTreeNode *branch,	/* Insert after this toplevel trunk. */
	gbool as_template
);
int EditorFileSave(
	editor_struct *editor,
	GtkCTreeNode *branch	/* Save data on this branch. */
);
int EditorFileSaveAs(
        editor_struct *editor, const char *filename,
        GtkCTreeNode *branch    /* Save data on this branch. */
);



/*
 *	Editor file load progress callback.
 *
 *	Updates the progress bar.
 */
static int EditorFileLoadProgressCB(
        long cur, long max, void *data
)
{
	gfloat percent;
	editor_struct *editor = (editor_struct *)data;
	if(editor == NULL)
	    return(0);

	if(max > 0)
	    percent = (gfloat)cur / (gfloat)max;
	else
	    percent = 1.0;

	EditorSetStatusProgress(editor, percent);

	return(0);
}

/*
 *      Editor file save progress callback.
 *
 *      Updates the progress bar.
 */
static int EditorFileSaveProgressCB(
        long cur, long max, void *data
)
{ 
        gfloat percent;
        editor_struct *editor = (editor_struct *)data;
        if(editor == NULL)
            return(0);

        if(max > 0)
            percent = (gfloat)cur / (gfloat)max;
        else
            percent = 1.0;

        EditorSetStatusProgress(editor, percent);

        return(0);
}

/*
 *	Parses the line containing the groff title heading tag.
 *
 *	Fetched values will be stored on the item_ptr structure which
 *	is assumed to be of type EditorItemTypeHeader.
 */
static void EditorFileLoadParseTitleHeading(
	editor_struct *editor, const char *line_ptr,
	editor_item_struct *item_ptr
)
{
	int line_len, seg_len;
	const char *line_end, *line_next;
	char *tmp_buf;
	int tmp_buf_len;


	while(ISBLANK(*line_ptr))
	    line_ptr++;

	line_len = strlen(line_ptr);
	line_end = (const char *)(line_ptr + line_len);


	/* Argument 1, name. */
	line_next = line_ptr;
	if((*line_next) == '"')
	{
            line_ptr++;
	    line_next++;
	    while(((*line_next) != '"') && ((*line_next) != '\0'))
	    {
		if((*line_next) == '\\')
		{
		    line_next++;
		    if((*line_next) != '\0')
			line_next++;
		}
		else
		{
		    line_next++;
		}
	    }
	}
	else
	{
	    while(!ISBLANK(*line_next) && ((*line_next) != '\0'))
		line_next++;
	}
	/* Allocate memory for new argument. */
	tmp_buf_len = seg_len = MAX(line_next - line_ptr, 0);
	tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
	if(tmp_buf != NULL)
	{
	    if(tmp_buf_len > 0)
		memcpy((void *)tmp_buf, (const void *)line_ptr, tmp_buf_len);
	    tmp_buf[tmp_buf_len] = '\0';
	}
	/* Set new value. */
	free(item_ptr->header_name);
	item_ptr->header_name = tmp_buf;
        /* Seek past current quote if any. */
        if((*line_next) == '"')
            line_next++;

        /* Argument 2, section number. */
        line_ptr = line_next;
	while(ISBLANK(*line_ptr))
	    line_ptr++;
	line_next = line_ptr;
        if((*line_next) == '"')
        {
            line_ptr++;
            line_next++;         
            while(((*line_next) != '"') && ((*line_next) != '\0'))
            {
                if((*line_next) == '\\')
                {
                    line_next++;
                    if((*line_next) != '\0')
                        line_next++;
                }
                else
                {
                    line_next++;
                }
            }
        }
        else      
        {    
            while(!ISBLANK(*line_next) && ((*line_next) != '\0'))
                line_next++;
        }
        /* Allocate memory for new argument. */
        tmp_buf_len = seg_len = MAX(line_next - line_ptr, 0);
        tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
        if(tmp_buf != NULL)
        {
            if(tmp_buf_len > 0)
                memcpy((void *)tmp_buf, (const void *)line_ptr, tmp_buf_len);
            tmp_buf[tmp_buf_len] = '\0';
        }
        /* Set new value. */
        free(item_ptr->header_section_number);
        item_ptr->header_section_number = tmp_buf;
        /* Seek past current quote if any. */
        if((*line_next) == '"')
            line_next++;

        /* Argument 3, version. */
        line_ptr = line_next;
        while(ISBLANK(*line_ptr))
            line_ptr++;
        line_next = line_ptr;
        if((*line_next) == '"')
        {
            line_ptr++;
            line_next++;
            while(((*line_next) != '"') && ((*line_next) != '\0'))
            {       
                if((*line_next) == '\\')
                {
                    line_next++;
                    if((*line_next) != '\0')
                        line_next++;
                }
                else
                {
                    line_next++;
                }
            }
        }   
        else
        {   
            while(!ISBLANK(*line_next) && ((*line_next) != '\0'))
                line_next++;
        }
        /* Allocate memory for new argument. */
        tmp_buf_len = seg_len = MAX(line_next - line_ptr, 0);
        tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
        if(tmp_buf != NULL)
        {
            if(tmp_buf_len > 0)
                memcpy((void *)tmp_buf, (const void *)line_ptr, tmp_buf_len);
            tmp_buf[tmp_buf_len] = '\0';
        }
        /* Set new value. */
        free(item_ptr->header_version);
        item_ptr->header_version = tmp_buf;
	/* Seek past current quote if any. */
	if((*line_next) == '"')
	    line_next++;

        /* Argument 4, author. */
        line_ptr = line_next;
        while(ISBLANK(*line_ptr))
            line_ptr++;
        line_next = line_ptr;
        if((*line_next) == '"')
        {
	    line_ptr++;
            line_next++;
            while(((*line_next) != '"') && ((*line_next) != '\0'))
            {
                if((*line_next) == '\\')
                {
                    line_next++;
                    if((*line_next) != '\0')
                        line_next++;
                }
                else
                {
                    line_next++;
                }
            }
        }   
        else
        {
            while(!ISBLANK(*line_next) && ((*line_next) != '\0'))
                line_next++;
        }
        /* Allocate memory for new argument. */
        tmp_buf_len = seg_len = MAX(line_next - line_ptr, 0);
        tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
        if(tmp_buf != NULL)
        {
            if(tmp_buf_len > 0)
                memcpy((void *)tmp_buf, (const void *)line_ptr, tmp_buf_len);
            tmp_buf[tmp_buf_len] = '\0';
        }
        /* Set new value. */
        free(item_ptr->header_author);
        item_ptr->header_author = tmp_buf;
        /* Seek past current quote if any. */
        if((*line_next) == '"')
            line_next++;

        /* Argument 5, catagory. */
        line_ptr = line_next;
        while(ISBLANK(*line_ptr))
            line_ptr++;
        line_next = line_ptr;
        if((*line_next) == '"')
        {
            line_ptr++;
            line_next++;
            while(((*line_next) != '"') && ((*line_next) != '\0'))
            {
                if((*line_next) == '\\')
                {
                    line_next++;
                    if((*line_next) != '\0')
                        line_next++;
                }
                else
                {
                    line_next++;
                }
            }
        }
        else
        {
            while(!ISBLANK(*line_next) && ((*line_next) != '\0'))
                line_next++;
        }
        /* Allocate memory for new argument. */
        tmp_buf_len = seg_len = MAX(line_next - line_ptr, 0);
        tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
        if(tmp_buf != NULL)
        {
            if(tmp_buf_len > 0)
                memcpy((void *)tmp_buf, (const void *)line_ptr, tmp_buf_len);
            tmp_buf[tmp_buf_len] = '\0';
        }
        /* Set new value. */
        free(item_ptr->header_catagory);
        item_ptr->header_catagory = tmp_buf;
        /* Seek past current quote if any. */
        if((*line_next) == '"')
            line_next++;

}

/*
 *	Post processing for loading of header item. Parses each line
 *	on the given item_ptr assumed to be of type EditorItemTypeHeader.
 */
static void EditorFileLoadParseHeader(
	editor_struct *editor, editor_item_struct *item_ptr
)
{
	char ***line, *line_ptr;
	int i, line_num, *total_lines;


	if((editor == NULL) || (item_ptr == NULL))
	    return;

	/* Get pointer to lines array on item data. */
	line = &item_ptr->line;
	total_lines = &item_ptr->total_lines;

	/* Go through each line. */
	for(line_num = 0; line_num < (*total_lines); line_num++)
	{
	    line_ptr = (*line)[line_num];
	    if(line_ptr == NULL)
		continue;

	    /* Seek past initial spaces. */
	    while(ISBLANK(*line_ptr))
		line_ptr++;

	    /* Title heading? */
	    if(strpfx(line_ptr, ".TH"))
	    {
		/* Seek line_ptr to first argument. */
		while(!ISBLANK(*line_ptr) && ((*line_ptr) != '\0'))
		    line_ptr++;
		while(ISBLANK(*line_ptr))
		    line_ptr++;

		/* Parse title heading, this will parse the title heading
		 * arguments pointed to by line_ptr and updates the
		 * values on the item_ptr.
		 */
		EditorFileLoadParseTitleHeading(
		    editor, line_ptr, item_ptr
		);

		/* Remove this line from the lines array. */
		free((*line)[line_num]);
		(*line)[line_num] = line_ptr = NULL;

		(*total_lines) = (*total_lines) - 1;
		for(i = line_num; i < (*total_lines); i++)
		    (*line)[i] = (*line)[i + 1];

		line_num--;	/* Go back one line. */
		continue;
	    }
	    /* Comment (dot backslash doublequote)? */
	    else if(strpfx(line_ptr, ".\\\""))
            {
		char *new_line_ptr;

                /* Seek line_ptr to first and only argument. */
                while(!ISBLANK(*line_ptr) &&
                      ((*line_ptr) != '\0')
                )
                    line_ptr++;

                while(ISBLANK(*line_ptr))
                    line_ptr++;

		/* Make a copy of the comment line as the new line. This
		 * strips the comment sequence prefix and creates a
		 * regular line with the comment's contents without
		 * the comment prefix.
		 */
		new_line_ptr = strdup(line_ptr);

                /* Free old line. */
                free((*line)[line_num]);
		line_ptr = NULL;

		/* Set new line into line array. */
		(*line)[line_num] = new_line_ptr;
		new_line_ptr = NULL;

		/* Get pointer to new line from array. */
		line_ptr = (*line)[line_num];

		continue;
	    }

	}
}


/*
 *	Preprocessing for the manual page header structure, coppies values
 *	from the item data structure to the manual page header structure
 *	and formats it.
 *
 *	Last line on the manual page header structure will be appended
 *	as a title heading statement and all other lines will be added
 *	first as comments.
 */
static void EditorFileSaveFormatHeader(
        editor_struct *editor, editor_item_struct *item_ptr, 
        mp_header_struct *mp_header
)
{
        const char *line_ptr;
	char *new_line_ptr;
        int i, new_line_len;


        if((editor == NULL) || (item_ptr == NULL) || (mp_header == NULL))
            return;

	/* Delete any existing lines on manual page header structure
	 * just in case.
	 */
	StringFreeArray(mp_header->line, mp_header->total_lines);
	mp_header->line = NULL;
	mp_header->total_lines = 0;


	/* Allocate lines pointer array on manual page header structure
	 * to match all lines on item data structure.
	 */
	mp_header->total_lines = item_ptr->total_lines;
	if(mp_header->total_lines > 0)
	{
	    mp_header->line = (char **)calloc(
		mp_header->total_lines, sizeof(char *)
	    );
	    if(mp_header->line == NULL)
	    {
		mp_header->total_lines = 0;
		return;
	    }
	}

	/* First copy each line on the item data structure to the manual
	 * page header structure. Prefixing a comment prefix to each
	 * line.
	 */
	for(i = 0; i < item_ptr->total_lines; i++)
	{
	    line_ptr = (const char *)item_ptr->line[i];
	    if(line_ptr == NULL)
		continue;

	    /* Create new line with a groff comment prefix. */
	    new_line_ptr = strdup("");
	    new_line_ptr = strcatalloc(new_line_ptr, ".\\\" ");
	    new_line_ptr = strcatalloc(new_line_ptr, line_ptr);

	    /* Set new line to manual page header structure line array. */
	    mp_header->line[i] = new_line_ptr;
	    new_line_ptr = NULL;
	}


	/* Append title heading line to manual page header structure. */
	new_line_len = 32 +
	    strlen((item_ptr->header_name == NULL) ?
		"(null)" : item_ptr->header_name
	    ) + 2 +
            strlen((item_ptr->header_section_number == NULL) ?
                "(null)" : item_ptr->header_section_number
            ) + 2 +
            strlen((item_ptr->header_version == NULL) ?
                "(null)" : item_ptr->header_version
            ) + 2 +
            strlen((item_ptr->header_author == NULL) ?
                "(null)" : item_ptr->header_author
            ) + 2 +
            strlen((item_ptr->header_catagory == NULL) ?
                "(null)" : item_ptr->header_catagory
            ) + 2;
	new_line_ptr = (char *)malloc((new_line_len + 1) * sizeof(char));
	if(new_line_ptr != NULL)
	{
	    sprintf(
		new_line_ptr,
		".TH \"%s\" \"%s\" \"%s\" \"%s\" \"%s\"",
		item_ptr->header_name,
		item_ptr->header_section_number,
		item_ptr->header_version,
		item_ptr->header_author,
		item_ptr->header_catagory
	    );

	    i = mp_header->total_lines;
	    mp_header->total_lines = i + 1;
	    mp_header->line = (char **)realloc(
		mp_header->line,
		mp_header->total_lines * sizeof(char *)
	    );
	    if(mp_header->line == NULL)
	    {
		mp_header->total_lines = 0;
		free(new_line_ptr);
	    }
	    else
	    {
		mp_header->line[i] = new_line_ptr;
	    }
	    new_line_ptr = NULL;
	}
}

/*
 *      Preprocessing for the manual page section structure, coppies
 *	values from the item data structure to the manual page section
 *	structure and formats it.
 *
 *	Section name will be coppied as well.
 */
static void EditorFileSaveFormatSection(
        editor_struct *editor, editor_item_struct *item_ptr,
        mp_section_struct *mp_section
)
{
        const char *line_ptr;
        char *new_line_ptr;
        int i;


        if((editor == NULL) || (item_ptr == NULL) || (mp_section == NULL))
            return;

        /* Delete any existing lines on manual page section structure
         * just in case.
         */
        StringFreeArray(mp_section->line, mp_section->total_lines);
        mp_section->line = NULL;
        mp_section->total_lines = 0;
        
         
        /* Allocate lines pointer array on manual page section structure
         * to match all lines on item data structure.
         */
        mp_section->total_lines = item_ptr->total_lines;
        if(mp_section->total_lines > 0)
        {
            mp_section->line = (char **)calloc(
                mp_section->total_lines, sizeof(char *)
            );
            if(mp_section->line == NULL)
            {
                mp_section->total_lines = 0;
                return;
            }
        }

        /* First copy each line on the item data structure to the manual
         * page section structure.
         */
        for(i = 0; i < item_ptr->total_lines; i++)
        {
            line_ptr = (const char *)item_ptr->line[i];
            if(line_ptr == NULL)
                continue;
            
            /* Create new line. */
	    new_line_ptr = strdup(line_ptr);

            /* Set new line to manual page section structure line array. */
            mp_section->line[i] = new_line_ptr;
        }

	/* Copy section name. */
	if(item_ptr->section_name != NULL)
	{
	    free(mp_section->section);
	    mp_section->section = strdup(item_ptr->section_name);
	}

	return;
}



/*
 *	Loads a manual page into the editor and adds a toplevel layout
 *	trunk after the given toplevel trunk branch.
 *
 *	The new branch will have branch item data type EditorItemTypeFile.
 *
 *	If the given branch is not of type EditorItemTypeFile and or it
 *	is not a toplevel trunk branch then it's toplevel parent will
 *	be recursed to and picked as the toplevel branch.
 *
 *	Existing layout trunks on the editor will not be modified.
 *
 *	If as_template is TRUE then the loaded maual page will not have
 *	its file_name_full set on the main trunk branch and the editor's
 *	last opened path will not be modified.
 *
 *	Returns non-zero on error, where -1 is a general error and -2 is
 *	file not found/not allowed to read.
 */
int EditorFileLoad(
	editor_struct *editor, const char *filename,
        GtkCTreeNode *branch,	/* Insert after this toplevel trunk. */
	gbool as_template
)
{
	int i, n, status;
	char *strptr;
	const char *cstrptr;
	mp_header_struct *mp_header;
	mp_section_struct **mp_section, *mp_section_ptr;
	int total_mp_sections;
	GtkCTree *ctree;
	GtkCList *clist;
	GtkCTreeNode *trunk, *new_trunk = NULL;
	editor_item_struct *editor_item_ptr;
	medit_core_struct *core_ptr;
	medit_pixmaps_list_struct *pixmaps_list;


	if((editor == NULL) || (filename == NULL))
	    return(-1);

	if(!editor->initialized)
	    return(-1);

	/* Already processing? */
	if(editor->processing)
	    return(-1);

	ctree = (GtkCTree *)editor->layout_ctree;
	clist = (GtkCList *)ctree;
	if(ctree == NULL)
	    return(-1);

	core_ptr = (medit_core_struct *)editor->core_ptr;
	if(core_ptr == NULL)
	    return(-1);

	/* Get pointer to pixmaps list on core structure. */
	pixmaps_list = &core_ptr->pixmaps_list;


        /* Be sure to apply values of currently selected branch (not
         * the given branch) before opening. This ensures we have data
         * up to date before loading new data.
         */
        EditorDoApplyValues(editor, editor->selected_branch);


	/* Find branch node to insert after. */
	trunk = NULL;
	if(branch != NULL)
	{
	    GtkCTreeRow *node_row;
	    GtkCTreeNode *next_node;

	    /* Start from given branch and itterate to toplevel. */
	    next_node = branch;
	    while(1)
	    {
		node_row = GTK_CTREE_ROW(next_node);
		if(node_row == NULL)
		{
		    break;
		}
		else if(node_row->parent == NULL)
		{
		    trunk = node_row->sibling;
		    break;
		}
		else
		{
		    next_node = node_row->parent;
		}
	    }
	}

	/* Value of trunk is now set as the branch node to be the
	 * sibling of where to insert the new trunk after (which may be
	 * NULL).
	 */

	/* Mark editor as processing. */
	editor->processing = TRUE;
        EditorSetBusy(editor);
        EditorSetStatusMessage(editor, "Loading manual page...");


	/* Load manual page file. */
	mp_header = NULL;
	mp_section = NULL;
	total_mp_sections = 0;
	status = MPLoad(
	    filename, NULL,
	    &mp_header,
	    &mp_section, &total_mp_sections,
	    editor, EditorFileLoadProgressCB
	);
	if(status)
	{
	    /* Error occured while loading. */

            /* Deallocate loaded header if any. */
            MPDeleteHeader(mp_header);
            mp_header = NULL;

            /* Deallocate loaded sections if any. */
            MPDeleteAllSections(mp_section, total_mp_sections);
            mp_section = NULL;
            total_mp_sections = 0;

            EditorSetStatusMessage(editor, "Loading done");
            EditorSetStatusProgress(editor, 0.0);

            /* Update menus. */
            EditorUpdateMenus(editor);

	    editor->processing = FALSE;
            EditorSetReady(editor);   

	    return(status);
	}
	else
	{
	    /* Loaded successfully. */

	    EditorSetStatusMessage(editor, "Load: Allocating memory...");

	    /* Add new trunk to layout ctree widget. */
	    if(editor->total_layout_trunks < 0)
		editor->total_layout_trunks = 0;

	    n = editor->total_layout_trunks;
	    editor->total_layout_trunks = n + 1;
	    editor->layout_trunk = (GtkCTreeNode **)realloc(
		editor->layout_trunk,
		editor->total_layout_trunks * sizeof(GtkCTreeNode *)
	    );
	    if(editor->layout_trunk == NULL)
	    {
		editor->total_layout_trunks = 0;
	    }
	    else
	    {
		gchar *tmp_path;
		gchar *tmp_text;
		gchar *text[1];

		/* Get only file name (without absolute path) from
		 * filename as tmp_path and make a copy of it.
		 */
		cstrptr = strrchr(filename, DIR_DELIMINATOR);
		tmp_path = strdup((cstrptr == NULL) ? filename : cstrptr + 1);
		text[0] = tmp_path;

		/* Create new trunk, insert before trunk which is the
                 * sibling of the toplevel trunk of given branch and
                 * may be NULL.
                 */
		new_trunk = branch = gtk_ctree_insert_node(
		    ctree,
		    NULL,		/* No parent, it's toplevel. */
		    trunk,		/* Insert before this sibling. */
		    text,
		    MEDIT_LIST_ICON_TEXT_SPACING,
		    pixmaps_list->manual_closed_20x20,
		    pixmaps_list->manual_closed_20x20_mask,
                    pixmaps_list->manual_opened_20x20,
                    pixmaps_list->manual_opened_20x20_mask,
		    FALSE,		/* Is leaf. */
		    TRUE		/* Expanded. */
		);
		editor->layout_trunk[n] = branch;
		trunk = branch;

		/* Create editor item structure. */
                editor_item_ptr = EditorItemNew(
		    EditorItemTypeFile,
		    editor,
		    ctree,
		    NULL,		/* Parent. */
		    branch,		/* This branch. */
		    FALSE,		/* Is leaf. */
		    NULL, 0
		);
		if(editor_item_ptr != NULL)
		{
		    /* Loading as template? */
		    if(as_template)
		    {
			char untitled_title[256];

			/* Loading as a template, give it an untitled
			 * name and increment untitled count on core.
			 */
			core_ptr->untitled_count++;
			sprintf(untitled_title, "Untitled%i.1",
			    core_ptr->untitled_count
			);
			free(editor_item_ptr->file_name);
                        editor_item_ptr->file_name = strdup(untitled_title);

			/* Mark as having changes. */
			editor_item_ptr->has_changes = TRUE;
		    }
		    else
		    {
			/* Loading regularly, so record file name as
			 * usual.
			 */
			free(editor_item_ptr->file_name);
			editor_item_ptr->file_name = strdup(tmp_path);
		    }

		    /* Record full file name only if not loading as
		     * a template, this is so that loaded templates will
		     * require a save-as if the user selects to save it.
		     */
		    if(!as_template)
		    {
		        free(editor_item_ptr->file_name_full);
		        editor_item_ptr->file_name_full = strdup(filename);
		    }
		}
		EditorBranchSetData(
		    ctree, branch,
		    editor_item_ptr, EditorItemDestroyCB
		);

                free(tmp_path);         /* Deallocate coppied tmp_path. */
                tmp_path = NULL;
		text[0] = NULL;

		/* Add header as first leaf. */
		if(mp_header != NULL)
		{
		    EditorSetStatusMessage(editor,
			"Load: Formatting header..."
		    );

		    text[0] = "Header";
		    /* Insert new branch on layout ctree. */
		    branch = gtk_ctree_insert_node(
			ctree, trunk, NULL,
			text,
			MEDIT_LIST_ICON_TEXT_SPACING,
			pixmaps_list->manpage_heading_20x20,
			pixmaps_list->manpage_heading_20x20_mask,
			pixmaps_list->manpage_heading_20x20,
			pixmaps_list->manpage_heading_20x20_mask,
			TRUE,		/* Is leaf. */
			FALSE		/* Expanded. */
		    );

		    /* Create new editor item structure. */
		    editor_item_ptr = EditorItemNew(
		        EditorItemTypeHeader,
			editor,
			ctree,
			trunk,		/* Parent. */
			branch,		/* This branch. */
			TRUE,		/* Is leaf. */
			mp_header->line,
			mp_header->total_lines
		    );

		    /* Header's lines are transfered, so mark them
		     * NULL and 0 on the source.
		     */
		    mp_header->line = NULL;
		    mp_header->total_lines = 0;

		    /* Set branch data as new editor_item_struct. */
		    EditorBranchSetData(
			ctree, branch,
			editor_item_ptr, EditorItemDestroyCB
		    );

		    /* Do post processing on header item data, need
		     * to fetch the header item's title heading.
		     */
		    EditorFileLoadParseHeader(
			editor, editor_item_ptr
		    );
		}

		/* Load each section. */
		for(i = 0; i < total_mp_sections; i++)
		{
		    mp_section_ptr = mp_section[i];
		    if(mp_section_ptr == NULL)
			continue;

		    /* Update status message. */
		    tmp_text = (char *)malloc(256 * sizeof(char));
		    if(tmp_text != NULL)
		    {
			sprintf(tmp_text,
			    "Load: Formatting section %i of %i...",
			    i + 1, total_mp_sections
			);
			EditorSetStatusMessage(editor, tmp_text);

			free(tmp_text);
			tmp_text = NULL;
		    }

		    /* Get section name and store into tmp_text, also
		     * text[0] will point to the new section name.
		     */
		    tmp_text = NULL;
		    if(mp_section_ptr->section == NULL)
		    {
			tmp_text = (char *)malloc(256 * sizeof(char));
			if(tmp_text != NULL)
			{
			    sprintf(tmp_text, "Section %i", i);
			    text[0] = tmp_text;
			}
			else
			{
			    text[0] = "Section";
			}
		    }
		    else
		    {
			text[0] = mp_section_ptr->section;
		    }
		    /* Insert new branch to layout ctree. */
                    branch = gtk_ctree_insert_node(
                        ctree, trunk, NULL,
                        text,
                        MEDIT_LIST_ICON_TEXT_SPACING,
                        pixmaps_list->manpage_section_20x20,
                        pixmaps_list->manpage_section_20x20_mask,
                        pixmaps_list->manpage_section_20x20,
                        pixmaps_list->manpage_section_20x20_mask,
			TRUE,		/* Is leaf. */
			FALSE		/* Expanded. */
		    );

                    /* Create new editor item structure. */  
                    editor_item_ptr = EditorItemNew(
                        EditorItemTypeSection,
                        editor,
                        ctree,
                        trunk,		/* Parent. */
                        branch,		/* This branch. */
                        TRUE,		/* Is leaf. */
                        mp_section_ptr->line,
                        mp_section_ptr->total_lines
                    );

                    /* Section's lines are transfered, so mark them
		     * NULL and 0 on the source.
		     */
                    mp_section_ptr->line = NULL;
                    mp_section_ptr->total_lines = 0;

		    if(editor_item_ptr != NULL)
		    {
			free(editor_item_ptr->section_name);
			editor_item_ptr->section_name = (
			    (text[0] == NULL) ? NULL : strdup(text[0])
			);
		    }

                    /* Set branch data as new editor_item_struct. */
                    EditorBranchSetData(
                        ctree, branch,
			editor_item_ptr, EditorItemDestroyCB
                    );

		    free(tmp_text);	/* Free tempory text. */
		    tmp_text = NULL;
		    text[0] = NULL;
		}
	    }
	}

        EditorSetStatusMessage(editor, "Load: Post processing...");

	/* Deallocate loaded header if any. */
	MPDeleteHeader(mp_header);
	mp_header = NULL;

	/* Deallocate loaded sections if any. */
	MPDeleteAllSections(mp_section, total_mp_sections);
	mp_section = NULL;
	total_mp_sections = 0;



	/* Update last opened path if not loading as template. */
	if(!as_template)
	{
	    free(editor->last_open_path);
	    editor->last_open_path = strdup(filename);
	    if(editor->last_open_path != NULL)
	    {
		strptr = strrchr(editor->last_open_path, DIR_DELIMINATOR);
		if(strptr != NULL)
		    (*strptr) = '\0';
	    }
	}


	/* Update layout ctree column width. */
	gtk_clist_set_column_width(
	    clist,
	    0,			/* Column. */
	    gtk_clist_optimal_column_width(clist, 0)
	);

	EditorSetStatusMessage(editor, "Loading done");
	EditorSetStatusProgress(editor, 0.0);

	/* Update menus. */
	EditorUpdateMenus(editor);

	/* Mark processing done. */
	editor->processing = FALSE;

        /* Force select new trunk, new_trunk is the toplevel trunk
         * node for this newly loaded file (do this after processing
	 * is marked done).
         */
	if(new_trunk != NULL)
	    EditorBranchSelect(editor, new_trunk);

        EditorSetReady(editor);

	return(0);
}



/*
 *	Saves the data on the given layout branch from it's toplevel
 *	trunk branch parent to all its children (thus including the
 *	given branch itself).
 *
 *	The given branch pointer may not be NULL, it may either be a child
 *	branch node or a toplevel trunk node. The file to be saved to will
 *	be the one stored on the toplevel trunk branch's item data
 *	file_name_full.
 *
 *	Has changes markers will not be modified in this function, it is
 *	up to the calling function to set that.
 *
 *      Returns non-zero on error, where -1 is a general error and -2 is
 *      cannot write to file.
 */
int EditorFileSave(
	editor_struct *editor,
	GtkCTreeNode *branch	/* Save data on this branch. */
)
{
	int i, status;
	const char *filename = NULL;
        mp_header_struct *mp_header;
        mp_section_struct **mp_section, *mp_section_ptr;
        int total_mp_sections;
        GtkCTree *ctree;
        GtkCTreeNode *trunk;
        editor_item_struct *item_ptr;
        medit_core_struct *core_ptr;


        if((editor == NULL) || (branch == NULL))
            return(-1);

	if(!editor->initialized)
	    return(-1);

	/* Already processing? */
	if(editor->processing)
	    return(-1);

        ctree = (GtkCTree *)editor->layout_ctree;
        if(ctree == NULL)
            return(-1);

        core_ptr = (medit_core_struct *)editor->core_ptr;
        if(core_ptr == NULL)
            return(-1);

	/* Be sure to apply values of currently selected branch (not
	 * the given branch) before saving. This ensures we save the most
	 * up to date data.
	 *
	 * This also needs to be done before we mark the editor as
	 * processing.
	 */
	EditorDoApplyValues(editor, editor->selected_branch);


	/* Find toplevel trunk branch of the given branch. */
	trunk = EditorItemGetToplevel(editor, branch);
	if(trunk != NULL)
	{
	    /* Get item pointer on trunk. */
	    item_ptr = EditorBranchGetData(ctree, trunk);
	    if(item_ptr != NULL)
	    {
		/* Skip type check. */

		/* Get absolute path of file name. */
		filename = item_ptr->file_name_full;
	    }
	}
	if(filename == NULL)
	{
	    /* Cannot save without filename. */
	    return(-1);
	}
	else
	{
	    char *tmp_buf;
	    int tmp_buf_len;


	    /* Check if file name is writeable. */
	    if(!access(filename, F_OK))
	    {
		if(access(filename, W_OK))
		{
		    /* Allocate and format message. */
		    tmp_buf_len = strlen(filename) + 256;
		    tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
		    if(tmp_buf != NULL)
			sprintf(
			    tmp_buf,
			    "Write access denied for file\n`%s'",
			    filename
			);

		    CDialogGetResponse(
"Write access denied!",
tmp_buf,
"Cannot write to the given path, check write\n\
permissions on the given path or try a\n\
different path.",
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );

		    /* Free message buffer. */
		    free(tmp_buf);
		    tmp_buf = NULL;

		    return(-2);
                }
	    }
	}


	/* Mark editor as processing. */
	editor->processing = TRUE;
	EditorSetBusy(editor);

        EditorSetStatusMessage(editor, "Formatting data...");
        EditorSetStatusProgress(editor, 0.0);


	/* Start on the trunk's first child and itterate through its
	 * siblings.
	 */
	branch = NULL;
	if(trunk != NULL)
	{
	    GtkCTreeRow *node_row;

            node_row = GTK_CTREE_ROW(trunk);
            if(node_row != NULL)
		branch = node_row->children;
	}

	/* Branch is now set to the first child branch node of the trunk. */
	mp_header = NULL;
	mp_section = NULL;
	total_mp_sections = 0;
	while(branch != NULL)
	{
	    GtkCTreeRow *node_row;

            item_ptr = EditorBranchGetData(ctree, branch);
            if(item_ptr != NULL)
            {
		/* Handle item data by type. */
		switch(item_ptr->type)
		{
		  case EditorItemTypeHeader:
		    /* Destroy current header if exists. */
		    MPDeleteHeader(mp_header);

		    /* Allocate new header structure. */		    
		    mp_header = (mp_header_struct *)calloc(
			1, sizeof(mp_header_struct)
		    );
		    if(mp_header != NULL)
		    {
                        /* Copy values from the item data structure
			 * to the manual page header structure formatted.
			 */
                        EditorFileSaveFormatHeader(
                            editor, item_ptr, mp_header
                        );
		    }
		    break;

                  case EditorItemTypeSection:
                    /* Allocate a new section structure. */

		    mp_section_ptr = NULL;

		    /* Increase total and allocate more pointers. */
		    if(total_mp_sections < 0)
			total_mp_sections = 0;
		    i = total_mp_sections;
		    total_mp_sections = i + 1;
                    mp_section = (mp_section_struct **)realloc(
                        mp_section,
			total_mp_sections * sizeof(mp_section_struct *)
                    );
                    if(mp_section == NULL)
                    {
			total_mp_sections = 0;   
                    }
		    else
		    {
			/* Allocate new section. */
			mp_section_ptr = (mp_section_struct *)calloc(
			    1, sizeof(mp_section_struct)
			);
			mp_section[i] = mp_section_ptr;
		    }
		    /* Successfully allocated a new section? */
		    if(mp_section_ptr != NULL)
		    {
                        /* Copy values from the item data structure
                         * to the manual page section structure formatted.
                         */
                        EditorFileSaveFormatSection(
                            editor, item_ptr, mp_section_ptr
                        );
		    }
                    break;
        

		}
	    }

	    /* Get sibling of branch. */
	    node_row = GTK_CTREE_ROW(branch);
            if(node_row == NULL)
		branch = NULL;
	    else
                branch = node_row->sibling;
	}


        EditorSetStatusMessage(editor, "Saving data...");

	/* Save to file. */
	status = MPSave(
	    filename, NULL,
	    mp_header,
	    mp_section, total_mp_sections,
	    (void *)editor, EditorFileSaveProgressCB
	);

	/* Deallocate data used for manual page format saving. */
	MPDeleteHeader(mp_header);
	mp_header = NULL;

	MPDeleteAllSections(mp_section, total_mp_sections);
	mp_section = NULL;
	total_mp_sections = 0;


        EditorSetStatusMessage(editor, "Saving done");
        EditorSetStatusProgress(editor, 0.0);



        /* Do not update last saved path. */

        /* Update menus. */
        EditorUpdateMenus(editor);

	editor->processing = FALSE;
        EditorSetReady(editor);

	return(status);
}

/*
 *	Updates the given branch toplevel trunk parent's item data's
 *	file names to the given filename. Then calls EditorFileSave().
 *
 *	The given branch and filename may not be NULL.
 *
 *	Returns non-zero on error, where -1 is a general error, -2 is
 *	cannot write to file, and -3 is user abort on confirmation of
 *	overwrite.
 */
int EditorFileSaveAs(
        editor_struct *editor, const char *filename,
        GtkCTreeNode *branch    /* Save data on this branch. */
)
{
	int status;
	char *strptr;
	char *tmp_buf;
	int tmp_buf_len;
	const char *cstrptr;
        GtkCTree *ctree;
	GtkCTreeRow *branch_row;
        GtkCTreeNode *trunk;
        editor_item_struct *item_ptr;
        medit_core_struct *core_ptr;
	medit_pixmaps_list_struct *pixmaps_list;
	struct stat stat_buf;


        if((editor == NULL) || (branch == NULL) || (filename == NULL))
            return(-1);

	if(!editor->initialized)
	    return(-1);

        ctree = (GtkCTree *)editor->layout_ctree;
        if(ctree == NULL) 
            return(-1);

        core_ptr = (medit_core_struct *)editor->core_ptr;
        if(core_ptr == NULL)
            return(-1);

	pixmaps_list = &core_ptr->pixmaps_list;

        /* Find toplevel trunk branch of the given branch which will
         * be the one to save data from.
         */
        trunk = NULL;
        if(branch != NULL)
        {
            GtkCTreeRow *node_row;
            GtkCTreeNode *next_node;  
        
            /* Start from given branch and itterate to toplevel. */
            next_node = branch;
            while(1)
            {
                node_row = GTK_CTREE_ROW(next_node);
                if(node_row == NULL)
                {
                    break;
                }
                else if(node_row->parent == NULL)
                {
                    trunk = next_node;
                    break;
                }
                else
                {
                    next_node = node_row->parent;
                }
            }
        }
	if(trunk == NULL)
	    return(-1);

	/* Get pointer to item data on trunk. */
	item_ptr = EditorBranchGetData(ctree, trunk);
	if(item_ptr == NULL)
	    return(-1);

	/* Check if file exists. */
	if(!stat(filename, &stat_buf))
	{
	    /* Allocate message buffer. */
	    tmp_buf_len = 256 + strlen(filename);
	    tmp_buf = (char *)malloc((tmp_buf_len + 1) * sizeof(char));
	    if(tmp_buf != NULL)
		sprintf(
		    tmp_buf,
"Overwrite existing file\n`%s'?",
		    filename
		);

	    /* Make confirmation. */
	    status = CDialogGetResponse(
"Confirm overwrite?",
		tmp_buf,
"You are being asked if you want to overwrite an existing\n\
file. If you say yes then any existing data on the file\n\
will be replaced with the data you are intending to save.\n\
This new data may not correspond to the data in the existing\n\
file, if you are unsure say cancel.",
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_CANCEL |
                CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_CANCEL
            );

	    /* Free message buffer. */
	    free(tmp_buf);
	    tmp_buf = NULL;

            switch(status)
            {
              case CDIALOG_RESPONSE_YES:
              case CDIALOG_RESPONSE_YES_TO_ALL:
              case CDIALOG_RESPONSE_OK:
                break;

              default:
                return(-3);
                break;
	    }
	}


	/* Got all required data, now update file name on item data. */

	/* Update item data's full file name. */
	free(item_ptr->file_name_full);
	item_ptr->file_name_full = strdup(filename);

	/* Update item data's file name. */
	cstrptr = strrchr(filename, DIR_DELIMINATOR);
	if(cstrptr == NULL)
	{
	    free(item_ptr->file_name);
	    item_ptr->file_name = strdup(filename);
	}
	else
	{
	    cstrptr += 1;	/* Seek past deliminator. */

	    free(item_ptr->file_name);
            item_ptr->file_name = strdup(cstrptr);
        }
	/* Update toplevel trunk branch name. */
	if(item_ptr->file_name != NULL)
	{
	    branch_row = GTK_CTREE_ROW(trunk);
	    if((branch_row == NULL) ? 0 : branch_row->expanded)
	    {
		gtk_ctree_node_set_pixtext(
		    ctree,
		    branch,
                    0,              /* Column. */
		    item_ptr->file_name,
		    MEDIT_LIST_ICON_TEXT_SPACING,
                    pixmaps_list->manual_opened_20x20,
                    pixmaps_list->manual_opened_20x20_mask
                );
	    }
	    else
	    {
                gtk_ctree_node_set_pixtext(
                    ctree,
                    branch,
                    0,              /* Column. */
                    item_ptr->file_name,
                    MEDIT_LIST_ICON_TEXT_SPACING,
                    pixmaps_list->manual_closed_20x20,
                    pixmaps_list->manual_closed_20x20_mask
                );
	    }
	}


	/* Call save procedure. */
	status = EditorFileSave(editor, trunk);

        /* Update last opened path. */
        free(editor->last_save_as_path);
        editor->last_save_as_path = strdup(filename);
        if(editor->last_save_as_path != NULL)
        {
            strptr = strrchr(editor->last_save_as_path, DIR_DELIMINATOR);
            if(strptr != NULL)
                (*strptr) = '\0';
        }

	return(status);
}
