/*************************************************************************/
/*                                                                       */
/*  MacroSystem - Powerful C++ template system.                          */
/*                                                                       */
/*  http://projects.nn.com.br/                                           */
/*                                                                       */
/*  Copyright (C) 2000 Gustavo Niemeyer <gustavo@nn.com.br>              */
/*                                                                       */
/*  This library is free software; you can redistribute it and/or        */
/*  modify it under the terms of the GNU Library General Public          */
/*  License as published by the Free Software Foundation; either         */
/*  version 2 of the License, or (at your option) any later version.     */
/*                                                                       */
/*  This library is distributed in the hope that it will be useful,      */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    */
/*  Library General Public License for more details.                     */
/*                                                                       */
/*  You should have received a copy of the GNU Library General Public    */
/*  License along with this library; if not, write to the                */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,         */
/*  Boston, MA  02111-1307, USA.                                         */
/*                                                                       */
/*************************************************************************/

// $Revision: 1.11 $
// $Date: 2000/02/03 06:25:49 $

#ifndef MACRO_SYSTEM_HH
#define MACRO_SYSTEM_HH

#include <map>
#include <string>
#include <list>
#include <iostream>
#include <functional>
#include <strstream>
#include <iomanip>

namespace macrosystem
{

  class MacroSystem;

  /** Manipulation of subscripting.
   *
   * Class used to manipulate MacroSystem's subscripting.
   *
   * @author Gustavo Niemeyer
   *
   */
  class MacroSystemManip
  {

  public:

    /// Method used to parse macro.
    inline string
    parse();

    /// Turns macro parsing on/off.
    inline void
    parse(const bool& _on_off);

    /// Verify if macro parsing is on
    inline const bool&
    parse_on();

    /** Set a macro.
     *
     *  This method is usefull if you want to turn macro parsing off
     *  when setting a macro.
     *
     */
    inline void
    set(const string& _value, const bool& _parse = true);

    /// Name of the current macro.
    inline const string&
    name();

    /// Value of the current macro.
    inline string&
    value();

    /// Erase a macro.
    inline void
    erase();

    /// Return if macro exists or not.
    inline bool
    exist() const;

    /// MacroSystemManip to string conversion.
    inline operator
    string() const;

    /// Overloaded operator to allow 'macro["name"] = string'.
    inline void
    operator=(const string& _s);

    /// Overloaded operator to allow 'macro["name"] += string'.
    inline void
    operator+=(const string& _s);

    /// Overloaded operator to allow 'string = macro["name"] + string'.
    inline string
    operator+(const string& _s);

    /// Overloaded operator to allow 'macro["name"] = char'.
    inline void
    operator=(const char& _c);

    /// Overloaded operator to allow 'macro["name"] += char'.
    inline void
    operator+=(const char& _c);

    /// Overloaded operator to allow 'string = macro["name"] + char'.
    inline string
    operator+(const char& _c);

    /** Overloaded operator to allow assignments from float.
     *
     *  This is a specialization of the template to create macros
     *  with 2 decimal digits.
     *
     */
    inline MacroSystemManip&
    operator=(const float& _f);

    /** Overloaded operator to allow assignments from double.
     *
     *  This is a specialization of the template to create macros
     *  with 2 decimal digits.
     *
     */
    inline MacroSystemManip&
    operator=(const double& _d);

    /// Operator to allow assignments from other types (int, etc).
    template<class T>
    inline MacroSystemManip&
    operator=(const T& _t);

    /// Operator to allow assignments from other types (int, float, etc).
    template<class T>
    inline MacroSystemManip&
    operator+=(const T& _t);

    friend inline istream&
    operator>>(istream& _is, MacroSystemManip& _msm);

  private:

    friend class MacroSystem;
    friend class MacroSystemIterator;

    /** Constructor.
     * @see MacroSystem::operator[]()
     */
    MacroSystemManip(MacroSystem& _ms)
      : m_ms(_ms)
      {}

    /** Method used by MacroSystem to set macro name when subscripting.
     * @see MacroSystem::operator[]()
     */
    inline MacroSystemManip&
    m_find(const string& _name);
    
    /// Verify if m_current is a valid iterator. Create one if not.
    inline void
    m_verify();

    /** Reference to MacroSystem using this MacroSystemManip.
     *
     * This is the key to subscripting. All operations done in this class
     * will be, indeed, done in m_ms MacroSystem.
     *
     * @see MacroSystemManip::MacroSystemManip()
     * @see MacroSystem::mm
     *
     */
    MacroSystem& m_ms;

    /// Current iterator.
    map< string, pair<string,bool> >::iterator m_current;
    
    /// Current macro name.
    string m_macro_name;

  };

  class MacroSystemIterator
  {
  public:

    inline
    MacroSystemIterator()
    {}

    inline MacroSystemManip*
    operator->()
    { m_msm->m_current = m_current; return m_msm; }

    inline MacroSystemManip*
    operator*()
    { m_msm->m_current = m_current; return m_msm; }
    
    inline bool
    operator==(const MacroSystemIterator& _msi) const
    { return (m_current == _msi.m_current); }

    inline MacroSystemIterator&
    operator++()
    { m_current++; return *this; }

    inline MacroSystemIterator&
    operator++(int)
    { m_current++; return *this; }

    inline MacroSystemIterator&
    operator--()
    { m_current--; return *this; }

    inline MacroSystemIterator&
    operator--(int)
    { m_current--; return *this; }

    inline MacroSystemIterator&
    operator=(const MacroSystemIterator& _msi)
    { m_msm = _msi.m_msm; m_current = _msi.m_current; return *this; }

  private:
    
    friend class MacroSystem;

    explicit inline
    MacroSystemIterator(MacroSystemManip& _msm,
		        const map< string, pair<string,bool> >::iterator& _current)
    { m_msm = &_msm; m_current = _current; }

    MacroSystemManip* m_msm;
    map< string, pair<string,bool> >::iterator m_current;

  };

  /** Main class.
   *
   * It is used to insert macros (nested or not) inside strings.
   * It was primarly developed to allow development of CGI systems that
   * separate the code from HTML. Indeed, it can be used with anything
   * that require a simple and fast method of parsing macros inside strings.
   *
   * @author Gustavo Niemeyer
   *
   */
  class MacroSystem {

  public:

    /** Constructor.
     *
     * @param _mstart Start of macro delimiter
     * @param _mend End of macro delimiter
     * @param _mquery Query symbol
     * @param _mnegate Negation symbol
     *
     */
    MacroSystem(const string& _mstart = "<%",
		const string& _mend = "%>",
		const string& _mquery = "?",
		const string& _mnegate = "!")
      : m_msm(*this), m_mstart(_mstart), m_mend(_mend),
	m_mquery(_mquery), m_mnegate(_mnegate)
      {}

    /** Parse macros from string.
     * @return Parsed string
     */
    string
    parse(const string& _s);

    /** Subscripting operator.
     *
     * This is the key of the system since almost everything should be
     * manipulated through subscripting.
     *
     * @return A reference to a MacroSystemManip object that will take care of
     * operators, assignment and etc.
     * @param _name Macro name. Shouldn't contain any white spaces.
     *
     */
    inline MacroSystemManip&
    operator[](const string& _name)
      { return m_msm.m_find(_name); };

    typedef MacroSystemIterator iterator;
    
    inline iterator
    begin()
      { return iterator(m_msm,m_macro_map.begin()); }

    inline const iterator
    end()
      { return iterator(m_msm,m_macro_map.end()); }

  private:

    friend class MacroSystemManip;

    /// Verify if string _s has the negation symbol.
    inline const bool
    m_is_negation(const string& _s) const;

    /// Remove negation symbol from string _s.
    inline void
    m_remove_negation(string& _s) const;

    /** MacroSystemManip that handles subscripting operations on this class.
     * @see MacroSystemManip
     */
    MacroSystemManip m_msm;

    /// Start of macro delimiter.
    string m_mstart;

    /// End of macro delimiter.
    string m_mend;

    /// Macro query symbol.
    string m_mquery;

    /// Macro negation symbol.
    string m_mnegate;

    /// Macro store.
    map< string, pair<string,bool> > m_macro_map;

  };

  //
  //  MacroSystemManip methods:
  //

  inline MacroSystemManip&
  MacroSystemManip::m_find(const string& _s)
  {
    m_macro_name = _s;
    m_current = m_ms.m_macro_map.find(_s);
    return *this;
  }

  inline void
  MacroSystemManip::m_verify()
  {
    if(m_current == m_ms.m_macro_map.end())
      m_current = (m_ms.m_macro_map.insert(map<string,pair<string,bool> >::value_type(m_macro_name, pair<string,bool>(string(),true)))).first;
  }
  
  inline string
  MacroSystemManip::parse()
  { 
    m_verify();
    return m_ms.parse((*m_current).second.first);
  }
  
  inline void
  MacroSystemManip::parse(const bool& _on_off)
  {
    m_verify();
    (*m_current).second.second = _on_off;
  }

  inline const bool&
  MacroSystemManip::parse_on()
  {
    m_verify();
    return (*m_current).second.second;
  }

  inline void
  MacroSystemManip::set(const string& _value, const bool& _parse)
  {
    if(m_current == m_ms.m_macro_map.end())
      m_current = (m_ms.m_macro_map.insert(map<string,pair<string,bool> >::value_type(m_macro_name, pair<string,bool>(_value,_parse)))).first;
    else
      {
	(*m_current).second.first = _value;
	(*m_current).second.second = _parse;
      }
  }    

  inline
  MacroSystemManip::operator string() const
  { 
    if(m_current == m_ms.m_macro_map.end())
      return string();
    else
      return (*m_current).second.first;
  }

  inline const string&
  MacroSystemManip::name()
  {
    m_verify();
    return m_current->first;
  }

  inline string&
  MacroSystemManip::value()
  {
    m_verify();
    return (*m_current).second.first;
  }

  inline void
  MacroSystemManip::erase()
  {
    if(m_current != m_ms.m_macro_map.end())
      m_ms.m_macro_map.erase(m_current);
  }
  
  inline bool
  MacroSystemManip::exist() const
  {
    return (m_current != m_ms.m_macro_map.end());
  }

  inline void
  MacroSystemManip::operator=(const string& _s)
  {
    m_verify();
    (*m_current).second.first = _s;
  }
  
  inline void
  MacroSystemManip::operator+=(const string& _s)
  {
    m_verify();
    (*m_current).second.first += _s;
  }
  
  inline string
  MacroSystemManip::operator+(const string& _s)
  {
    m_verify();
    return (*m_current).second.first + _s;
  }
  
  inline void
  MacroSystemManip::operator=(const char& _c)
  {
    m_verify();
    (*m_current).second.first = _c;
  }
  
  inline void
  MacroSystemManip::operator+=(const char& _c)
  { 
    m_verify();
    (*m_current).second.first += _c;
  }
  
  inline string
  MacroSystemManip::operator+(const char& _c)
  {
    m_verify();
    return (*m_current).second.first + _c;
  }

  inline MacroSystemManip&
  MacroSystemManip::operator=(const float& _f)
  {
    ostrstream os;
    os << setprecision(2) << setiosflags(ios::showpoint|ios::fixed)
       << _f << ends;
    string s(os.str());
    os.rdbuf()->freeze(0);
    m_verify();
    (*m_current).second.first = s;
    return *this;
  }

  inline MacroSystemManip&
  MacroSystemManip::operator=(const double& _d)
  {
    ostrstream os;
    os << setprecision(2) << setiosflags(ios::showpoint|ios::fixed)
       << _d << ends;
    string s(os.str());
    os.rdbuf()->freeze(0);
    m_verify();
    (*m_current).second.first = s;
    return *this;
  }

  template<class T>
  inline MacroSystemManip&
  MacroSystemManip::operator=(const T& _t)
  {
    ostrstream os;
    os << _t << ends;
    string s(os.str());
    os.rdbuf()->freeze(0);
    m_verify();
    (*m_current).second.first = s;
    return *this;
  }
  
  template<class T>
  inline MacroSystemManip&
  MacroSystemManip::operator+=(const T& _t)
  {
    ostrstream os;
    os << _t << ends;
    string s(os.str());
    os.rdbuf()->freeze(0);
    m_verify();
    (*m_current).second.first += s;
    return *this;
  }
  

  //
  // MacroSystem methods:
  //
  
  inline const bool
  MacroSystem::m_is_negation(const string& s) const
  { 
    return (string(s,0,m_mnegate.length()) == m_mnegate);
  }
  
  inline void
  MacroSystem::m_remove_negation(string& s) const
  {
    if(this->m_is_negation(s))
      s.erase(0,m_mnegate.length());
  }
  
  
  //
  // Other functions:
  //
  
  /** Operator to allow operations with strings.
   *
   * Example: string_a = string_b + macro["name"]
   *
   * @see MacroSystemManip
   *
   */
  inline string
  operator+(const string& _s, const MacroSystemManip& _msm)
  {
    return (_s+string(_msm));
  }
  
  /** Operator to allow output of macros.
   *
   * Example: cout << macro["name"];
   *
   * @see MacroSystemManip
   *
   */
  inline ostream& operator<<(ostream& _s, const MacroSystemManip& _msm)
  {
    return _s << string(_msm);
  }
  
  /** Operator to read a macro from an istream.
   *
   * It'll read the whole istream into that macro, not just until a
   * space as it usually does. That way, it's very easy to parse a
   * file.
   *
   * @see MacroSystemManip
   *
   */
  inline istream& operator>>(istream& _is, MacroSystemManip& _msm)
  {
    char c;
    _msm.m_verify();
    while(_is.get(c)) (*(_msm.m_current)).second.first += c;
    return _is;
  }

  /// Operator to import macro file.
  istream& operator>>(istream& _is, MacroSystem& _ms);

  /// Operator to export macro file.
  ostream& operator<<(ostream& _os, MacroSystem& _ms);

}

#endif // MACRO_SYSTEM_HH
