/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIBLENDING_HPP
  #define TEPDIBLENDING_HPP

  #include "TePDIAlgorithm.hpp"
  #include "TePDIParameters.hpp"
  
  #include <TeSharedPtr.h>
  #include <TeGeometry.h>
  #include <TeThreadFunctor.h>

  /**
   * @brief This is the class for digital image blending.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup TePDIGeneralAlgoGroup
   *
   * @note The general required parameters:
   * @param blending_type ( std::string ) - Blending type
   * ( see TePDIBlendStratFactory for reference ).
   * @param input_raster1 ( TePDITypes::TePDIRasterPtrType ) - 
   *  Input raster 1.
   * @param input_raster2 ( TePDITypes::TePDIRasterPtrType ) - 
   *  Input raster 2.
   * @param output_raster ( TePDITypes::TePDIRasterPtrType ) - Output raster.
   * @param channels1 ( std::vector< int > ) - The channels to process from 
   * input_raster1.
   * @param channels2 ( std::vector< int > ) - The channels to process from 
   * input_raster2.
   * @param raster1_pol_ptr ( TePDITypes::TePDIPolygonPtrType ) - 
   * Raster 1 polygon pointer ( related to
   * input_raster1 world reference - Just 1 linear ring allowed ).
   * @param raster2_pol_ptr ( TePDITypes::TePDIPolygonPtrType ) - 
   * Raster 2 polygon pointer ( related to 
   * input_raster2 world reference - Just 1 linear ring allowed ).
   * @param raster2_pol_offset_x ( double ) - Raster 2 polygon box X offset 
   * related to raster1 polygon box in input_raster1 matricial reference
   * ( offset_x = pol2_x - pol1_x ).
   * @param raster2_pol_offset_y ( double ) - Raster 2 polygon box Y offset 
   * related to raster1 polygon box in input_raster1 matricial reference
   * ( offset_y = pol2_y - pol1_y ). 
   *
   * @note The optional parameters are:
   *
   * @param auto_equalize ( int ) - If present ( any value ), input_raster2
   * auto-equalizing will be made ( using overlap area reference ).
   * @param dummy_value (double) - A dummy pixel value for use 
   * in pixels where no data is available ( input pixels with
   * dummy values will be ignored ).  
   * @param enable_multi_thread (int) - If present(any value) will
   * enable multi-threading.  
   *
   * @note The specific parameters: See each blending strategy for reference.
   *
   * @note The output_raster parameters will follow input_raster1 parameters
   * ( box, resolution, projection ).
   *
   * @note All rasters will be converted to multi-band.
   */
  class PDI_DLL TePDIBlending : public TePDIAlgorithm {
    public :
    
      /**
       * @brief Default Constructor.
       *
       */
      TePDIBlending();

      /**
       * @brief Default Destructor
       */
      ~TePDIBlending();
      
      /**
       * @brief Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;      
      
      /**
       * @brief Calculate the polygons for rendering output image.
       * @param params Algorithm parameters.
       * @param new_raster1_pol_ref1 The new raster1 polygon
       * ( reference - raster1 projected coords ).
       * @param new_raster2_pol_ref1 The new raster2 polygon
       * ( reference - raster1 projected coords ).
       * @param inter_pol_ref1 The calculated intersection polygon
       * ( reference - raster1 projected coords ).
       * @param inter_pol_ref2 The calculated intersection polygon
       * ( reference - raster2 projected coords ).
       * @param relation The calculated polygons relation.
       * @param raster1_rel_index_offset_x X offset between output and input
       * raster 1 ( reference output_raster indexed units -> 
       * input_x = output_x + offset ).
       * @param raster1_rel_index_offset_y Y offset between output and input
       * raster 1 ( reference output_raster indexed units -> 
       * input_y = output_y + offset ).
       * @param raster2_rel_index_offset_x X offset between output and input
       * raster 2 ( reference output_raster indexed units -> 
       * input_x = output_x + offset ).
       * @param raster2_rel_index_offset_y Y offset between output and input
       * raster 2 ( reference output_raster indexed units -> 
       * input_y = output_y + offset ).
       * @return true if ok, false on errors.
       */
      static bool extractPolygons( const TePDIParameters& params,
        TePolygon& new_raster1_pol_ref1, TePolygon& new_raster2_pol_ref1,
        TePolygon& inter_pol_ref1, TePolygon& inter_pol_ref2,
        short& relation,
        double& raster1_rel_index_offset_x,
        double& raster1_rel_index_offset_y,
        double& raster2_rel_index_offset_x,
        double& raster2_rel_index_offset_y );

    protected :
    
      /**
       * @brief Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );    
     
      /**
       * @brief Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();      

      /**
       * @brief Render into output raster the non-intersected areas.
       * @param pols_relation The found raster polygons relation (output).  
       * @return true if ok, false on errors.
       */
      bool renderNonIntersectedAreas( short& pols_relation );
      
      /**
       * @brief Equalize the input_raster2.
       * @return true if ok, false on errors.
       */
      bool equalizeInputRaster2();
        
      /**
       * @brief Reset the output raster to the new geometry.
       * @return true if ok, false on errors.
       */
      bool resetOuputRaster(); 
      
      /**
       * @brief The mean and variance thread entry function.
       * @param pars Thread parameters.
       * @return true if ok, false on errors.
       * @note The needed thread parameters are:
       * * TePDITypes::TePDIRasterPtrType input_raster
       * * TePolygon input_pol
       * * bool progress_enabled_flag
       * * std::vector< int > input_channels
       * * std::vector< double >* mean_vector_ptr
       * * std::vector< double >* variance_vector_ptr
       */      
      static bool getMeanAndVarianceThreadEntry( 
        const TeThreadParameters& pars );
  };
  
/** @example TePDIBlending_test.cpp
 *    Shows how to use this class.
 */    

#endif
