/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeSTElementSet.h"
#include "TeGeneralizedProxMatrix.h"


bool
TeGeneralizedProxMatrix::IsValid () const
{
	if ((imp_) && (sc_) && (ss_) && (sw_))
		return true;

	return false;
}


TeProxMatrixImplementation*
TeGeneralizedProxMatrix::getImplementation (const string type)
{
	 if (imp_ == 0) 
		imp_ = TeProxMatrixAbstractFactory::MakeConcreteImplementation (type); 
    return imp_;
}



TeGeneralizedProxMatrix::TeGeneralizedProxMatrix(TeSTElementSet* objects, TeGeomRep geom_type, const string imp_type)
{
	imp_ = 0;
	imp_ = getImplementation(imp_type);
	total_slices_ = 1;

	if ((geom_type==TePOLYGONS) || (geom_type==TeCELLS))
		sc_ = new TeProxMatrixLocalAdjacencyStrategy (objects, geom_type);
	else {
			imp_ = 0;
			sc_ = 0;
			ss_ = 0;
			sw_ = 0;
		}
	ss_ = new TeProxMatrixNoSlicingStrategy;
	sw_ = new TeProxMatrixNoWeightsStrategy;

	if (IsValid()) {
		if (sc_->Construct (imp_)) {
			ss_->Slice (imp_);
			sw_->ComputeWeigths (imp_);
		} 
		else {

			imp_ = 0;
			sc_ = 0;
			ss_ = 0;
			sw_ = 0;
		}
	}
}


TeGeneralizedProxMatrix::TeGeneralizedProxMatrix (TeProxMatrixConstructionStrategy* sc, TeProxMatrixWeightsStrategy* sw, TeProxMatrixSlicingStrategy* ss, const string imp_type)
{
	imp_ = 0;
	imp_ = getImplementation(imp_type);
	total_slices_ = 1;
	sc_ = sc;


  if(ss)
		ss_ = ss;
	else
		ss_ = new TeProxMatrixNoSlicingStrategy();

	if(sw)
		sw_ = sw;
	else
		sw_ = new TeProxMatrixNoWeightsStrategy();


	if (IsValid()) {
		if (sc_->Construct (imp_)) {
			ss_->Slice (imp_);
			sw_->ComputeWeigths (imp_);
		} 
		else {
			imp_ = 0;
			sc_ = 0;
			ss_ = 0;
			sw_ = 0;
		}
	}
}


TeGeneralizedProxMatrix::TeGeneralizedProxMatrix(const TeGeneralizedProxMatrix& pm)
{
	if (pm.imp_ == 0)
		imp_ = 0;
	else 
		imp_ = pm.imp_->CreateCopy ();

	total_slices_ = pm.total_slices_;
	sc_ = pm.sc_;
	ss_ = pm.ss_;
	sw_ = pm.sw_;

	if (IsValid() == false)
	{
		imp_ = 0;
		sc_ = 0;
		ss_ = 0;
		sw_ = 0;
	}
}


TeGeneralizedProxMatrix& 
TeGeneralizedProxMatrix::operator=(const TeGeneralizedProxMatrix& pm) 
{
	if (*this == pm) return *this;

	if (imp_) delete imp_;
	imp_ = 0;
	if (pm.imp_)	imp_ = pm.imp_->CreateCopy ();

	total_slices_ = pm.total_slices_;
	sc_ = pm.sc_;
	ss_ = pm.ss_;
	sw_ = pm.sw_;

	if (IsValid() == false)
	{
		imp_ = 0;
		sc_ = 0;
		ss_ = 0;
		sw_ = 0;
	}

	return *this;

}

bool 
TeGeneralizedProxMatrix::operator==(const TeGeneralizedProxMatrix& pm) const
{
	if (IsValid() && pm.IsValid()) 
	{	
		if ((sc_->IsEqual (*(pm.sc_))) &&
			(ss_->IsEqual (*(pm.ss_))) &&
			(sw_->IsEqual (*(pm.sw_))) &&
			(total_slices_ == pm.total_slices_) &&

			(imp_->IsEqual(*(pm.imp_)))) return true;
	} else 	
		if ((IsValid() == false) && (pm.IsValid() == false)) 
			return true;
	
	return false;

}

TeNeighbours
TeGeneralizedProxMatrix:: getNeighbours (const string& object_id, int slice) 
{
	TeNeighbours neigh2;
	if (slice > total_slices_) return neigh2;

	if (IsValid()) 
	{
		TeNeighbours neigh1;
		imp_->getNeighbours (object_id, neigh1);
		for (int i=0; i < neigh1.size(); i++) 
			if (neigh1.Attributes(i).Slice() == slice) 
				neigh2.Insert (neigh1.ObjectId(i), neigh1.Attributes(i));
	}

	return neigh2;
}


TeNeighbours 
TeGeneralizedProxMatrix::operator[](const string& object_id) 
{
	return getNeighbours(object_id);
}


TeNeighboursMap
TeGeneralizedProxMatrix::getMapNeighbours (const string& object_id, int slice) 
{
	TeNeighboursMap map;
	if (slice > total_slices_) return map;

	if (IsValid()) 
	{
		TeNeighbours neigh;
		imp_->getNeighbours (object_id, neigh);

		for (int i=0; i < neigh.size(); i++) 
			if (neigh.Attributes(i).Slice() == slice) 
				map[neigh.ObjectId(i)] = neigh.Attributes(i);
	}

	return map;
}


TeSTElementSet 
TeGeneralizedProxMatrix::getSTENeighbours(const string& object_id)
{
	TeSTElementSet selected_objects;
	if (imp_) 
	{
		TeNeighbours neigh;
		imp_->getNeighbours (object_id, neigh);
		
		for (int i = 0; i < neigh.size(); i++) 
		{
			// Construct a sto instance with its attributes
			TeSTInstance obj;
			obj.objectId(neigh.ObjectId(i));  
			
			//load the attributes
			TePropertyVector propVector;
			sc_->objects()->getPropertyVector(object_id, propVector);
			obj.properties(propVector);
				
			// insert object in the return vector
			selected_objects.insertSTInstance(obj);
		}
	}
	return selected_objects;
}


bool 
TeGeneralizedProxMatrix::setCurrentConstructionStrategy (TeProxMatrixConstructionStrategy* sc)  
{	
	if (sc == 0) 
		return false; 
	sc_ = sc; 
	return true;
}


bool 
TeGeneralizedProxMatrix::setCurrentWeightsStrategy (TeProxMatrixWeightsStrategy* sw) 
{	
	if (sw == 0) 
		return false; 
	sw_ = sw; 
	return true;
}

bool 
TeGeneralizedProxMatrix::setCurrentSlicingStrategy (TeProxMatrixSlicingStrategy* ss) 
{
	if (ss == 0) 
		return false; 
	ss_ = ss; 
	return true;
}

bool
TeGeneralizedProxMatrix:: ConstructMatrix ()
{

	//	ClearImplementation();
	imp_ = 0;
	imp_ = getImplementation();

	if (sc_) 
	{
		if (sc_->Construct (imp_)) 
		{
			if (ss_) ss_->Slice (imp_);
			if (sw_) sw_->ComputeWeigths (imp_); 
			return true;
		} 
	}
	return false;
}


bool
TeGeneralizedProxMatrix:: ClearImplementation ()
{
	TeProxMatrixImplementation* aux;
	if (imp_ == 0)  aux = getImplementation ();
	else	aux = getImplementation (imp_->Type().c_str());

	if (aux == 0) return false;
	delete imp_;
	imp_ = aux;

	return true;
}

bool
TeGeneralizedProxMatrix:: RecomputeWeights ()
{
	if (IsValid()){
		sw_->ComputeWeigths (imp_); return true;
	} 
	return false;
}


bool
TeGeneralizedProxMatrix:: RecomputeSlicing ()
{
	if (IsValid()){
		ss_->Slice (imp_); return true;
	} 
	return false;
}


bool 
TeGeneralizedProxMatrix::IsConnected (const string& object_id1, const string& object_id2) 
{
	if (imp_ == 0) 
		return false;  
	return imp_->isConnected (object_id1, object_id2);
}

bool 
TeGeneralizedProxMatrix::ConnectObjects (const string& object_id1, const string& object_id2, const TeProxMatrixAttributes& attr)
{
	if (!imp_) 
		getImplementation();  
	imp_->connectObjects (object_id1, object_id2, attr);
	return true;
}

bool 
TeGeneralizedProxMatrix::ConnectObjects (const string& object_id1, const string& object_id2)
{
	if (!imp_) 
		getImplementation();  
	TeProxMatrixAttributes attr;
	imp_->connectObjects (object_id1, object_id2, attr);
	return true;
}

bool 
TeGeneralizedProxMatrix::DisconnectObjects (const string& object_id1, const string& object_id2)
{
	if (imp_ == 0) 
		return false;  
	return imp_->disconnectObjects (object_id1, object_id2);
}

bool 
TeGeneralizedProxMatrix::RemoveObject (const string& object_id)
{
	if (imp_ == 0) 
		return false;  
	return imp_->removeObject (object_id);
}

bool 
TeGeneralizedProxMatrix::GetConnectionAttributes (const string& object_id1, string& object_id2, TeProxMatrixAttributes& attr)
{
	if (imp_ == 0) 
		return false;
	return imp_->getConnectionAttributes (object_id1, object_id2, attr); 
}

int  
TeGeneralizedProxMatrix::NumberOfObjects () 
{
	if (imp_ == 0) 
		return 0;  
	return imp_->NumberOfObjects ();
}

bool 
TeGeneralizedProxMatrix::SaveTextFile (string name) 
{	
	if (imp_ == 0) 
		return false;
	return imp_->SaveTextFile (name); 
}


bool 
TeGeneralizedProxMatrix::SaveGALFile (string name) 
{	
	if (imp_ == 0) 
		return false;
	return imp_->SaveGALFile (name); 
}

bool 
TeGeneralizedProxMatrix::SaveGWTFile (string name) 
{	
	if (imp_ == 0) 
		return false;
	return imp_->SaveGWTFile (name); 
}

TeGeneralizedProxMatrix::~TeGeneralizedProxMatrix()
{
    if (imp_ !=  0) 
		delete imp_; //It is not counted.
}






