/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeOracleSpatial.h"

#include <zlib.h>
#include <jpeg.h>
#include <sys/stat.h>
#include "TeSpatialOperations.h"
#include "TeGeometryAlgorithms.h"


#define MAX(a,b) a>b?a:b
#define MIN(a,b) a<b?a:b

string 
getOracleSpatialRelation(int relation)
{
	string spatialRel="";
	switch (relation)
	{
		case TeDISJOINT:
		case TeINTERSECTS:
			spatialRel = "ANYINTERACT";	
		break;

		case TeTOUCHES:
			spatialRel = "TOUCH";
		break;

		case TeOVERLAPS:
			spatialRel = "OVERLAPBDYINTERSECT";
		break;

		case TeCOVERS:
			spatialRel = "COVERS"; 
		break;

		case TeCOVEREDBY:
			spatialRel = "COVEREDBY";
		break;
		
		case TeCONTAINS:
			spatialRel = "CONTAINS";
		break;
		
		case TeWITHIN:
			spatialRel = "INSIDE";
		break;
		
		case TeEQUALS:
			spatialRel = "EQUAL";
		break;
		
		case TeCROSSES:
			spatialRel = "OVERLAPBDYDISJOINT";
		break;
		default:
			spatialRel = "";
		break;
	}
	
	return spatialRel;
}


TeOracleSpatial::TeOracleSpatial() 
{
	Conn = new OCIConnection();  
	dbmsName_ = "OracleSpatial";
}


TeOracleSpatial::~TeOracleSpatial() 
{
	delete (Conn); //disconnect
	Conn = NULL;
}

void
TeOracleSpatial::close () 
{
	clear();
	Conn->Disconnect();
	isConnected_ = false;
}


bool
TeOracleSpatial::connect(const string& host, const string& user, const string& password, const string& database, int port)
{
	if (Conn->IsConnected())
	{ 
		delete (Conn);
		Conn = new OCIConnection();
	}

	isConnected_ = false;
	if (Conn->Connect(host.c_str(),user.c_str(),password.c_str()))
	{
		isConnected_ = true;
		host_ = host;
		user_ = user;
		password_ = password;
		database_ = database;
		portNumber_ = port;
		return true;
	}
	else
	{
		isConnected_ = false;
		errorMessage_ = "Error connecting to database server!";
		delete (Conn); //disconect
		return false;
	}
}


bool 
TeOracleSpatial::tableExist(const string& table)
{
	bool	status;
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string exist = " SELECT table_name FROM all_tables WHERE";
	exist += " TABLE_NAME = '" + TeConvertToUpperCase(table) + "'";
	exist += " AND OWNER = '" + TeConvertToUpperCase(user_) + "'";

	if(!ocip->query(exist))
	{
		delete ocip;
		return false;
	}

	if(ocip->fetchRow())
		status = true;
	else
		status = false;

	delete ocip;
	return (status);
}


bool 
TeOracleSpatial::columnExist(const string& table, const string& column, TeAttribute& attr)
{
	bool	status = false;
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string exist = " SELECT DATA_TYPE, DATA_LENGTH, DATA_SCALE FROM ALL_TAB_COLUMNS WHERE";
	exist += " TABLE_NAME = '" + TeConvertToUpperCase(table) + "'";
	exist += " AND COLUMN_NAME = '" + TeConvertToUpperCase(column) + "'";
	exist += " AND OWNER = '" + TeConvertToUpperCase(user_) + "'";

	if(!ocip->query(exist))
	{
		delete ocip;
		return false;
	}
	
	if(ocip->fetchRow())
	{	
		attr.rep_.name_ = column;
		
		string	dataType = string(ocip->getData(0)); 
		int		dataLength = atoi(ocip->getData(1)); 
		int		dataScale = atoi(ocip->getData(2));
		bool    number = false;
		
		if(dataType=="VARCHAR2")
		{
			attr.rep_.type_ = TeSTRING;
			attr.rep_.numChar_ = dataLength;
		}
		else if (dataType=="BLOB")
		{
			attr.rep_.type_ = TeBLOB;
			attr.rep_.numChar_ = dataLength;
		}
		else if (dataType=="NUMBER")
		{
			number = true;
		}
		else if (dataType=="SDO_GEOMETRY")
		{
			attr.rep_.type_ = TeOBJECT;
			attr.rep_.numChar_ = dataLength;
		}
		else if (dataType== "CHAR")
		{	
			attr.rep_.type_ = TeCHARACTER;
			attr.rep_.numChar_ = dataLength;
		}
		else if (dataType=="DATE")
		{
			attr.rep_.type_ = TeDATETIME;
		}
		else
		{
			attr.rep_.type_ = TeSTRING;
			attr.rep_.numChar_ = dataLength;
		}

		if(number)
		{
			if(dataScale > 0)
				attr.rep_.type_ = TeREAL;
			else
				attr.rep_.type_ = TeINT;
		}
		status = true;
	}
	
	delete ocip;
	return (status);
}

bool 
TeOracleSpatial::execute (const string &q)
{
	bool result = Conn->Execute(q);
	
	if(!result)
		errorMessage_ = Conn->getErrorMessage();

	return result;
}


bool
TeOracleSpatial::createTable(const string& table, TeAttributeList &attr)
{
	short	cont=0;
	string pkeys ="";
	bool	hasAutoNumber=false;
	string	fieldName="";

	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}

	TeAttributeList::iterator it = attr.begin();
	string tablec;
	tablec = "CREATE TABLE " + table +" (";
	
	while ( it != attr.end())
	{
		if (cont)
			tablec += ", ";
			
		switch ((*it).rep_.type_)
		{
			case TeSTRING:
				tablec += (*it).rep_.name_ + " VARCHAR2(" + Te2String((*it).rep_.numChar_) + ")";
			break;
			
			case TeREAL:
				if((*it).rep_.decimals_>0)
					tablec += (*it).rep_.name_ +" NUMBER(*,"+ Te2String((*it).rep_.decimals_) +") ";
				else
					tablec += (*it).rep_.name_ +" NUMBER(*,38) ";
			break;
			
			case TeINT:
				tablec += (*it).rep_.name_ + " NUMBER(32) ";
				if((*it).rep_.isAutoNumber_)
				{
					hasAutoNumber=true;
					fieldName=(*it).rep_.name_;
				}
			break;

			case TeDATETIME:
				tablec += (*it).rep_.name_ + " DATE ";
			break;

			case TeCHARACTER:
				tablec += (*it).rep_.name_ + " CHAR ";
			break;

			case TeBLOB:
				tablec += (*it).rep_.name_ + " BLOB ";
			break;

			default:
				tablec += (*it).rep_.name_ + " VARCHAR2(255) ";
			break;
		}

		// check if column is part of primary key
		if ((*it).rep_.isPrimaryKey_ && (*it).rep_.type_ != TeBLOB )
		{
			if (!pkeys.empty())
				pkeys += ", ";
			pkeys += (*it).rep_.name_;
			tablec += " NOT NULL";
		}

		++it;
		cont++;
	}

	if(!pkeys.empty())
		tablec += ", PRIMARY KEY (" + pkeys + ") ";

	tablec += ")";

	if(!execute(tablec))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table " + table;

		return false;
	}

	if(hasAutoNumber)
	{
		string dropSql = " DROP TRIGGER "+ getNameTrigger(table); 
		execute(dropSql);
		dropSql = " DROP SEQUENCE "+ getNameSequence(table); 
		execute(dropSql);

		if(!createSequence(table))
		{
			deleteTable(table);
			return false;
		}
		
		if(!createAutoIncrementTrigger(table,fieldName))
		{
			deleteTable(table);
			string sql= "DROP SEQUENCE "+ getNameSequence(table);
			execute(sql); 
			return false;
		}
	}
	return true;
}


TeDatabasePortal*  
TeOracleSpatial::getPortal ()
{
	TeOracleSpatialPortal* ocip = new TeOracleSpatialPortal (this);
	return ocip;
}


bool 
TeOracleSpatial::addColumn (const string& table, TeAttributeRep &rep)
{
	if(!tableExist(table))
		return false;

	string field = TeGetExtension(rep.name_.c_str());
	if(field.empty())
		field = rep.name_;

	string tab;
	tab = " ALTER TABLE " + table + " ADD ( ";
	tab += field + "  ";
	
	switch (rep.type_)
	{
		case TeSTRING:
			tab += "VARCHAR2(" + Te2String(rep.numChar_) + ") ";
			break;
			
		case TeREAL:
			tab += "NUMBER(*,38)";	
			break;
			
		case TeINT:
			tab += "NUMBER(32)";
			break;

		case TeDATETIME:
			tab += "DATE";

		case TeCHARACTER:
			tab += "CHAR";
		
		case TeBLOB:
			tab += "BLOB";
		
		default:
			tab += "VARCHAR2(" + Te2String(rep.numChar_) + ") ";
			break;
	}

	tab += " ) ";

	if(!Conn->Execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting a column to table " + table + " !";
		return false;
	}

	return true;
}

bool
TeOracleSpatial::deleteTable (const string& table)
{
	int f =	table.find ("te_collection", std::string::npos);

	if( table=="te_theme" ||
		table=="te_layer" ||
		table=="te_representation" ||
		table=="te_tables_relation" ||
		table=="te_layer_table" ||
		table=="te_color_scheme" ||
		table=="te_raster_metadata" ||
		table=="te_projection" ||
		table=="te_view" ||
		table=="te_chart_params" ||
		table=="te_legend" ||
		table=="te_visual" ||
		f == 0)
	{
		errorMessage_ = "No  possvel deletar tabelas do modelo!";
		return false;
	}

	string del = "DROP TABLE " + table;
	if(!execute(del))
		return false;

	string seq = "DROP SEQUENCE " + getNameSequence(table);
	if(!execute(seq))
		return false;

	return true;
}
 

bool 
TeOracleSpatial::createViewTable ()
{
	string table;
	table = "CREATE TABLE te_view ( ";
	table += "view_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", projection_id ";
	table += "NUMBER(32) NOT NULL";
	table += " CONSTRAINT view_proj UNIQUE ";
	table += ", name ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", user_name ";
	table += "VARCHAR2(255) ";
	table += ", visibility ";
	table += "NUMBER(32) ";

	table += ", PRIMARY KEY(view_id))";
	
	if(!execute(table))
	{	
		errorMessage_ = "Error creating table te_view!";
		return false;
	}
	
	//creates a sequence to table 
	if(!createSequence("te_view"))
	{
		errorMessage_ = "Error creating sequence to table te_view!";
		return false;
	}

	//creates index
	createIndex ("te_view", "view_index", "name, user_name");
	return true;
}

bool 
TeOracleSpatial::createThemeTable ()
{
	string table;
	table = "CREATE TABLE te_theme (";
	table += "theme_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", layer_id ";
	table += "NUMBER(32) NULL";
	table += ", view_id ";
	table += "NUMBER(32) NULL";
	table += ", name ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", parent_id ";
	table += "NUMBER(32) ";
	table += ", priority ";
	table += "NUMBER(32) ";	
	table += ", node_type ";
	table += "NUMBER(32) NOT NULL";	
		
	table += ", min_scale ";
	table += "NUMBER(*,38)";
	table += ", max_scale ";
	table += "NUMBER(*,38)";

	table += ", generate_attribute_where ";
	table += "VARCHAR2(500)";
	table += ", generate_spatial_where ";
	table += "VARCHAR2(500)";
	table += ", generate_temporal_where ";
	table += "VARCHAR2(500)";
	
	table += ", collection_table ";
	table += "VARCHAR2(255)";

	table += ", visible_rep ";
	table += "NUMBER(32) NULL";
	table += ", enable_visibility ";
	table += "NUMBER(32) NULL";

	table += ", PRIMARY KEY (theme_id))";
	
	if(!execute(table))
	{	
		errorMessage_ = "Error creating table te_theme!";
		return false;
	}

	//creates a sequence to the tabela
	if(!createSequence("te_theme"))
	{
		errorMessage_ = "Error creating sequence to table te_theme!";
		return false;
	}

	//creates index
	createIndex ("te_theme", "theme_view_index", "view_id");
	return true;
}


bool
TeOracleSpatial::createGroupingTable()
{
	string table;
	table = "CREATE TABLE te_grouping (";
	table += "theme_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", grouping_number ";
	table += "NUMBER(32)";
	table += ", grouping_attr ";
	table += "VARCHAR2(255)";
	table += ", grouping_attr_type ";
	table += "NUMBER(32)";
	table += ", grouping_mode ";
	table += "NUMBER(32)";
	table += ", grouping_norm_attr ";
	table += "VARCHAR2(255)";
	table += ", grouping_std_dev ";
	table += "NUMBER(*,38) DEFAULT 0.0";
	table += ", grouping_precision ";
	table += "NUMBER(32)";
	table += ", grouping_function ";
	table += "VARCHAR2(20)";
	table += ", PRIMARY KEY (theme_id))";
	
	if(!execute(table))
	{	
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table te_grouping!";
		return false;
	}
	return true;
}


bool 
TeOracleSpatial::createThemeTablesTable()
{
	string table = "CREATE TABLE te_theme_table  (";
	table += " theme_table_id ";
	table += " NUMBER(32) NOT NULL";
	table += ", theme_id ";
	table += " NUMBER(32) NOT NULL ";
	table += ", table_id ";
	table += " NUMBER(32) NOT NULL ";
	table += ", relation_id ";
	table += " NUMBER(32)";
	table += ", table_order ";
	table += " NUMBER(32) ";
	table += ", PRIMARY KEY (theme_table_id) ";
	table += ")";
	
	if(!execute(table))
	{	
		errorMessage_ = "Error creating table te_theme_table!";
		return false;
	}

	//creates a sequence to the tabela
	if(!createSequence("te_theme_table"))
	{
		errorMessage_ = "Error creating sequence to table te_theme_table!";
		return false;
	}

	//creates index
	createIndex ("te_theme_table", "theme_table_index", "theme_id");
	return true;
}


bool 
TeOracleSpatial::createLayerTable()
{
	string table;
	table = "CREATE TABLE te_layer (";
	table += "layer_id ";
	table += "NUMBER(32) NOT NULL ";
	table += ", projection_id ";
	table += "NUMBER(32) NOT NULL ";
	table += "CONSTRAINT layer_proj UNIQUE ";
	table += ", name ";
	table += "VARCHAR2(255) NOT NULL ";
	table += ", lower_x ";
	table += "NUMBER(*,38)";
	table += ", lower_y ";
	table += "NUMBER(*,38)";
	table += ", upper_x ";
	table += "NUMBER(*,38) ";
	table += ", upper_y ";
	table += "NUMBER(*,38) ";
	table += ", initial_time ";
	table += "DATE ";
	table += ", final_time ";
	table += "DATE ";

	table += ", PRIMARY KEY (layer_id))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_layer!";
		return false;
	}

	//creates a sequence to table 
	if(!createSequence("te_layer"))
	{
		errorMessage_ = "Error creating sequence to table te_layer!";
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createLayerTableTable()
{
	string table;
	table = "CREATE TABLE te_layer_table (";
	table += "table_id ";
	table += " NUMBER(32) NOT NULL ";
	table += ", layer_id ";
	table += "NUMBER(32) ";
	table += ", attr_table ";
	table += "VARCHAR2(255) NOT NULL ";
	table += ", unique_id ";
	table += "VARCHAR2(255) ";
	table += ", attr_link ";
	table += "VARCHAR2(255) ";
	table += ", attr_initial_time ";
	table += "VARCHAR2(255) ";
	table += ", attr_final_time ";
	table += "VARCHAR2(255) ";
	table += ", attr_time_unit ";
	table += "NUMBER(32) ";
	table += ", attr_table_type ";
	table += "NUMBER(32) ";
	table += ", user_name ";
	table += "VARCHAR2(255) ";
	table += ", initial_time ";
	table += "DATE ";
	table += ", final_time ";
	table += "DATE ";

	table += ", PRIMARY KEY (table_id))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_layer_table!";
		return false;
	}

	//creates a sequence to table 
	if(!createSequence("te_layer_table"))
	{
		errorMessage_ = "Error creating sequence to table te_layer_table!";
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createTablesRelationTable()
{
	string table;
	table = "CREATE TABLE te_tables_relation (";
	table += "relation_id ";
	table += "NUMBER(32) NOT NULL ";
	table += ", related_table_id ";
	table += "NUMBER(32) NOT NULL ";
	table += ", related_attr ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", external_table_name ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", external_attr ";
	table += "VARCHAR2(255) NOT NULL ";

	table += ", PRIMARY KEY (relation_id))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_tables_relation!";
		return false;
	}

	//creates a sequence to table 
	if(!createSequence("te_tables_relation"))
	{
		errorMessage_ = "Error creating sequence to table te_tables_relation!";
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createRepresentationTable ()
{
	string table;
	table = "CREATE TABLE te_representation (";
	table += " repres_id ";
	table += " NUMBER(32) NOT NULL "; 
	table += ", layer_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", geom_type ";
	table += "NUMBER(32) NOT NULL";
	table += ", geom_table ";
	table += "VARCHAR2(255) ";
	table += ", description "; 
	table += "VARCHAR2(255) ";
	table += ", lower_x ";
	table += "NUMBER(*,38) ";
	table += ", lower_y ";
	table += "NUMBER(*,38) ";
	table += ", upper_x ";
	table += "NUMBER(*,38) ";
	table += ", upper_y ";
	table += "NUMBER(*,38) ";
	table += ", res_x ";
	table += "NUMBER(*,38) ";
	table += ", res_y ";
	table += "NUMBER(*,38) ";
	table += ", num_cols ";
	table += "NUMBER(32)";
	table += ", num_rows ";
	table += "NUMBER(32)";
	table += ", initial_time ";
	table += "DATE";
	table += ", final_time ";
	table += "DATE";
	
	table += ", PRIMARY KEY (repres_id))";

	if(!execute(table))
	{	
		errorMessage_ = "Error creating table te_representation!";
		return false;
	}
	//creates a sequence to table 
	if(!createSequence("te_representation"))
	{
		errorMessage_ = "Error creating sequence to table te_representation!";
		return false;
	}

	//creates index
	createIndex ("te_representation", "representation_index", "layer_id, geom_type");
	return true;
}


bool 
TeOracleSpatial::createLegendTable ()
{
	string table;
	table = "CREATE TABLE te_legend (";
	table += "  legend_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", theme_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", group_id ";
	table += "NUMBER(32)";
	table += ", num_objs ";
	table += "NUMBER(32)";
	table += ", lower_value ";
	table += "VARCHAR2(255)";
	table += ", upper_value ";
	table += "VARCHAR2(255)";
	table += ", label ";
	table += "VARCHAR2(255)";

	table += ", PRIMARY KEY (legend_id))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_legend!";
		return false;
	}
	
	//creates a sequence to the table 
	if(!createSequence("te_legend"))
	{
		errorMessage_ = "Error creating sequence to table te_legend!";
		return false;
	}

	//creates index
	createIndex ("te_legend", "legend_index", "theme_id");
	return true;
}



bool 
TeOracleSpatial::createVisualTable()
{
	string table;
	table = "CREATE TABLE te_visual (";
	table += "  legend_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", geom_type ";
	table += "NUMBER(32) NOT NULL";

	table += ", lib_name ";
	table += "VARCHAR2(255) ";
	table += ", symb_id ";
	table += "NUMBER(32) ";
	table += ", red ";
	table += "NUMBER(32) ";
	table += ", green ";
	table += "NUMBER(32) ";
	table += ", blue ";
	table += "NUMBER(32) ";
	table += ", transparency ";
	table += "NUMBER(32) ";
	table += ", width ";
	table += "NUMBER(32) ";
	
	table += ", contour_lib_name ";
	table += "VARCHAR2(255) ";
	table += ", contour_symb_id ";
	table += "NUMBER(32) ";
	table += ", contour_red ";
	table += "NUMBER(32) ";
	table += ", contour_green ";
	table += "NUMBER(32) ";
	table += ", contour_blue ";
	table += "NUMBER(32) ";
	table += ", contour_transp ";
	table += "NUMBER(32) ";
	table += ", contour_width ";
	table += "NUMBER(32) ";

	table += ", size_value ";
	table += "NUMBER(32) ";
	table += ", pt_angle ";
	table += "NUMBER(32) ";
	
	table += ", family ";
	table += "VARCHAR2(100)";
	table += ", bold ";
	table += "NUMBER(32) ";
	table += ", italic ";
	table += "NUMBER(32) ";
	table += ", alignment_vert ";
	table += "NUMBER(*,38)";
	table += ", alignment_horiz ";
	table += "NUMBER(*,38)";
	table += ", tab_size ";
	table += "NUMBER(32) ";
	table += ", line_space ";
	table += "NUMBER(32) ";
	table += ", fixed_size ";
	table += "NUMBER(32) ";

	table += ", PRIMARY KEY (legend_id, geom_type))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_visual!";
		return false;
	}
	
	return true;
}

bool 
TeOracleSpatial::createVisualRasterTable()
{
	string create = "CREATE TABLE te_visual_raster (";
	create += " theme_id		NUMBER(32) NOT NULL,";
	create += " band_in			NUMBER(32) NOT NULL,";
	create += " band_out		NUMBER(32), ";
	create += " transf_type		NUMBER(32), ";
	create += " param1			FLOAT,";
	create += " param2			FLOAT,";
	create += " lut_table		VARCHAR2(255), ";
	create += " PRIMARY KEY (theme_id, band_in))";
	
	if(!execute(create))
	{
		errorMessage_ = "Error creating table te_visual_raster!";
		return false;
	}
	
	return true;
}

bool 
TeOracleSpatial::createDatabaseTable()
{
	string create = "CREATE TABLE te_database (";
	create += "db_version		VARCHAR(50) NOT NULL,";
	create += "db_creation		DATE,";
	create += "PRIMARY KEY (db_version))";
	return (execute (create));
}

bool 
TeOracleSpatial::createProjectionTable ()
{
	string table;
	table = "CREATE TABLE te_projection (";
	table += "projection_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", name ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", long0 ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", lat0 ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", offx ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", offy ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", stlat1 ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", stlat2 ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", unit ";
	table += "VARCHAR2(100) ";
	table += ", scale ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", hemis ";
	table += "NUMBER(32) ";
	table += ", datum ";
	table += "VARCHAR2(100) ";
	table += ", radius ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", flattening ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", dx ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", dy ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	table += ", dz ";
	table += "NUMBER(*,38) DEFAULT 0.0000 NOT NULL ";
	
	table += ", PRIMARY KEY (projection_id))";

	if(!execute(table))
	{
		errorMessage_ = "Error creating table te_projection!";
		return false;
	}
	
	//creates a sequence to table 
	if(!createSequence("te_projection"))
	{
		errorMessage_ = "Error creating sequence to table te_projection!";
		return false;
	}

	return true;
}

bool
TeOracleSpatial::createRasterMetadataTable(const string& tableName)
{
	if(tableExist(tableName))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string table;
	table = "CREATE TABLE "+ tableName +" (";
	table += " geom_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", band_id ";
	table += "NUMBER(32) NOT NULL";
	table += ", min_value ";
	table += "NUMBER(*,38) DEFAULT 0.0 ";
	table += ", max_value ";
	table += "NUMBER(*,38) DEFAULT 0.0 ";
	table += ", num_bits ";
	table += "NUMBER(32)";
	table += ", data_type ";
	table += "NUMBER(32)";
	table += ", photometric_type ";
	table += "NUMBER(32)";
	table += ", compression_type ";
	table += "NUMBER(32)";
	table += ", dummy ";
	table += "NUMBER(*,38) DEFAULT 0.0";
	
	table += ", PRIMARY KEY (geom_id, band_id))";

	if(!execute(table))
	{	
		errorMessage_ = "Error creating table "+ tableName +"!";
		return false;
	}
	
	return true;
}


bool 
TeOracleSpatial::createLUTTable(const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string tab;
	tab = "CREATE TABLE " + table + " ( ";
	tab += "index_id ";
	tab += "NUMBER(32) NOT NULL ";
	tab += ", r_val ";
	tab += "NUMBER(32) NOT NULL ";
	tab += ", g_val ";
	tab += "NUMBER(32) NOT NULL ";
	tab += ", b_val ";
	tab += "NUMBER(32) NOT NULL ";
	
	tab += ", PRIMARY KEY (index_id))";

	if(!execute(tab))
	{
		errorMessage_ = "Error creating table " + table ;
		return false;
	}

	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"index_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool
TeOracleSpatial::createCollectionTable(const string& tableName)
{
	
	if(tableExist(tableName))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string table = "CREATE TABLE "+ tableName +" (";
	table += " c_object_id ";
	table += "VARCHAR2(255) NOT NULL";
	table += ", c_legend_id  ";
	table += "NUMBER(32) ";
	table += ", label_x ";
	table += "NUMBER(*,38) ";
	table += ", label_y ";
	table += "NUMBER(*,38) ";
	table += ", c_legend_own ";
	table += "NUMBER(32) ";
	table += ", c_object_status ";
	table += "NUMBER(32) ";
	
	table += ", PRIMARY KEY (c_object_id))";

	if(!execute(table))
	{	
		errorMessage_ = "Error creating table " + tableName + "!";
		return false;
	}

	createIndex (tableName, tableName+"_index_1", "c_legend_id");
	createIndex (tableName, tableName+"_index_2", "c_legend_own");
	return true;
}


bool 
TeOracleSpatial::createCellGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string tab;
	tab = "CREATE TABLE " + table + "(";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", object_id ";
	tab += "VARCHAR2(255) NULL";
	tab += ", col_number ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", row_number ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", spatial_data ";
	tab += "MDSYS.SDO_GEOMETRY";	
	tab += ", PRIMARY KEY (geom_id))";

	if (!execute(tab))
	{
		errorMessage_ = "Error creating table " + table + " !" ;
		return false;
	}

	//creates indexes
	createIndex (table, table + "obj_idx", "object_id"); 	
	createIndex(table, table + "rc_idx", "row_number, col_number"); 

	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createTextGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string tab;
	tab = "CREATE TABLE " + table +"(";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", object_id ";
	tab += "VARCHAR2(255) ";
	tab += ", x ";
	tab += "NUMBER(*,38) NULL";
	tab += ", y ";
	tab += "NUMBER(*,38) NULL";
	tab += ", text_value ";
	tab += "VARCHAR2(255) NULL";
	tab += ", angle ";
	tab += "NUMBER(*,38) ";
	tab += ", height ";
	tab += "NUMBER(*,38) ";
	tab += ", alignment_vert ";
	tab += "NUMBER(*,38) ";
	tab += ", alignment_horiz ";
	tab += "NUMBER(*,38) ";
	
	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table  " + table + " !";
		return false;
	}

	//creates indexes
	createIndex(table, table + "obj_idx", "object_id");
	createIndex(table, table + "xy_idx", "x,y"); 
		
	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table, "geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createArcGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}

	string tab;
	tab = "CREATE TABLE " + table +"(";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", object_id ";
	tab += "VARCHAR2(255) NULL";
	tab += ", from_node ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", to_node ";
	tab += "NUMBER(32) NOT NULL";
	
	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		errorMessage_ = "Error creating table " + table ;
		return false;
	}

	//creates indexes
	createIndex(table, table + "obj_idx", "object_id");
	createIndex(table, table + "fn_idx", "from_node"); 
	createIndex(table, table + "tn_idx", "to_node"); 
		
	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}
	return true;
}


bool 
TeOracleSpatial::createNodeGeometry (const string& table)
{
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
		
	string tab;
	tab = "CREATE TABLE " + table + " (";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL ";
	tab += ", object_id ";
	tab += "VARCHAR2(255) NULL";
	tab += ", spatial_data ";
	tab += "MDSYS.SDO_GEOMETRY";
	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		errorMessage_ = "Error creating table  " + table + " !";
		return false;
	}
	
	//creates indexes
	createIndex(table, table + "obj_idx", "object_id"); 
				
	///creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createRasterGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}

	string create = "CREATE TABLE " + table + " (";
	create += "geom_id		NUMBER(32) NOT NULL,";
	create += "object_id	VARCHAR2(255) NOT NULL,";
	create += "raster_table VARCHAR2(255) NOT NULL,";
	create += "lut_table	VARCHAR2(255) ,";
	create += "res_x		NUMBER(*,38) DEFAULT 0.0,";
	create += "res_y		NUMBER(*,38) DEFAULT 0.0,";
	create += "num_bands	NUMBER(32),";
	create += "num_cols		NUMBER(32),";
	create += "num_rows		NUMBER(32),";
	create += "block_height NUMBER(32),";
	create += "block_width	NUMBER(32),";
	create += "lower_x		NUMBER(*,38) DEFAULT 0.0,";
	create += "lower_y		NUMBER(*,38) DEFAULT 0.0,";
	create += "upper_x		NUMBER(*,38) DEFAULT 0.0,";
	create += "upper_y		NUMBER(*,38) DEFAULT 0.0,";
	create += "tiling_type  NUMBER(32),";
	create += "PRIMARY KEY (geom_id))";
	
	if(!execute (create))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table " + table + " !" ;
		return false;
	}
	
	createIndex(table, table + "obj_idx", "object_id"); 
	
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}
	
	return true;
}

bool 
TeOracleSpatial::createRasterTable (const string& tableName)
{
	string tab;
	tab = "CREATE TABLE "+ tableName + " ( ";
	tab += "block_id ";
	tab += " VARCHAR2(50) NOT NULL";
	tab += ", band_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", resolution_factor ";
	tab += "NUMBER(32) ";
	tab += ", subband ";
	tab += "NUMBER(32) ";
	tab += ", block_box ";
	tab += "MDSYS.SDO_GEOMETRY ";
	tab += ", block_size ";
	tab += "NUMBER(32) ";
	tab += ", spatial_data ";
	tab += "BLOB ";
	
	tab += ", PRIMARY KEY (block_id))";

	if(!execute(tab))
	{
		errorMessage_ = "Error creating table " + tableName ;
		return false;
	}
	
	createIndex(tableName, tableName + "band_idx", "band_id"); 
	createIndex(tableName, tableName + "sbnd_idx", "subband");
	createIndex(tableName, tableName + "rf_idx", "resolution_factor");
	return true;
}


bool 
TeOracleSpatial::createPolygonGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string tab;
	tab = "CREATE TABLE " + table + " ( ";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL ";
	tab += ", object_id ";
	tab += "VARCHAR2(255) NULL";
	tab += ", spatial_data ";
	tab += "MDSYS.SDO_GEOMETRY";

	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table " + table ;
		return false;
	}

	//creates indexes
	createIndex(table, table + "obj_idx", "object_id");
		
	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	//creates trigger
	if(!createAutoIncrementTrigger(table, "geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createLineGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}
	
	string tab;
	tab = "CREATE TABLE " + table + " ( ";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", object_id ";
	tab += "VARCHAR2(255)  NULL";
	tab += ", spatial_data ";
	tab += "MDSYS.SDO_GEOMETRY";
	
	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table " + table + " !" ;
		return false;
	}

	//creates indexes
	createIndex(table, table + "obj_idx", "object_id");
			
	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	//creates trigger
	if(!createAutoIncrementTrigger(table, "geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::createPointGeometry (const string& table)
{
	
	if(tableExist(table))
	{
		errorMessage_= "Table already exist!";
		return false;
	}

	string tab;
	tab = "CREATE TABLE " + table + " ( ";
	tab += "geom_id ";
	tab += "NUMBER(32) NOT NULL";
	tab += ", object_id ";
	tab += "VARCHAR2(255) NULL";
	tab += ", spatial_data ";
	tab += "MDSYS.SDO_GEOMETRY";
	
	tab += ", PRIMARY KEY (geom_id))";

	if(!execute(tab))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error creating table " + table + " !" ;
		return false;
	}	
		
	//creates indexes
	createIndex(table, table + "obj_idx", "object_id");
		
	//creates a sequence to table 
	if(!createSequence(table))
	{
		deleteTable(table);
		return false;
	}

	if(!createAutoIncrementTrigger(table,"geom_id"))
	{
		deleteTable(table);
		string sql= "DROP SEQUENCE "+ getNameSequence(table);
		execute(sql); 
		return false;
	}

	return true;
}

bool 
TeOracleSpatial::insertRelationInfo(const int tableId, const string& tField,
						  const string& rName, const string& rField, int& relId)
{
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	relId = -1;
	string sel = "SELECT relation_id FROM te_tables_relation WHERE";
	sel += " table_id = " + Te2String(tableId);
	sel += " AND attr_link = '" + tField + "'";
	sel += " AND related_table = '" + rName + "'";
	sel += " AND related_link = '" + rField + "'";
	if (!ocip->query(sel))
	{
		delete ocip;
		return false;
	}

	if (ocip->fetchRow())
	{
		relId = atoi(ocip->getData(0));
		delete ocip;
		return true;
	}

	string seq, ins;
	seq = getNameSequence("te_tables_relation");	
	ins = " INSERT INTO te_tables_relation ( "; 
	ins += " relation_id, attr_link, table_id, ";
	ins += " related_table, related_link) VALUES ( ";
	ins += seq +".NEXTVAL "; 
	ins += ",  " + Te2String(tableId);
	ins += ", '" + escapeSequence(tField) + "'";
	ins += ", '" +  escapeSequence(rName) + "'";
	ins += ", '" +  escapeSequence(rField) + "'";
	ins += ")";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting tables information!";   
		delete ocip;
		return false;
	}

	ins = "SELECT "+ seq +".CURRVAL FROM DUAL";
	if (!ocip->query(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_table_relation_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	relId = atoi((const char*)ocip->getData(0));
	delete ocip;
	return true;
}

bool 
TeOracleSpatial::insertTableInfo (int layerId, TeTable &table, const string& user)
{
	string ins, seq;

	seq = getNameSequence("te_layer_table");
	ins = "INSERT INTO te_layer_table ( "; 
	ins += " table_id, layer_id, attr_table, unique_id, attr_link, ";
	ins += " attr_initial_time, attr_final_time, attr_time_unit, ";
	ins += " attr_table_type, user_name) VALUES ( ";
	ins += seq + ".NEXTVAL "; 
	
	if(layerId>0)
		ins += ", " + Te2String(layerId);
	else
		ins += ", " + NULL;
	
	ins += ", '" + escapeSequence(table.name()) + "'";
	ins += ", '" + escapeSequence(table.uniqueName()) + "'";
	ins += ", '" + escapeSequence(table.linkName()) + "'";
	ins += ", '" + escapeSequence(table.attInitialTime()) + "'";
	ins += ", '" + escapeSequence(table.attFinalTime()) + "'";
	ins += ",  " + Te2String(table.attTimeUnit());
	ins += ",  " + Te2String(table.tableType());
	ins += ", '" +  escapeSequence(user) + "'";
	ins += ")";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting tables information!";   
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	ins = "SELECT "+ seq +".CURRVAL FROM DUAL";
	if (!ocip->query(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_layer_table_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0));
	table.setId(index);
	delete ocip;
	return true;
}


TeDBRelationType 
TeOracleSpatial::existRelation(const string& tableName, const string& relName)
{
	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return TeNoRelation;
	
	string integ = " SELECT DELETE_RULE  ";
	integ += " FROM   all_constraints ";
	integ += " WHERE  CONSTRAINT_TYPE = 'R'"; 
	integ += " AND    OWNER = '" + TeConvertToUpperCase(user_) + "'";
	integ += " AND    CONSTRAINT_NAME = '" + TeConvertToUpperCase(relName) + "'";
	
	string temp = " AND    TABLE_NAME = '" + TeConvertToUpperCase(tableName) + "' ";   
	integ += temp;
	
	if(!ocip->query(integ))
	{
		delete ocip;
		return TeNoRelation;
	}

	string cascade;
	if(ocip->fetchRow())
	{
		cascade = string(ocip->getData(0));
		if(cascade== "CASCADE")
		{
			delete ocip;
			return TeRICascadeDeletion;
		}
		
		delete ocip;
		return TeRINoCascadeDeletion;
	}

	delete ocip;
	return TeNoRelation;	
}


bool 
TeOracleSpatial::createRelation (const string& name, const string& table, const string& fieldName, const string& relatedTable, const string& relatedField, bool cascadeDeletion)
{
	string relation = "ALTER TABLE " + table + " ADD ";
	relation += " CONSTRAINT " + name;
	relation += " FOREIGN KEY ( " + fieldName + ") "; 
	relation += " REFERENCES " + relatedTable + "(" + relatedField + ")";
		
	if (cascadeDeletion)
		relation += " ON DELETE CASCADE ";

	if(!execute(relation))
		return false;

	return true;
}

string 
TeOracleSpatial::getSQLBoxWhere (TeBox &box, TeGeomRep rep)
{
	string wherebox;
	string colname = "spatial_data";

	if(rep == TeRASTER)
		colname = "block_box";

	if(rep == TeTEXT)
	{
		wherebox = TeDatabase::getSQLBoxWhere (box, rep);
		return wherebox;
	}

	wherebox = "mdsys.sdo_filter (" + colname +",";
	wherebox += "mdsys.sdo_geometry(2003,null,null,";
	wherebox += "mdsys.sdo_elem_info_array(1,1003,3),";
	wherebox += "mdsys.sdo_ordinate_array(";
	wherebox += Te2String(box.x1(),10) + ", " + Te2String(box.y1(),10);
	wherebox += ", " + Te2String(box.x2(),10) + ", " + Te2String(box.y2(),10) + ")),";
	wherebox += "'mask=anyinteract querytype = window') = 'TRUE'";

	return wherebox;
}


string 
TeOracleSpatial::getSQLBoxWhere (const string& table1, const string& table2, TeGeomRep rep2, TeGeomRep rep1)
{
	string wherebox;
	string colname1, colname2; 
	colname1 = colname2 = "spatial_data";
	
	if(rep1 == TeRASTER) 
		colname1 = "block_box";

	if(rep2 == TeRASTER)
		colname2 = "block_box";
		
	wherebox =  "MDSYS.SDO_FILTER ("+ table1 +"."+ colname1 +",";
	wherebox += table2 +"."+ colname2 +", 'querytype = window') = 'TRUE'";  

	return wherebox;
}


string 
TeOracleSpatial::getSQLBoxSelect (const string& tableName, TeGeomRep rep)
{
	
	string select;
	string colname = "spatial_data";

	if(rep == TeRASTER)
		colname = "block_box";

	select =  tableName +".* , ";
	select += " SDO_GEOM.SDO_MIN_MBR_ORDINATE("+ tableName +"."+ colname +", 1) as lower_x,";
	select += " SDO_GEOM.SDO_MIN_MBR_ORDINATE("+ tableName +"."+ colname +", 2) as lower_y,";
	select += " SDO_GEOM.SDO_MAX_MBR_ORDINATE("+ tableName +"."+ colname +", 1) as upper_x,"; 
	select += " SDO_GEOM.SDO_MAX_MBR_ORDINATE("+ tableName +"."+ colname +", 2) as upper_y ";
	return select;
}

bool 
TeOracleSpatial::getMBRSelectedObjects(string geomTable,string colGeom, string fromClause, string whereClause, string afterWhereClause, TeGeomRep /* repType */,TeBox &bout, const double& tol)
{

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	if (!portal)
		return false;

	string sql = "SELECT SDO_AGGR_MBR(" + colGeom + ") ";
	sql += " FROM " + fromClause;
	
	if(!whereClause.empty())
		sql += " WHERE " + whereClause;  
	
	if(!portal->query(sql)) 
	{
		delete portal;
		return false;
	}

	if(!portal->fetchRow())
	{
		delete portal;
		return false;
	}

	try
	{
		TeCoord2D coord1,coord2;
		portal->GetCoordinates (1, coord1);
		portal->GetCoordinates (2, coord2);
		TeBox b(coord1.x()-tol, coord1.y()-tol, coord2.x()+tol, coord2.y()+tol);
		bout = b;
	}
	
	catch(...)
	{
		delete portal;
		return false;
	}
	
	delete portal;
	return true;
}


string
TeOracleSpatial::getSQLStatistics (TeGroupingAttr& attrs)
{
	string sql = "";
	string virg = "";

	TeGroupingAttr::iterator it = attrs.begin();
	int count = 0;
	while(it != attrs.end())
	{
		if(count>0)
			virg = ",";

		switch ((*it).second)
		{
			case TeSUM:
				sql += virg +" SUM( "+ (*it).first.name_ +") AS SUM_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMAXVALUE:
				sql += virg +" MAX( "+ (*it).first.name_ +") AS MAX_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMINVALUE:
				sql += virg +" MIN( "+ (*it).first.name_ +") AS MIN_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeCOUNT:
				sql += virg +" COUNT( "+ (*it).first.name_ +") AS COUNT_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeMEAN:
				sql += virg +" AVG( "+ (*it).first.name_ +") AS AVG_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeSTANDARDDEVIATION:
				sql += virg +" STDDEV( "+ (*it).first.name_ +") AS STDDEV_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			case TeVARIANCE:
				sql += virg +" VARIANCE( "+ (*it).first.name_ +") AS VAR_"+ Te2String(count);
				(*it).second = TeNOSTATISTIC;
				++count;
				break;
			default:
				break;
		}
		++it;
	}
	return sql;
}

string 
TeOracleSpatial::getSQLAutoNumber(const string& table)
{
	string aut = getNameSequence(table) +".NEXTVAL";
	return aut;
}

bool 
TeOracleSpatial::getMBRGeom(string tableGeom, string object_id, TeBox& box, string colGeom)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	if (!portal)
		return false;

	string sql = "SELECT SDO_GEOM.SDO_MBR(" + tableGeom + "." + colGeom + ") ";
	sql += " FROM " + tableGeom;
	sql += " WHERE object_id = '" + object_id + "'";

	if((!portal->query(sql)) || (!portal->fetchRow()))
	{
		delete portal;
		return false;
	}
	
	TeCoord2D coord1,coord2;
	portal->GetCoordinates (1, coord1);
	portal->GetCoordinates (2, coord2);
	TeBox b(coord1.x(), coord1.y(), coord2.x(), coord2.y());
	box = b;
	
	delete portal;
	return true;
}


bool 
TeOracleSpatial::insertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, 
								   unsigned char *buf,unsigned long size, int band, unsigned int res, unsigned int subband)
{

	if (blockId.empty()) // no block identifier provided
	{
		errorMessage_ = "bloco sem identificador";
		return false;
	}

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) this->getPortal();
	if (!portal)
		return false;

	bool update = false;
	string q ="SELECT * FROM " + table; 
	q += " WHERE block_id='" + blockId + "' FOR UPDATE ";

	if (!portal->query (q))
	{
		delete portal;
		return false;
	}
		// check if this block is alread in the database
	if (portal->fetchRow())
		update = true;

	string sdo_geom = " MDSYS.SDO_GEOMETRY(2003, NULL, NULL";
	sdo_geom += ", MDSYS.SDO_ELEM_INFO_ARRAY( 1, 1003, 3 )";
	sdo_geom += ", MDSYS.SDO_ORDINATE_ARRAY( " ;
	sdo_geom += Te2String(ll.x());
	sdo_geom += ", " + Te2String(ll.y());
	sdo_geom += ", " + Te2String(ur.x());
	sdo_geom += ", " + Te2String(ur.y());
	sdo_geom += ")) ";

	try
	{
		if (!update)
		{
			portal->freeResult();

			q = "INSERT INTO "+ table +" (block_id, band_id, subband, ";
			q += " resolution_factor, block_box, block_size, spatial_data) VALUES ( ";
			q += "'" + blockId + "'";
			q += ", " + Te2String(band);
			q += ", " + Te2String(subband);
			q += ", " + Te2String(res);
			q += ", " + sdo_geom;
			q += ", " + Te2String(size);
			q += ", EMPTY_BLOB()";
			q += ")";
	
			if (!this->Conn->Execute(q))
				return false;
			
			q = "SELECT * FROM " + table; 
			q += " WHERE block_id= '" + blockId + "' FOR UPDATE ";
			
			if((!portal->query(q)) || (!portal->fetchRow()))
			{
				delete portal;
				return false;
			}
		}
		
		portal->getCursor()->WriteBlob(buf, size);
		portal->freeResult();		
	}
	catch(...)
	{
		errorMessage_ = "Oppps !";
		delete portal;
		return false;
	}
	delete portal;
	return true;
	
}

bool 
TeOracleSpatial::insertBlob (const string& tableName, const string& /* columnBlob */, TeAttributeRep& columnId, const string& valueId, unsigned char* data, int size)
{
	
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	if (!portal)
		return false;

	try
	{
		string q = "SELECT * FROM "+ tableName +" WHERE "+ columnId.name_ +" = ";
	
		switch (columnId.type_ )
		{
			case TeSTRING:
				q += "'"+ valueId + "'";
				break;
			default:
				q += valueId;
				break;
		}
		q += " FOR UPDATE ";

		if((!portal->query(q)) || (!portal->fetchRow()))
		{
			delete portal;
			return false;
		}
				
		portal->getCursor()->WriteBlob(data, size);
		portal->freeResult();		
	}

	catch(...)
	{
		errorMessage_ = "Error!";
		delete portal;
		return false;
	}

	delete portal;
	if(data)
		delete []data;
	return true;
}
	

bool 
TeOracleSpatial::insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, const string& fileName)
{
	unsigned char	*cdata = 0;
	int		size;
	FILE	*fp = 0;
	
	struct	_stat stat;
	int		result;
	
	result = _stat(fileName.c_str(), &stat);
	
	if( result != 0 )
		return false;
	
	size = stat.st_size;

	cdata = new unsigned char[size];
	fp = fopen(fileName.c_str(), "rb");
	fread(cdata, sizeof(unsigned char), size, fp); 

	bool status = insertBlob (tableName, columnBlob, columnId,  valueId, cdata, size);

	if(fp)
		fclose(fp);

	return status;
}
	

bool 
TeOracleSpatial::dropConceptualModel()
{
	//drop tables
	string drop = "	DROP TABLE te_raster_metadata";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_representation";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_tables_relation";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_layer_table";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_visual";
	if(!execute(drop))
		return false;
	
	drop = "DROP TABLE te_legend";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_theme";
	if(!execute(drop))
		return false;
	
	drop = "DROP TABLE te_theme_table";
	if(!execute(drop))
		return false;
	
	drop = "DROP TABLE te_view";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_layer";
	if(!execute(drop))
		return false;

	drop = "DROP TABLE te_projection";
	if(!execute(drop))
		return false;

	//drop sequences
	drop = "DROP SEQUENCE "+ getNameSequence("te_tables_relation"); 
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_layer_table");  
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_legend");
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_theme");
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_theme_table");
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_view"); 
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_layer");
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_projection");
	if(!execute(drop))
		return false;

	drop = "DROP SEQUENCE "+ getNameSequence("te_representation");
	if(!execute(drop))
		return false;

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)this->getPortal();
		
	//tabelas geometricas com SDO_GEOMETRY
	string sql = "SELECT TABLE_NAME FROM USER_SDO_GEOM_METADATA ";
	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	while(portal->fetchRow())
	{
		string tableName = portal->getData(0);
		string del = "drop table " + tableName;
		this->execute (del);
	}
	
	portal->freeResult();

	sql = "DELETE USER_SDO_GEOM_METADATA ";
	this->execute(sql);
	
	//spatial index
	sql = "SELECT INDEX_NAME FROM USER_SDO_INDEX_INFO";
	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	while(portal->fetchRow())
	{
		string indexName = portal->getData(0);
		string del = "drop index " + indexName;
		this->execute(del);
	}
	
	delete portal;
	return true;
}

TeOracleSpatialPortal::TeOracleSpatialPortal ( TeOracleSpatial*  pDatabase) 
{
	cursor = new OCICursor(pDatabase->Conn);
	db_ = pDatabase;
}


TeOracleSpatialPortal::~TeOracleSpatialPortal () 
{
	delete cursor;
	cursor = NULL;
}


bool TeOracleSpatialPortal::isConnected()
{
	return (cursor->Conn()->IsConnected());
}


bool TeOracleSpatialPortal::isEOF() 
{
	if (!isConnected())
		return true;

	return false;
}


bool
TeOracleSpatialPortal::moveFirst() 
{
	if (cursor->MoveFirst())
		return true;
	else
		return false;
}


bool 
TeOracleSpatialPortal::moveNext() 
{
	if(cursor->MoveNext())
		return true;
	return false;
}


bool TeOracleSpatialPortal::query (const string &q,TeCursorLocation /* l */ , TeCursorType /* t */, TeCursorEditType /* e */, TeCursorDataType /* dt */)  //Ok
{
	errorMessage_.clear ();

	if (!cursor->isOpen())
	{
		if(!cursor->Open())
		{
			numRows_ = 0;
			return false;
		}
	}

	if (!cursor->Query(q))
	{
		this->errorMessage_ = cursor->getErrorMessage();
		numRows_ = 0;
		return false;
	}
	
	numFields_= this->cursor->NumCol();
	
	attList_.clear ();
	int i;
	for(i = 1; i <= numFields_ ; i++)
	{
		TeAttribute attribute;

		switch (cursor->ColType(i))
		{
			case 3: //INTEGER
			attribute.rep_.type_ = TeINT;
			break;

			case 2:  //NUMBER
			case 4: //FLOAT DOUBLE
			attribute.rep_.type_ = TeREAL;
			break;

			case 12: //Date
			attribute.rep_.type_ = TeDATETIME;
			attribute.dateChronon_ = TeSECOND;
			attribute.dateTimeFormat_ = "DD/MM/YYYY HH24:MI:SS";
			break;

			case 113: //Blob
			attribute.rep_.type_ = TeBLOB;
			break;

			case 96: //CHAR
			case 9: //VARCHAR:
			case 1: //VARCHAR2:
			attribute.rep_.type_ = TeSTRING;
			break;

			case 108: //OBJECT: // SDO_GEOMETRY
			attribute.rep_.type_ = TeOBJECT;
			break;
			default :
			attribute.rep_.type_ = TeUNKNOWN;
			break;
		} 
		
		//attribute.rep_.name_ = TeConvertToLowerCase(cursor->ColName(i)); 
		attribute.rep_.name_ = cursor->ColName(i); 
		attribute.rep_.numChar_ = cursor->ColSize(i);
		attList_.push_back ( attribute );

	}

	curRow_=-1;
	return true;
}



bool TeOracleSpatialPortal::querySDO (const string &q)  
{
	errorMessage_.clear ();

	if (!cursor->isOpen())
	{
		if(!cursor->Open())
		{
			numRows_ = 0;
			return false;
		}
	}

	if (!cursor->QuerySDO(q))
	{
		this->errorMessage_ = cursor->getErrorMessage();
		numRows_ = 0;
		return false;
	}
	
	numFields_= this->cursor->NumCol();
	
	attList_.clear ();
	int i;
	for(i = 1; i <= numFields_ ; i++)
	{
		TeAttribute attribute;

		switch (cursor->ColType(i))
		{
			case 3: //INTEGER
			attribute.rep_.type_ = TeINT;
			break;

			case 2:  //NUMBER
			case 4: //FLOAT DOUBLE
			attribute.rep_.type_ = TeREAL;
			break;

			case 12: //Date
			attribute.rep_.type_ = TeDATETIME;
			break;

			case 113: //Blob
			attribute.rep_.type_ = TeBLOB;
			break;

			case 96: //CHAR
			case 9: //VARCHAR:
			case 1: //VARCHAR2:
			attribute.rep_.type_ = TeSTRING;
			break;

			case 108: //OBJECT: // SDO_GEOMETRY
			attribute.rep_.type_ = TeOBJECT;
			break;
			default :
			attribute.rep_.type_ = TeUNKNOWN;
			break;
		} 
		
		attribute.rep_.name_ = cursor->ColName(i); 
		attribute.rep_.numChar_ = cursor->ColSize(i);
		attList_.push_back ( attribute );
	}

	curRow_=-1;
	return true;
}


void TeOracleSpatialPortal::freeResult () 
{
	cursor->FreeResult();
}


bool TeOracleSpatialPortal::fetchRow () 
{
	try
	{	
		if( !isConnected() )
			return false;
		if( numFields_ == 0/* || numRows_ < 1*/)
			return false;
		
		if (curRow_ == -1)
		{
			if(moveFirst())
				curRow_++;
			else
				return false;
		}
		else
		if (moveNext())
		{
			curRow_++;
			return true;
		}
		else
			return false;
	}
	
	catch(...) //_com_error &e)
	{
		return false;
	}

	return true;
}

bool TeOracleSpatialPortal ::fetchRow (int i)
{
	
	if(curRow_ == i)
		return true;

	if (cursor->MoveTo(i))
	{
		curRow_ = cursor->CurrentRow()
			;
		return true;
	}
	
	return false;
}

string  
TeOracleSpatial::escapeSequence (const string& from)
{
	string aux = "";
	unsigned int size = from.length ();
	unsigned int i;

	for(i=0; i < size; i++)
	{
		if(from[i] == 39) 
			aux += from[i] + unsigned char(39);
		else
			aux += from[i];
    }
	return aux;
}

bool 
TeOracleSpatial::insertTable(TeTable &table)
{
	string tableName = table.name();
	int size = table.size();
	TeAttributeList att = table.attributeList();
	TeAttributeList::iterator it = att.begin();
	TeAttributeList::iterator itEnd = att.end();

	TeTableRow row;
	int i;
	unsigned int j;
	for ( i = 0; i < size; i++  )
	{
		row = table[i];
		it = att.begin();
		string q = 	"INSERT INTO "+tableName+" VALUES(";
		j = 1;
		int jj = 0;
		while ( it != itEnd )
		{
			if (j != 1)
				q += ", ";

  			string oracleFormat="";
			string dateTime="";
			if(((*it).rep_.type_==TeDATETIME) && (!row[jj].empty()))
			{
				TeTime t  (row[jj], (*it).dateChronon_, (*it).dateTimeFormat_, (*it).dateSeparator_,
						   (*it).timeSeparator_, (*it).indicatorPM_);
				
				dateTime = t.getDateTime("DDsMMsYYYYsHHsmmsSS");
				oracleFormat = " TO_DATE('" + dateTime + "', 'DD/MM/YYYY HH24:MI:SS')";
			}
			
			switch ((*it).rep_.type_)
  			{
  				case TeSTRING:
					q += "'"+ escapeSequence(row[jj]) +"'";
  				break;
  				case TeREAL:
					q += row[jj]; 
  				break;
  				case TeINT:
					q += row[jj];
  				break;
				case TeDATETIME:
					q += oracleFormat;
  				break;
				case TeCHARACTER:
					q += "'" + escapeSequence(row[jj]) + "'";
  				break;
				case TeBLOB:
					q += " EMPTY_BLOB() ";
				break;
  				default:
					q += "'"+ escapeSequence(row[jj]) +"'";
  				break;
  			}
			++it;
			j++;
			jj++;
		}
		q += ")";
		
		if (!execute(q))
			continue;
	}
	return true;
}
	
	
bool
TeOracleSpatial::updateTable (TeTable &table)
{
	string tableName = table.name();
	TeAttributeList att = table.attributeList();
	TeAttributeList::iterator it = att.begin();

	TeTableRow row;
	unsigned int i;
	unsigned int j;
	string uniqueName = table.uniqueName();
	string uniqueVal;
	for ( i = 0; i < table.size(); i++  )
	{
		row = table[i];
		it = att.begin();
		string q = 	"UPDATE "+tableName+" SET ";
		j = 1;
		int jj = 0;
		while ( it != att.end() )
		{
			string oracleFormat;
			string dateTime; 
			if((*it).rep_.type_==TeDATETIME)
			{
				TeTime t  (row[jj], (*it).dateChronon_, (*it).dateTimeFormat_, (*it).dateSeparator_,
						   (*it).timeSeparator_, (*it).indicatorPM_);
				
				dateTime = t.getDateTime();
				oracleFormat = "DD/MM/YYYY HH24:MI:SS"; 
			}
			
			if (uniqueName != (*it).rep_.name_)
			{
				q += (*it).rep_.name_ + "=";
  				switch ((*it).rep_.type_)
  				{
  					case TeSTRING:
						q += "'"+escapeSequence(row[jj])+"'";
  					break;
  					case TeREAL:
						q += row[jj];
  					break;
  					case TeINT:
						q += row[jj];
  					break;
					case TeDATETIME:
						q += " TO_DATE('" + dateTime + "', '"+ oracleFormat +"')";
  					break;
					case TeCHARACTER:
						q += "'" + escapeSequence(row[jj]) + "'";
  					break;
  					default:
						q += "'"+escapeSequence(row[jj])+"'";
  					break;
  				}
				if (j<att.size())
					q+= ",";
			}
			else
				uniqueVal = row[jj];

			++it;
			j++;
			jj++;
		}
		q += " WHERE " + uniqueName + " = " + uniqueVal;
		if (!execute(q))
			continue;
	}
	return true;
}
	

bool 
TeOracleSpatial::insertProjection (TeProjection *proj)
{
	string insert = "INSERT INTO te_projection (projection_id, name, long0, lat0,";
	insert += " offx, offy, stlat1, stlat2, unit, scale, hemis, datum, ";
	insert += " radius, flattening, dx , dy, dz ) VALUES ( ";
	insert += "te_projection_seq.NEXTVAL";
	insert += ", '" + escapeSequence(proj->name()) + "'";
	insert += ", " + Te2String(proj->params().lon0*TeCRD,10);
	insert += ", " + Te2String(proj->params().lat0*TeCRD,10);
	insert += ", " + Te2String(proj->params().offx,10);
	insert += ", " + Te2String(proj->params().offy,10);
	insert += ", " + Te2String(proj->params().stlat1*TeCRD,10);
	insert += ", " + Te2String(proj->params().stlat2*TeCRD,10);
	insert += ", '" + escapeSequence(proj->params().units) + "'";
	insert += ", " + Te2String(proj->params().scale,10);
	insert += ", " + Te2String(proj->params().hemisphere);
	insert += ", '" + escapeSequence(proj->params().datum.name()) + "'";
	insert += ", " + Te2String(proj->params().datum.radius(),10);
	insert += ", " + Te2String(proj->params().datum.flattening(),10);
	insert += ", " + Te2String(proj->params().datum.xShift(),10);
	insert += ", " + Te2String(proj->params().datum.yShift(),10);
	insert += ", " + Te2String(proj->params().datum.zShift(),10);
	insert += ")";

	if(!execute(insert))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_projection!";   
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*) getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT te_projection_seq.CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_projection_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	proj->id(atoi((const char*)ocip->getData(0))); 
	delete ocip;

	return true;
}



bool
TeOracleSpatial::insertRepresentation (int layerId, TeRepresentation& rep)
{
	if (layerId <= 0)
		return false;
	
	string ins;
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*) getPortal();
	if(!ocip)
		return false;

	try
	{
		ins = " INSERT INTO te_representation (repres_id, layer_id, geom_type, geom_table, ";
		ins += " description, lower_x, lower_y, upper_x, upper_y, res_x, res_y, num_cols, ";
		ins += " num_rows) VALUES (";
		ins += " te_representation_seq.NEXTVAL ";
		ins += ", " + Te2String(layerId);
		ins += ", " + Te2String(static_cast<int>(rep.geomRep_));
		ins += ", '" + escapeSequence(rep.tableName_) + "'";
		ins += ", '" + escapeSequence(rep.description_) + "'";
		ins += ", " + Te2String(rep.box_.x1(),10);
		ins += ", " + Te2String(rep.box_.y1(),10);
		ins += ", " + Te2String(rep.box_.x2(),10);
		ins += ", " + Te2String(rep.box_.y2(),10);
		ins += ", " + Te2String(rep.resX_,10);
		ins += ", " + Te2String(rep.resY_,10);
		ins += ", " + Te2String(rep.nCols_);
		ins += ", " + Te2String(rep.nLins_);
		ins += ")";

		if(!execute(ins))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error inserting in the table te_representation!";   
			return false;
		}
		
		string seq = "SELECT te_representation_seq.CURRVAL FROM DUAL";
		if(!ocip->query(seq))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error in the sequence te_representation_seq!";  
			delete ocip;
			return false;
		}

		if(!ocip->fetchRow())
		{
			errorMessage_ = "Sequence value not found!";
			delete ocip;
			return false;
		}

		int id = atoi((const char*)ocip->getData(0)); 
		rep.id_ = id;
	}

	catch(...)
	{
		errorMessage_ = "Error inserting in the table te_representation!"; 
		delete ocip;
		return false;
	}

	delete ocip;
	return true;
}


bool
TeOracleSpatial::insertLegend (TeLegendEntry* leg)
{
	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	try
	{
		string ins = "INSERT INTO te_legend (legend_id, theme_id, group_id, ";
		ins += " num_objs, lower_value, upper_value, label) VALUES ( ";
		ins += "te_legend_seq.NEXTVAL";
		ins += ", " + Te2String(leg->theme());
		ins += ", " + Te2String(leg->group());
		ins += ", " + Te2String(leg->count());
		ins += ", '" + escapeSequence(leg->from()) + "'";
		ins += ", '" + escapeSequence(leg->to()) + "'";
		ins += ", '" + escapeSequence(leg->label()) + "'";
		ins += ")";
			
		if (!execute(ins))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error inserting in the table te_legend!"; 
			return false;
		}

		string seq = "SELECT te_legend_seq.CURRVAL FROM DUAL";
		if(!ocip->query(seq))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error in the sequence te_theme_seq!";  
			delete ocip;
			return false;
		}

		if(!ocip->fetchRow())
		{
			errorMessage_ = "Sequence value not found!";;
			delete ocip;
			return false;
		}

		int index = atoi((const char*)ocip->getData(0)); 
		leg->id(index);
	}
	catch(...)
	{
		errorMessage_ = "Error inserting in the table te_legend!"; 
		delete ocip;
		return false;
	}
		
	delete ocip;
	legendMap_[leg->id()] = leg;
	return insertVisual(leg);
}


bool 
TeOracleSpatial::AllocateOrdinatesObject(TePolygon &poly, string& elInfo, OCICursor* cursor)
{
	int		totalsize, ni, size;
	double	xult, yult;
	short	orient;

	ni = poly.size () - 1;

	xult = -9999.99;
	yult = -9999.99;

	totalsize = 0;

	try
	{
		//OCI: create the ordinates array
		if(!cursor)
			Conn->AllocateObjectOrdinates();
		
		for (int k = 0; k <= ni; k++ )
		{
			TeLinearRing ring ( poly[k] );
			totalsize += ring.size();
			size = ring.size();
			orient = TeOrientation(ring);
			
			if (k==0)  //external polygon: UNCLOCKWISE
			{
				elInfo = "1, 1003, 1";
				if(orient == TeCOUNTERCLOCKWISE)   
				{
					for (int i=0;i<size;i++)
					{
						if(xult != ring[i].x() || yult != ring[i].y())
						{
							if(cursor)
							{
								cursor->AppendOrdinates(ring[i].x());
								cursor->AppendOrdinates (ring[i].y());	
							}
							else
							{
								Conn->AppendOrdinates(ring[i].x());
								Conn->AppendOrdinates (ring[i].y());	
							}
						
							xult = ring[i].x();
							yult = ring[i].y();
						}
					}
				}
				//keep UNCLOCKWISE ring 
				else
				{
					for (int i=0;i<size;i++)
					{
						if(xult != ring[size-1-i].x() || yult != ring[size-1-i].y())
						{
							if(cursor)
							{
								cursor->AppendOrdinates(ring[size-1-i].x());
								cursor->AppendOrdinates (ring[size-1-i].y());
							}
							else
							{
								Conn->AppendOrdinates(ring[size-1-i].x());
								Conn->AppendOrdinates (ring[size-1-i].y());
							}
							
							xult = ring[size-1-i].x();
							yult = ring[size-1-i].y();
						}
					}
				}
			}

			else  //internal polygon: CLOCKWISE
			{
				int pos = ((totalsize - size) * 2) + 1; 
				elInfo += ", " + Te2String(pos) + ", 2003, 1";
						
				if(orient == TeCLOCKWISE)   
				{
					for (int i=0;i<size;i++)
					{
						if(xult != ring[i].x() || yult != ring[i].y())
						{
							if(cursor)
							{
								cursor->AppendOrdinates(ring[i].x());
								cursor->AppendOrdinates (ring[i].y());
							}
							else
							{
								Conn->AppendOrdinates(ring[i].x());
								Conn->AppendOrdinates (ring[i].y());
							}

							xult = ring[i].x();
							yult = ring[i].y();
						}
					}
				}
				//keep CLOCKWISE ring
				else
				{
					for (int i=0;i<size;i++)
					{
						if(xult != ring[size-1-i].x() || yult != ring[size-1-i].y())
						{
							Conn->AppendOrdinates(ring[size-1-i].x());
							Conn->AppendOrdinates (ring[size-1-i].y());
							
							xult = ring[size-1-i].x();
							yult = ring[size-1-i].y();
						}
					}
				}
			}
		}//for all rings
	}
	catch(...)
	{
		return false;
	}
	
	return true;	
}

bool 
TeOracleSpatial::AllocateOrdinatesObject(TeLine2D &line, OCICursor* cursor)
{
	int size = line.size();
	double	xult, yult;
	xult = -9999.99;
	yult = -9999.99;

	try
	{
		//OCI: create the ordinates array
		if(!cursor)
			Conn->AllocateObjectOrdinates ();

		for (int i=0;i<size;i++)
		{		
			if(xult != line[i].x() || yult != line[i].y())
			{
				if(cursor)
				{
					cursor->AppendOrdinates(line[i].x());
					cursor->AppendOrdinates(line[i].y());
				}
				else
				{
					Conn->AppendOrdinates(line[i].x());
					Conn->AppendOrdinates(line[i].y());
				}
				xult = line[i].x();
				yult = line[i].y();
			}
		}
	}
	catch(...)
	{
		return false;
	}

	return true;
}


bool 
TeOracleSpatial::insertPolygon (const string& table, TePolygon &poly)
{
	
	string	elinfo;
	
	if(!AllocateOrdinatesObject(poly, elinfo))
		return false;
	

	string ins = "INSERT INTO " + table + " ( ";
	ins += " geom_id, object_id, spatial_data) VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(poly.objectId()) + "'";
	ins += ", MDSYS.SDO_GEOMETRY(2003, NULL, NULL";
	ins += ", MDSYS.SDO_ELEM_INFO_ARRAY( " + elinfo + " )";
	ins += ", :ordinates) ";
	ins += " )";

	if(!Conn->ExecuteSDOSTM(ins))
	{
		errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	/*
	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	poly.geomId(index); 
	delete ocip;
	*/

	return true;
}

bool 
TeOracleSpatial::updatePolygon (const string& table, TePolygon &poly)
{
	if(!tableExist(table))
	{	
		errorMessage_ = "Table not exist!";
		return false;
	}
	
	string elinfo;

	if(!AllocateOrdinatesObject(poly, elinfo))
		return false;
		
	string sql;
	sql =  "UPDATE " + table + " SET ";
	sql += ", object_id = '" + poly.objectId() + "'";
	sql += ", spatial_data = ";
	sql += " MDSYS.SDO_GEOMETRY(2003, NULL, NULL";
	sql += ", MDSYS.SDO_ELEM_INFO_ARRAY( " + elinfo + " )";
	sql += ", :ordinates) ";
	sql += " WHERE geom_id = " + poly.geomId();

	//OCI
	if(!Conn->ExecuteSDOSTM(sql))
	{
		errorMessage_ = "Error updating in the table " + table + "!"; 
		return false;
	}
	
	return true;
}

bool 
TeOracleSpatial::selectPolygonSet (const string& table, const string& criteria, TePolygonSet &ps)
{
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string sql ="SELECT * FROM " + table;
	if (!criteria.empty())
		sql += " WHERE " + criteria;
	sql += " ORDER BY object_id ASC ";
	 
	if (!ocip->query(sql) || !ocip->fetchRow())
	{
		delete ocip;
		return false;
	}
	
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = ocip->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);

	delete ocip;
	return true;
}

bool 
TeOracleSpatial::loadPolygonSet (const string& table, const string& geoid, TePolygonSet &ps)
{
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string q ="SELECT * FROM " + table;

	if (geoid != "")
		q += " WHERE object_id = '" + geoid +"'";
	
	if (!ocip->query(q) || !ocip->fetchRow())
	{	
		delete ocip;
		return false;
	}

	bool flag = true;
	do
	{
		TePolygon poly;
		flag = ocip->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);
	delete ocip;
	return true;
}


bool 
TeOracleSpatial::loadPolygonSet (const string& table, TeBox &box, TePolygonSet &ps)
{
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string q = "SELECT * FROM " + table;
	q += this->getSQLBoxWhere (box, TePOLYGONS);
	q += " ORDER BY object_id ";

	if (!ocip->query(q) || !ocip->fetchRow())
	{	
		delete ocip;
		return false;
	}
	bool flag = true;
	do
	{
		TePolygon poly;
		flag = ocip->fetchGeometry(poly);
		ps.add(poly);
	}
	while (flag);
	delete ocip;
	return true;
}

bool 
TeOracleSpatial::loadPolygonSet(TeTheme* theme, TePolygonSet &ps)
{
	string collTable = theme->collectionTable();
	if (collTable.empty())
		return false;

	TeLayer* themeLayer = theme->layer();
	if (!themeLayer->hasGeometry(TePOLYGONS))
		return false;
	
	string polygonTable = themeLayer->tableName(TePOLYGONS);
	if (polygonTable.empty())
		return false;

	string sql = "SELECT * FROM (" + polygonTable + " RIGHT JOIN " + collTable;
	sql += " ON " + polygonTable + ".object_id = " + collTable + ".object_id)";
	
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	if (!ocip->query(sql) || !ocip->fetchRow())
	{
		delete ocip;
		return false;
	}

	bool flag = true;
	do
	{
		TePolygon poly;
		flag = ocip->fetchGeometry(poly);
		ps.add ( poly );
	}
	while (flag);		
	delete ocip;
	return true;
}

TeDatabasePortal* 
TeOracleSpatial::loadPolygonSet(const string& table, TeBox &box)
{
	TeOracleSpatialPortal *portal = (TeOracleSpatialPortal*)this->getPortal();
	if (!portal)
		return 0;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (box, TePOLYGONS);
	q += " ORDER BY object_id ";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	else 
		return portal;
}


//Spatial query
//retornam um portal
bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, TeDatabasePortal *portal, int relate, const string& actCollTable)
{
	string Ids = getStringIds(actIdsIn);
	string actGeomColl = "spatial_data";
	
	string spatialRel = getOracleSpatialRelation(relate);
	
	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable1.* ";
	sql += " FROM "+ actGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!actCollTable.empty())
	{
		sql += ", "+ actCollTable +" collTable ";
		sql += " WHERE geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	sql += " geomTable2.object_id IN (" + Ids + ") AND ";

	if(relate==TeEQUALS)
		sql += " geomTable1.object_id NOT IN (" + Ids + ") AND ";  
	
	if(relate==TeDISJOINT)
		sql += " NOT ";  // NOT ANYINTERACT

	sql += " SDO_RELATE(geomTable1."+ actGeomColl +", geomTable2."+ actGeomColl +", 'mask= "; 
	sql += spatialRel + " querytype=WINDOW') = 'TRUE'";

	portal->freeResult();
	if(!((TeOracleSpatialPortal*)portal)->querySDO (sql)) 
		return false;

	return (portal->fetchRow());
}

bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, const string& visGeomTable, TeGeomRep /* visRep */, TeDatabasePortal *portal, int relate, const string& visCollTable)
{
	string Ids = getStringIds(actIdsIn);
	string spatialRel = getOracleSpatialRelation(relate);
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";
	
	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable1.* ";
	sql += " FROM "+ visGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!visCollTable.empty())
	{
		sql += ", "+ visCollTable +" collTable";
		sql += " WHERE geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	sql += " geomTable2.object_id IN (" + Ids + ") AND ";

	if(relate==TeDISJOINT)
		sql += " NOT ";  // NOT ANYINTERACT

	sql += " SDO_RELATE(geomTable1."+ visGeomColl +", geomTable2."+ actGeomColl +", 'mask= "; 
	sql += spatialRel + " querytype=WINDOW') = 'TRUE'";

	portal->freeResult();
	if(!((TeOracleSpatialPortal*)portal)->querySDO(sql))
		return false;
	
	return (portal->fetchRow());
		
}

bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep /* actRep */, TeGeometry* geom, TeDatabasePortal *portal, int relate, const string& actCollTable) 
{
	portal->freeResult();
	string elinfo, sdo;
	
	OCICursor	*cursor = ((TeOracleSpatialPortal*)portal)->getCursor();	
	string spatialRel = getOracleSpatialRelation(relate);
	string actGeomColl = "spatial_data";

	if(geom->elemType()==TePOLYGONS)
	{
		TePolygon poly, *pPoly;
		pPoly = new TePolygon();
		pPoly = (TePolygon*)geom;
		poly = *pPoly;

		if(!AllocateOrdinatesObject(poly, elinfo, cursor))
		{
			delete cursor;
			return false;
		}

		sdo = " MDSYS.SDO_GEOMETRY(2003, NULL, NULL, ";
		sdo += " MDSYS.SDO_ELEM_INFO_ARRAY(" + elinfo + "), ";
		sdo += " :ordinates)";

		//delete pPoly;  //delete tambm o geom, talvez deixar para aplicacao
	}
	
	else if (geom->elemType()==TeLINES)
	{
		TeLine2D line, *pLine;
		pLine = new TeLine2D();
		pLine = (TeLine2D*)geom;
		line = *pLine;

		if(!AllocateOrdinatesObject(line, cursor))
		{
			delete cursor;
			return false;
		}

		elinfo = "1, 2, 1";

		sdo = " MDSYS.SDO_GEOMETRY(2002, NULL, NULL, ";
		sdo += " MDSYS.SDO_ELEM_INFO_ARRAY(" + elinfo + "), ";
		sdo += " :ordinates)";

		//delete pLine;  //delete tambm o geom
	}
	
	else if (geom->elemType()==TePOINTS)
	{
		TePoint point, *pPoint;
		pPoint = new TePoint();
		pPoint = (TePoint*)geom;
		point = *pPoint;
		
		sdo = " MDSYS.SDO_GEOMETRY(2001, NULL, ";
		sdo += " MDSYS.SDO_POINT_TYPE( ";
		sdo += Te2String(point.location().x());
		sdo += ", " + Te2String(point.location().y());
		sdo += ", NULL )";
		sdo += ", NULL, NULL))";
		
		//delete pPoint;
	}

	else if (geom->elemType()==TeCELLS)
	{
		TeCell cell, *pCell;
		pCell = new TeCell();
		pCell = (TeCell*)geom;
		cell = *pCell;

		TeBox b = cell.box();
		
		sdo = " MDSYS.SDO_GEOMETRY(2003, NULL, NULL ";
		sdo += ", MDSYS.SDO_ELEM_INFO_ARRAY( 1, 1003, 3 )";
		sdo += ", MDSYS.SDO_ORDINATE_ARRAY( " ;
		sdo += Te2String(b.lowerLeft().x());
		sdo += ", " + Te2String(b.lowerLeft().y());
		sdo += ", " + Te2String(b.upperRight().x());
		sdo += ", " + Te2String(b.upperRight().y());
		sdo += ")) ";

		//delete pCell;
	}
	
	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable.* ";
	sql += " FROM " + actGeomTable + " geomTable ";

	if(!actCollTable.empty())
	{
		sql += ", "+ actCollTable +" collTable ";
		sql += " WHERE geomTable.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	
	if(relate==TeDISJOINT)
		sql += " NOT ";  // NOT ANYINTERACT

	sql += " MDSYS.SDO_RELATE(geomTable."+ actGeomColl +", "+ sdo +", 'mask= "; 
	sql += spatialRel + " querytype=WINDOW') = 'TRUE'";

	if(!((TeOracleSpatialPortal*)portal)->querySDO(sql))
		return false;

	return (portal->fetchRow());
	
}


//retornam um vetor de object_ids resultantes da consulta
bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, Keys& actIdsOut, int relate, const string& actCollTable)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
		
	string Ids = getStringIds(actIdsIn);
	string spatialRel = getOracleSpatialRelation(relate);
	string actGeomColl = "spatial_data";

	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable1.object_id ";
	sql += " FROM "+ actGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!actCollTable.empty())
	{
		sql += ", "+ actCollTable +" collTable ";
		sql += " WHERE geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	sql += " geomTable2.object_id IN (" + Ids + ") AND ";

	if(relate==TeEQUALS)
		sql += " geomTable1.object_id NOT IN (" + Ids + ") AND ";  
	
	if(relate==TeDISJOINT)
		sql += " NOT ";  // NOT ANYINTERACT

	sql += " SDO_RELATE(geomTable1."+ actGeomColl +", geomTable2."+ actGeomColl +", 'mask= "; 
	sql += spatialRel + " querytype=WINDOW') = 'TRUE'";

	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	actIdsOut.clear();
	while(portal->fetchRow())
	{
		string objId = portal->getData (0);
		actIdsOut.push_back(objId);
	}

	sort(actIdsOut.begin(), actIdsOut.end());
	unique(actIdsOut.begin(), actIdsOut.end());

	delete portal;
	return true;
}


bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, const string& visGeomTable, TeGeomRep /* visRep */, Keys& visIdsOut, int relate, const string& visCollTable)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
	
	string Ids = getStringIds(actIdsIn);
	string spatialRel = getOracleSpatialRelation(relate);
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";
	
	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable1.object_id ";
	sql += " FROM "+ visGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!visCollTable.empty())
	{
		sql += ", "+ visCollTable +" collTable";
		sql += " WHERE geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	sql += " geomTable2.object_id IN (" + Ids + ") AND ";

	if(relate==TeDISJOINT)
		sql += " NOT ";  // NOT ANYINTERACT

	sql += " SDO_RELATE(geomTable1."+ visGeomColl +", geomTable2."+ actGeomColl +", 'mask= "; 
	sql += spatialRel + " querytype=WINDOW') = 'TRUE'";

	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	visIdsOut.clear();
	while(portal->fetchRow())
	{
		string objId = portal->getData (0);
		visIdsOut.push_back(objId);
	}

	sort(visIdsOut.begin(), visIdsOut.end());
	unique(visIdsOut.begin(), visIdsOut.end());

	delete portal;
	return true;
}


bool 
TeOracleSpatial::spatialRelation(const string& actGeomTable, TeGeomRep actRep, TeGeometry* geom, Keys& actIdsOut, int relate, const string& actCollTable)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
		
	if(!spatialRelation(actGeomTable, actRep, geom, portal, relate, actCollTable))
	{
		delete portal;
		return false;
	}
		
	actIdsOut.clear();
	do
	{
		string objId = portal->getData ("object_id");
		actIdsOut.push_back(objId);
	}while(portal->fetchRow());

	sort(actIdsOut.begin(), actIdsOut.end());
	unique(actIdsOut.begin(), actIdsOut.end());

	delete portal;
	return true;
}

// metric functions
bool
TeOracleSpatial::calculateArea(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, double &area)
{
	string Ids = getStringIds(actIdsIn);
	string actGeomColl = "spatial_data";
	
	string sql = "SELECT SUM(SDO_GEOM.SDO_AREA(g."+ actGeomColl +", m.diminfo))";
	sql += " FROM "+ actGeomTable +" g, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	area = portal->getDouble(0);
	delete portal;
	return true;
}

bool
TeOracleSpatial::calculateLength(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIdsIn, double &length)
{
	string Ids = getStringIds(actIdsIn);
	string actGeomColl = "spatial_data";
	
	string sql = "SELECT SUM(SDO_GEOM.SDO_LENGTH(g."+ actGeomColl +", m.diminfo))";
	sql += " FROM "+ actGeomTable +" g, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	length = portal->getDouble(0);
	delete portal;
	return true;
}


//distancia entre objetos de um mesma tabela
bool
TeOracleSpatial::calculateDistance(const string& actGeomTable, TeGeomRep /* actRep */, Keys& Ids, double& distance)
{
	string Id1 = Ids[0];
	string Id2 = Ids[1];
	string actGeomColl = "spatial_data";

	string sql = "SELECT MIN(SDO_GEOM.SDO_DISTANCE(g1."+ actGeomColl +", m.diminfo, ";
	sql += " g2."+ actGeomColl +", m.diminfo))";
	sql += " FROM "+ actGeomTable +" g1,"+ actGeomTable +" g2, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ Id1 +"'"; 
	sql += " AND g2.object_id = '"+ Id2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	distance = portal->getDouble(0);
	delete portal;
	return true;
}


//distancia entre objetos de duas tabelas distintas
bool
TeOracleSpatial::calculateDistance(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, TeGeomRep /* visRep */, const string& objId2, double& distance)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";
	
	string sql = "SELECT MIN(SDO_GEOM.SDO_DISTANCE(g1."+ actGeomColl +", m1.diminfo, ";
	sql += " g2."+ visGeomColl +", m2.diminfo))";
	sql += " FROM "+ actGeomTable +" g1,"+ visGeomTable +" g2, ";
	sql += " USER_SDO_GEOM_METADATA m1, USER_SDO_GEOM_METADATA m2 ";
	sql += " WHERE m1.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m1.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND m2.table_name = '"+ TeConvertToUpperCase(visGeomTable) +"'";
	sql += " AND m2.column_name = '"+ TeConvertToUpperCase(visGeomColl) +"'";
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	distance = portal->getDouble(0);
	delete portal;
	return true;
}

/*
bool 
TeOracleSpatial::withinDistance(const string& actGeomTable, TeGeomRep actRep, const TeCoord2D& point, KeysToDist& IdsDistOut, const double& max_distance, const string& actCollTable)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
		
	string Ids = getStringIds(actIdsIn);
	string actGeomColl = "spatial_data";

	//Montar a sql para passar para o Oracle
	string sql = "SELECT geomTable1.object_id ";
	sql += " FROM "+ actGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!actCollTable.empty())
	{
		sql += ", "+ actCollTable +" collTable ";
		sql += " WHERE geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
	
	sql += " geomTable2.object_id IN (" + Ids + ") AND ";
	sql += " SDO_WITHIN_DISTANCE(geomTable1."+ actGeomColl +", geomTable2."+ actGeomColl +", 'distance= "; 
	sql += Te2String(max_distance) + " querytype=WINDOW') = 'TRUE'";

	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	actIdsOut.clear();
	while(portal->fetchRow())
	{
		string objId = portal->getData (0);
		actIdsOut.push_back(objId);
	}

	sort(actIdsOut.begin(), actIdsOut.end());
	unique(actIdsOut.begin(), actIdsOut.end());

	delete portal;
	return true;
}
*/

// functions that return a new geometry

//Euclidean distance value: dist
bool
TeOracleSpatial::Buffer(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIds, TePolygonSet& bufferSet, double dist)
{
	string Ids = getStringIds(actIds);
	string actGeomColl = "spatial_data";

	string sql = "SELECT g.geom_id, g.object_id,";
	sql += " SDO_GEOM.SDO_BUFFER(g."+ actGeomColl +", m.diminfo, "+ Te2String(dist) +")";
	sql += " FROM "+ actGeomTable +" g, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		TePolygonSet polySet;
		flag = portal->fetchGeometry(polySet);
		//teste c/ buffer com filhos
		for(unsigned int i=0; i<polySet.size(); i++)
			bufferSet.add(polySet[i]);

	}while(flag);

	delete portal;
	return true;
}

bool 
TeOracleSpatial::ConvexHull(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIds, TePolygonSet& convexHullSet)
{
	string Ids = getStringIds(actIds);
	string actGeomColl = "spatial_data";
	
	string sql = "SELECT g.geom_id, g.object_id,";
	sql += " SDO_GEOM.SDO_CONVEXHULL(g."+ actGeomColl +", m.diminfo )";
	sql += " FROM "+ actGeomTable +" g, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		TePolygon poly;
		flag = portal->fetchGeometry(poly);
		convexHullSet.add(poly);
	}while(flag);

	delete portal;
	return true;
}

bool 
TeOracleSpatial::Centroid(const string&  actGeomTable , TeGeomRep /* actRep */, TePointSet& centroidSet, Keys actIds, const string& /* actCollTable */)
{
	string Ids = getStringIds(actIds);
	string actGeomColl = "spatial_data";
	
	string sql = "SELECT g.geom_id, g.object_id,";
	sql += " SDO_GEOM.SDO_CENTROID(g."+ actGeomColl +", m.diminfo )";
	sql += " FROM "+ actGeomTable +" g, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	
	//if empty it calculates the centroids to all geometries 
	if(!Ids.empty())
		sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
	{
		delete portal;
		return false;
	}

	bool flag = true;
	do 
	{
		TePoint point;
		flag = portal->fetchGeometry(point);
		centroidSet.add(point);
	}while(flag);

	delete portal;
	return true;
	
}

bool 
TeOracleSpatial::nearestNeighbors(const string& actGeomTable, const string& actCollTable, TeGeomRep actRep, const string& objId1, Keys& actIdsOut, int numRes)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
	
	if(!nearestNeighbors(actGeomTable, actCollTable, actRep, objId1, portal, numRes))
	{
		delete portal;
		return false;
	}
		
	actIdsOut.clear();
	while(portal->fetchRow())
	{
		string objId = portal->getData ("object_id");
		actIdsOut.push_back(objId);
	}

	sort(actIdsOut.begin(), actIdsOut.end());
	unique(actIdsOut.begin(), actIdsOut.end());

	delete portal;
	return true;
}


bool 
TeOracleSpatial::nearestNeighbors(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, const string& visCollTable, TeGeomRep visRep, Keys& visIdsOut, int numRes)
{
	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) getPortal(); 
	
	if(!nearestNeighbors(actGeomTable, actRep, objId1, visGeomTable, visCollTable, visRep, portal, numRes))
	{
		delete portal;
		return false;
	}
		
	visIdsOut.clear();
	while(portal->fetchRow())
	{
		string objId = portal->getData ("object_id");
		visIdsOut.push_back(objId);
	}

	sort(visIdsOut.begin(), visIdsOut.end());
	unique(visIdsOut.begin(), visIdsOut.end());

	delete portal;
	return true;
}

bool 
TeOracleSpatial::nearestNeighbors(const string& actGeomTable, const string& actCollTable, TeGeomRep /* actRep */, const string& objId1, TeDatabasePortal* portal, int numRes)
{
	string actGeomColl = "spatial_data";

	//select the spatial index
	string index = " SELECT INDEX_NAME FROM USER_SDO_INDEX_INFO";
	index += " WHERE TABLE_NAME = '"+ TeConvertToUpperCase(actGeomTable) +"'"; 

	portal->freeResult();
	if(!portal->query(index) || !portal->fetchRow())
		return false;

	string indexName = string(portal->getData(0));
	string perf = "/*+ INDEX("+ TeConvertToUpperCase(actGeomTable) +" "+ indexName +") */ ";
	
	string sql = "SELECT "+ perf +"  geomTable1.* ";
	sql += " FROM "+ actGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!actCollTable.empty())
	{
		sql += ", "+ actCollTable +" collTable ";
		sql += " WHERE ";
		sql += " geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
		
	sql += " SDO_NN(geomTable1."+ actGeomColl +", geomTable2."+ actGeomColl +", 'sdo_batch_size=10')='TRUE' AND ";
	sql += " geomTable2.object_id = '" + objId1 + "' AND ";
	sql += " geomTable1.object_id <> '"+ objId1 +"' AND ";
	sql += " ROWNUM <= "+ Te2String(numRes);

	portal->freeResult();
	if(!((TeOracleSpatialPortal*)portal)->querySDO(sql))
		return false;
	
	return true;
}
	
bool 
TeOracleSpatial::nearestNeighbors(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, const string& visCollTable, TeGeomRep /* visRep */, TeDatabasePortal* portal, int numRes)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";

	//select the spatial index
	string index = " SELECT INDEX_NAME FROM USER_SDO_INDEX_INFO";
	index += " WHERE TABLE_NAME = '"+ TeConvertToUpperCase(visGeomTable) +"'"; 

	portal->freeResult();
	if(!portal->query(index) || !portal->fetchRow())
		return false;

	string indexName = string(portal->getData(0));
	string perf = "/*+ INDEX("+ TeConvertToUpperCase(visGeomTable) +" "+ indexName +") */ ";
	string nres = " ROWNUM <= "+ numRes;
	
	string sql = "SELECT "+ perf +"  geomTable1.* ";
	sql += " FROM "+ visGeomTable +" geomTable1,";
	sql += actGeomTable + " geomTable2 ";
	
	if(!visCollTable.empty())
	{
		sql += ", "+ visCollTable +" collTable";
		sql += " WHERE ";
		sql += " geomTable1.object_id = collTable.c_object_id AND ";
	}
	else
		sql += " WHERE ";
		
	sql += " SDO_NN(geomTable1."+ actGeomColl +", geomTable2."+ actGeomColl;
	sql += ", 'sdo_batch_size=10') = 'TRUE' AND ";
	sql += " geomTable2.object_id = '" + objId1 + "' AND ";
	sql += " ROWNUM <= "+ Te2String(numRes);
	
	portal->freeResult();
	if(!((TeOracleSpatialPortal*)portal)->querySDO(sql))
		return false;
	
	return true;
}


//Intersection entre dois objetos geogrficos de uma mesma tabela


bool 
TeOracleSpatial::geomIntersection(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIds, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string Ids = getStringIds(actIds);

	string sql = "SELECT SDO_GEOM.SDO_INTERSECTION(g1."+ actGeomColl +", m.diminfo, ";
	sql += " g2."+ actGeomColl +", m.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ actGeomTable +" g2, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if((!portal->query(sql))||(!portal->fetchRow()))
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}


bool 
TeOracleSpatial::geomIntersection(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, TeGeomRep /* visRep */, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";
	
	string sql = "SELECT SDO_GEOM.SDO_INTERSECTION(g1."+ actGeomColl +", m1.diminfo, ";
	sql += " g2."+ visGeomColl +", m2.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ visGeomTable +" g2, ";
	sql += " USER_SDO_GEOM_METADATA m1, USER_SDO_GEOM_METADATA m2 ";
	sql += " WHERE m1.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m1.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND m2.table_name = '"+ TeConvertToUpperCase(visGeomTable) +"'";
	sql += " AND m2.column_name = '"+ TeConvertToUpperCase(visGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;        

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}


bool 
TeOracleSpatial::geomDifference(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, TeGeomRep /* visRep */, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";

	string sql = "SELECT SDO_GEOM.SDO_DIFFERENCE(g1."+ actGeomColl +", m1.diminfo, ";
	sql += " g2."+ visGeomColl +", m2.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ visGeomTable +" g2, ";
	sql += " USER_SDO_GEOM_METADATA m1, USER_SDO_GEOM_METADATA m2 ";
	sql += " WHERE m1.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m1.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND m2.table_name = '"+ TeConvertToUpperCase(visGeomTable) +"'";
	sql += " AND m2.column_name = '"+ TeConvertToUpperCase(visGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}

bool 
TeOracleSpatial::geomDifference(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";

	string sql = "SELECT SDO_GEOM.SDO_DIFFERENCE(g1."+ actGeomColl +", m.diminfo, ";
	sql += " g2."+ actGeomColl +", m.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ actGeomTable +" g2, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}

bool 
TeOracleSpatial::geomXOr(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";

	string sql = "SELECT SDO_GEOM.SDO_XOR(g1."+ actGeomColl +", m.diminfo, ";
	sql += " g2."+ actGeomColl +", m.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ actGeomTable +" g2, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);

	delete portal;
	return true;
}


bool 
TeOracleSpatial::geomXOr(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, TeGeomRep /* visRep */, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";

	string sql = "SELECT SDO_GEOM.SDO_XOR(g1."+ actGeomColl +", m1.diminfo, ";
	sql += " g2."+ visGeomColl +", m2.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ visGeomTable +" g2, ";
	sql += " USER_SDO_GEOM_METADATA m1, USER_SDO_GEOM_METADATA m2 ";
	sql += " WHERE m1.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m1.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND m2.table_name = '"+ TeConvertToUpperCase(visGeomTable) +"'";
	sql += " AND m2.column_name = '"+ TeConvertToUpperCase(visGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}


bool 
TeOracleSpatial::geomUnion(const string& actGeomTable, TeGeomRep /* actRep */, Keys& actIds, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string Ids = getStringIds(actIds);

	string sql = "SELECT SDO_GEOM.SDO_UNION(g1."+ actGeomColl +", m.diminfo, ";
	sql += " g2."+ actGeomColl +", m.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ actGeomTable +" g2, USER_SDO_GEOM_METADATA m";
	sql += " WHERE m.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND object_id IN ("+ Ids +")";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
			
	delete portal;
	return true;
}


bool 
TeOracleSpatial::geomUnion(const string& actGeomTable, TeGeomRep /* actRep */, const string& objId1, const string& visGeomTable, TeGeomRep /* visRep */, const string& objId2, TeGeometryVect& geomVect)
{
	string actGeomColl = "spatial_data";
	string visGeomColl = "spatial_data";

	string sql = "SELECT SDO_GEOM.SDO_UNION(g1."+ actGeomColl +", m1.diminfo, ";
	sql += " g2."+ visGeomColl +", m2.diminfo)";
	sql += " FROM "+ actGeomTable +" g1,"+ visGeomTable +" g2, ";
	sql += " USER_SDO_GEOM_METADATA m1, USER_SDO_GEOM_METADATA m2 ";
	sql += " WHERE m1.table_name = '"+ TeConvertToUpperCase(actGeomTable) +"'";
	sql += " AND m1.column_name = '"+ TeConvertToUpperCase(actGeomColl) +"'"; 
	sql += " AND m2.table_name = '"+ TeConvertToUpperCase(visGeomTable) +"'";
	sql += " AND m2.column_name = '"+ TeConvertToUpperCase(visGeomColl) +"'"; 
	sql += " AND g1.object_id = '"+ objId1 +"'"; 
	sql += " AND g2.object_id = '"+ objId2 +"'";

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*)getPortal();
	
	if(!portal->query(sql) || !portal->fetchRow())
		return false;

	bool flag = true;
	do 
	{
		bool result=false;
		TeGeometry* geom = new TeGeometry(); 
		TeGeometry* geom2 = geom;
		flag = portal->getGeometry(&geom, result);
		if(result)
			geomVect.push_back (geom);
		delete geom2;
	}while(flag);
		
	delete portal;
	return true;
}


//End Spatial Query


bool
TeOracleSpatialPortal::fetchGeometry (TePolygon& poly)
{
	int			elem,elemnext,i,k,elemType, sdoInterp;

	try
	{
		int ndim = GetDimArraySize();

		if(ndim==0)
			return false;

		vector<TeCoord2D> coordinates;
		if(!cursor->GetCoordinates (coordinates))
			return false;

		//number of the oordinates in the SDO_GEOMETRY
		int noords = NumberOfOrdinates();
	
		//Indicates the type of the SDO_GEOMETRY
		int sdoEType;  
		GetDimElement(2,sdoEType);
	
		int geomId = atoi(getData("geom_id"));
		string objId = string(getData("object_id"));
		
		// 1005: compound polygon: Compound polygon with some vertices connected by straight
		//line segments and some by circular arcs.
		if((sdoEType==1005) || (sdoEType==2005))  
		{					   
			TeLine2D	lin;   
			//nelem: number of elemnts
			int nelem; 
			GetDimElement(3,nelem);  
			
			//posinit is initial position in the vector 
			int posinit=0; //1;
			//ipoxmax: second element position in the ordinates array
			int iposmax=7;			
			
			int posmax;
			bool thisElemArc = false; 

			//keep the last point of the element
			//TePoint lastPoint;
			
			//for each element
			for(elem=1; elem<=nelem; elem++)  
			{
				//to catch the last position of the element: iposmax
				if(elem==nelem)
					posmax = noords+1;
				else
					GetDimElement(iposmax, posmax);

				//verify if the element is a arc 
				GetDimElement((iposmax-1), elemType);
				if(elemType == 2)
					thisElemArc = true;   
				else 
					thisElemArc = false;

				//ptSet: ordinates of the element
				TePointSet ptSet;

				//to catch all coords of the element
				int pos=0;
									
				for(pos=posinit; pos<((posmax-1)/2);++pos)
				{
					TePoint pt(coordinates[pos]);
					ptSet.add(pt);
				}

				if(thisElemArc) 
				{
					//pegar o prximo
					if(elem!=nelem)
					{
						TePoint pt(coordinates[pos]);
						ptSet.add(pt);
					}

					int size = ptSet.size();
					for(int s=1; (s+1)<size; s=s+2)
					{
						TeLine2D arc;
						TeGenerateArc (ptSet[s-1], ptSet[s], ptSet[s+1], arc, 10);

						int sz = arc.size();
						for(int i=0; i<sz; i++)
							lin.add(arc[i]);
					}
				}

				else //line segment
				{
					int size = ptSet.size();
					for(int s=0; s<size; s++)
						lin.add(ptSet[s].location());
				}

				iposmax+=3;
				posinit=pos;  //skip first coordinate: equal last point of previous element 
			
			} //for each element

			TeLinearRing rg(lin);
			rg.objectId(objId);
			rg.geomId(geomId);
			poly.add(rg);
		}

		else if((sdoEType==1003)||(sdoEType==2003))  //no complex
		{
			for(i=1;i<=ndim;i+=3)
			{
				TeLine2D	line;
				GetDimElement(i,elem);	// oordinate position
				if((i+3) > ndim)
					elemnext = noords+1;
				else
					GetDimElement(i+3,elemnext);

				GetDimElement(i+2,sdoInterp);	// sdo interpretation 
				// sdoInterp = 3: Rectangle type
				// sdoInterp = 2: Polygon made up of a connected sequence of circular arcs 

				if(sdoInterp==2)
				{
					//para gerar o arco
					TePoint pt1;
					TePoint pt2;
					TePoint pt3;
										
					int cont=0;
					int k = elem/2;
					while (k<(elemnext/2))
					{
						for(int i=0; i<3; i++)
						{
							TeCoord2D pt = coordinates[k];
						
							if(i==0)
							{
								if(!cont)
									pt1.add(pt);
								else
								{
									pt1.add(pt3.location());
									pt2.add(pt);
									++i;
								}
							}
							else if (i==1)
								pt2.add(pt);
							else if (i==2)
								pt3.add(pt);

							++cont;
							++k;
						}

						TeLine2D arc;
						TeGenerateArc (pt1, pt2, pt3, arc, 20);
						
						int s = arc.size();
						for(int j=0; j<s; j++)
							line.add(arc[j]);
					}
				}
				else
				{
					// ler os pontos
					for(k=(elem/2);k<(elemnext/2);k++)
					{
						TeCoord2D pt = coordinates[k];
						line.add(pt);
					}

					if(sdoInterp==3) // rectangle
					{
						double xmin,ymin, xmax, ymax;
						xmin = line.box().x1();
						ymin = line.box().y1();
						xmax = line.box().x2();
						ymax = line.box().y2();

						line.clear();
						TeCoord2D pt1(xmin,ymin);
						line.add(pt1);
						TeCoord2D pt2(xmin,ymax);
						line.add(pt2);
						TeCoord2D pt3(xmax,ymax);
						line.add(pt3);
						TeCoord2D pt4(xmax,ymin);
						line.add(pt4);
						line.add(pt1);
					}
				}

				TeLinearRing ring(line);
				ring.objectId (objId);
				ring.geomId(geomId);
				poly.add(ring);
			}
		}
		poly.objectId(objId);
		poly.geomId(geomId);
		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}


bool
TeOracleSpatialPortal::getGeometry (TeGeometry** geom, bool& result)
{
	TeSDOGType gType;
	bool flag=true;	
	
	try
	{
		this->GetGeometryType(gType);
			
		if(gType==TeSDOPOLYGON)
		{
			TePolygon   poly, *pol;
			flag = this->fetchGeometry(poly); 
			pol = new TePolygon();
			*pol = poly;
			*geom = pol;
			result = true;
			return flag;
		}
		else if(gType==TeSDOLINE)
		{
			TeLine2D	line, *lin;
			flag = this->fetchGeometry(line); 
			lin = new TeLine2D();
			*lin = line;
			*geom = lin;
			result = true;
			return flag;
		}
		else if(gType==TeSDOPOINT)
		{
			TePoint		point, *pnt;
			flag = this->fetchGeometry(point); 
			pnt = new TePoint();
			*pnt = point;
			*geom = pnt;
			result = true;
			return flag;
		}

		else if(gType==TeSDOMULTIPOLYGON)
		{
			TePolygonSet   polySet, *polSet;
			flag = this->fetchGeometry(polySet); 
			polSet = new TePolygonSet();
			*polSet = polySet;
			*geom = polSet;
			result = true;
			return flag;
		}
		else if(gType==TeSDOMULTILINE)
		{
			TeLineSet	lineSet, *linSet;
			flag = this->fetchGeometry(lineSet); 
			linSet = new TeLineSet();
			*linSet = lineSet;
			*geom = linSet;
			result = true;
			return flag;
		}
		else if(gType==TeSDOMULTIPOINT)
		{
			TePointSet		pointSet, *pntSet;
			flag = this->fetchGeometry(pointSet); 
			pntSet = new TePointSet();
			*pntSet = pointSet;
			*geom = pntSet;
			result = true;
			return flag;
		}
		
	}
	catch(...)
	{
		result = false;
		return false;
	}
		
	result = false;
	return (this->fetchRow());
}


bool
TeOracleSpatialPortal::fetchGeometry (TePolygonSet& polySet)
{
	int					elem,elemnext,i,k,sdoInterp;
	vector<TeCoord2D>	coordinates;
	
	try
	{
		int ndim = GetDimArraySize();
		if(ndim==0)
			return false;
		
		TePolygonSet polyHoles;
		TeSDOGType gType;
		
		GetGeometryType(gType);

		if(gType==TeSDOPOLYGON)  
		{
			TePolygon poly;
			bool res = fetchGeometry(poly);
			polySet.add(poly);
			return res;
		}

		if(!cursor->GetCoordinates(coordinates))
			return false;

		int noords = NumberOfOrdinates();
	
		int geomId = atoi(getData("geom_id"));
		string objId = string(getData("object_id"));
		
		bool hasHole = false;

		for(i=1;i<=ndim;i+=3)
		{
			int			eType;
			TeLine2D	line;
			TePolygon	poly;
			GetDimElement(i,elem);	// oordinate position
			if((i+3) > ndim)
				elemnext = noords+1;
			else
				GetDimElement(i+3,elemnext);

			GetDimElement(i+1, eType);		// sdo_etype do proximo elemento
			GetDimElement(i+2,sdoInterp);	// sdo_interpretation
			
			if(eType==2003)
				poly = polyHoles.last();

			// ler os pontos
			for(k=(elem/2);k<(elemnext/2);k++)
			{
				TeCoord2D pt(coordinates[k]);
				line.add(pt);
			}

			if(sdoInterp == 3) // rectangle
			{
				
				double xmin,ymin, xmax, ymax;
				xmin = line.box().x1();
				ymin = line.box().y1();
				xmax = line.box().x2();
				ymax = line.box().y2();

				line.clear();
				TeCoord2D pt1(xmin,ymin);
				line.add(pt1);
				TeCoord2D pt2(xmin,ymax);
				line.add(pt2);
				TeCoord2D pt3(xmax,ymax);
				line.add(pt3);
				TeCoord2D pt4(xmax,ymin);
				line.add(pt4);
				line.add(pt1);
			}
			TeLinearRing ring(line);
			ring.objectId (objId);
			ring.geomId(geomId);
			
			poly.add(ring);
	
			//verificar se o proximo  hole
			hasHole=false;
			if(i+4<ndim)
			{
				GetDimElement(i+4, eType);		// sdo_etype do proximo elemento
				if(eType == 2003)
					hasHole = true;
			}
			poly.objectId(objId);
			poly.geomId(geomId);
			
			if(!hasHole)
				polySet.add(poly);
			else	
				polyHoles.add(poly);
		} //for

		polySet.objectId(objId);
		polySet.geomId(geomId);

		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}


bool
TeOracleSpatial::insertLine (const string& table, TeLine2D &line)
{
	
	string	elinfo = "1, 2, 1";
	
	if(!AllocateOrdinatesObject(line))
		return false;

	string ins = "INSERT INTO " + table + " ( ";
	ins += " geom_id, object_id, spatial_data) VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(line.objectId()) + "'";
	ins += ", MDSYS.SDO_GEOMETRY(2002, NULL, NULL";
	ins += ", MDSYS.SDO_ELEM_INFO_ARRAY( " + elinfo + " )";
	ins += ", :ordinates) ";
	ins += " )";

	//OCI
	if(!Conn->ExecuteSDOSTM(ins))
	{
		errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	/*
	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	line.geomId(index); 
	delete ocip;
	*/

	return true;
}


bool 
TeOracleSpatial::updateLine(const string& table, TeLine2D &line)
{
	string	elinfo = "1, 2, 1";
	
	if(!AllocateOrdinatesObject(line))
		return false;
	
	string sql;
	sql =  "UPDATE " + table + " SET ";
	sql += "  object_id= '" + line.objectId() + "'";
	sql += ", spatial_data = ";
	sql += " MDSYS.SDO_GEOMETRY(2002, NULL, NULL";
	sql += ", MDSYS.SDO_ELEM_INFO_ARRAY( " + elinfo + " )";
	sql += ", :ordinates) ";
	sql += " WHERE geom_id = " +  line.geomId ();

	//OCI
	if(!Conn->ExecuteSDOSTM(sql))
	{
		errorMessage_ = "Error updating in the table " + table + "!"; 
		return false;
	}
	return true;
}


bool 
TeOracleSpatial::loadLineSet (const string& table, const string& geoid, TeLineSet &ls)
{
	
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string q ="SELECT * FROM " + table;
	if (geoid != "")
		q += " WHERE object_id = '" + geoid +"'";
	
	if (!ocip->query(q) || !ocip->fetchRow())
	{	
		delete ocip;
		return false;
	}

	bool flag = true;
	do 
	{
		TeLine2D line;
		flag = ocip->fetchGeometry(line);
		ls.add ( line );
	}while(flag);

	delete ocip;
	return true;
}

bool 
TeOracleSpatial::loadLineSet (const string& table, TeBox &bb, TeLineSet &linSet)
{
	TeOracleSpatialPortal *portal = (TeOracleSpatialPortal*)getPortal();
	if (!portal)
		return false;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (bb, TeLINES);
	q += " ORDER BY object_id";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	bool flag = true;
	do
	{
		TeLine2D lin;
		flag = portal->fetchGeometry(lin);
		linSet.add(lin);
	}
	while (flag);
	delete portal;
	return true;
}

TeDatabasePortal* 
TeOracleSpatial::loadLineSet (const string& table, TeBox &box)
{
	TeOracleSpatialPortal *portal = (TeOracleSpatialPortal*) getPortal();
	if (!portal)
		return 0;

	string q;
	q = "SELECT * FROM " + table + " WHERE ";
	q += this->getSQLBoxWhere (box, TeLINES);
	q += " ORDER BY object_id";

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return 0;
	}
	return portal;
}

bool 
TeOracleSpatialPortal::fetchGeometry (TeLine2D& line)
{
	int					i;
	vector<TeCoord2D>	coordinates;

	try
	{
		if(!cursor->GetCoordinates(coordinates))
			return false;
		
		int noords = NumberOfOrdinates();
		if(noords==0)
			return false;

		for(i=0;i<noords/2;i++)
			line.add(coordinates[i]);
			
		line.geomId (atoi(getData("geom_id")));
		line.objectId(string(getData("object_id")));
		return (this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}


bool 
TeOracleSpatialPortal::fetchGeometry (TeLineSet& lineSet)
{
	vector<TeCoord2D>	coordinates;
	int					elem,elemnext,i,k;

	try
	{
		int ndim = GetDimArraySize();
		if(ndim==0)
			return false;
		
		int noords = NumberOfOrdinates();
		int geomId = atoi(getData("geom_id"));
		string objId = string(getData("object_id"));
		
		TeSDOGType gType;
		GetGeometryType(gType);

		if(gType==TeSDOLINE)  
		{
			TeLine2D line;
			fetchGeometry(line);
			lineSet.add(line);
			return true;
		}

		if(!cursor->GetCoordinates(coordinates))
			return false;

		for(i=1;i<=ndim;i+=3)
		{
			TeLine2D	line;
			GetDimElement(i,elem);		// oordinate position
			if((i+3) > ndim)
				elemnext = noords+1;
			else
				GetDimElement(i+3,elemnext);
			
			// ler os pontos
			for(k=(elem/2);k<(elemnext/2);k++)
			{
				TeCoord2D pt(coordinates[k]);
				line.add(pt);
			}
			line.objectId (objId);
			line.geomId(geomId);
			lineSet.add(line);
		}

		lineSet.objectId(objId);
		lineSet.geomId(geomId);
		
		return (this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}


bool 
TeOracleSpatial::insertPoint(const string& table, TePoint &point)
{
	
	string ins = "INSERT INTO " + table + " ( ";
	ins += " geom_id, object_id, spatial_data) ";
	ins += " VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(point.objectId()) + "'";
	ins += ", MDSYS.SDO_GEOMETRY(2001, NULL, ";
	ins += "MDSYS.SDO_POINT_TYPE( ";
	ins += Te2String(point.location().x());
	ins += ", " + Te2String(point.location().y());
	ins += ", NULL )";
	ins += ", NULL, NULL)";
	ins += " ) ";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	point.geomId(index); 
	delete ocip;

	return true;
}


bool 
TeOracleSpatial::updatePoint (const string& table, TePoint &point)
{
	string sql;
	sql =  "UPDATE " + table + " SET ";
	sql += "object_id = '" + point.objectId() + "'";
	sql += ", spatial_data = ";
	sql += " MDSYS.SDO_GEOMETRY(2001, NULL";
	sql += ", MDSYS.SDO_POINT_TYPE( ";
	sql += Te2String(point.location ().x());
	sql += ", " + Te2String(point.location ().y());
	sql += ", NULL )";
	sql += ", NULL, NULL)";
	sql += " WHERE geom_id = " + Te2String(point.geomId());

	if(!execute(sql))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error updating in the table " + table + "!"; 
		return false;
	}

	return true;
}
	

bool 
TeOracleSpatialPortal::fetchGeometry(TePoint& p)
{
	double		x,y;
	
	try
	{
		if(!GetPointXYZ(x,y))
			return false;		//point in SDO_POINT

		TeCoord2D c(x,y);
		p.add(c);
		p.objectId(string(getData("object_id")));
		p.geomId(atoi(getData("geom_id")));
		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}



bool 
TeOracleSpatial::insertText	(const string& table, TeText &text)
{
	
	string ins = "INSERT INTO " + table + " (geom_id, ";
	ins += " object_id, x, y, text_value, angle, height, alignment_vert, ";
	ins += " alignment_horiz) VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(text.objectId()) + "'";
	ins += ",  " + Te2String(text.location().x(),10);
	ins += ",  " + Te2String(text.location().y(),10);
	ins += ", '" + escapeSequence(text.textValue()) + "'";
	ins += ",  " + Te2String(text.angle(),10);
	ins += ",  " + Te2String(text.height(),10);
	ins += ",  " + Te2String(text.alignmentVert(),10);
	ins += ",  " + Te2String(text.alignmentHoriz(),10);
	ins += " )";
	
	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	text.geomId(index); 
	delete ocip;
	return true;
}


bool 
TeOracleSpatial::insertArc (const string& table, TeArc &arc)
{
	
	string ins = "INSERT INTO " + table + " (geom_id, ";
	ins += " object_id, from_node, to_node ) ";
	ins += " VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(arc.objectId()) + "'";
	ins += ",  " + Te2String(arc.fromNode().geomId());
	ins += ",  " + Te2String(arc.toNode().geomId());
	ins += " )";
	
	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	arc.geomId(index); 
	delete ocip;

	return true;
}

bool 
TeOracleSpatialPortal::fetchGeometry(TePointSet& pointSet)
{
	double		x,y;
	vector<TeCoord2D> coordinates;

	x=y=-99999.;
	try
	{
		int noords = NumberOfOrdinates();
		if(noords==0)
			return false;

		TeSDOGType gType;
		GetGeometryType(gType);

		if(gType==TeSDOPOINT)  
		{
			TePoint point;
			fetchGeometry(point);
			pointSet.add(point);
			return true;
		}

		if(!cursor->GetCoordinates(coordinates))
			return false;

		int geomId = atoi(getData("geom_id"));
		string objId = string(getData("object_id"));
		
		for(int i=1;i<=noords/2;i++)
		{
			TePoint pt(coordinates[i-1]);
			pt.geomId (geomId);
			pt.objectId (objId);
			pointSet.add(pt);
		}
		
		pointSet.objectId(objId);
		pointSet.geomId(geomId);
		
		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}
	

bool 
TeOracleSpatial::insertNode (const string& table, TeNode &node)
{	
	string ins = "INSERT INTO " + table + " ( ";
	ins += " geom_id, object_id, spatial_data) ";
	ins += " VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(node.objectId()) + "'";
	ins += ", MDSYS.SDO_GEOMETRY(2001, NULL, ";
	ins += "MDSYS.SDO_POINT_TYPE( ";
	ins += Te2String(node.location().x());
	ins += ", " + Te2String(node.location ().y());
	ins += ", NULL )";
	ins += ", NULL, NULL)";
	ins += " ) ";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT "+ getNameSequence(table) + ".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	node.geomId(index); 
	delete ocip;

	return true;
}
	

bool
TeOracleSpatial::updateNode (const string& table, TeNode &node)
{	
	string sql;
	sql =  "UPDATE " + table + " SET ";
	sql += " object_id = '" + node.objectId() + "'";
	sql += ", spatial_data = ";
	sql += " MDSYS.SDO_GEOMETRY(2001, NULL";
	sql += ", MDSYS.SDO_POINT_TYPE( ";
	sql += Te2String(node.location ().x());
	sql += ", " + Te2String(node.location ().y());
	sql += ", NULL )";
	sql += ", NULL, NULL)";
	sql += " WHERE geom_id = " + Te2String(node.geomId());
	
	if(!execute(sql))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error updating in the table " + table + "!"; 
		return false;
	}
	return true;
}

bool 
TeOracleSpatialPortal::fetchGeometry(TeNode& n)
{
	double	x,y;
	x=y=-99999.;
	try
	{
		if(!GetPointXYZ(x,y))
			return false;

		TeCoord2D point(x,y);
		n.add(point);
		n.geomId(atoi(getData("geom_id")));
		n.objectId(string(getData("object_id")));
		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}



bool 
TeOracleSpatial::insertCell (const string& table, TeCell &cell )
{
	
	TeBox b = cell.box();

	string ins = "INSERT INTO " + table + " ( ";
	ins += " geom_id, object_id, col_number, row_number, spatial_data) ";
	ins += " VALUES ( ";
	ins += getNameSequence(table) +".NEXTVAL";
	ins += ", '" + escapeSequence(cell.objectId ()) + "'";
	ins += ",  " + Te2String(cell.column ());
	ins += ",  " + Te2String(cell.line ());
	ins += ", MDSYS.SDO_GEOMETRY(2003, NULL, NULL";
	ins += ", MDSYS.SDO_ELEM_INFO_ARRAY( 1, 1003, 3 )";
	ins += ", MDSYS.SDO_ORDINATE_ARRAY( " ;
	ins += Te2String(b.lowerLeft().x());
	ins += ", " + Te2String(b.lowerLeft().y());
	ins += ", " + Te2String(b.upperRight().x());
	ins += ", " + Te2String(b.upperRight().y());
	ins += ")) ";
	ins += " )";
		
	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table " + table + "!"; 
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT "+ getNameSequence(table) +".CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence " + table + "_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	cell.geomId(index); 
	delete ocip;
	
	return true;
}


bool 
TeOracleSpatial::updateCell(const string& table, TeCell &cell)
{
	TeBox b = cell.box ();
	
	string sql;
	sql =  "UPDATE " + table + " SET ";
	sql += " object_id= '" + cell.objectId() + "'";
	sql += " col_number= " + Te2String(cell.column ());
	sql += " row_number= " + Te2String(cell.line ());
	sql += " spatial_data= ";
	sql += " MDSYS.SDO_GEOMETRY(2003, NULL, NULL";
	sql += ", MDSYS.SDO_ELEM_INFO_ARRAY( 1, 1003, 3 )";
	sql += ", MDSYS.SDO_ORDINATE_ARRAY( " ;
	sql += Te2String(b.lowerLeft().x());
	sql += ", " + Te2String(b.lowerLeft().y());
	sql += ", " + Te2String(b.upperRight().x());
	sql += ", " + Te2String(b.upperRight().y());
	sql += ")) ";
	sql += " WHERE geom_id = " +  cell.geomId ();
	
	if(!execute(sql))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error updating in the table " + table + "!"; 
		return false;
	}
	
	return true;
}


char* TeOracleSpatialPortal ::getData (int i) 
{

	char* result;
	if (i > numFields_ || i < 0)
		return "";
	
	try
	{
		result = cursor->GetFieldValue(i+1);
	}
	catch(...)
	{
		errorMessage_ = "Error!";
		return "";
	}
	return result;
	
}

char* TeOracleSpatialPortal ::getData (const string &s)
{
	string	attstr;
	char* result;

	string fieldName;
	size_t pos = s.find(".", string::npos,1);
	if (pos != string::npos)
		fieldName = s.substr(pos+1);
	else
		fieldName = s;

	attstr = strupr((char*)(fieldName.c_str()));
	int index = getColumnIndex (attstr);
	if(index == -1)
		return "";
	else
		result = getData(index);

	return result;
}


int TeOracleSpatialPortal ::getInt (int i) 
{
	double field;
	char* fieldChar;
	
	fieldChar = cursor->GetFieldValue(i+1);
	field = atoi((const char*)fieldChar);

	return (int)field;
}

int  
TeOracleSpatialPortal::getInt (const string& s)
{
	string	attstr;

	attstr = strupr((char*)(s.c_str()));
	int index = getColumnIndex (attstr);
	if(index == -1)
		return (int)-1.;

	return (getInt(index));
}

double TeOracleSpatialPortal ::getDouble (int i) 
{
	double field;
	char* fieldChar;
	
	fieldChar = cursor->GetFieldValue(i+1);
	field = atof((const char*)fieldChar);

	return field;
}

double  
TeOracleSpatialPortal::getDouble (const string& s)
{
	string	attstr;

	attstr = strupr((char*)(s.c_str()));
	int index = getColumnIndex (attstr);
	if(index == -1)
		return -1.;

	return (getDouble(index));
}


bool
TeOracleSpatialPortal::getBool(int i)
{
	char* fieldChar=0;
	
	fieldChar = cursor->GetFieldValue(i+1);
	if(fieldChar == 0)
		return false;
	
	return true;
}


bool    
TeOracleSpatialPortal::getBool (const string& s)
{
	string	attstr;

	attstr = strupr((char*)(s.c_str()));
	int index = getColumnIndex (attstr);
	if(index == -1)
		return 0;

	return (getBool(index));
}

bool
TeOracleSpatial::insertMetadata(const string &table, const string &column, double tolx,double toly,TeBox &box,short /* srid */)
{	
	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string check = "SELECT * FROM USER_SDO_GEOM_METADATA ";
	check += " WHERE TABLE_NAME = '" + TeConvertToUpperCase(table);
	check += "' AND COLUMN_NAME = '" + TeConvertToUpperCase(column) + "'";
	if(!ocip->query(check))
	{
		delete ocip;
		return false;
	}

	if(ocip->fetchRow())
	{
		delete ocip;
		return false;
	}
			
	delete ocip;
	
	double xmin = box.x1();
	double ymin = box.y1();
	double xmax = box.x2();
	double ymax = box.y2();

	string inser = "INSERT INTO USER_SDO_GEOM_METADATA VALUES ( ";
	inser += "'" + TeConvertToUpperCase(table) + "' ," ;
	inser += "'" + TeConvertToUpperCase(column) + "' ," ; 
	inser += " MDSYS.SDO_DIM_ARRAY(";
	inser += " MDSYS.SDO_DIM_ELEMENT('X',";
	inser += Te2String(xmin) + "," + Te2String(xmax,10) + "," + Te2String(tolx,10) + "), ";
	inser += " MDSYS.SDO_DIM_ELEMENT('Y',";
	inser += Te2String(ymin) + "," + Te2String(ymax,10) + "," + Te2String(toly,10) + ")), ";
	inser += " NULL ) ";

	if(!(execute(inser.c_str()))) 
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table USER_SDO_GEOM_METADATA! "; 
		return false;
	}

	return true;
}

bool
TeOracleSpatial::DeleteMetadata(const string &table, const string &column)
{
	string del = "DELETE FROM USER_SDO_GEOM_METADATA ";
	del += " WHERE TABLE_NAME = '" + TeConvertToUpperCase(table);
	del += "' AND COLUMN_NAME = '" + TeConvertToUpperCase(column) + "'";
	if(!(execute(del.c_str ())))
		return false;
	return true;
}


bool
TeOracleSpatial::createSpatialIndex(const string &table, const string &column,TeSpatialIndexType type, short level, short tile)
{
	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	
	string name;
	if(table.size()>21)
		name = TeConvertToUpperCase(table.substr(0,20)) + "_IDX";
	else
		name = TeConvertToUpperCase(table) + "_IDX";

	
	string index = "SELECT * FROM USER_SDO_INDEX_INFO";
	index += " WHERE table_name = '" + TeConvertToUpperCase(table) + "'";
	if(!ocip->query(index))
	{
		delete ocip;
		return false;
	}
	
	if(ocip->fetchRow())
	{
		errorMessage_ = "Spatial Index table already exist!";
		delete ocip;
		return false;
	}
		
	delete ocip;

	index = " CREATE INDEX " + name;
	index += " ON " + table + "(" + column + ")";
	index += " INDEXTYPE IS MDSYS.SPATIAL_INDEX ";
	
	if(type == TeQUADTREE)
	{
		if(level==0)
			return false;

		index += " PARAMETERS ('";
		index += "SDO_LEVEL = " + Te2String(level);
		index += " SDO_NUMTILES = " + Te2String(tile) + "'";
	}

	if(!execute(index))
		return false;
	
	return true;
}

bool
TeOracleSpatial::RebuildSpatialIndex(const string &table)
{
	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string tabIndex;

	string ind = "SELECT index_name FROM USER_SDO_INDEX_INFO";
	ind += " WHERE table_name = '" + TeConvertToUpperCase(table) + "'";
	if (!ocip->query(ind))
	{
		delete ocip;
		return false;
	}
	
	if(!ocip->fetchRow())
	{
		delete ocip;
		return false;
	}

	tabIndex = string(ocip->getData(0));
	delete ocip;
		
	string reb = "ALTER INDEX ";
	reb += tabIndex + " REBUILD";
	if(!execute(reb))
		return false;

	return true;
}

bool
TeOracleSpatial::DeleteSpatialIndex(const string &table)
{
	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string tabIndex;

	string ind = "SELECT index_name FROM USER_SDO_INDEX_INFO";
	ind += " WHERE table_name = '"+ TeConvertToUpperCase(table) +"'";
	if (!ocip->query(ind))
	{
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		delete ocip;
		return false;
	}

	tabIndex = string(ocip->getData(0));
	ocip->freeResult();
	delete ocip;

	string drop = "DROP INDEX "+ tabIndex;
	if (!(execute(drop.c_str ())))
		return false;
	return true;
}

bool	
TeOracleSpatial::createSequence(const string &tableName)
{
	string nameSeq = getNameSequence(tableName);
	
	string seq = " CREATE SEQUENCE " + nameSeq;
	seq += " START WITH 1 INCREMENT BY 1 ORDER ";
	if (!execute(seq))
	{	
		if(errorMessage_.empty())
			errorMessage_ = "Error creating sequence to table " + tableName + " !";
		return false;
	}

	return true;
}

bool
TeOracleSpatial::createAutoIncrementTrigger(const string &tableName, const string &fieldName)
{
	string nameTri = getNameTrigger(tableName);
	string nameSeq = getNameSequence(tableName);

	string tri;
	tri = "CREATE TRIGGER " + nameTri; 
	tri += " BEFORE INSERT ON "+tableName; 
	tri += " for each row";
	tri += " begin";
	tri += " select "+nameSeq+".nextval";
	tri += " into :new."+fieldName;
	tri += " from dual;";
	tri += " end;";

	if(!execute(tri))
	{	
		if(errorMessage_.empty())
			errorMessage_ = "Error creating trigger to table " + tableName + " !";
		return false;
	}
	return true;
}

string
TeOracleSpatial::getNameSequence(const string &tableName)
{
	string name;
	if(tableName.size()>21)
		name = tableName.substr(0,20) + "_seq";
	else
		name = tableName + "_seq";

	return name;
}

string
TeOracleSpatial::getNameTrigger(const string &tableName)
{
	string name;
	if(tableName.size()>21)
		name = tableName.substr(0,20) + "_tri";
	else
		name = tableName + "_tri";

	return name;
}


int 
TeOracleSpatialPortal::GetDimArraySize()
{
	if(!cursor)
		return 0;

	return(cursor->GetDimArraySize());
}

bool 
TeOracleSpatialPortal::GetDimElement(int i,int &elem)
{
	if(!cursor)
		return false;

	return (cursor->GetDimElement(i,elem));
}

int 
TeOracleSpatialPortal::NumberOfOrdinates()
{
	if(!cursor)
		return 0;

	return(cursor->GetNumberOrdinates());
}

bool
TeOracleSpatialPortal::GetCoordinates(int i,TeCoord2D& coord)
{
	if(!cursor)
		return false;

	return (cursor->GetCoordinates((i*2)-1,coord));
}

bool
TeOracleSpatialPortal::GetGeometryType(TeSDOGType& gType)
{
	if(!cursor)
		return false;

	int type = cursor->GetGeometryType(); 
	
	switch(type)
	{ 
		case 2000:
			gType=TeSDOUNKNOWN;
			break;

		case 2001:
			gType=TeSDOPOINT;
			break;

		case 2002:
			gType=TeSDOLINE;
			break;

		case 2003:
			gType=TeSDOPOLYGON;
			break;

		case 2004:
			gType=TeSDOCOLLECTION;
			break;

		case 2005:
			gType=TeSDOMULTIPOINT;
			break;

		case 2006:
			gType=TeSDOMULTILINE;
			break;
		
		case 2007:
			gType=TeSDOMULTIPOLYGON;
			break;
		default:
			return false;
	};
	return true;
}

int
TeOracleSpatialPortal::GetSpatialReferenceId()
{
	if(!cursor)
		return -1;

	return (cursor->GetSpatialReferenceId());
}

bool
TeOracleSpatialPortal::GetPointXYZ (double& x,double& y)
{
	if(!cursor)
		return false;

	return (cursor->GetXYZcoord(x,y));
}


bool 
TeOracleSpatial::insertLayer(TeLayer* layer)
{	
	int index;
	TeProjection* proj = layer->projection();
	if (!proj || !insertProjection(proj))
	{
		errorMessage_ = "No  possvel inserir layer sem projeo";
		return false;
	}
	string ins = "INSERT INTO te_layer (layer_id, projection_id, name ";
	ins += ", lower_x, lower_y, upper_x, upper_y) ";
	ins += " VALUES ( ";
	ins += "te_layer_seq.NEXTVAL";
	ins += ", " + Te2String(proj->id());
	ins += ", '" + escapeSequence(layer->name()) + "'";
	ins += ", " + Te2String(layer->box().x1(),10);
	ins += ", " + Te2String(layer->box().y1(),10);
	ins += ", " + Te2String(layer->box().x2(),10);
	ins += ", " + Te2String(layer->box().y2(),10);
	ins += ")";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_layer!";   
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	ins = "SELECT te_layer_seq.CURRVAL FROM DUAL";
	if (!ocip->query(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_layer_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	index = atoi((const char*)ocip->getData(0));
	layer->id(index);
	delete ocip;
	layerMap_[layer->id()] = layer;
	return true;
}

bool 
TeOracleSpatial::deleteLayer (int layerId)
{
	//Delete attributes tables
	if(!deleteLayerTable(layerId))
		return false;

	TeOracleSpatialPortal* portal = (TeOracleSpatialPortal*) this->getPortal();
	if (!portal)
		return false;

	string geomTable;
	string sql = "SELECT projection_id FROM te_layer WHERE layer_id = ";
	sql += Te2String(layerId);

	if (!portal->query(sql))
	{	
		delete portal;
		return false;
	}

	if (!portal->fetchRow())
	{
		delete portal;
		return false;
	}
	string projId = portal->getData("projection_id");
	portal->freeResult();

	// Get all representations that are associated to this layer
	sql = "SELECT * FROM te_representation WHERE layer_id = "+ Te2String(layerId);
	if (!portal->query (sql))
	{
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{	// Delete the geometry tables
		geomTable = portal->getData("geom_table");
	
		// Delete lut table
		TeGeomRep rep = TeGeomRep(atoi(portal->getData("geom_type")));
		if (rep == TeRASTER)
		{
			TeOracleSpatialPortal* portal2 = (TeOracleSpatialPortal*)this->getPortal();
			sql = "SELECT lut_table, raster_table FROM " + geomTable;
			string tabName;
			if (!portal2->query (sql))
			{
				delete portal2;
				continue;
			}

			while (portal2->fetchRow())
			{
				// remove lut table
				tabName = portal2->getData(0);
				if (!tabName.empty() && this->tableExist(tabName))
				{
					sql = "DROP TABLE " + tabName;
					this->execute(sql);

					sql= "DROP SEQUENCE "+ getNameSequence(tabName);
					this->execute(sql); 
				}
				// remove raster table
				tabName = portal2->getData(1);
				if (!tabName.empty() && this->tableExist(tabName))
				{
					sql = "DROP TABLE " + tabName;
					this->execute(sql);

					DeleteMetadata(tabName, "block_box");
					DeleteSpatialIndex(tabName);
				}
			}
			delete portal2;
			// remove raster metadata table
			tabName = geomTable + "_metadata";
			if (!tabName.empty() && this->tableExist(tabName))
			{
				sql = "DROP TABLE " + tabName;
				this->execute(sql);
			}
		}
		if (this->tableExist(geomTable))
		{
			sql = "DROP TABLE " + geomTable;
			if (!this->execute(sql) )
			{
				delete portal;
				return false;
			}
			
			sql= "DROP SEQUENCE "+ getNameSequence(geomTable);
			this->execute(sql);
			DeleteMetadata(geomTable, "spatial_data");
			DeleteSpatialIndex(geomTable);
		}
	}

	portal->freeResult();
	if (existRelation("te_representation","fk_rep_layer_id") != TeRICascadeDeletion)
	{
		// Delete entries into representations table
		sql = "DELETE FROM te_representation WHERE layer_id = " +Te2String(layerId);
		if (!this->execute(sql) )
		{
			delete portal;
			return false;
		}
	}

	// delete layer themes
	sql = "SELECT theme_id FROM te_theme WHERE layer_id=" + Te2String(layerId);
	if (!portal->query (sql))
	{
		delete portal;
		return false;
	}
	
	int themeId;
	while (portal->fetchRow())
	{	
		themeId = atoi(portal->getData("theme_id"));
		this->deleteTheme(themeId);
	}
	
	sql = "DELETE FROM te_layer WHERE layer_id=" + Te2String(layerId);
	if (!this->execute(sql))
	{
		delete portal;
		return false;
	}

	// delete layer projection
	sql = "DELETE FROM te_projection WHERE projection_id = "+ projId;
	if (!this->execute(sql))
	{	
		delete portal;
		return false;
	}

	// remove all the items themes associated to the layer to be removed
	TeThemeMap::iterator it;
	for (it = themeMap_.begin(); it != themeMap_.end(); ++it)
	{
		TeTheme *theme = it->second;
		if (theme->layer()->id() == layerId)
		{
			themeMap_.erase(theme->id());
			delete theme;
		}
	}

	// Delete layer and its entry in the layer map
	TeLayer* layer = layerMap_[layerId];
	layerMap_.erase(layerId);
	delete layer;

	delete portal;
	return true;
}


bool
TeOracleSpatial::insertTheme (TeTheme *theme)
{
	double maxScale = theme->maxScale ();
	if(maxScale==TeMAXFLOAT)
		maxScale = 0.;

	string ins = "INSERT INTO te_theme (theme_id, layer_id, view_id, name, ";
	ins += " parent_id, priority, node_type, min_scale, max_scale, ";
	ins += " generate_attribute_where, generate_spatial_where, generate_temporal_where, ";
	ins += " collection_table, visible_rep, enable_visibility) VALUES (";
	ins += "te_theme_seq.NEXTVAL";
	ins += ", " + Te2String(theme->layerId());
	ins += ", " + Te2String(theme->view());
	ins += ", '" + escapeSequence(theme->name()) + "'";
	ins += ", " + Te2String(theme->parentId ());
	ins += ", " + Te2String(theme->priority());
	ins += ", " + Te2String(theme->type ());
	ins += ", " + Te2String (theme->minScale(),9);
	ins += ", " + Te2String (maxScale);
	ins += ", '" + escapeSequence(theme->attributeRest()) + "'";
	ins += ", '" + escapeSequence(theme->spatialRest()) + "'";
	ins += ", '" + escapeSequence(theme->temporalRest()) + "'";
	ins += ", '" + escapeSequence(theme->collectionTable()) + "'";
	ins += ", " + Te2String(theme->visibleRep());
	ins += ", " + Te2String(theme->visibility());
	ins += ")";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_theme!";   
		return false;
	}

	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT te_theme_seq.CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_theme_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	theme->id(index);
	delete ocip;
	string colName = theme->collectionTable(); 

	//insert collection name
	if(colName.empty())
	{
		colName = "te_collection_" + Te2String(index);
		theme->collectionTable(colName);	
		
		ins = "UPDATE te_theme SET ";
		ins += " collection_table = '" + escapeSequence(colName) + "'";
		ins += " WHERE theme_id = " + Te2String (index);
		
		if(!execute(ins))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error inserting in the table te_theme!";   
			return false;
		}
	}
	
	bool status;

	// insert grouping
	int numSlices = 0;
	if(theme->grouping())
	{
		if(!insertGrouping (theme->id(), theme->grouping()))
			return false;
		numSlices = theme->grouping()->groupNumSlices_;
	}

	// insert legend
	theme->outOfCollectionLegend().group(-1); 
	theme->outOfCollectionLegend().theme(theme->id()); 
	status = insertLegend (&(theme->outOfCollectionLegend())); 
	if (!status)
		return status;

	theme->withoutDataConnectionLegend().group(-2); 
	theme->withoutDataConnectionLegend().theme(theme->id()); 
	status = insertLegend (&(theme->withoutDataConnectionLegend())); 
	if (!status)
		return status;

	theme->defaultLegend().group(-3); 
	theme->defaultLegend().theme(theme->id()); 
	status = insertLegend (&(theme->defaultLegend())); 
	if (!status)
		return status;

	theme->pointingLegend().group(-4); 
	theme->pointingLegend().theme(theme->id()); 
	status = insertLegend (&(theme->pointingLegend())); 
	if (!status)
		return status;

	theme->queryLegend().group(-5); 
	theme->queryLegend().theme(theme->id()); 
	status = insertLegend (&(theme->queryLegend())); 
	if (!status)
		return status;

	theme->queryAndPointingLegend().group(-6); 
	theme->queryAndPointingLegend().theme(theme->id()); 
	status = insertLegend (&(theme->queryAndPointingLegend())); 
	if (!status)
		return status;

	for (int i = 0; i < numSlices; i++)
	{
		theme->legend()[i].group(i);
		theme->legend()[i].theme(theme->id());
		status = insertLegend (&(theme->legend()[i]));
		if (!status)
			return status;
	}
	if (!status)
		return status;

	themeMap_[theme->id()] = theme;
	return updateThemeTable(theme);
}


bool 
TeOracleSpatial::insertThemeTable (int themeId, int tableId, int relationId, int tableOrder)
{
	string ins = "INSERT INTO te_theme_table ";
	ins += " (theme_table_id, theme_id, table_id, relation_id, table_order)";
	ins += " VALUES ( ";
	ins += getNameSequence("te_theme_table") +".NEXTVAL ";
	ins += ", "+ Te2String(themeId);
	ins += ", "+ Te2String(tableId);
	
	if(relationId>0)
		ins += ", "+ Te2String(relationId);
	else
		ins += ", null ";

	ins += ", "+ Te2String(tableOrder);
	ins += ")";

	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_theme_table!";   
		return false;
	}

	return true;
}

bool 
TeOracleSpatial::insertThemeGroup(TeViewTree* tree)
{
	string ins = "INSERT INTO te_theme (theme_id, view_id, name, ";
	ins += " parent_id, priority, node_type ) VALUES (";
	ins += "te_theme_seq.NEXTVAL";
	ins += ", " + Te2String(tree->view());
	ins += ", '" + escapeSequence(tree->name()) + "'";
	ins += ", " + Te2String(tree->parentId ());
	ins += ", " + Te2String(tree->priority());
	ins += ", " + Te2String(1);
	ins += ")";

	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;
	
	try
	{
		if(!execute(ins))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error inserting in the table te_theme!";   
			return false;
		}

		string seq = "SELECT te_theme_seq.CURRVAL FROM DUAL";
		if(!ocip->query(seq))
		{
			if(errorMessage_.empty())
				errorMessage_ = "Error in the sequence te_theme_seq!";  
			delete ocip;
			return false;
		}

		if(!ocip->fetchRow())
		{
			errorMessage_ = "Sequence value not found!";
			delete ocip;
			return false;
		}

		int index = atoi((const char*)ocip->getData(0)); 
		tree->id(index);
		
	}
	catch(...)
	{
		errorMessage_ = "Error inserting in the table te_theme!";
		if(ocip)
			delete ocip;
		return false;
	}

	delete ocip;
	return true;
}

bool 
TeOracleSpatial::generateLabelPositions	(TeTheme *theme)
{
	
	string	geomTable, upd;
	string	collTable = theme->collectionTable();
	
	if((collTable.empty()) || (!tableExist(collTable)))
		return false;

	if( theme->layer()->hasGeometry(TeCELLS)    || 
		theme->layer()->hasGeometry(TePOLYGONS) ||
		theme->layer()->hasGeometry(TeLINES) )
	{
		geomTable = theme->layer()->tableName(TeCELLS);
		
		if(geomTable.empty())
		{
			geomTable = theme->layer()->tableName(TePOLYGONS);
			if(geomTable.empty())
				geomTable = theme->layer()->tableName(TeLINES);
		}
		
		upd= " UPDATE " + collTable + " SET ";
		upd += " label_x = (SELECT MAX(SDO_GEOM.SDO_MIN_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 1) + (SDO_GEOM.SDO_MAX_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 1) -  SDO_GEOM.SDO_MIN_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 1))/2) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id), ";
		
		upd += " label_y = (SELECT MAX(SDO_GEOM.SDO_MIN_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 2) + (SDO_GEOM.SDO_MAX_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 2) -  SDO_GEOM.SDO_MIN_MBR_ORDINATE( ";
		upd += geomTable + ".spatial_data, 2))/2) ";
		upd += "FROM " + geomTable + " WHERE object_id = c_object_id) ";

		upd += " WHERE label_x IS NULL OR label_y IS NULL";

	}
	
	else if (theme->layer()->hasGeometry(TePOINTS))
	{
		geomTable = theme->layer()->tableName(TePOINTS);
		
		upd= " UPDATE "+ collTable +" SET ";
		upd += " label_x = (SELECT MAX(p.spatial_data.SDO_POINT.X) ";
		upd += " FROM " + geomTable + " p WHERE object_id = c_object_id), ";
		
		upd += " label_y = (SELECT MAX(p.spatial_data.SDO_POINT.Y) ";
		upd += " FROM " + geomTable + " p WHERE object_id = c_object_id) ";
		upd += " WHERE label_x IS NULL OR label_y IS NULL";
	}

	if(!execute(upd))
		return false;

	return true;
}


bool 
TeOracleSpatial::insertView(TeView *view)
{
	// save its projection
	TeProjection* proj = view->projection();
	if ( !proj || !insertProjection(proj))
	{
		errorMessage_ = "No  possvel inserir vista sem projeo";
		return false;
	}

	string ins = "INSERT INTO te_view (view_id, projection_id, name, user_name, visibility)";
	ins += " VALUES (";
	ins += "te_view_seq.NEXTVAL";
	ins += ", " + Te2String(proj->id());
	ins += ", '" + escapeSequence(view->name ()) + "'";
	ins += ", '" + escapeSequence(view->user ()) + "'";
	ins += ", " + Te2String((int)view->isVisible());
	ins += " )";
	
	if(!execute (ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_view!";   
		return false;
	}

	TeOracleSpatialPortal *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	ins = "SELECT te_view_seq.CURRVAL FROM DUAL";
	if (!ocip->query(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_view_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0));
	view->id(index);
	delete ocip;

	int size = view->size();

	for (int th=0; th<size; th++)
	{
		TeViewNode* node = view->get(th);
		if (node->type() == TeTHEME)
		{
			TeTheme *theme = (TeTheme*) node;
			insertTheme (theme);
		}
		else
		{
			TeViewTree* tree = (TeViewTree*)node;
			insertViewTree (tree);
		}
	}

	// Insert view in the view map
	viewMap_[view->id()] = view;
	return true;
}



bool
TeOracleSpatial::insertViewTree (TeViewTree *tree)
{
	
	string ins = "INSERT INTO te_theme (theme_id, view_id, name, ";
	ins += " parent_id, node_type, priority) VALUES (";
	ins += " te_theme_seq.NEXTVAL";
	ins += ", " + Te2String(tree->view());
	ins += ", '" + escapeSequence(tree->name()) + "'";
	ins += ", " + Te2String(tree->parentId());
	ins += ", " + Te2String(tree->type());
	ins += ", " + Te2String(tree->priority());
	ins += ")";
	
	if(!execute(ins))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error inserting in the table te_theme!";
		return false;
	}

	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string seq = "SELECT te_theme_seq.CURRVAL FROM DUAL";
	if(!ocip->query(seq))
	{
		if(errorMessage_.empty())
			errorMessage_ = "Error in the sequence te_theme_seq!";  
		delete ocip;
		return false;
	}

	if(!ocip->fetchRow())
	{
		errorMessage_ = "Sequence value not found!";
		delete ocip;
		return false;
	}

	int index = atoi((const char*)ocip->getData(0)); 
	tree->id(index);
	delete ocip;
	
	return true;
}

bool 
TeOracleSpatial::locatePolygon (const string& table, TeCoord2D &pt, TePolygon & /* polygon */, const double& /* tol */)
{
	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string q = "SELECT * FROM " + table;
	q += " WHERE MDSYS.SDO_RELATE (spatial_data,";
	q += "MDSYS.SDO_GEOMETRY(2001,NULL, MDSYS.SDO_POINT_TYPE(";
	q += Te2String(pt.x(),10) + ", " + Te2String(pt.y(), 10);
	q += ", NULL), NULL, NULL), ";
	q += " 'mask=contains querytype = window') = 'TRUE'";

	if (!ocip->query(q) || !ocip->fetchRow())
	{	
		delete ocip;
		return false;
	}
	
//	bool flag = ocip->fetchGeometry(polygon);
	delete ocip;
	return true;
}

bool 
TeOracleSpatial::locateLine (const string& table, TeCoord2D &pt, TeLine2D & /* line */, const double& tol)
{

	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	
	string sdoGeom = "mdsys.sdo_geometry(2003,null,null,";
	sdoGeom += "mdsys.sdo_elem_info_array(1,1003,3),";
	sdoGeom += "mdsys.sdo_ordinate_array(";
	sdoGeom += Te2String(box.x1(),10) + ", " + Te2String(box.y1(),10);
	sdoGeom += ", " + Te2String(box.x2(),10) + ", " + Te2String(box.y2(),10) + "))";
		
	string q = "SELECT * FROM " + table;
	q += " WHERE MDSYS.SDO_RELATE (spatial_data,";
	q += sdoGeom +","; 
	q += "'mask=anyinteract querytype = window') = 'TRUE'";

	if (!ocip->query(q) || !ocip->fetchRow())
	{
		delete ocip;
		return false;
	}

//	bool flag = ocip->fetchGeometry(line);
	delete ocip;
	return true;
}

bool 
TeOracleSpatial::locatePoint (const string& table, TeCoord2D &pt, TePoint & /* point */, const double& tol)
{
	TeOracleSpatialPortal	*ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	
	TeBox box (pt.x()-tol,pt.y()-tol,pt.x()+tol,pt.y()+tol);
	
	string sdoGeom = "mdsys.sdo_geometry(2003,null,null,";
	sdoGeom += "mdsys.sdo_elem_info_array(1,1003,3),";
	sdoGeom += "mdsys.sdo_ordinate_array(";
	sdoGeom += Te2String(box.x1(),10) + ", " + Te2String(box.y1(),10);
	sdoGeom += ", " + Te2String(box.x2(),10) + ", " + Te2String(box.y2(),10) + "))";
		
	string q = "SELECT * FROM " + table;
	q += " WHERE MDSYS.SDO_RELATE (spatial_data,";
	q += sdoGeom +","; 
	q += "'mask=anyinteract querytype = window') = 'TRUE'";
	
	if (!ocip->query(q) || !ocip->fetchRow())
	{
		delete ocip;
		return false;
	}

//	bool flag = ocip->fetchGeometry(point);
	delete ocip;
	return true;
}

bool	
TeOracleSpatialPortal::fetchGeometry (TeCell& cell)
{
	TeCoord2D coord1,coord2;
	try
	{
		GetCoordinates (1, coord1);
		GetCoordinates (2, coord2);
		TeBox b(coord1.x(), coord1.y(), coord2.x(), coord2.y());
		
		cell.setBox (b);
		cell.geomId(atoi(getData("geom_id")));
		cell.objectId (string(getData("object_id")));
		cell.column(atoi(getData("col_number")));
		cell.line(atoi(getData("row_number")));
		return(this->fetchRow());
	}
	catch(...)
	{
		errorMessage_ = cursor->getErrorMessage();
		return false;
	}
}

bool 
TeOracleSpatial::locateCell(const string& table, TeCoord2D &pt, TeCell & /* cell */, const double& /* tol */)
{
	TeOracleSpatialPortal  *ocip = (TeOracleSpatialPortal*)getPortal();
	if(!ocip)
		return false;

	string q = "SELECT * FROM " + table;
	q += " WHERE MDSYS.SDO_RELATE (spatial_data,";
	q += "MDSYS.SDO_GEOMETRY(2001,NULL,MDSYS.SDO_POINT_TYPE(";
	q += Te2String(pt.x(),10) + ", " + Te2String(pt.y(), 10);
	q += ", NULL), NULL, NULL), ";
	q += " 'mask=contains querytype = window') = 'TRUE'";

	if (!ocip->query(q) || !ocip->fetchRow())
	{	
		delete ocip;
		return false;
	}
	
//	bool flag = ocip->fetchGeometry(cell);
	delete ocip;
	return true;
}


bool		
TeOracleSpatialPortal::getBlob(const string& /* s */, unsigned char* &data, long& size)
{
	try
	{
		size = getCursor()->SizeBlob();
		if(size>0)
		{
			data = new unsigned char[size];
			cursor->ReadBlob((unsigned char*)data, size);
		}
	}
	catch(...)
	{
		errorMessage_ = "Error getting media!";
		return false;
	}
	return true;
}


TeTime 
TeOracleSpatialPortal::getDate (int i) 
{ 
	TeTime temp;
	string result;

	if (i > numFields_ || i < 0)
		return temp;
	
	try
	{
		result = cursor->GetFieldValue(i+1);
	}
	catch(...)
	{
		errorMessage_ = "Error!";
		return temp;
	}
		
	TeTime t(result, TeSECOND, "DDsMMsYYYYsHHsmmsSS");
	return t;
}
	

TeTime 
TeOracleSpatialPortal::getDate (const string& s) 
{ 
	string	attstr;
	string  result;
	TeTime temp;

	string fieldName;
	size_t pos = s.find(".", string::npos,1);
	if (pos != string::npos)
		fieldName = s.substr(pos+1);
	else
		fieldName = s;

	attstr = strupr((char*)(fieldName.c_str()));
	int index = getColumnIndex (attstr);
	if(index == -1)
		return temp;
	else
		result = getData(index);

	TeTime t(result, TeSECOND, "DDsMMsYYYYsHHsmmsSS") ;
	return t;
}

string
TeOracleSpatialPortal::getDateAsString(int i)
{
	TeTime t = this->getDate(i);
	string date = t.getDateTime ();

	if (!date.empty())
	{		string tval = " TO_DATE ('"+ date +"','DDsMMsYYYYsHHsmmsSS') ";
		return tval;
	}
	else
		return "";
}

string 
TeOracleSpatialPortal::getDateAsString(const string& s)
{
	TeTime t = this->getDate(s);
	string date = t.getDateTime ();

	if (!date.empty())
	{		string tval = " TO_DATE ('"+ date +"','DDsMMsYYYYsHHsmmsSS') ";
		return tval;
	}
	else
		return "";
}

 
bool
TeOracleSpatialPortal::getRasterBlock(unsigned long& size, unsigned char* ptData)
{
	// gets the spatial data
	// expects that the data size will be a block of double maximum
	unsigned long	len;
	try
	{
		// get the actual length of the compressed data
		len = getCursor()->SizeBlob();
		size = len;
		if(len > 0)
			getCursor()->ReadBlob(ptData, len);
	}
	catch(...)
	{
		size = 0;
		errorMessage_ = "ERRO!";
		return false;
	}
	return true;
}

string TeOracleSpatial::concatValues(vector<string>& values, const string& unionString)
{
	string concat = "";
	
	for(unsigned int i = 0; i < values.size(); ++i)
	{
		if(i != 0)
		{
			concat += " || ";

			if(!unionString.empty())
			{
				concat += "'";
				concat += unionString;
				concat += "'";
				concat += " || ";
			}
		}

		concat += values[i];
	}

	return concat;
}

string TeOracleSpatial::toUpper(const string& value)
{
	string result  = "upper(";
	       result += value;
		   result += ")";

	return result;
}