/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeDatabaseUtils.h
    This contains some utilitary functions to create some tables useful for 
	TerraView like applications.
*/
#ifndef __TERRALIB_INTERNAL_DATABASEUTILS_H
#define __TERRALIB_INTERNAL_DATABASEUTILS_H

#include "TeThemeApplication.h"
#include <TeDatabase.h>
#include <string>
#include <map>

class	TeView;
struct	TeKernelParams;

 
/*!
	\fn bool createColorSchemeTable(TeDatabase* db, const string& tname="te_color_scheme")
	\brief Creates a standard TerraView table to store color ramps 
 	\param db pointer to an opened database connection
 	\note The table created will have the following scheme:
 
   \par  --------------------------------------------------
   \par | ramp_name  |  color_order  | red  | green | blue |
   \par ----------------------------------------------------
   \par | red        |      1        | 255  |  256  | 256  |
   \par | red        |      2        | 196  |  195  | 256  |
   \par | red        |      3        | 123  |  124  | 256  |
   \par | green      |      1        | 0    |  256  | 123  |
   \par | ...        |     ...       | ...  |  ...  | ...  |
   \par ---------------------------------------------------
*/
bool createColorSchemeTable(TeDatabase* db, const string& tname="te_color_scheme");

//! Populates a color scheme table with 30 ramp colors, each ramp with 13 shades
bool populateColorSchemeTable(TeDatabase* db, const string& tname="te_color_scheme");

//! Insert a ramp color entry in the database
bool insertRampColor(TeDatabase* bd, const string& name, int order, TeColor& color, const string& tableName = "te_color_scheme");

//! Insert some default colors a grid color table
bool updateGridColor(TeDatabase* bd, int themeId, TeObjectSelectionMode sel, TeColor& color);

//! Creates table to store information about the results of Kernel algorithm
bool createKernelMetadataTable(TeDatabase* db);

//! Inserts kernel parameters
bool insertKernelMetadata(TeDatabase* db, TeKernelParams& kernelPar);

//! Updates kernel parameters in a database -- inserts if it does not exist
bool updateKernelMetadata(TeDatabase* db, TeKernelParams& kernelPar);

//! Creates a table to store multimedia documents 
bool createMediaTable(TeDatabase* db, const string& tname);

//! Creates a table to store the association between objects of a layer and its multimedia attributes
bool createLayerMediaTable(TeDatabase* db, int layerId, const string& tname);

//! Inserts a new media file in a database
bool insertMedia(TeDatabase* db, const string& tableName, string fileName, const string& description, const string& type, bool blob=true);

//! Associates a media data to an object of a layer
bool insertMediaObjLayer (TeDatabase* db, TeLayer* layer, const string& objectId, const string& mediaName, const string& mediaTable); 

//! Sets a media as a default media of an object of a layer
bool setDefaultMedia(TeDatabase* db, TeLayer* layer, const string& objectId, const string& mediaName, const string& mediaTable);

//! Updates a media description in a database
bool updateMediaDescription(TeDatabase* db, const string& mediaName, const string& mediaTable, const string& desc);

//! Load the kernel parameters 
bool loadKernelParams (TeDatabase* db, int supThemeId, const string& supTableName, const string& supAttrName, TeKernelParams& KernelPar);

//! Load the kernel parameters 
bool loadKernelParams (TeDatabase* db, int LayerId, TeKernelParams& KernelPar);

//! Creates a table to store parameters of bar and pie charts 
bool createChartParamsTable(TeDatabase* db, bool integrity=false);	

//! Retrieves the location of pie/bar charts over the objects of a theme
bool locatePieBar (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt, double delta);

//! Load the charts parameters of a theme in the database
bool loadChartParams (TeDatabase* db, TeThemeApplication *theme);

//! Sets the location of pie/bar charts over the objects of a theme
bool updatePieBarLocation (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt); 

//! Sets the location of point of a layer
bool updatePointLocation (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt); 

//! Insert the chart parameters of a theme in the database
bool insertChartParams(TeDatabase* bd, TeThemeApplication *theme);

//! Delete the chart parameters of a theme in the database
bool deleteChartParams (TeDatabase* bd, TeThemeApplication *theme);

//! Loads a set of views that belongs to a user
bool loadViewSet(TeDatabase* db, const string& user);

//! Updates the parameters of a view
bool updateView(TeDatabase* db, TeView *view);

//! Creates a table to store other information about the theme 
bool createThemeApplicationTable(TeDatabase* db, bool integrity=false);

//! Populates the te_theme_application table with default parameters
bool populateThemeApplicationTable(TeDatabase* db);

//! Loads the parameters of a ThemeApplication
bool loadThemeApplication (TeDatabase* db, TeThemeApplication *theme, const string& tableName="te_theme_application");

//! Saves the parameters of a ThemeApplication
bool insertThemeApplication (TeDatabase* db, TeThemeApplication *theme, const string& tableName="te_theme_application");

//! Updates the parameters of a theme application
bool updateThemeApplication (TeDatabase* db, TeThemeApplication *theme);

//! Delete a theme application
bool deleteThemeApplication (TeDatabase* db, int themeId);

//! Recursive load view tree from a database
TeViewTree* loadViewTree(TeDatabase* db, TeView* view, int id);

//! Updates a tree view structure in a database
bool updateViewTree (TeDatabase* db, TeViewTree *tree);

//! Creates a table to store metadata about layers 
int createTeLayerMetadata(TeDatabase* db);

//! Creates a table to store metadata about attribute tables 
int createTeTableMetadata(TeDatabase* db);

//! Removes tables that visual of texts associated to a layer
bool deleteTextVisual(TeLayer* layer);

//! Returns true whether a database contains the TerraLib data model
bool isTerralibModel(TeDatabase* db);

//! insert a new object into collection
string insertObjectIntoCollection(TeTheme* theme, TeCoord2D p, string newId, string newTId = "");

//! delete pointed objects
string deletePointedObjects(TeTheme* theme, TeBox& box);

//! \fn bool TeCopyDatabase(TeDatabase* dbFrom, TeDatabase* dbTo)
/*
    \brief This function copies the contents of a TerraLib database to another TerraLib database
	\param dbFrom A pointer to the source database
	\param dbTo	  A pointer to the destination database
*/
bool TeCopyDatabase(TeDatabase* dbFrom, TeDatabase* dbTo);


#endif

